C Button press call back functions for GETWIN32
C =============================================
C
C The variable NPRESS is needed to provide for reverse
C communication between w_clearwin.dll and w_graphics.dll
C to identify the button that has been pressed.
C
C Call back actions: (numbers were logical in the original scheme)
C ==================
C NPRESS =  0: Quit
C NPRESS =  1: menu .... just sets NPRESS = 1
C NPRESS =  2: titles
C NPRESS =  3: legends
C NPRESS =  4: axes
C NPRESS =  5: style
C NPRESS =  6: data
C NPRESS =  7: panel
C NPRESS =  8: select text
C NPRESS =  9: select A/L/B
C NPRESS = 10: colours
C NPRESS = 11: transform
C NPRESS = 12: configure
C NPRESS = 13: select object
C NPRESS = 14: help
C NPRESS = 15: drag text
C NPRESS = 16: drag A/L/B^
C NPRESS = 17: drag A/L/B_
C NPRESS = 18: drag object
C NPRESS = 19: drag panel
C NPRESS = 20: menu or Quit
C NPRESS = 21: save ... just sets NPRESS = 21
C NPRESS = 22: PostScript ... just sets NPRESS = 22
C NPRESS = 23: Windows copy
C NPRESS = 24: Quit 3D graphics
C NPRESS = 25: SVG
C
C The idea is that the graphics menus can be called (e.g. from SIMPLOT)
C as normal menus or by button press. After button press, the call back
C function can be returned as 0 to close the window, or 2 to leave it open
C if a Cancel has been selected. The value of N_BUTTON_PRESSED is then
C communicated to the control subroutine (e.g. GKSDEC) which then returns
C to SIMPLOT (or similar) to control the next action. Also the graphics
C widow which has been opened (in GETWIN32) is closed (in GETWIN32) as
C dictated by the value of N_BUTTON_PRESSED. The individual routines
C could be made to indicate a Cancel by returning a value of NPRESS
C diferent from the value supplied, as the call backs are only closed
C when the graphics routine return NPRESS unchanged. This mechanism
C can fail, as in GRAF06$ which calls GKSEB4 directly and thereby alters
C N_BUTTON_PRESSED independently.
C
C 25/01/2014 added GRAPHICS_BUTTON_PRESS$ to set module parameters without a
C            button press. Note: The call-backs can now return -2 which allows
C            screen refresh by placing a new window on top of the old one before
C            the old one is deleted. This results when PROCEED_FLAG = 1 which
C            terminates the loop in the FINISH* routines and allows NPRESS to be
C            returned to the calling subroutine.
C 16/03/2014 major overhaul to correct screen refresh anomalies and make sure
C            QUERY_EXIT is set to .TRUE. where appropriate 
C 12/05/2014 added SVG format
C 10/12/2018 added call to SAVRES$ before call to W_SVGINI   
C 18/07/2019 added calls to W_SVGDIS, HELP_EDITSVG, W_SVGEDI, and SVGPAR  
C 02/08/2019 added the logical variable call_i_press_25 to module_clearwin to
C            allow k = i_press_25 from within finish2$ after i_press_25 has requested
C            to create a svg file or view the current svg       
C

      SUBROUTINE GRAPHICS_BUTTON_PRESS$(ISEND)
      USE MODULE_CLEARWIN, ONLY : PROCEED_FLAG
C
C This subroutine is provided so that when called with ISEND = K from other
C routines (such as STYGKS$ in W_GRAPHICS.DLL with ISEND = 0) it acts as if 
C button K has been pressed i.e. instead of using the call back I_PRESS_K
C      
      IMPLICIT NONE
      include <windows.ins>
      INTEGER, INTENT (IN) :: ISEND 
      INTEGER  N_BUTTON_PRESSED
      LOGICAL  SURFACE
      EXTERNAL CLOSE_GR_WINDOWS
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      IF (ISEND.EQ.0) THEN
         CALL TEMPORARY_YIELD@()
         N_BUTTON_PRESSED = 0
         PROCEED_FLAG = 1
         CALL TEMPORARY_YIELD@()
         CALL CLOSE_GR_WINDOWS
      ENDIF   
      END

C...0 Quit
      recursive integer function I_PRESS_0()
      USE MODULE_CLEARWIN, ONLY : PROCEED_FLAG
      IMPLICIT NONE
      include <windows.ins>
      INTEGER  N_BUTTON_PRESSED
      LOGICAL  SURFACE
      EXTERNAL CLOSE_GR_WINDOWS
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 0
      PROCEED_FLAG = 1
      I_PRESS_0 = 0
      CALL TEMPORARY_YIELD@()
      CALL CLOSE_GR_WINDOWS
      END
      
C...1 Edit from simple graphics interface
      recursive integer function I_PRESS_1()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT NONE
      include <windows.ins>
      INTEGER  N_BUTTON_PRESSED
      LOGICAL  SURFACE
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 1
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1
      I_PRESS_1 = 1
      CALL TEMPORARY_YIELD@()
      END
      
C...2 Titles
      recursive integer function I_PRESS_2()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   X_GRAF02
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 2
      NPRESS = N_BUTTON_PRESSED
      CALL X_GRAF02 (NPRESS)
      CALL TEMPORARY_YIELD@()
      IF (NPRESS.EQ.N_BUTTON_PRESSED) THEN
         I_PRESS_2 = 1
      ELSE
         I_PRESS_2 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1      
      CALL TEMPORARY_YIELD@()
      END
      
C...3 Legends
      recursive integer function I_PRESS_3()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   X_GRAF02
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 3
      NPRESS = N_BUTTON_PRESSED
      CALL X_GRAF02 (NPRESS)
      CALL TEMPORARY_YIELD@()
      IF (NPRESS.EQ.N_BUTTON_PRESSED) THEN
         I_PRESS_3 = 1
      ELSE
         I_PRESS_3 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1      
      CALL TEMPORARY_YIELD@()
      END
      
C...4 Axes
      recursive integer function I_PRESS_4()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   X_GRAF04
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 4
      NPRESS = N_BUTTON_PRESSED
      CALL X_GRAF04 (NPRESS)
      CALL TEMPORARY_YIELD@()
      IF (NPRESS.EQ.N_BUTTON_PRESSED) THEN
         I_PRESS_4 = 1
      ELSE
         I_PRESS_4 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1      
      CALL TEMPORARY_YIELD@()
      END
      
C...5 Style
      recursive integer function I_PRESS_5()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   X_GRAF05
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 5
      NPRESS = N_BUTTON_PRESSED
      CALL X_GRAF05 (NPRESS)
      CALL TEMPORARY_YIELD@()
      IF (NPRESS.EQ.N_BUTTON_PRESSED) THEN
         I_PRESS_5 = 1
      ELSE
         I_PRESS_5 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1      
      CALL TEMPORARY_YIELD@()
      END
      
C...6 Data
C     NOTE: GRAF06$ changes N_BUTTON_PRESSED if error bars are created
C     as GKSEB4 can be called to display the error bars
      recursive integer function I_PRESS_6()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   X_GRAF06
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      NPRESS = 6
      CALL X_GRAF06 (NPRESS)
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 6
      IF (NPRESS.EQ.6) THEN
         I_PRESS_6 = 1   
      ELSE
         I_PRESS_6 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1 
      CALL TEMPORARY_YIELD@()
      END
      
C...7 Panel
      recursive integer function I_PRESS_7()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   X_GRAF07
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 7
      NPRESS = N_BUTTON_PRESSED
      CALL X_GRAF07 (NPRESS)
      CALL TEMPORARY_YIELD@()
      IF (NPRESS.EQ.N_BUTTON_PRESSED) THEN
         I_PRESS_7 = 1
      ELSE
         I_PRESS_7 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1 
      CALL TEMPORARY_YIELD@()
      END
      
C...8 Select text
      recursive integer function I_PRESS_8()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            QUERY_EXIT, USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    ISEND
      PARAMETER (ISEND = 2)
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   OBJPLT$
      COMMON   / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 8
      NPRESS = N_BUTTON_PRESSED
      CALL OBJPLT$(ISEND, NPRESS)
      CALL TEMPORARY_YIELD@()
      IF (NPRESS.EQ.N_BUTTON_PRESSED) THEN
         I_PRESS_8 = 1
      ELSE
         I_PRESS_8 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1 
      QUERY_EXIT = .TRUE.
      CALL TEMPORARY_YIELD@()
      END
      
C...9 Select A/L/B
      recursive integer function I_PRESS_9()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            QUERY_EXIT, USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    ISEND
      PARAMETER (ISEND = 2)
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   OBJPLT$
      COMMON   / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 9
      NPRESS = N_BUTTON_PRESSED
      CALL OBJPLT$(ISEND, NPRESS)
      CALL TEMPORARY_YIELD@()
      IF (NPRESS.EQ.N_BUTTON_PRESSED) THEN
         I_PRESS_9 = 1
      ELSE
         I_PRESS_9 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1 
      QUERY_EXIT = .TRUE.
      CALL TEMPORARY_YIELD@()
      END
      
C...10 Colours
      recursive integer function I_PRESS_10()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   X_GRAF10
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 10
      NPRESS = N_BUTTON_PRESSED
      CALL X_GRAF10 (NPRESS)
      CALL TEMPORARY_YIELD@()
      IF (NPRESS.EQ.N_BUTTON_PRESSED) THEN
         I_PRESS_10 = 1
      ELSE
         I_PRESS_10 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1 
      CALL TEMPORARY_YIELD@()
      END
      
C...11 Transform
      recursive integer function I_PRESS_11()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   X_GRAF11
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 11
      NPRESS = N_BUTTON_PRESSED
      CALL X_GRAF11 (NPRESS)
      CALL TEMPORARY_YIELD@()
      IF (NPRESS.EQ.N_BUTTON_PRESSED) THEN
         I_PRESS_11 = 1
      ELSE
         I_PRESS_11 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1 
      CALL TEMPORARY_YIELD@()
      END
      
C...12 Configure
      recursive integer function I_PRESS_12()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT NONE
      INTEGER  N_BUTTON_PRESSED
      LOGICAL  SURFACE
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      N_BUTTON_PRESSED = 12
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1 
      I_PRESS_12 = 1
      END
      
C...13 Select an Object
      recursive integer function I_PRESS_13()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            QUERY_EXIT, USING_CTRL
      IMPLICIT   NONE
      INCLUDE   <WINDOWS.INS>
      INTEGER    ISEND
      PARAMETER (ISEND = 2)
      INTEGER    NPRESS
      INTEGER    N_BUTTON_PRESSED
      LOGICAL    SURFACE
      EXTERNAL   OBJPLT$
      COMMON   / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()
      N_BUTTON_PRESSED = 13
      NPRESS = N_BUTTON_PRESSED
      CALL OBJPLT$(ISEND, NPRESS)
      CALL TEMPORARY_YIELD@()
      IF (NPRESS.EQ.N_BUTTON_PRESSED) THEN
         I_PRESS_13 = 1
      ELSE
         I_PRESS_13 = 2
      ENDIF
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1 
      QUERY_EXIT = .TRUE.
      CALL TEMPORARY_YIELD@()
      END
      
C...14 Help
      recursive integer function I_PRESS_14()
      IMPLICIT NONE
      INCLUDE <WINDOWS.INS>
      INTEGER  N_BUTTON_PRESSED
      LOGICAL  SURFACE
      EXTERNAL HELP_SIMHLP
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      N_BUTTON_PRESSED = 14
      CALL HELP_SIMHLP ('simhlp')
      CALL TEMPORARY_YIELD@()
      I_PRESS_14 = 2
      END
      
C...15 Drag text
      recursive integer function i_press_15()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            QUERY_EXIT, USING_CTRL
      implicit   none
      include   <windows.ins>
      integer    isend, npress, n1
      parameter (isend = 3, n1 = 1)
      integer    n_button_pressed
      integer    mprev, nprev
      logical    store
      parameter (store = .false.)
      logical    surface
      external   savemn$
      external   w_putall
      external   objplt$
      common   / plot_action / n_button_pressed, surface
      n_button_pressed = 15
      npress = n_button_pressed
      call savemn$(mprev, nprev, store)
      call temporary_yield@()
      if (mprev.lt.1) then
         call w_putall (n1,
     +   'No text string is currently selected for dragging')
         i_press_15 = 2
      else
         if (surface) call objplt$(isend, npress)
         i_press_15 = 1
         CTRLVAR(USING_CTRL) = NEG_VAL
         PROCEED_FLAG = 1
         QUERY_EXIT = .TRUE.
      endif
      call temporary_yield@()
      end
      
C...16 Drag arrow head
      recursive integer function i_press_16()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            QUERY_EXIT, USING_CTRL
      implicit   none
      include   <windows.ins>
      integer    isend, npress, n1
      parameter (isend = 3, n1 = 1)
      integer    n_button_pressed
      integer    mprev, nprev
      logical    store
      parameter (store = .false.)
      logical    surface
      external   savemn$
      external   w_putall
      external   objplt$
      common   / plot_action / n_button_pressed, surface
      n_button_pressed = 16
      npress = n_button_pressed
      call savemn$(mprev, nprev, store)
      call temporary_yield@()
      if (nprev.lt.1) then
         call w_putall (n1,
     +   'No arrow head is currently selected for dragging')
         i_press_16 = 2
      else
         if (surface) call objplt$(isend, npress)
         i_press_16 = 1
         CTRLVAR(USING_CTRL) = NEG_VAL
         PROCEED_FLAG = 1
         QUERY_EXIT = .TRUE.
      endif
      call temporary_yield@()
      end
      
C...17 Drag arrow tail
      recursive integer function i_press_17()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            QUERY_EXIT, USING_CTRL
      implicit   none
      include   <windows.ins>
      integer    isend, npress, n1
      parameter (isend = 3, n1 = 1)
      integer    n_button_pressed
      integer    mprev, nprev
      logical    store
      parameter (store = .false.)
      logical    surface
      external   savemn$
      external   w_putall
      external   objplt$
      common   / plot_action / n_button_pressed, surface
      n_button_pressed = 17
      npress = n_button_pressed
      call savemn$(mprev, nprev, store)
      call temporary_yield@()
      if (nprev.lt.1) then
         call w_putall (n1,
     +   'No arrow tail is currently selected for dragging')
         i_press_17 = 2
      else
         if (surface) call objplt$(isend, npress)
         i_press_17 = 1
         CTRLVAR(USING_CTRL) = NEG_VAL
         PROCEED_FLAG = 1
         QUERY_EXIT = .TRUE.
      endif
      call temporary_yield@()
      end
      
C...18 Drag an object
      recursive integer function i_press_18()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            QUERY_EXIT, USING_CTRL
      implicit   none
      include   <windows.ins>
      integer    isend, npress, n1
      parameter (isend = 3, n1 = 1)
      integer    n_button_pressed
      integer    kprev
      logical    store
      parameter (store = .false.)
      logical    surface
      external   saveob$
      external   w_putall
      external   objplt$
      common   / plot_action / n_button_pressed, surface
      n_button_pressed = 18
      npress = n_button_pressed
      call saveob$(kprev, store)
      call temporary_yield@()
      if (kprev.lt.1) then
         call w_putall (n1,
     +   'No graphical object is currently selected for dragging')
         i_press_18 = 2
      else
         if (surface) call objplt$(isend, npress)
         i_press_18 = 1
         CTRLVAR(USING_CTRL) = NEG_VAL
         PROCEED_FLAG = 1
         QUERY_EXIT = .TRUE.
      endif
      call temporary_yield@()
      end
      
C...19 Drag a side panel
      recursive integer function i_press_19()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            QUERY_EXIT, USING_CTRL
      implicit   none
      include   <windows.ins>
      integer    n_button_pressed
      integer    npanel  
      integer    n1
      parameter (n1 = 1)
      logical    store
      parameter (store = .false.)
      logical    surface
      external   saveit$
      external   w_putall
      common   / plot_action / n_button_pressed, surface
      save       npanel
      n_button_pressed = 19
      call saveit$(npanel, store)
      call temporary_yield@()
      if (npanel.eq.1) then
         i_press_19 = 1
         CTRLVAR(USING_CTRL) = NEG_VAL
         PROCEED_FLAG = 1
         QUERY_EXIT = .TRUE.
      else
         call w_putall (n1,
     +   'No side panel is currently displayed for dragging')
         i_press_19 = 2
      endif
      call temporary_yield@()
      end
      
C...20 Menu/Quit
      recursive integer function I_PRESS_20()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL, CALL_I_PRESS_25   
      use        module_defngks, only : dotmat, hard_copy, hpgl, meta, 
     +                                  pcl, ps                 
      IMPLICIT   NONE
      INTEGER    N_BUTTON_PRESSED
      INTEGER    N1, N24
      PARAMETER (N1 = 1, N24 = 24)
      INTEGER    KVAL_12, X_NKLCFG
      LOGICAL    SURFACE
      LOGICAL    ABORT
      EXTERNAL   QUERY_GRAPH_EXIT, X_NKLCFG, CLOSE_GR_WINDOWS   
      COMMON   / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
C
C Quit the plot ... First warn the user
C
      N_BUTTON_PRESSED = 20
      IF (SURFACE) THEN
         N_BUTTON_PRESSED = 23
         ABORT = .FALSE.
         I_PRESS_20 = 1
         DOTMAT = .FALSE.
         HARD_COPY = .FALSE.
         HPGL = .FALSE.
         META = .FALSE.
         PCL = .FALSE.
         PS = .FALSE. 
         CTRLVAR(USING_CTRL) = NEG_VAL
         PROCEED_FLAG = 1
      ELSE
         IF (CALL_I_PRESS_25.EQ.1) THEN
            ABORT = .TRUE.
         ELSE   
            KVAL_12 = X_NKLCFG(N24)
            IF (KVAL_12.EQ.N1) THEN
               CALL QUERY_GRAPH_EXIT (ABORT)
            ELSE
               ABORT = .TRUE.
            ENDIF
         ENDIF     
         IF (ABORT) THEN
            PROCEED_FLAG = 1
            I_PRESS_20 = 0
            CALL CLOSE_GR_WINDOWS
         ELSE
            I_PRESS_20 = 2
         ENDIF
      ENDIF
      END
      
C...21 Quit to save files or transfer to Simplot
      recursive integer function I_PRESS_21()
      USE MODULE_CLEARWIN, ONLY : PROCEED_FLAG
      IMPLICIT NONE
      include <windows.ins>
      INTEGER  N_BUTTON_PRESSED
      LOGICAL  SURFACE
      EXTERNAL CLOSE_GR_WINDOWS
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      call temporary_yield@()
      PROCEED_FLAG = 1
      N_BUTTON_PRESSED = 21
      I_PRESS_21 = 0
      CALL CLOSE_GR_WINDOWS
      call temporary_yield@()
      END
      
C...22 [PS]
      recursive integer function I_PRESS_22()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            USING_CTRL
      IMPLICIT NONE
      include <windows.ins>
      INTEGER  N_BUTTON_PRESSED
      LOGICAL  SURFACE
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      CALL TEMPORARY_YIELD@()  
      N_BUTTON_PRESSED = 22
      CTRLVAR(USING_CTRL) = NEG_VAL
      PROCEED_FLAG = 1
      I_PRESS_22 = 1
      CALL TEMPORARY_YIELD@()  
      END
      
C...23 [Windows]
      RECURSIVE INTEGER FUNCTION I_PRESS_23()
      USE MODULE_CLEARWIN, ONLY : CTRLVAR, NEG_VAL, PROCEED_FLAG,
     +                            QUERY_EXIT, USING_CTRL
      use        module_defngks, only : dotmat, hard_copy, hpgl, meta, 
     +                                  pcl, ps,
     +                                  jpg, png, svg     
      IMPLICIT   NONE
      INCLUDE   <windows.ins>
      INTEGER    N_BUTTON_PRESSED
      INTEGER    ICOLOR, IX, IY, JCOLOR, LSHADE, NUMDEC, NUMOPT, NUMSTA,
     +           NUMTXT
      PARAMETER (ICOLOR = 9, IX = 4, IY = 4, JCOLOR = 1,LSHADE = 0,
     +           NUMOPT = 12, NUMSTA = 6) 
      INTEGER    IOPEN_SVG, IX_SVG, IY_SVG
      PARAMETER (IOPEN_SVG = 0)
      INTEGER    ISEND, JSEND
      PARAMETER (ISEND = 5, JSEND = 0)
      INTEGER    NUMBLD(30)
      INTEGER    XRES, YRES
      INTEGER    I, L, X_LEN200
      INTEGER    N1, N2, N4
      PARAMETER (N1 = 1, N2 = 2, N4 = 4)              
      DOUBLE PRECISION Q, Q1, X_SVG, Y_SVG
      DOUBLE PRECISION ONE
      PARAMETER (ONE = 1.0D+00) 
      CHARACTER (LEN = 6   ) SVG_STATE 
      CHARACTER (LEN = 1024) SVG_FILE
      CHARACTER  FNAME*1024, LINE*100, TEXT(30)*100, WORD4*4,
     +           WORD5*5   
      CHARACTER  EXT*3, TYPE1*100
      CHARACTER  INFO(7)*10
      CHARACTER  BLANK*1, NA*10
      PARAMETER (BLANK = ' ', NA = '[NA]')
      LOGICAL    ACTIVE_SVG
      LOGICAL    SURFACE, YES
      LOGICAL    LINUX_OS, X_LINUX3, NOTAV(7)
      LOGICAL    ABORT, AGAIN, BORDER, STORE
      PARAMETER (BORDER = .FALSE., STORE = .FALSE.)
      EXTERNAL   W_PUTALL, SAVRES$, X_LSTBOX, X_LCASE1, W_GETFIL,
     +           X_LEN200, W_PATCH1, X_YESNO2, X_LINUX3,
     +           X_EDICFG 
      EXTERNAL   W_SVGINI, SVGPAR
      SAVE       SVG_STATE, SVG_FILE
      COMMON   / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      DATA       NUMBLD / 30*0 / 
      DATA       SVG_STATE, SVG_FILE / 'NOFILE', 'NOFILE' /

      CALL TEMPORARY_YIELD@()  
C
C Check for the Linux operating system
C       
      LINUX_OS = X_LINUX3('*')
      IF (LINUX_OS) THEN 
         DO I = 1, 7 
            IF (I.GE.7) THEN
               INFO(I) = NA
               NOTAV(I) = .TRUE.
            ELSE
               INFO(I) = BLANK
               NOTAV(I) = .FALSE.   
            ENDIF   
         ENDDO  
      ELSE 
         DO I = 1, 7
            INFO(I) = BLANK 
            NOTAV(I) = .FALSE.
         ENDDO        
      ENDIF     
C
C Initialise
C      
      N_BUTTON_PRESSED = 23
      I_PRESS_23 = 2
      DOTMAT = .FALSE.
      HARD_COPY = .FALSE.
      HPGL = .FALSE.
      META = .FALSE.
      PCL = .FALSE.
      PS = .FALSE. 
      SVG = .FALSE.
C
C Main loop
C      
      AGAIN = .TRUE.
      DO WHILE (AGAIN)
         ABORT = .FALSE.
         WRITE (TEXT,200) (INFO(I), I = 1, 7)
         NUMTXT = NUMSTA + NUMOPT - 1
         NUMDEC = NUMOPT - 1
         NUMBLD(1) = 4
         CALL X_LSTBOX (NUMBLD, NUMDEC, NUMOPT, NUMSTA, NUMTXT,
     +                  TEXT) 
         NUMBLD(1) = 0     
         IF (NUMDEC.LE.8) THEN
C 18/11/2013 ... now there is no change in NUMDEC           
            IF (NOTAV(NUMDEC)) THEN
               WRITE (LINE,300)
               CALL W_PUTALL(N4,
     +                      LINE)
            ENDIF
         ENDIF           
         IF (NUMDEC.EQ.5 .OR. NUMDEC.EQ.6) THEN  
C 18/11/2013 ... now only warns users about bitmaps           
            WRITE (LINE,400)
            YES = .TRUE.
            CALL X_YESNO2 (JCOLOR, IX, IY,
     +                     LINE,
     +                     YES)
            IF (YES) NUMDEC = NUMOPT + 1
         ENDIF
         IF (NUMDEC.EQ.1) THEN
C
C Print
C
            AGAIN = .FALSE.
            HARD_COPY = .TRUE.
            PCL = .TRUE.
            CTRLVAR(USING_CTRL) = NEG_VAL
            PROCEED_FLAG = 1
         ELSEIF (NUMDEC.EQ.2) THEN
C
C svg
C
            EXT = 'svg'
            TYPE1 = 'svg image file'
            SVG_FILE = 'c:\svg\default_simfit.svg'
            CALL W_GETFIL (JSEND,
     +                     EXT, SVG_FILE, TYPE1,
     +                     ABORT)                         
            L = 1
            IF (.NOT.ABORT) L = X_LEN200(SVG_FILE)
            IF (L.LT.5) ABORT = .TRUE.
            IF (.NOT.ABORT) THEN
               WORD4 = SVG_FILE(L - 3:L)
               CALL X_LCASE1 (WORD4)
C
C Check for .svg
C
                  IF (WORD4.NE.'.svg') THEN
                     WRITE (LINE,500) '.svg'
                     CALL W_PUTALL (N4,
     +                             LINE)
                     ABORT = .TRUE.
                  ENDIF
            ENDIF
            IF (ABORT) THEN
               AGAIN = .TRUE.
            ELSE   
               AGAIN = .FALSE.
               HARD_COPY = .TRUE.
               SVG = .TRUE.
               PCL = .TRUE.
               CTRLVAR(USING_CTRL) = NEG_VAL
               PROCEED_FLAG = 1
C
C Make sure scaling is switched off (ISEND = 5 in call to SVGPAR) then set SVG size to current XRES, YRES dimensions
C  
               CALL SVGPAR (ISEND,               
     +                      X_SVG, Y_SVG,
     +                      ACTIVE_SVG)
               CALL SAVRES$(IX_SVG, IY_SVG,
     +                      STORE)
               CALL W_SVGINI (IOPEN_SVG, IX_SVG, IY_SVG,
     +                        SVG_FILE, SVG_STATE) 
            ENDIF 
         ELSEIF (NUMDEC.LE.5 .OR. NUMDEC.EQ.7) THEN
C
C Open a file for .png, .jpg, .jpeg, .bmp or .emf
C                               
            IF (NUMDEC.EQ.3) THEN
               EXT = 'png'       
               TYPE1 = 'png graphic image file'
            ELSEIF (NUMDEC.EQ.4) THEN
               EXT = 'jpg'      
               TYPE1 = 'jpg graphic image file'
            ELSEIF (NUMDEC.EQ.5) THEN
               EXT = 'bmp'
               TYPE1 = 'bitmap graphic image file'
            ELSE
               EXT = 'emf' 
               TYPE1 = 'Windows enhanced metafile'
            ENDIF
            CALL W_GETFIL (JSEND,
     +                     EXT, FNAME, TYPE1,
     +                     ABORT)                         
            L = 1
            IF (.NOT.ABORT) L = X_LEN200(FNAME)
            IF (L.LT.5) ABORT = .TRUE.
            IF (.NOT.ABORT) THEN
               WORD4 = FNAME(L - 3:L)
               CALL X_LCASE1 (WORD4)
               IF (NUMDEC.EQ.3) THEN
C
C Check for .png
C
                  IF (WORD4.NE.'.png') THEN
                     WRITE (LINE,500) '.png'
                     CALL W_PUTALL (N4,
     +                             LINE)
                     ABORT = .TRUE.
                  ENDIF
               ELSEIF (NUMDEC.EQ.4) THEN
C
C Check for .jpg or .jpeg
C
                 IF (WORD4.NE.'.jpg') THEN
                     IF (L.LT.6) THEN
                        ABORT = .TRUE. 
                        WRITE (LINE,600)
                        CALL W_PUTALL (N4,
     +                                 LINE)
                     ELSE
                        WORD5 = FNAME(L - 4:L)
                        CALL X_LCASE1 (WORD5)
                        IF (WORD5.NE.'.jpeg') THEN
                           ABORT = .TRUE.
                           WRITE (LINE,500) '.jpg or .jpeg'
                           CALL W_PUTALL (N4, 
     +                                    LINE)
                        ENDIF
                     ENDIF
                  ENDIF
               ELSEIF (NUMDEC.EQ.5) THEN
C
C Check for .bmp
C
                  IF (WORD4.NE.'.bmp') THEN
                     ABORT = .TRUE. 
                     WRITE (LINE,500) '.bmp'
                     CALL W_PUTALL (N4,
     +                              LINE)
                  ENDIF
               ELSEIF (NUMDEC.EQ.7) THEN
C
C Check for .wmf or .emf
C
                  IF (WORD4.NE.'.wmf' .AND. WORD4.NE.'.emf') THEN
                     ABORT = .TRUE.
                     WRITE (LINE,500) '.wmf or .emf' 
                     CALL W_PUTALL (N4,
     +                              LINE)
                  ENDIF
               ENDIF
            ENDIF
         ENDIF
         IF (NUMDEC.GT.2 .AND. NUMDEC.LT.9 .AND. .NOT.ABORT) THEN
            IF (NUMDEC.LE.5 .OR. NUMDEC.EQ.7) THEN
C
C Try to create .png, .jpg, .jpeg, .bmp, or .emf file
C                               
               if (numdec.eq.4) then
                  q = one
                  q1 = set_jpeg_quality@(q)
                  q = q1!to silence ftn95
               endif
               i = export_image@(fname)
            ELSEIF (NUMDEC.EQ.6) THEN
C
C Try to copy to clipboard
C
               call savres$(XRES, YRES,
     +                      STORE)
               i = graphics_to_clipboard@(N1, N1, XRES - N1, YRES - N1)
            ELSEIF (NUMDEC.EQ.8) THEN
C
C Metafile to clipboard
C
                i = metafile_to_clipboard@()
            ENDIF
            IF (I.NE.0) THEN
               AGAIN = .FALSE.  
               WRITE (LINE,700)
               CALL W_PUTALL (N1,
     +                        LINE)
            ELSE
               AGAIN = .TRUE. 
               WRITE (LINE,800) I
               CALL W_PUTALL (N4,
     +                        LINE)
            ENDIF
         ELSEIF (NUMDEC.EQ.9) THEN
C
C Fonts
C
            AGAIN = .FALSE.
            CALL X_EDICFG (N1)
            CTRLVAR(USING_CTRL) = NEG_VAL
            PROCEED_FLAG = 1
            QUERY_EXIT = .TRUE.
         ELSEIF (NUMDEC.EQ.10) THEN
C
C Line thickness
C
            AGAIN = .FALSE.
            CALL X_EDICFG (N2)
            CTRLVAR(USING_CTRL) = NEG_VAL
            PROCEED_FLAG = 1
            QUERY_EXIT = .TRUE.
         ELSEIF (NUMDEC.EQ.11) THEN
C
C Help
C
            AGAIN = .TRUE.
            WRITE (TEXT,100)
            NUMBLD(1) = 1
            NUMBLD(12) = 1
            NUMTXT = 24
            CALL W_PATCH1 (ICOLOR, IX, IY, LSHADE, NUMBLD, NUMTXT,
     +                     TEXT,
     +                     BORDER)
            NUMBLD(1) = 0
            NUMBLD(12) = 0
         ELSEIF (NUMDEC.EQ.12) THEN
            AGAIN = .FALSE.
         ELSEIF (NUMDEC.EQ.NUMOPT + 1) THEN
            AGAIN = .TRUE.
         ENDIF
      ENDDO
      
      CALL TEMPORARY_YIELD@()  
C
C format statements
C      
  100 FORMAT (
     + 'The Simfit Windows hardcopy options'
     +/
     +/'These are to get hardcopy from a printer, or to save the graph'
     +/'as a compressed bitmap, or metafile. Only three of the Windows'
     +/'graphic image file-types are worth considering.'
     +/'1)`Scaleable vector graphics (.svg) are the W3C recommended XML'
     +/'  `standard for line drawings, i.e. Simfit graphics.'
     +/'2)`Portable network graphics files (.png) are the industry'
     +/'  `standard for compressed bitmaps.'
     +/'3)`Enhanced metafiles (.emf) are obsolete but are still used.'
     +/
     +/'The Simfit PostScript hardcopy option'
     +/
     +/'This creates professional quality encapsulated PostScript files'
     +/'(.eps) in a structured format that allows retrospective editing'
     +/'in text editors (e.g. Notepad). If you use a PostScript printer'
     +/'no further action is needed. Archive the .eps files for reading'
     +/'into documents, or to edit/re-size/rotate/overlay/make-collages'
     +/'using Editps. If you do not have a PostScript printer, make'
     +/'.png, .svg, .pdf, .jpg, .tif, .emf, or .bmp files as the .eps'
     +/'file is created, or later from Simfit using Ghostscript.'
     +/
     +/'Read the Simfit manual w_manual.pdf or ms_office.pdf for'
     +/'further details.')
  200 FORMAT (
     + 'Windows graphics options for the current plot displayed'
     +/ 
     +/'The overall character scheme also applies to Simfit PostScript'
     +/'but the overall line thickness only applies to Windows graphs.'
     +/
     +/'Save As ...pdf`or print as a high-resolution bit-map',1X,A
     +/'Save As ...svg`High resolution internet vector-type file'
     +/'Save As ...png`Low resolution compressed bitmap',1X,A
     +/'Save As ...jpg`Low resolution compressed bitmap',1X,A
     +/'Save As ...bmp`uncompressed bitmap',1X,A
     +/'Copy As ...bmp`uncompressed bitmap to clipboard',1X,A
     +/'Save As ...emf`enhanced metafile',1X,A
     +/'Copy As ...emf`enhanced metafile to clipboard',1X,A
     +/'Font type     `Set overall character scheme'
     +/'Line width    `Set overall line thickness'
     +/'Help          `Explain these options'
     +/'Quit          `Exit Simfit Windows graphics options')
  300 FORMAT ('May not be available in Linux ... Use the [PS] options')
  400 FORMAT (
     +'PS options or PNG files are better...Cancel bad choice')
  500 FORMAT ('File extension must be',1X,A)
  600 FORMAT ('File name is unacceptable')   
  700 FORMAT ('The plot has now been copied')
  800 FORMAT ('The plot cannot be copied ... function returns',i4)
      END
C
C...24
      recursive integer function I_PRESS_24()
      USE MODULE_CLEARWIN, ONLY : PROCEED_FLAG
      IMPLICIT   NONE
      INTEGER    N_BUTTON_PRESSED
      INTEGER    N1, N24
      PARAMETER (N1 = 1, N24 = 24)
      INTEGER    KVAL_12, X_NKLCFG
      LOGICAL    ABORT, SURFACE
      EXTERNAL   QUERY_GRAPH_EXIT, X_NKLCFG, CLOSE_GR_WINDOWS   
      COMMON / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      KVAL_12 = X_NKLCFG(N24)
      IF (KVAL_12.EQ.N1) THEN
         CALL QUERY_GRAPH_EXIT (ABORT)
      ELSE
         ABORT = .TRUE.
      ENDIF     
      IF (ABORT) THEN
         PROCEED_FLAG = 1
         I_PRESS_24 = 0
         CALL CLOSE_GR_WINDOWS
      ELSE
         I_PRESS_24 = 2
      ENDIF
      N_BUTTON_PRESSED = 24
      END
C
C      
C...25 SVG
      RECURSIVE INTEGER FUNCTION I_PRESS_25()
      USE MODULE_CLEARWIN, ONLY : CALL_I_PRESS_25, CTRLVAR, NEG_VAL, 
     +                            PROCEED_FLAG, QUERY_EXIT, USING_CTRL
      use        module_defngks, only : dotmat, hard_copy, hpgl, meta, 
     +                                  pcl, ps,
     +                                  jpg, png, svg     
      IMPLICIT   NONE
      INCLUDE   <windows.ins>
      INTEGER    N_BUTTON_PRESSED
      INTEGER    NUMDEC, NUMOPT, NUMSTA, NUMTXT
      PARAMETER (NUMOPT = 10, NUMSTA = 6) 
      INTEGER    IOPEN_SVG, IX_SVG, IY_SVG
      PARAMETER (IOPEN_SVG = 0)
      INTEGER    JSEND
      PARAMETER (JSEND = 0)
      INTEGER    NUMBLD(30)
      INTEGER    L, X_LEN200
      INTEGER    N0, N1, N2, N3, N4, N5
      PARAMETER (N0 = 0, N1 = 1, N2 = 2, N3 = 3, N4 = 4, N5 = 5)    
      DOUBLE PRECISION X_SVG, Y_SVG    
      DOUBLE PRECISION ONE
      PARAMETER (ONE = 1.0D+00)       
      CHARACTER (LEN = 6   ) SVG_STATE 
      CHARACTER (LEN = 1024) SVG_FILE
      CHARACTER (LEN = 1024) FNAME, F$SIMFIT_SVG
      CHARACTER (LEN = 100 ) LINE, TEXT(30), TYPE1
      CHARACTER (LEN = 40  ) CIPHER
      CHARACTER (LEN = 4   ) WORD4
      CHARACTER (LEN = 3   ) EXT
      CHARACTER (LEN = 1   ) BLANK
      PARAMETER (BLANK = ' ')
      LOGICAL    ACTIVE_SVG, SURFACE
      LOGICAL    ABORT, AGAIN, FIRST, STORE
      PARAMETER (STORE = .FALSE.)
      EXTERNAL   W_PUTALL, SAVRES$, X_LSTBOX, X_LCASE1, W_GETFIL,
     +           X_LEN200, W_V7PATH, X_EDICFG 
      EXTERNAL   W_SVGINI, HELP_EDITSVG, SVGPAR
      EXTERNAL   HELP_SVG_SCALING
      SAVE       SVG_STATE, SVG_FILE
      SAVE       FNAME, F$SIMFIT_SVG
      COMMON   / PLOT_ACTION / N_BUTTON_PRESSED, SURFACE
      DATA       NUMBLD / 30*0 / 
      DATA       SVG_STATE, SVG_FILE / 'NOFILE', 'NOFILE' /
      DATA       FNAME, F$SIMFIT_SVG / 'NOFILE', 'NOFILE' / 
      DATA       FIRST /.TRUE. /

      CALL TEMPORARY_YIELD@()  
C
C Initialise
C      
      IF (FIRST) THEN
         FIRST = .FALSE.
         CALL W_V7PATH (L,
     +                  'tmp', F$SIMFIT_SVG)
         IF (F$SIMFIT_SVG(L:L).NE.'\') THEN
            L = L + 1
            F$SIMFIT_SVG(L:L) = '\'
         ENDIF
         F$SIMFIT_SVG(L + 1:L + 12) = 'f$simfit.svg'
      ENDIF   
      CALL_I_PRESS_25 = 0    
      N_BUTTON_PRESSED = 25
      I_PRESS_25 = 2
      DOTMAT = .FALSE.
      HARD_COPY = .FALSE.
      HPGL = .FALSE.
      META = .FALSE.
      PCL = .FALSE.
      PS = .FALSE. 
      SVG = .FALSE.
      CALL SVGPAR (N0,
     +             X_SVG, Y_SVG,
     +             ACTIVE_SVG)  
C
C Main loop
C      
      AGAIN = .TRUE.
      DO WHILE (AGAIN)
         ABORT = .FALSE.
         IF (X_SVG.GT.ONE .AND. Y_SVG.LE.ONE) THEN
            WRITE (CIPHER,'(A18,F7.2,A1)') '(but using x_svg =',
     +                                      X_SVG,')'
         ELSEIF (X_SVG.LE.ONE  .AND. Y_SVG.GT.ONE) THEN   
            WRITE (CIPHER,'(A18,F7.2,A1)') '(but using y_svg =',
     +                                      Y_SVG,')'
         ELSEIF (X_SVG.GT.ONE .AND. Y_SVG.GT.ONE) THEN
            WRITE (CIPHER,'(A13,F7.2,A13,F7.2)') 'using x_svg =', x_svg,
     +                                           ', and y_svg =', y_svg    
         ELSE
            CIPHER = BLANK
         ENDIF      
         WRITE (TEXT,100) CIPHER, CIPHER, X_SVG, Y_SVG
         NUMTXT = NUMSTA + NUMOPT - 1
         NUMDEC = NUMOPT - 1
         NUMBLD(1) = 4
         CALL X_LSTBOX (NUMBLD, NUMDEC, NUMOPT, NUMSTA, NUMTXT,
     +                  TEXT) 
         NUMBLD(1) = 0     
C
C make sure i_press_25 = 0 unless a file is created or a file is viewed
C
         CALL_I_PRESS_25 = 0
         IF (NUMDEC.EQ.1) THEN
C
C svg
C
            EXT = 'svg'
            TYPE1 = 'svg image file'
            SVG_FILE = 'c:\svg\default_simfit.svg'
            CALL W_GETFIL (JSEND,
     +                     EXT, SVG_FILE, TYPE1,
     +                     ABORT)                         
            L = 1
            IF (.NOT.ABORT) L = X_LEN200(SVG_FILE)
            IF (L.LT.5) ABORT = .TRUE.
            IF (.NOT.ABORT) THEN
               WORD4 = SVG_FILE(L - 3:L)
               CALL X_LCASE1 (WORD4)
C
C Check for .svg
C
                  IF (WORD4.NE.'.svg') THEN
                     WRITE (LINE,200) '.svg'
                     CALL W_PUTALL (N4,
     +                             LINE)
                     ABORT = .TRUE.
                  ENDIF
            ENDIF
            IF (ABORT) THEN
               AGAIN = .TRUE.
            ELSE   
               AGAIN = .FALSE.
               HARD_COPY = .TRUE.
               SVG = .TRUE.
C**************PCL = .TRUE.
               PCL = .false.
               CTRLVAR(USING_CTRL) = NEG_VAL
               PROCEED_FLAG = 1
C
C set SVG size to current XRES, YRES dimenions then set call_i_press_25 = 1
C               
               CALL SAVRES$(IX_SVG, IY_SVG,
     +                      STORE)
               CALL W_SVGINI (IOPEN_SVG, IX_SVG, IY_SVG,
     +                        SVG_FILE, SVG_STATE) 
               CALL_I_PRESS_25 = 1
            ENDIF 
         ELSEIF (NUMDEC.EQ.2) THEN
C
C Save as f$simfit.svg in the temp folder then view then set call_i_press_25 = 1
C
            AGAIN = .FALSE.
            HARD_COPY = .TRUE.
            SVG = .TRUE.
            PCL = .TRUE.
            CTRLVAR(USING_CTRL) = NEG_VAL
            PROCEED_FLAG = 1
C***********CALL PERMIT_ANOTHER_CALLBACK@
C
C set SVG size to current XRES, YRES dimensions then define FNAME
C               
            CALL SAVRES$(IX_SVG, IY_SVG,
     +                   STORE)
            FNAME = F$SIMFIT_SVG
            CALL W_SVGINI (IOPEN_SVG, IX_SVG, IY_SVG,
     +                     FNAME, SVG_STATE)
            CALL_I_PRESS_25 = 1
        ELSEIF (NUMDEC.EQ.3) THEN
C
C Explain the SVG scaling factors
C         
            CALL HELP_SVG_SCALING ('svg_scaling')
         ELSEIF (NUMDEC.EQ.4) THEN
C
C Change X_SVG
C         
            CALL SVGPAR (N3,
     +                   X_SVG, Y_SVG,
     +                   ACTIVE_SVG) 
C            IF (X_SVG .GT. ONE) Y_SVG = ONE
         ELSEIF (NUMDEC.EQ.5) THEN
C
C Change Y_SVG
C         
            CALL SVGPAR (N4,
     +                   X_SVG, Y_SVG,
     +                   ACTIVE_SVG) 
C            IF (Y_SVG .GT. ONE) X_SVG = ONE
         ELSEIF (NUMDEC.EQ.6) THEN
C
C Restore defaults
C         
            CALL SVGPAR (N5,
     +                   X_SVG, Y_SVG,
     +                   ACTIVE_SVG)            
         ELSEIF (NUMDEC.EQ.7) THEN
C
C Fonts
C
            CALL X_EDICFG (N1)
            CTRLVAR(USING_CTRL) = NEG_VAL
            PROCEED_FLAG = 1
            QUERY_EXIT = .TRUE.
         ELSEIF (NUMDEC.EQ.8) THEN
C
C Line thickness
C
            CALL X_EDICFG (N2)
            CTRLVAR(USING_CTRL) = NEG_VAL
            PROCEED_FLAG = 1
            QUERY_EXIT = .TRUE.
         ELSEIF (NUMDEC.EQ.9) THEN
C
C Help
C
            CALL HELP_EDITSVG ('editsvg')
         ELSEIF (NUMDEC.EQ.NUMDEC) THEN
            AGAIN = .FALSE.
         ENDIF
      ENDDO
      
      CALL TEMPORARY_YIELD@()  
C
C format statements
C      
  100 FORMAT (
     + 'The Simfit SVG graphics options'
     +/ 
     +/'The overall character scheme also applies to Simfit PostScript'
     +/'but the overall line thickness only applies to Windows graphs.'
     +/
     +/'Save As ...`Create a new user-named SVG file',2X,A
     +/'Display    `View the current graph as SVG',2X,A
     +/'Explain    `The X,Y-scaling factors x_svg and y_svg'
     +/'Change     `X_scaling factor: x_svg =',f7.2
     +/'Change     `Y_scaling factor: y_svg =',f7.2
     +/'Restore    `Use default X,Y-scaling factors'
     +/'Font type  `Set the overall character scheme'
     +/'Line width `Set the overall line thickness'
     +/'Help       `Describe the Simfit SVG file format'
     +/'Quit       `Exit Simfit SVG graphics options')   
  200 FORMAT ('File extension must be',1X,A)
        END
C      
C

     