c
c subroutines
c cdfsam
c cdplot
c
c---------------------------------------------------------------------
c
      subroutine cdfsam (i, na, ndist, ni, nsamp, 
     +                   a, sample,
     +                   title)
c
c action: submit a sample and pdf to cdplot
c author: w.g.bardsley, university of manchester, u.k., 24/03/2019
c
c Note: the cdf plotted is calculated from the pdf by triangulation in cdplot 
c       and it will look unrepresentative if the range of the sample is less
c       than the range over which the random variable is defined. Probably
c       sample sizes of at least 100 are required to satisfy this criterion,
c       particularly with distributions like the normal or Cauchy 
c       distribution which have long tails.
c 
c          i: integer parameters
c         na: dimension of a
c      ndist: distribution 1 to 12
c      nsamp: sample size 
c          a: double precision parameters
c     sample: random sample
c      title: name of distribution
c     
c ndist = 1: Cauchy
c ndist = 2: Chi-square
c ndist = 3: Negative Exponential
c ndist = 4: Gamma
c ndist = 5: Logistic
c ndist = 6: Lognormal
c ndist = 7: Normal
c ndist = 8: Uniform(A, B)
c ndist = 9: Weibull
c ndist = 10: F
c ndist = 11: t
c ndist = 12: Beta
c
      implicit none
c
c arguments
c      
      integer,             intent (in) :: na, ndist, ni, nsamp
      integer,             intent (in) :: i(ni) 
      double precision,    intent (in) :: a(na), sample(nsamp) 
      character (len = *), intent (in) :: title
c
c allocatable
c      
      double precision, allocatable :: rdata(:), x(:), y(:), z(:) 
c
c local
c      
      integer    ifail, loop, nrmax
      integer    npdf
      parameter (npdf = 120)
      double precision delta, pdf(npdf), t(npdf), 
     +                 tmax, tmin 
      character (len = 100) line
      logical    abort
      external   putfat, putadv
      external   cdplot, pdfval, nxsortg
      intrinsic  dble
c
c check input
c      
      if (na.lt.2 .or. ni.lt.2) then
         call putfat ('NA < 2  or NI < 2 in call to CDFSAM')
         return
      endif
      if (nsamp.lt.10) then
         call putfat ('NSAMP < 10 in call to CDFSAM')
         return
      endif
      if (ndist.lt.1 .or. ndist.gt.12) then 
         write (line,100) ndist
         call putadv (line)
         return
      endif
c
c allocate then sort
c      
      nrmax = max(npdf,nsamp) 
      allocate (rdata(nsamp), stat = ifail)
      if (ifail.eq.0) allocate (x(2*nrmax), stat = ifail)
      if (ifail.eq.0) allocate (y(2*nrmax), stat = ifail)
      if (ifail.eq.0) allocate (z(2*nrmax), stat = ifail)
      if (ifail.ne.0) then
         call putfat ('Failure to allocate in CDFSAM')
         return
      endif 
      do loop = 1, nsamp
         rdata(loop) = sample(loop)
      enddo
      call nxsortg (nsamp,
     +              rdata)    
c
c get the sample range then prepare the t values
c      
      tmax = rdata(nsamp)
      tmin = rdata(1)
      delta = (tmax - tmin)/dble(npdf - 1)  
      t(1) = tmin
      do loop = 2, npdf - 1
         t(loop) = t(loop - 1) + delta
      enddo
      t(npdf) = tmax  
c
c now prepare the pdf values
c      
      call pdfval (i, na, ndist, ni, npdf,  
     +             a, pdf, t,
     +             abort)
      if (abort) then
         write (line,200) ndist
         call putfat (line)
         return
      endif   
c
c finally plot the results
c      
      call cdplot (npdf, nrmax, nsamp,
     +             pdf, rdata, t, x, y, z,
     +             title, 'x', 'cdf(x)')
c
c deallocate
c         
      deallocate (rdata, stat = ifail) 
      deallocate (x, stat = ifail) 
      deallocate (y, stat = ifail) 
      deallocate (z, stat = ifail) 
c
c format statements
c
  100 format ('Distribution',i3,' is not supported by CDFSAM')  
  200 format ('Error calculating with distribution',i3,' in CDFVAL')
      end
c
c--------------------------------------------------------------       
c
      subroutine cdplot (npdf, nrmax, nsamp,
     +                   pdf, sample, t, x, y, z,
     +                   ptitle, xtitle, ytitle)
c
c action: plot step function and best-fit cdf
c author: w.g.bardsley, university of manchester, u.k., 24/08/2005 
c         19/05/2007 added intents 
c
c Note: the cdf plotted is calculated from the pdf by triangulation in cdplot 
c       and it will look unrepresentative if the range of the sample is less
c       than the range over which the random variable is defined. Probably
c       sample sizes of at least 100 are required to satisfy this criterion,
c       particularly with distributions like the normal or Cauchy 
c       distribution which have long tails.
c
c          npdf: (input/unchanged) number of pdf values
c         nrmax: (input/unchanged) dimension of workspace 
c         nsamp: (input/unchanged) dimension of sample 
c           pdf: (input/unchanged) pdf values at t values (below)
c        sample: (input/unchanged) sample values in nondecreasing order
c             t: (input/unchanged) t values for pdf(t) in nondecreasing order
c             x:  workspace for plotting
c             y:  workspace for plotting
c             z:  workspace for plotting
c        ptitle: (input/unchanged) plot title
c        xtitle: (input/unchanged) x-axis title
c        ytitle: (input/unchanged) y-axis title
c
      implicit   none
c
c arguments
c
      integer,          intent (in)    :: npdf, nrmax, nsamp
      double precision, intent (in)    :: pdf(npdf), sample(nsamp),
     +                                    t(npdf)
      double precision, intent (inout) :: x(2*nrmax), y(2*nrmax), 
     +                                    z(2*nrmax)
      character (len = *), intent (in) :: ptitle, xtitle, ytitle
c
c locals
c
      integer    i, icount
      integer    l1, l2, l3, l4, m1, m2, m3, m4, n1, n2, n3, n4
      parameter (l1 = 1, l2 = 2, l3 = 0, l4 = 0, m1 = 0, m2 = 0, m3 = 0,
     +           m4 = 0, n3 = 2, n4 = 2)
      double precision addon, base, delta, value, xmax, xmin
      double precision x3(n3), x4(n4), y3(n3), y4(n4)
      double precision zero, half, one
      parameter (zero = 0.0d+00, half = 0.5d+00, one = 1.0d+00)
      character  line*100
      logical    axes, gsave
      parameter (axes = .true., gsave = .true.)
      external   putfat, gks004
      intrinsic  max, min, dble
c
c check
c
      if (npdf.lt.10 .or. nsamp.lt.5) then
         write (line,100)
         call putfat (line)
         return
      endif
      do i = 2, nsamp
         if (sample(i).lt.sample(i - 1)) then
            write (line,200)
            call putfat (line)
            return
         endif
      enddo
      do i = 2, npdf
         if (t(i).lt.t(i - 1)) then
            write (line,300)
            call putfat (line)
            return
         endif
      enddo
c
c cdf calculation for sample
c
      xmax = max(sample(nsamp), t(npdf))
      xmin = min(sample(1), t(1))
      icount = 1
      x(icount) = xmin
      z(icount) = zero
      value = sample(1)
      delta = one/dble(nsamp)
      base = zero
      addon = zero
      do i = 1, nsamp - 1
         if (sample(i).gt.value) then
c
c start a new step
c
            icount = icount + 1
            x(icount) = value
            z(icount) = base
            icount = icount + 1
            x(icount) = value
            base = base + addon
            z(icount) = base
            addon = delta
            value = sample(i)
         else
c
c add to an existing step
c
            addon = addon + delta
         endif
      enddo
c
c finish off the last data point
c
      if (icount.lt.2*nrmax - 2) then
         icount = icount + 1
         x(icount) = value
         z(icount) = base
         addon = addon + delta
         icount = icount + 1
         x(icount) = value
         base = base + addon
         z(icount) = base
      endif
c
c extend final step
c
      if (icount.lt.2*nrmax) then
         icount = icount + 1
         x(icount) = xmax
         z(icount) = one
      endif
c
c cdf calculation from pdf using triangulation for area
c
      delta = t(2) - t(1)
      addon = half*delta*pdf(1)
      y(1) = addon
      base = addon
      do i = 2, npdf
         delta = t(i) - t(i - 1)
         addon = half*delta*(pdf(i) + pdf(i - 1))
         base = base + addon
         y(i) = base
      enddo
c
c plot
c
      n1 = icount
      n2 = npdf
      call gks004 (l1, l2, l3, l4, m1, m2, m3, m4, n1, n2, n3, n4,
     +             x, t, x3, x4,
     +             z, y, y3, y4,
     +             ptitle, xtitle, ytitle,
     +             axes, gsave) 
c
c format statements
c     
  100 format ('Must have sample size > 5, pdf size > 5')
  200 format ('Must have sample in nondecreasing order')
  300 format ('Must have pdf in nondecreasing order')
      end
c
c
