C
C G_DRAWING ... primitive drawing subroutines
C =========
C
C FILL_POLYGON$(N, X, Y, COLOUR_INDEX)
C LINE$(X1, Y1, X2, Y2, COLOUR_INDEX)
C SOLID_LINE$(X1, Y1, X2, Y2, COLOUR_INDEX)
C DASHED_LINE$(X1, Y1, X2, Y2, COLOUR_INDEX)
C POLYLINE$(N, X, Y, COLOUR_INDEX)
C SOLID_POLYLINE$(N, X, Y, COLOUR_INDEX)
C DASHED_POLYLINE$(N, X, Y, COLOUR_INDEX)
C ARROW_POLYLINE$(ISEND, N, X, Y, COLOUR_INDEX)
C MITER_POLYLINE$(N, X, Y, COLOUR_INDEX)
C TEXT$(X, Y, STR, COLOUR_INDEX)
C
C These are the routines that call Salford graphics using the w_salfgr
C interface primitives to draw lines and fill polygons.
C These graphics calls may have to be intercepted and re-routed.
C They also call Salford routines for text drawing. If it is only
C required to create PostScript, then make sure PS = .TRUE. and
C none of these calls will then be necessary.
C
C Note that the Salford routines now only use long integers
C
C ACTION : Primitive drawing functions
C AUTHOR : W. G. Bardsley, University of Manchester, U.K., 21/11/94
C          23/02/1997 Win32 version
C          24/07/1997 Added SHORT to swap between short/long integers
C                     EDIT at SHORT = to swap colour integers
C          10/12/1997 Added new line types and ARROW_POLYLINE$
C          07/02/1999 Added NUMRGB$ to re-define colours.
C          24/10/1999 Edited TEXT$ to deal with underscore
C          26/11/1999 Edited TEXT$ for caret and backslash and added SLASHB$
C
C          ******************************************************
C          From now on all colours must be defined using NUMRGB$.
C          ******************************************************
C
C          27/09/2000 added linetype 9 = filled polygon
C          12/08/2003 revised for new win32 routines using normal integers, etc.
C                     with no remaining confusion over long and short integers
C                     OLD              NEW
C                     ===              ===
C                     DRAW_LINE@       DRAW_LINE_BETWEEN@
C                     RECTANGLE@       DRAW_RECTANGLE@
C                     FILL_RECTANGLE@  DRAW_FILLED_RECTANGLE@
C                     DRAW_TEXT@       DRAW_CHARACTERS@
C                     POLYLINE@        DRAW_POLYLINE@
C                     FILL_POLYGON@    DRAW_FILLED_POLYGON@
C          01/10/2005 moved NUMRGB$ to w_menus.dll  
C          07/07/2006 replaced IXTEMP and IYTEMP by IX and IY in SOLID_POLYLINE$
C                     and FILL_POLYGON$, and allocatables introduced in POLYLINE$
C          20/03/2007 replaced the Salford primitives with the w_salfgr interface, 
C                     deleted inclusion of <windows.ins> and added INTENTS         
C          14/06/2007 removed defngks.ins and added GETGKS calls
C          12/04/2010 introduced SET_LINE_STYLE to use for dash/dot cases if line thickness = 1
C          25/08/2014 added MITER_POLYLINE  
C
C  
C------------------------------------------------------
C
      SUBROUTINE FILL_POLYGON$(N, X, Y, COLOUR_INDEX)
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: N, COLOUR_INDEX 
      DOUBLE PRECISION, INTENT (IN) :: X(N), Y(N)
C
C Local allocatable arrays
C                         
      INTEGER, ALLOCATABLE :: IX(:), IY(:)
C
C Locals
C      
      INTEGER    IERR, NTEMP
      INTEGER    N0, N1, N2, N6
      PARAMETER (N0 = 0, N1 = 1, N2 = 2, N6 = 6)
      INTEGER    COLOUR_LONG
      INTEGER    NUMRGB$  
      logical    ps
      EXTERNAL   GKSA2I$, PSPOLY$, NUMRGB$, GETGKS_LGL
      EXTERNAL   DRAW_FILLED_POLYGON
      IF (N.LT.N2) RETURN   
      CALL GETGKS_LGL (N6,
     +                 PS)      
      IF (PS) THEN
         NTEMP = N - N1
         CALL PSPOLY$(COLOUR_INDEX, NTEMP, X, Y, 'pf')
      ELSE                  
         IERR = N0
         IF (ALLOCATED(IX)) DEALLOCATE(IX, STAT = IERR)
         IF (IERR.NE.N0) RETURN
         IF (ALLOCATED(IY)) DEALLOCATE(IY, STAT = IERR)
         IF (IERR.NE.N0) RETURN           
         ALLOCATE(IX(N), STAT = IERR)
         IF (IERR.NE.N0) RETURN 
         ALLOCATE(IY(N), STAT = IERR)
         IF (IERR.NE.N0) RETURN
         CALL GKSA2I$(IX, IY, N, X, Y)
         COLOUR_LONG = NUMRGB$(COLOUR_INDEX)
         CALL DRAW_FILLED_POLYGON (IX, IY, N, COLOUR_LONG)
         DEALLOCATE(IX, STAT = IERR) 
         DEALLOCATE(IY, STAT = IERR)
      ENDIF
      END
C 
C---------------------------------------------------
C
      SUBROUTINE LINE$(X1, Y1, X2, Y2, COLOUR_INDEX)
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: COLOUR_INDEX  
      DOUBLE PRECISION, INTENT (IN) :: X1, Y1, X2, Y2
C
C Locals
C                  
      INTEGER    LINE_TYPE
      INTEGER    N2, N4, N6
      PARAMETER (N2 = 2, N4 = 4, N6 = 6)
      DOUBLE PRECISION X(2), Y(2)
      DOUBLE PRECISION X1TEMP, X2TEMP, Y1TEMP, Y2TEMP 
      LOGICAL    PS
      EXTERNAL   PSPOLY$, PSLINE$, SOLID_LINE$, DASHED_LINE$, POLYLINE$
      EXTERNAL   GETGKS_INT, GETGKS_LGL
      CALL GETGKS_INT (N4, LINE_TYPE)
      CALL GETGKS_LGL (N6,
     +                 PS) 
      IF (LINE_TYPE.LT.1) THEN
         RETURN
      ELSEIF (LINE_TYPE.EQ.5 .OR. LINE_TYPE.EQ.6) THEN
         X(1) = X1
         X(2) = X2
         Y(1) = Y1
         Y(2) = Y2
         CALL POLYLINE$(N2, X, Y, COLOUR_INDEX)
      ELSEIF (PS) THEN
         IF (LINE_TYPE.EQ.1) THEN
            CALL PSLINE$(COLOUR_INDEX, X1, Y1, X2, Y2)
         ELSE
            X(1) = X1
            X(2) = X2
            Y(1) = Y1
            Y(2) = Y2
            IF (LINE_TYPE.EQ.2) THEN
               CALL PSPOLY$(COLOUR_INDEX, N2, X, Y, 'da')
            ELSEIF (LINE_TYPE.EQ.3) THEN
               CALL PSPOLY$(COLOUR_INDEX, N2, X, Y, 'do')
            ELSEIF (LINE_TYPE.EQ.4) THEN
               CALL PSPOLY$(COLOUR_INDEX, N2, X, Y, 'dd')
            ELSEIF (LINE_TYPE.EQ.7) THEN
               X1TEMP = X1
               X2TEMP = X1
               Y1TEMP = Y1
               Y2TEMP = Y2
               CALL PSLINE$(COLOUR_INDEX, X1TEMP, Y1TEMP, X2TEMP,
     +                      Y2TEMP)
               X2TEMP = X2
               Y1TEMP = Y2
               CALL PSLINE$(COLOUR_INDEX, X1TEMP, Y1TEMP, X2TEMP,
     +                      Y2TEMP)
            ELSEIF (LINE_TYPE.EQ.8) THEN
               X1TEMP = X1
               X2TEMP = X2
               Y1TEMP = Y1
               Y2TEMP = Y1
               CALL PSLINE$(COLOUR_INDEX, X1TEMP, Y1TEMP, X2TEMP,
     +                      Y2TEMP)
               X1TEMP = X2
               Y2TEMP = Y2
               CALL PSLINE$(COLOUR_INDEX, X1TEMP, Y1TEMP, X2TEMP,
     +                      Y2TEMP)
            ENDIF
         ENDIF
      ELSE
         IF (LINE_TYPE.EQ.1) THEN
            CALL SOLID_LINE$(X1, Y1, X2, Y2, COLOUR_INDEX)
         ELSEIF (LINE_TYPE.LT.5) THEN
            CALL DASHED_LINE$(X1, Y1, X2, Y2, COLOUR_INDEX)
         ELSEIF (LINE_TYPE.EQ.7) THEN
            X1TEMP = X1
            X2TEMP = X1
            Y1TEMP = Y1
            Y2TEMP = Y2
            CALL SOLID_LINE$(X1TEMP, Y1TEMP, X2TEMP, Y2TEMP,
     +                       COLOUR_INDEX)
            X2TEMP = X2
            Y1TEMP = Y2
            CALL SOLID_LINE$(X1TEMP, Y1TEMP, X2TEMP, Y2TEMP,
     +                       COLOUR_INDEX)
         ELSEIF (LINE_TYPE.EQ.8) THEN
            X1TEMP = X1
            X2TEMP = X2
            Y1TEMP = Y1
            Y2TEMP = Y1
            CALL SOLID_LINE$(X1TEMP, Y1TEMP, X2TEMP, Y2TEMP,
     +                       COLOUR_INDEX)
            X1TEMP = X2
            Y2TEMP = Y2
            CALL SOLID_LINE$(X1TEMP, Y1TEMP, X2TEMP, Y2TEMP,
     +                       COLOUR_INDEX)
         ENDIF
      ENDIF
      END
C 
C---------------------------------------------------------
C
      SUBROUTINE SOLID_LINE$(X1, Y1, X2, Y2, COLOUR_INDEX)
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: COLOUR_INDEX 
      DOUBLE PRECISION, INTENT (IN) :: X1, Y1, X2, Y2
C
C Locals
C      
      INTEGER    NUMRGB$
      INTEGER    IX1, IX2, IY1, IY2
      DOUBLE PRECISION DX1, DX2, DY1, DY2
      INTEGER    COLOUR_LONG 
      INTEGER    N1, N6
      PARAMETER (N1 = 1, N6 = 6)
      LOGICAL    PS
      LOGICAL    STORE, USE_GDIPLUS
      PARAMETER (STORE = .FALSE.)
      EXTERNAL   GKSR2D$, GKSR2I$, PSLINE$, NUMRGB$, GETGKS_LGL  
      EXTERNAL   DRAW_LINE_BETWEEN, DRAW_LINE_BETWEEN_D, USE_GDIPLUS,
     +           SET_LINE_STYLE
      CALL GETGKS_LGL (N6,
     +                 PS) 
      IF (PS) THEN
         CALL PSLINE$(COLOUR_INDEX, X1, Y1, X2, Y2)
      ELSEIF (USE_GDIPLUS(STORE)) THEN
         CALL GKSR2D$(DX1, DY1, X1, Y1)
         CALL GKSR2D$(DX2, DY2, X2, Y2) 
         COLOUR_LONG = NUMRGB$(COLOUR_INDEX)
         CALL SET_LINE_STYLE (N1)
         CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2, COLOUR_LONG)
      ELSE   
         CALL SET_LINE_STYLE (N1)
         CALL GKSR2I$(IX1, IY1, X1, Y1)
         CALL GKSR2I$(IX2, IY2, X2, Y2)
         COLOUR_LONG = NUMRGB$(COLOUR_INDEX)
         CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2, COLOUR_LONG) 
      ENDIF
      END
C                               
C---------------------------------------------------------------------
C
C DASHED_LINE$ for FTN77 GKS look alike
C
C ***************
C ****WARNING**** parameters must agree with those in DASHED_POLYLINE$
C ***************
C
      SUBROUTINE DASHED_LINE$(X1, Y1, X2, Y2, COLOUR_INDEX)
C
C ACTION : Map coordinates into 3:4 space then draw dashed/dotted line
C AUTHOR : W. G. Bardsley, University of Manchester, U.K.,16/12/93
C          HYPOT: hypotenuse = 5 in 3:4 space
C          NDIVS: no. divisions of HYPOT constitutes a unit of length
C          EPSI:  minimum size to detect a vertical line
C          NBITS: no. of pieces to make up a line segment
C          FDASH, FDOT, FSPACE: define lengths of dashes, dots and spaces
C          10/04/2000 special cases for vertical and horizontal lines
C          12/04/2010 new code using SET_LINE_STYLE
C          23/06/2017 tested for Linux 
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: COLOUR_INDEX 
      DOUBLE PRECISION, INTENT (IN) :: X1, Y1, X2, Y2
C
C Locals
C      
      INTEGER    COLOUR_LONG
      INTEGER    N1, N2, N3, N4, N6, NDIVS
      PARAMETER (N1 = 1, N2 = 2, N3 = 3, N4 = 4, N6 = 6, NDIVS = 60)
      INTEGER    I, NBITS
      INTEGER    LINE_TYPE, NUMRGB$ 
      INTEGER    IX1, IX2, IY1, IY2
      DOUBLE PRECISION DX1, DX2, DY1, DY2, X(2), Y(2)      
      DOUBLE PRECISION FRACX1, FRACX2, FRACY1, FRACY2
      DOUBLE PRECISION XDIFF1, XMAX01, XMIN01, YDIFF1, YMAX01, YMIN01
      DOUBLE PRECISION EPSI, HYPOT
      PARAMETER (EPSI = 1.0D-300, HYPOT = 5.0D+00)
      DOUBLE PRECISION FDASH, FDOT, FSPACE
      PARAMETER (FDASH = 0.4D+00, FDOT = 0.05D+00, FSPACE = 0.3D+00)
      DOUBLE PRECISION ZERO, TWO, THREE, FOUR
      PARAMETER (ZERO = 0.0D+00, TWO = 2.0D+00, THREE = 3.0D+00,
     +           FOUR = 4.0D+00)
      DOUBLE PRECISION DASH, DELTA, DOT, SPACE, XYDIST
      DOUBLE PRECISION A, B
      DOUBLE PRECISION XSTART, XSTOP, YSTART, YSTOP
      LOGICAL    WINDOWS, PS, STORE, USE_GDIPLUS
      PARAMETER (WINDOWS = .TRUE., STORE = .FALSE.)
      LOGICAL    DRAW, TYPE2, TYPE3
      LOGICAL    LINUX_OS, X_LINUX3
      LOGICAL    GDIPLUS, USE_WINDOWS
      EXTERNAL   GKSR2D$, GKSR2I$, PSLINE$, PSPOLY$, NUMRGB$,
     +           GETGKS_LGL, GETGKS_INT, GETGKS_CX1, GETGKS_FF1 
      EXTERNAL   DRAW_LINE_BETWEEN, DRAW_LINE_BETWEEN_D, SET_LINE_STYLE
      EXTERNAL   USE_GDIPLUS, X_LINUX3
      INTRINSIC  NINT, SQRT, ABS, MIN
C
C Check
C
      IF (ABS(X2 - X1).LE.EPSI .AND. ABS(Y2 - Y1).LE.EPSI) RETURN
C
C PS
C               
      CALL GETGKS_INT (N4, LINE_TYPE) 
      CALL GETGKS_LGL (N6,
     +                 PS) 
      IF (PS) THEN
         X(1) = X1
         X(2) = X2
         Y(1) = Y1
         Y(2) = Y2
         IF (LINE_TYPE.EQ.1) THEN
            CALL PSLINE$(COLOUR_INDEX, X1, Y1, X2, Y2)
         ELSEIF (LINE_TYPE.EQ.2) THEN
            CALL PSPOLY$(COLOUR_INDEX, N2, X, Y, 'da')
         ELSEIF (LINE_TYPE.EQ.3) THEN
            CALL PSPOLY$(COLOUR_INDEX, N2, X, Y, 'do')
         ELSEIF (LINE_TYPE.EQ.4) THEN
            CALL PSPOLY$(COLOUR_INDEX, N2, X, Y, 'dd')
         ENDIF
         RETURN
      ENDIF
C
C Initialise COLOUR_LONG and TYPE?
C
      GDIPLUS = USE_GDIPLUS(STORE)
      COLOUR_LONG = NUMRGB$(COLOUR_INDEX)
C
C Check whether to use windows or my technique
C
      USE_WINDOWS = WINDOWS
      LINUX_OS = X_LINUX3('*')
      IF (LINUX_OS) USE_WINDOWS = .NOT.USE_WINDOWS
             
      IF (USE_WINDOWS) THEN
         IF (LINE_TYPE.GE.N1 .AND. LINE_TYPE.LE.N4) THEN
            IF (LINE_TYPE.EQ.N1) THEN
               CALL SET_LINE_STYLE (N1)
            ELSEIF (LINE_TYPE.EQ.N2) THEN
               CALL SET_LINE_STYLE (N2)
            ELSEIF (LINE_TYPE.EQ.N3) THEN
               CALL SET_LINE_STYLE (N3)
            ELSE
               CALL SET_LINE_STYLE (N4)
            ENDIF
            IF (GDIPLUS) THEN
               CALL GKSR2D$(DX1, DY1, X1, Y1)
               CALL GKSR2D$(DX2, DY2, X2, Y2)
               CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                   COLOUR_LONG)
            ELSE
               CALL GKSR2I$(IX1, IY1, X1, Y1)
               CALL GKSR2I$(IX2, IY2, X2, Y2)
               CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                 COLOUR_LONG)
            ENDIF  
            CALL SET_LINE_STYLE (N1)
            RETURN    
         ENDIF
      ENDIF
      
      TYPE2 = .FALSE.
      TYPE3 = .FALSE.
C
C Find length of line segment in 3:4 space
C                                                  
      CALL GETGKS_CX1 (FRACX1, FRACX2, FRACY1, FRACY2)
      CALL GETGKS_FF1 (XDIFF1, XMAX01, XMIN01, YDIFF1, YMAX01, YMIN01)
      XSTART = FOUR*(FRACX2 - FRACX1)*(X1 - XMIN01)/XDIFF1
      XSTOP = FOUR*(FRACX2 - FRACX1)*(X2 - XMIN01)/XDIFF1
      YSTART = THREE*(FRACY2 - FRACY1)*(Y1 - YMIN01)/YDIFF1
      YSTOP = THREE*(FRACY2 - FRACY1)*(Y2 - YMIN01)/YDIFF1
      XYDIST = SQRT((XSTOP - XSTART)**2 + (YSTOP - YSTART)**2)
C
C Define lengths of dashes, dots, spaces in 3:4 space
C
      DELTA = HYPOT/NDIVS
      DASH = FDASH*DELTA
      SPACE = FSPACE*DELTA
      IF (LINE_TYPE.EQ.2) THEN
         TYPE2 = .TRUE.
         DOT = ZERO
         DELTA = DASH + SPACE
      ELSEIF (LINE_TYPE.EQ.3) THEN
         TYPE3 = .TRUE.
         DOT = FDOT*DELTA
         DELTA = DOT + SPACE
      ELSE
         DOT = FDOT*DELTA
         DELTA = DASH + DOT + TWO*SPACE
      ENDIF
      NBITS = NINT(XYDIST/DELTA)
      IF (NBITS.LT.1) RETURN

        
      IF (ABS(X2 - X1).GT.EPSI .AND. ABS(Y2 - Y1).GT.EPSI) THEN
C
C Draw a non vertical line segment
C
         XYDIST = ABS((X2 - X1)/NBITS)
         DASH = DASH*XYDIST/DELTA
         SPACE = SPACE*XYDIST/DELTA
         DOT = DOT*XYDIST/DELTA
         A = (Y2 - Y1)/(X2 - X1)
         B = Y1 - A*X1
         XSTART = MIN(X2, X1)
         XSTOP = XSTART
         YSTART = A*XSTART + B
         YSTOP = YSTART
         DRAW = .FALSE.
         DO I = 1, 2*NBITS
            DRAW = .NOT.DRAW
            IF (DRAW) THEN
               IF (GDIPLUS) THEN
                  IF (TYPE2) THEN
                     XSTOP = XSTOP + DASH
                     YSTOP = A*XSTOP + B
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                  ELSEIF (TYPE3) THEN
                     XSTOP = XSTOP + DOT
                     YSTOP = A*XSTOP + B
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                  ELSE
                     XSTOP = XSTOP + DASH
                     YSTOP = A*XSTOP + B
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                     XSTART = XSTOP + SPACE
                     YSTART = A*XSTART + B
                     XSTOP = XSTART + DOT
                     YSTOP = A*XSTOP + B
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                  ENDIF
               ELSE  
                  IF (TYPE2) THEN
                     XSTOP = XSTOP + DASH
                     YSTOP = A*XSTOP + B
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                  ELSEIF (TYPE3) THEN
                     XSTOP = XSTOP + DOT
                     YSTOP = A*XSTOP + B
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                  ELSE
                     XSTOP = XSTOP + DASH
                     YSTOP = A*XSTOP + B
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                     XSTART = XSTOP + SPACE
                     YSTART = A*XSTART + B
                     XSTOP = XSTART + DOT
                     YSTOP = A*XSTOP + B
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                  ENDIF 
               ENDIF
            ELSE
               XSTOP = XSTOP + SPACE
               YSTOP = A*XSTOP + B
            ENDIF
            XSTART = XSTOP
            YSTART = YSTOP
         ENDDO
      ELSEIF (ABS(X2 - X1).LE.EPSI) THEN
C
C Draw a vertical line segment
C
         XYDIST = ABS((Y2 - Y1)/NBITS)
         DASH = DASH*XYDIST/DELTA
         SPACE = SPACE*XYDIST/DELTA
         DOT = DOT*XYDIST/DELTA
         XSTART = (X1 + X2)/TWO
         XSTOP = XSTART
         YSTART = MIN(Y2, Y1)
         YSTOP = YSTART
         DRAW = .FALSE.
         DO I = 1, 2*NBITS
            DRAW = .NOT.DRAW
            IF (DRAW) THEN
               IF (GDIPLUS) THEN
                  IF (TYPE2) THEN
                     YSTOP = YSTART + DASH
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                  ELSEIF (TYPE3) THEN
                     YSTOP = YSTART + DOT
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                  ELSE
                     YSTOP = YSTART + DASH
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                     YSTART = YSTOP + SPACE
                     YSTOP = YSTART + DOT
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                  ENDIF 
               ELSE
                  IF (TYPE2) THEN
                     YSTOP = YSTART + DASH
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                  ELSEIF (TYPE3) THEN
                     YSTOP = YSTART + DOT
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                  ELSE
                     YSTOP = YSTART + DASH
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                     YSTART = YSTOP + SPACE
                     YSTOP = YSTART + DOT
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                  ENDIF
               ENDIF
            ELSE
               YSTOP = YSTART + SPACE
            ENDIF
            YSTART = YSTOP
         ENDDO
      ELSEIF (ABS(Y2 - Y1).LE.EPSI) THEN
C
C Draw a horizontal line segment
C
         XYDIST = ABS((X2 - X1)/NBITS)
         DASH = DASH*XYDIST/DELTA
         SPACE = SPACE*XYDIST/DELTA
         DOT = DOT*XYDIST/DELTA
         XSTART = MIN(X2,X1)
         XSTOP = XSTART
         YSTART = (Y1 + Y2)/TWO
         YSTOP = YSTART
         DRAW = .FALSE.
         DO I = 1, 2*NBITS
            DRAW = .NOT.DRAW
            IF (DRAW) THEN
               IF (GDIPLUS) THEN
                  IF (TYPE2) THEN
                     XSTOP = XSTART + DASH
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                  ELSEIF (TYPE3) THEN
                     XSTOP = XSTART + DOT
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                  ELSE
                     XSTOP = XSTART + DASH
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                     XSTART = XSTOP + SPACE
                     XSTOP = XSTART + DOT
                     CALL GKSR2D$(DX1, DY1, XSTART, YSTART)
                     CALL GKSR2D$(DX2, DY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN_D (DX1, DY1, DX2, DY2,
     +                                         COLOUR_LONG)
                  ENDIF
               ELSE
                  IF (TYPE2) THEN
                     XSTOP = XSTART + DASH
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                  ELSEIF (TYPE3) THEN
                     XSTOP = XSTART + DOT
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                  ELSE
                     XSTOP = XSTART + DASH
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                     XSTART = XSTOP + SPACE
                     XSTOP = XSTART + DOT
                     CALL GKSR2I$(IX1, IY1, XSTART, YSTART)
                     CALL GKSR2I$(IX2, IY2, XSTOP, YSTOP)
                     CALL DRAW_LINE_BETWEEN (IX1, IY1, IX2, IY2,
     +                                       COLOUR_LONG)
                  ENDIF
               ENDIF  
            ELSE
               XSTOP = XSTART + SPACE
            ENDIF
            XSTART = XSTOP
         ENDDO
      ENDIF
      END
C  
C------------------------------------------------
C
      SUBROUTINE POLYLINE$(N, X, Y, COLOUR_INDEX)
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: N, COLOUR_INDEX 
      DOUBLE PRECISION, INTENT (IN) :: X(N), Y(N)
C
C Local allocatable arrays
C       
      DOUBLE PRECISION, ALLOCATABLE :: XGRAF(:), YGRAF(:)                  
C
C Locals
C
      INTEGER    NGRAF
      INTEGER    I, ICOUNT, IERR, NTEMP
      INTEGER    LINE_TYPE 
      INTEGER    N4, N6
      PARAMETER (N4 = 4, N6 = 6)
      LOGICAL    PS
      EXTERNAL   PSPOLY$, SOLID_POLYLINE$, DASHED_POLYLINE$,
     +           ARROW_POLYLINE$, FILL_POLYGON$
      EXTERNAL   GETGKS_INT, GETGKS_LGL
      INTRINSIC  MIN  
      CALL GETGKS_INT (N4, LINE_TYPE)  
      CALL GETGKS_LGL (N6,
     +                 PS)  
      IF (LINE_TYPE.LT.1 .OR. LINE_TYPE.GT.9) THEN
C
C No line if LINE_TYPE.LT.1 OR .GT.9
C
         RETURN  
      ELSEIF (LINE_TYPE.GE.7 .AND. LINE_TYPE.LE.9) THEN
C
C Allocate workspace for line types 7, 8, and 9
C      
         NGRAF = N
         IERR = 0
         IF (ALLOCATED(XGRAF)) DEALLOCATE(XGRAF, STAT = IERR)
         IF (IERR.NE.0) RETURN   
         IF (ALLOCATED(YGRAF)) DEALLOCATE(YGRAF, STAT = IERR)
         IF (IERR.NE.0) RETURN   
         ALLOCATE(XGRAF(NGRAF + 5), STAT = IERR)
         IF (IERR.NE.0) RETURN 
         ALLOCATE(YGRAF(NGRAF + 5), STAT = IERR)
         IF (IERR.NE.0) RETURN
      ENDIF   
      IF (LINE_TYPE.LT.5) THEN
C
C Line is solid, dashed, dotted or dot/dashed if LINE_TYPE.LT.5
C
         IF (PS) THEN
             IF (LINE_TYPE.EQ.1) THEN
                CALL PSPOLY$(COLOUR_INDEX, N, X, Y, 'pl')
             ELSEIF (LINE_TYPE.EQ.2) THEN
                CALL PSPOLY$(COLOUR_INDEX, N, X, Y, 'da')
             ELSEIF (LINE_TYPE.EQ.3) THEN
                CALL PSPOLY$(COLOUR_INDEX, N, X, Y, 'do')
             ELSEIF (LINE_TYPE.EQ.4) THEN
                CALL PSPOLY$(COLOUR_INDEX, N, X, Y, 'dd')
            ENDIF
         ELSE
            IF (LINE_TYPE.EQ.1) THEN
               CALL SOLID_POLYLINE$(N, X, Y, COLOUR_INDEX)
            ELSEIF (LINE_TYPE.LT.5) THEN
               CALL DASHED_POLYLINE$(N, X, Y, COLOUR_INDEX)
            ENDIF
         ENDIF
      ELSEIF (LINE_TYPE.EQ.5) THEN
C
C Arrow type
C
         CALL SOLID_POLYLINE$(N, X, Y, COLOUR_INDEX)
         I = 1
         CALL ARROW_POLYLINE$(I, N, X, Y, COLOUR_INDEX)
      ELSEIF (LINE_TYPE.EQ.6) THEN
C
C Arrow type
C
         CALL SOLID_POLYLINE$(N, X, Y, COLOUR_INDEX)
         I = 2
         CALL ARROW_POLYLINE$(I, N, X, Y, COLOUR_INDEX)
      ELSEIF (LINE_TYPE.EQ.7) THEN
C
C Stepped line if LINE_TYPE.eq.7 (first y steps then x steps)
C
         ICOUNT = 0
         DO I = 1, N - 1
            ICOUNT = ICOUNT + 1
            XGRAF(ICOUNT) = X(I)
            YGRAF(ICOUNT) = Y(I)
            ICOUNT = ICOUNT + 1
            XGRAF(ICOUNT) = X(I)
            YGRAF(ICOUNT) = Y(I + 1)
            IF (ICOUNT.GE.NGRAF) THEN
               IF (PS) THEN
                  CALL PSPOLY$(COLOUR_INDEX, ICOUNT, XGRAF, YGRAF, 'pl')
               ELSE
                  CALL SOLID_POLYLINE$(ICOUNT, XGRAF, YGRAF,
     +                                 COLOUR_INDEX)
               ENDIF
               XGRAF(1) = XGRAF(ICOUNT)
               YGRAF(1) = YGRAF(ICOUNT)
               ICOUNT = 1
            ENDIF
         ENDDO
         IF (ICOUNT.NE.0) THEN
            ICOUNT = ICOUNT + 1
            XGRAF(ICOUNT) = X(N)
            YGRAF(ICOUNT) = Y(N)
            IF (PS) THEN
               CALL PSPOLY$(COLOUR_INDEX, ICOUNT, XGRAF, YGRAF, 'pl')
            ELSE
               CALL SOLID_POLYLINE$(ICOUNT, XGRAF, YGRAF,
     +                              COLOUR_INDEX)
            ENDIF
         ENDIF
      ELSEIF (LINE_TYPE.EQ.8) THEN
C
C Stepped line if LINE_TYPE.eq.8 (First x steps then y steps)
C
         ICOUNT = 0
         DO I = 1, N - 1
            ICOUNT = ICOUNT + 1
            XGRAF(ICOUNT) = X(I)
            YGRAF(ICOUNT) = Y(I)
            ICOUNT = ICOUNT + 1
            XGRAF(ICOUNT) = X(I + 1)
            YGRAF(ICOUNT) = Y(I)
            IF (ICOUNT.GE.NGRAF) THEN
               IF (PS) THEN
                  CALL PSPOLY$(COLOUR_INDEX, ICOUNT, XGRAF, YGRAF, 'pl')
               ELSE
                  CALL SOLID_POLYLINE$(ICOUNT, XGRAF, YGRAF,
     +                                 COLOUR_INDEX)
               ENDIF
               XGRAF(1) = XGRAF(ICOUNT)
               YGRAF(1) = YGRAF(ICOUNT)
               ICOUNT = 1
            ENDIF
         ENDDO
         IF (ICOUNT.NE.0) THEN
            ICOUNT = ICOUNT + 1
            XGRAF(ICOUNT) = X(N)
            YGRAF(ICOUNT) = Y(N)
            IF (PS) THEN
               CALL PSPOLY$(COLOUR_INDEX, ICOUNT, XGRAF, YGRAF, 'pl')
            ELSE
               CALL SOLID_POLYLINE$(ICOUNT, XGRAF, YGRAF,
     +                              COLOUR_INDEX)
            ENDIF
         ENDIF
      ELSEIF (LINE_TYPE.EQ.9) THEN
C
C Filled polygon
C
         NTEMP = MIN(N,NGRAF) + 1
         DO I = 1, NTEMP - 1
            XGRAF(I) = X(I)
            YGRAF(I) = Y(I)
         ENDDO
         XGRAF(NTEMP) = X(1)
         YGRAF(NTEMP) = Y(1)
         IF (PS) THEN
            CALL PSPOLY$(COLOUR_INDEX, NTEMP, XGRAF, YGRAF, 'pl')
         ELSE
            CALL SOLID_POLYLINE$(NTEMP, XGRAF, YGRAF, COLOUR_INDEX)
         ENDIF
         CALL FILL_POLYGON$(NTEMP, XGRAF, YGRAF, COLOUR_INDEX)
      ENDIF  
      IF (LINE_TYPE.GE.7 .AND. LINE_TYPE.LE.9) THEN
C
C Deallocate workspaces
C      
         DEALLOCATE(XGRAF, STAT = IERR)
         DEALLOCATE(YGRAF, STAT = IERR)
      ENDIF   
      END
C     
C------------------------------------------------------
C
      SUBROUTINE SOLID_POLYLINE$(N, X, Y, COLOUR_INDEX)
C      
C     07/07/2006 introduced allocatable arrays ix and iy
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: N, COLOUR_INDEX
      DOUBLE PRECISION, INTENT (IN) :: X(N), Y(N)
C
C Local allocatable arrays
C                         
      INTEGER,          ALLOCATABLE :: IX(:), IY(:)
      DOUBLE PRECISION, ALLOCATABLE :: DX(:), DY(:)
C
C Locals
C      
      INTEGER    COLOUR_LONG, IERR
      INTEGER    NUMRGB$
      INTEGER    N1, N6
      PARAMETER (N1 = 1, N6 = 6)
      LOGICAL    PS
      LOGICAL    STORE, USE_GDIPLUS
      PARAMETER (STORE = .FALSE.)
      EXTERNAL   GKSA2D$, GKSA2I$, PSPOLY$, NUMRGB$ 
      EXTERNAL   GETGKS_LGL
      EXTERNAL   DRAW_POLYLINE, DRAW_POLYLINE_D, USE_GDIPLUS,
     +           SET_LINE_STYLE  
      IF (N.LT.1) RETURN   
      CALL GETGKS_LGL (N6,
     +                 PS) 
      IF (PS) THEN
         CALL PSPOLY$(COLOUR_INDEX, N, X, Y, 'pl')
      ELSEIF (USE_GDIPLUS(STORE)) THEN       
         IERR = 0
         IF (ALLOCATED(DX)) DEALLOCATE(DX, STAT = IERR)
         IF (IERR.NE.0) RETURN
         IF (ALLOCATED(DY)) DEALLOCATE(DY, STAT = IERR)
         IF (IERR.NE.0) RETURN   
         ALLOCATE(DX(N), STAT = IERR)
         IF (IERR.NE.0) RETURN  
         ALLOCATE(DY(N), STAT = IERR)
         IF (IERR.NE.0) RETURN
         COLOUR_LONG = NUMRGB$(COLOUR_INDEX)
         CALL GKSA2D$(N, DX, DY, X, Y)
         CALL SET_LINE_STYLE (N1)
         CALL DRAW_POLYLINE_D (DX, DY, N, COLOUR_LONG)
         DEALLOCATE(DX, STAT = IERR)
         DEALLOCATE(DY, STAT = IERR)
      ELSE   
         IERR = 0
         IF (ALLOCATED(IX)) DEALLOCATE(IX, STAT = IERR)
         IF (IERR.NE.0) RETURN
         IF (ALLOCATED(IY)) DEALLOCATE(IY, STAT = IERR)
         IF (IERR.NE.0) RETURN   
         ALLOCATE(IX(N), STAT = IERR)
         IF (IERR.NE.0) RETURN  
         ALLOCATE(IY(N), STAT = IERR)
         IF (IERR.NE.0) RETURN
         COLOUR_LONG = NUMRGB$(COLOUR_INDEX)
         CALL GKSA2I$(IX, IY, N, X, Y)
         CALL SET_LINE_STYLE (N1)
         CALL DRAW_POLYLINE (IX, IY, N, COLOUR_LONG)
         DEALLOCATE(IX, STAT = IERR)
         DEALLOCATE(IY, STAT = IERR)    
      ENDIF
      END
C   
C-----------------------------------------------------------------
C
C DASHED_POLYLINE$ for FTN77 GKS look alike
C
C ***************
C ****WARNING**** parameters must agree with those in DASHED_LINE$
C ***************
C
      SUBROUTINE DASHED_POLYLINE$(N, X, Y, COLOUR_INDEX)
C
C ACTION : Map coordinates into 3:4 space then draw dashed/dotted polyline
C AUTHOR : W. G. Bardsley, University of Manchester, U.K.,16/12/93
C          HYPOT: hypotenuse = 5 in 3:4 space
C          NDIVS: no. divisions of HYPOT constitutes a unit of length
C          EPSI:  minimum size to detect a vertical line
C          NBITS: no. of pieces to make up a line segment
C          FDASH, FDOT, FSPACE: define lengths of dashes, dots and spaces
C          10/04/2000 Altered mechanism to accumulate to avoid singular
C                     behaviour with cdf plots from KS 1-sample testing
C          07/02/2001 restored original mechanism
C          12/04/2010 introduced SET_LINE_STYLE for use with dash/dot cases when line thickness = 1  
C          23/06/2017 checked for Linux 
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: N, COLOUR_INDEX 
      DOUBLE PRECISION, INTENT (IN) :: X(N), Y(N)
C
C Allocatable
C
      INTEGER,          ALLOCATABLE :: IX(:), IY(:)
      DOUBLE PRECISION, ALLOCATABLE :: DX(:), DY(:)  
C
C Locals
C       
      INTEGER    N1, N2, N3, N4, N6, NDIVS
      PARAMETER (N1 = 1, N2 = 2, N3 = 3, N4 = 4, N6 = 6, NDIVS = 60)
      INTEGER    I, IERR, LINE_TYPE 
      INTEGER    COLOUR_LONG, NUMRGB$
      DOUBLE PRECISION FRACX1, FRACX2, FRACY1, FRACY2
      DOUBLE PRECISION XDIFF1, XMAX01, XMIN01, YDIFF1, YMAX01, YMIN01
      DOUBLE PRECISION HYPOT
      PARAMETER (HYPOT = 5.0D+00)
      DOUBLE PRECISION FDASH, FDOT, FSPACE
      PARAMETER (FDASH = 0.4D+00, FDOT = 0.05D+00, FSPACE = 0.3D+00)
      DOUBLE PRECISION ZERO, TWO, THREE, FOUR
      PARAMETER (ZERO = 0.0D+00, TWO = 2.0D+00, THREE = 3.0D+00,
     +           FOUR = 4.0D+00)
      DOUBLE PRECISION DASH, DELTA, DOT, SPACE, XYDIST
      DOUBLE PRECISION A, B, XSAV, YSAV
      DOUBLE PRECISION XSTART, XSTOP, YSTART, YSTOP
      LOGICAL    WINDOWS, PS, STORE
      PARAMETER (WINDOWS = .TRUE., STORE = .FALSE.)
      LOGICAL    GDIPLUS, USE_GDIPLUS, USE_WINDOWS
      LOGICAL    LINUX_OS, X_LINUX3
      EXTERNAL   DASHED_LINE$, PSPOLY$, NUMRGB$, GKSA2D$, GKSA2I$,
     +           USE_GDIPLUS, DRAW_POLYLINE, DRAW_POLYLINE_D,
     +           SET_LINE_STYLE, X_LINUX3       
      EXTERNAL   GETGKS_INT, GETGKS_LGL, GETGKS_CX1, GETGKS_FF1
      INTRINSIC  SQRT
      
      CALL GETGKS_INT (N4, LINE_TYPE)
      IF (LINE_TYPE.LT.1) RETURN     
      CALL GETGKS_LGL (N6,
     +                 PS) 
      IF (PS) THEN
          IF (LINE_TYPE.EQ.1) THEN
             CALL PSPOLY$(COLOUR_INDEX, N, X, Y, 'pl')
          ELSEIF (LINE_TYPE.EQ.2) THEN
             CALL PSPOLY$(COLOUR_INDEX, N, X, Y, 'da')
          ELSEIF (LINE_TYPE.EQ.3) THEN
             CALL PSPOLY$(COLOUR_INDEX, N, X, Y, 'do')
          ELSEIF (LINE_TYPE.EQ.4) THEN
             CALL PSPOLY$(COLOUR_INDEX, N, X, Y, 'dd')
         ENDIF
         RETURN
      ENDIF
      
      GDIPLUS = USE_GDIPLUS(STORE)
            
      IF (GDIPLUS) THEN
         IERR = 0
         IF (ALLOCATED(DX)) DEALLOCATE(DX, STAT = IERR)
         IF (IERR.NE.0) RETURN
         IF (ALLOCATED(DY)) DEALLOCATE(DY, STAT = IERR)
         IF (IERR.NE.0) RETURN   
         ALLOCATE(DX(N), STAT = IERR)
         IF (IERR.NE.0) RETURN  
         ALLOCATE(DY(N), STAT = IERR)
         IF (IERR.NE.0) RETURN
      ELSE
         IERR = 0
         IF (ALLOCATED(IX)) DEALLOCATE(IX, STAT = IERR)
         IF (IERR.NE.0) RETURN
         IF (ALLOCATED(IY)) DEALLOCATE(IY, STAT = IERR)
         IF (IERR.NE.0) RETURN   
         ALLOCATE(IX(N), STAT = IERR)
         IF (IERR.NE.0) RETURN  
         ALLOCATE(IY(N), STAT = IERR)
         IF (IERR.NE.0) RETURN     
      ENDIF        

      USE_WINDOWS = WINDOWS
      LINUX_OS = X_LINUX3('*')
      IF (LINUX_OS) USE_WINDOWS = .NOT.USE_WINDOWS 
      
      IF (USE_WINDOWS) THEN
         IF (LINE_TYPE.GE.N1 .AND. LINE_TYPE.LE.N4) THEN
            IF (LINE_TYPE.EQ.N1) THEN
               CALL SET_LINE_STYLE (N1)
            ELSEIF (LINE_TYPE.EQ.N2) THEN
               CALL SET_LINE_STYLE (N2)
            ELSEIF (LINE_TYPE.EQ.N3) THEN
               CALL SET_LINE_STYLE (N3)
            ELSE
               CALL SET_LINE_STYLE (N4)
            ENDIF  
            COLOUR_LONG = NUMRGB$(COLOUR_INDEX)
            IF (GDIPLUS) THEN
               CALL GKSA2D$(N, DX, DY, X, Y)
               CALL DRAW_POLYLINE_D (DX, DY, N, COLOUR_LONG)
               DEALLOCATE(DX, STAT = IERR)
               DEALLOCATE(DY, STAT = IERR)
               CALL SET_LINE_STYLE (N1)
               RETURN
            ELSE  
               CALL GKSA2I$(IX, IY, N, X, Y)
               CALL DRAW_POLYLINE (IX, IY, N, COLOUR_LONG)
               DEALLOCATE(IX, STAT = IERR)
               DEALLOCATE(IY, STAT = IERR)
               CALL SET_LINE_STYLE (N1)
               RETURN 
            ENDIF       
         ENDIF
      ENDIF
C
C Define lengths of dashes, dots, spaces in 3:4 space
C                          
      CALL GETGKS_CX1 (FRACX1, FRACX2, FRACY1, FRACY2)
      CALL GETGKS_FF1 (XDIFF1, XMAX01, XMIN01, YDIFF1, YMAX01, YMIN01) 
      DELTA = HYPOT/NDIVS
      DASH = FDASH*DELTA
      SPACE = FSPACE*DELTA
      IF (LINE_TYPE.EQ.2) THEN
         DOT = ZERO
         DELTA = DASH + SPACE
      ELSEIF (LINE_TYPE.EQ.3) THEN
         DOT = FDOT*DELTA
         DELTA = DOT + SPACE
      ELSE
         DOT = FDOT*DELTA
         DELTA = DASH + DOT + TWO*SPACE
      ENDIF
      A = FOUR*(FRACX2 - FRACX1)/XDIFF1
      B = THREE*(FRACY2 - FRACY1)/YDIFF1
      XSAV = X(1)
      YSAV = Y(1)
      XSTART = A*(XSAV - XMIN01)
      YSTART = B*(YSAV - YMIN01)
C
C Find length of line segment in 3:4 space and draw if long enough
C
      DO I = 2, N
         XSTOP = A*(X(I) - XMIN01)
         YSTOP = B*(Y(I) - YMIN01)
         XYDIST = SQRT((XSTOP - XSTART)**2 + (YSTOP - YSTART)**2)
         IF (XYDIST.GE.DELTA) THEN
C
C Draw and always re-set parameters
C
            CALL DASHED_LINE$(XSAV, YSAV, X(I), Y(I), COLOUR_INDEX)
            XSAV = X(I)
            YSAV = Y(I)
            XSTART = XSTOP
            YSTART = YSTOP
C********ELSEIF (N.LT.80) THEN
C
C Accumulate if n >= 80 otherwise always re-set parameters
C
C***********XSAV = X(I)
C***********YSAV = Y(I)
C***********XSTART = XSTOP
C***********YSTART = YSTOP
         ENDIF
      ENDDO
      END
C       
C-----------------------------------------------------------------
C
C ARROW_POLYLINE$ for FTN77 GKS look alike
C
C ***************
C ****WARNING**** parameters must agree with those in DASHED_LINE$
C ***************
C
      SUBROUTINE ARROW_POLYLINE$(ISEND, N, X, Y, COLOUR_INDEX)
C
C ACTION : Map coordinates into 3:4 space then draw arrows
C AUTHOR : W. G. Bardsley, University of Manchester, U.K.,16/12/93
C          HYPOT: hypotenuse = 5 in 3:4 space
C          NDIVS: no. divisions of HYPOT constitutes a unit of length
C          EPSI:  minimum size to detect a vertical line
C          NBITS: no. of pieces to make up a line segment
C          FDASH, FDOT, FSPACE: define lengths of dashes, dots and spaces
C          JSEND = 11 in call to WGB_ARROW to get vector arropw head
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: ISEND, N, COLOUR_INDEX  
      DOUBLE PRECISION, INTENT (IN) :: X(N), Y(N)
C
C Locals
C      
      INTEGER    JSEND, L, NDIVS, N0, N1, N4
      PARAMETER (JSEND = 11, L = 15, NDIVS = 60, N0 = 0, N1 = 1, N4 = 4)
      INTEGER    I, ISAV  
      INTEGER    I_GSELNT$, LINE_TYPE
      DOUBLE PRECISION FRACX1, FRACX2, FRACY1, FRACY2
      DOUBLE PRECISION XDIFF1, XMAX01, XMIN01, YDIFF1, YMAX01, YMIN01
      DOUBLE PRECISION HYPOT
      PARAMETER (HYPOT = 5.0D+00)
      DOUBLE PRECISION FDASH, FDOT, FSPACE, HEAD
      PARAMETER (FDASH = 0.4D+00, FDOT = 0.05D+00, FSPACE = 0.3D+00,
     +           HEAD = 0.01D+00)
      DOUBLE PRECISION TWO, THREE, FOUR
      PARAMETER (TWO = 2.0D+00, THREE = 3.0D+00, FOUR = 4.0D+00)
      DOUBLE PRECISION DASH, DELTA, DOT, SPACE, XYDIST
      DOUBLE PRECISION X1, X2, Y1, Y2
      DOUBLE PRECISION A, B, XSAV, YSAV
      DOUBLE PRECISION XSTART, XSTOP, YSTART, YSTOP
      EXTERNAL   GKSD2D$, WGB_ARROW$, GSELNT$ 
      EXTERNAL   GETGKS_CX1, GETGKS_FF1, GETGKS_INT, GETGKS_TRN
      INTRINSIC  SQRT  
      CALL GETGKS_INT (N4, LINE_TYPE)
      IF (LINE_TYPE.LT.1) RETURN
C
C Save current transformation
C                      
      CALL GETGKS_TRN (N1, I_GSELNT$)
      ISAV = I_GSELNT$
C
C Define lengths of dashes, dots, spaces in 3:4 space
C                                   
      CALL GETGKS_CX1 (FRACX1, FRACX2, FRACY1, FRACY2)
      CALL GETGKS_FF1 (XDIFF1, XMAX01, XMIN01, YDIFF1, YMAX01, YMIN01) 
      DELTA = HYPOT/NDIVS
      DASH = FDASH*DELTA
      SPACE = FSPACE*DELTA
      DOT = FDOT*DELTA
      DELTA = THREE*(DASH + DOT + TWO*SPACE)/TWO
      A = FOUR*(FRACX2 - FRACX1)/XDIFF1
      B = THREE*(FRACY2 - FRACY1)/YDIFF1
      XSAV = X(1)
      YSAV = Y(1)
      XSTART = A*(XSAV - XMIN01)
      YSTART = B*(YSAV - YMIN01)
C
C Find length of line segment in 3:4 space and draw if long enough
C
      DO I = 2, N
         XSTOP = A*(X(I) - XMIN01)
         YSTOP = B*(Y(I) - YMIN01)
         XYDIST = SQRT((XSTOP - XSTART)**2 + (YSTOP - YSTART)**2)
         IF (XYDIST.GE.DELTA) THEN
            IF (ISEND.EQ.1) THEN
               CALL GKSD2D$(X(I), X1, Y(I), Y1)
               CALL GKSD2D$(XSAV, X2, YSAV, Y2)
            ELSE
               CALL GKSD2D$(XSAV, X1, YSAV, Y1)
               CALL GKSD2D$(X(I), X2, Y(I), Y2)
            ENDIF
            CALL GSELNT$(N0)
            CALL WGB_ARROW$(JSEND, COLOUR_INDEX, L, HEAD, X1, X2, Y1,
     +                      Y2)
            CALL GSELNT$(ISAV)
            XSTART = XSTOP
            YSTART = YSTOP
         ENDIF
         XSAV = X(I)
         YSAV = Y(I)
      ENDDO
      END
C     
C------------------------------------------------------
C
      SUBROUTINE MITER_POLYLINE$(N, X, Y, COLOUR_INDEX)
C      
C 25/08/2014 derived from SOLID_POLYLINE$
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: N, COLOUR_INDEX
      DOUBLE PRECISION, INTENT (IN) :: X(N), Y(N)
C
C Local allocatable arrays
C                         
      INTEGER,          ALLOCATABLE :: IX(:), IY(:)
      DOUBLE PRECISION, ALLOCATABLE :: DX(:), DY(:)
C
C Locals
C      
      INTEGER    COLOUR_LONG, IERR
      INTEGER    NUMRGB$
      INTEGER    N0, N1, N6
      PARAMETER (N0 = 0, N1 = 1, N6 = 6)
      LOGICAL    PS
      LOGICAL    STORE, USE_GDIPLUS
      EXTERNAL   GKSA2D$, GKSA2I$, PSPOLY$, NUMRGB$ 
      EXTERNAL   GETGKS_LGL
      EXTERNAL   DRAW_POLYLINE, DRAW_POLYLINE_D, USE_GDIPLUS,
     +           SET_LINE_STYLE  
      EXTERNAL   PSJOIN$, SET_LINE_JOIN
      IF (N.LT.1) RETURN   
      CALL GETGKS_LGL (N6,
     +                 PS) 
      IF (PS) THEN
         CALL PSJOIN$(N0)
         CALL PSPOLY$(COLOUR_INDEX, N, X, Y, 'pl')
         CALL PSJOIN$(N1)
         RETURN
      ENDIF   
      STORE = .TRUE.
      CALL SET_LINE_JOIN (N0,
     +                    STORE)
      STORE = .FALSE.
      IF (USE_GDIPLUS(STORE)) THEN       
         IERR = 0
         IF (ALLOCATED(DX)) DEALLOCATE(DX, STAT = IERR)
         IF (IERR.NE.0) RETURN
         IF (ALLOCATED(DY)) DEALLOCATE(DY, STAT = IERR)
         IF (IERR.NE.0) RETURN   
         ALLOCATE(DX(N), STAT = IERR)
         IF (IERR.NE.0) RETURN  
         ALLOCATE(DY(N), STAT = IERR)
         IF (IERR.NE.0) RETURN
         COLOUR_LONG = NUMRGB$(COLOUR_INDEX)
         CALL GKSA2D$(N, DX, DY, X, Y)
         CALL SET_LINE_STYLE (N1)
         CALL DRAW_POLYLINE_D (DX, DY, N, COLOUR_LONG)
         DEALLOCATE(DX, STAT = IERR)
         DEALLOCATE(DY, STAT = IERR)
      ELSE   
         IERR = 0
         IF (ALLOCATED(IX)) DEALLOCATE(IX, STAT = IERR)
         IF (IERR.NE.0) RETURN
         IF (ALLOCATED(IY)) DEALLOCATE(IY, STAT = IERR)
         IF (IERR.NE.0) RETURN   
         ALLOCATE(IX(N), STAT = IERR)
         IF (IERR.NE.0) RETURN  
         ALLOCATE(IY(N), STAT = IERR)
         IF (IERR.NE.0) RETURN
         COLOUR_LONG = NUMRGB$(COLOUR_INDEX)
         CALL GKSA2I$(IX, IY, N, X, Y)
         CALL SET_LINE_STYLE (N1)
         CALL DRAW_POLYLINE (IX, IY, N, COLOUR_LONG)
         DEALLOCATE(IX, STAT = IERR)
         DEALLOCATE(IY, STAT = IERR)    
      ENDIF
      STORE = .TRUE.
      CALL SET_LINE_JOIN (N1,
     +                    STORE)
      END
C
C----------------------------------------------
C
      SUBROUTINE TEXT$(X, Y, STR, COLOUR_INDEX)
C
C ACTION: draw a text string
C AUTHOR: W.G.Bardsley, University of Manchester, U.K.
C         17/11/2003 removed restriction that 32 =< characters =< 126
C         25/08/2009 added calls to LEN200 and TRIML1 
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,             INTENT (IN) :: COLOUR_INDEX 
      DOUBLE PRECISION,    INTENT (IN) :: X, Y
      CHARACTER (LEN = *), INTENT (IN) :: STR
C
C Locals
C      
      INTEGER    NUMRGB$
      INTEGER    IX, IY, L2, LEN200
      INTEGER    COLOUR_LONG   
      INTEGER    L1, N6
      PARAMETER (L1 = 1, N6 = 6)
      CHARACTER  COPY*80
      CHARACTER  BLANK*1
      PARAMETER (BLANK = ' ')
      LOGICAL    PS
      EXTERNAL   GKSR2I$, PSTEXT$, NUMRGB$, SLASHB$
      EXTERNAL   GETGKS_LGL, LEN200, TRIML1
      EXTERNAL   DRAW_CHARACTERS
      IF (STR.EQ.BLANK) RETURN
      CALL GETGKS_LGL (N6,
     +                 PS)  
      COPY = STR
      CALL TRIML1 (COPY)
      L2 = LEN200 (COPY)
      IF (PS) THEN
         CALL PSTEXT$(COLOUR_INDEX, X, Y, COPY(L1:L2), 'tl')
      ELSE
         CALL SLASHB$(COPY)
         CALL GKSR2I$(IX, IY, X, Y)
         COLOUR_LONG = NUMRGB$(COLOUR_INDEX)
         CALL DRAW_CHARACTERS (COPY(L1:L2), IX, IY, COLOUR_LONG)
      ENDIF
      END
C
C
