c
c-------------------------------------------------------------
c Collected subroutines to plot a pdf on a sample histogram. 
c Also contains a call to help_rannum to allow allow rannum
c and related code to be run from the simfit.dll without
c making the simfit dll link to the clearwin dll. 
c-------------------------------------------------------------
c rannum_help
c pdfsam ... prepare a random sample 
c pdfval ... prepare the corresponding pdf values 
c pdfplt ... display a simplified interface to pdplot
c pdplot ... display a pdf on sample histogram 
c-------------------------------------------------------------
c

c
c-------------------------------------------------------------
c
      subroutine rannum_help
      implicit none
      external help_rannum
      call help_rannum ('rannum')
      end
c
c-------------------------------------------------------------
c
      subroutine pdfsam (i, na, ndist, ni, nsamp, 
     +                   a, sample,
     +                   title)
c
c action: submit a sample and pdf to pdfplt
c author: w.g.bardsley, university of manchester, u.k., 11/03/2019
c
c      i: integer parameters
c     na: dimension of a
c  ndist: distribution 1 to 12
c  nsamp: sample size 
c      a: double precision parameters
c sample: random sample
c  title: name of distribution
c     
c
c ndist = 1: Cauchy
c ndist = 2: Chi-square
c ndist = 3: Negative Exponential
c ndist = 4: Gamma
c ndist = 5: Logistic
c ndist = 6: Lognormal
c ndist = 7: Normal
c ndist = 8: Uniform(A, B)
c ndist = 9: Weibull
c ndist = 10: F
c ndist = 11: t
c ndist - 12: Beta
c
      implicit none
c
c arguments
c      
      integer,             intent (in) :: na, ndist, ni, nsamp
      integer,             intent (in) :: i(ni) 
      double precision,    intent (in) :: a(na), sample(nsamp) 
      character (len = *), intent (in) :: title
c
c local
c      
      integer    loop
      integer    npdf
      parameter (npdf = 120)
      double precision delta, pdf(npdf), t(npdf), 
     +                 tmax, tmin 
      character (len = 100) line
      logical    abort
      external   putfat, putadv
      external   pdfplt, pdfval
      intrinsic  dble
c
c check input
c      
      if (na.lt.2 .or. ni.lt.2) then
         call putfat ('NA < 2  or NI < 2 in call to PDFSAM')
         return
      endif
      if (nsamp.lt.10) then
         call putfat ('NSAMP < 10 in call to PDFSAM')
         return
      endif
      if (ndist.lt.1 .or. ndist.gt.12) then 
         write (line,100) ndist
         call putadv (line)
         return
      endif
c
c get the sample range then prepare the t values
c      
      tmax = -1.0d-250
      tmin = 1.0d+250
      do loop = 1, nsamp
         if (sample(loop).gt.tmax) tmax = sample(loop)
         if (sample(loop).lt.tmin) tmin = sample(loop)
      enddo   
      delta = (tmax - tmin)/dble(npdf - 1)  
      t(1) = tmin
      do loop = 2, npdf - 1
         t(loop) = t(loop - 1) + delta
      enddo
      t(npdf) = tmax  
c
c now prepare the pdf values
c      
      call pdfval (i, na, ndist, ni, npdf, 
     +             a, pdf, t,
     +             abort)
      if (abort) then
         write (line,200) ndist
         call putfat (line)
         return
      endif   
c
c finally plot the results
c      
      call pdfplt (npdf, nsamp,
     +             pdf, t, sample,
     +             title)     
c
c format statement
c
  100 format ('Distribution',i3,' is not supported by PDFSAM')  
  200 format ('Error calculating with distribution',i3,' in PDFVAL')
      end
c
c--------------------------------------------------------------       
c
      subroutine pdfval (i, na, ndist, ni, npdf, 
     +           a, pdf, t,
     +           abort)
c
c action: return pdf values
c author: w.g.bardsley, university of manchester, u.k., 12/03/2019
c 
c
c ndist = 1: Cauchy
c ndist = 2: Chi-square
c ndist = 3: Negative Exponential
c ndist = 4: Gamma
c ndist = 5: Logistic
c ndist = 6: Lognormal
c ndist = 7: Normal
c ndist = 8: Uniform(A, B)
c ndist = 9: Weibull
c ndist = 10: F
c ndist = 11: t
c ndist = 12: Beta
c    
      implicit none   
c
c arguments
c                  
      integer,          intent (in)  :: na, ndist, ni, npdf
      integer,          intent (in)  :: i(ni)
      double precision, intent (in)  :: a(na)
      double precision, intent (in)  :: t(npdf)
      double precision, intent (out) :: pdf(npdf)
      logical,          intent (out) :: abort 
c
c locals
c       
      integer ifail, loop
      double precision arg, dummy, dn, factor, pi, temp1, temp2, temp3
      double precision x01aafg, s14abfg
      double precision epsi
      parameter (epsi = 1.0d-07)
      double precision one, two
      parameter (one = 1.0d+00, two = 2.0d+00)
      external   x01aafg, s14abfg
      intrinsic  abs, dble, exp, sqrt, log 
c
c initialise abort then check data supplied
c      
      abort = .true.
      if (na.lt.2 .or. ni.lt.2 .or. npdf.lt.2 .or. ndist.lt.1 .or.
     +    ndist.gt.12) return
c
c calculate factors required
c     
      ifail = 0
      if (ndist.eq.1) then
c ndist = 1: Cauchy        
         if (abs(a(2)).lt.epsi) return
         pi = x01aafg(dummy)
      elseif (ndist.eq.2) then
c ndist = 2: Chi-square      
         if (i(1).lt.2) return
         temp1 = dble(i(1)/two)            !n/2
         temp2 = temp1 - one               !n/2 - 1
         temp3 = exp(s14abfg(temp1, ifail))!(n/2 - 1) factorial = Gamma(n)
         factor = (two**temp1)*temp3  
      elseif (ndist.eq.3) then
c ndist = 3: Negative Exponential      
         factor = one/a(1)          !2^(n/2)*(n/2 - 1)!
      elseif (ndist.eq.4) then
c ndist = 4: Gamma      
         temp1 = exp(s14abfg(a(1), ifail))
         temp2 = a(2)**a(1)
         temp3 = a(1) - one
         factor = one/(temp2*temp1)  
      elseif (ndist.eq.6) then
c ndist = 6: Lognormal      
         if (a(2).lt.epsi) return
         pi = x01aafg(dummy)
         temp1 = a(2)*sqrt(two*pi)
         temp2 = two*a(2)**2  
      elseif (ndist.eq.7) then
c ndist = 7: Normal      
         if (a(2).lt.epsi) return
         pi = x01aafg(dummy)
         arg = a(2)*sqrt(two*pi)
         factor = one/arg
      elseif (ndist.eq.8) then
c ndist = 8: Uniform      
         factor = one/(a(2) - a(1)) 
      elseif (ndist.eq.9) then
c ndist = 9: Weibull      
         temp1 = a(1)/a(2)     
         temp2 = a(1) - one
      elseif (ndist.eq.10) then
c ndist = 10: F      
         arg = (dble(i(1) + i(2)) - two)/two + one
         temp1 = s14abfg(arg, ifail)                              !log{(m + n - 2)/2}! 
         arg = dble(i(1))/two     
         temp2 = s14abfg(arg, ifail)                              !log(m/2 - 1)!
         arg = dble(i(2))/two
         temp3 = s14abfg(arg, ifail)                              !log(n/2 - 1)!
         arg = dble(i(1))/dble(i(2))                              !m/n
         factor = arg**(dble(i(1))/two)*exp(temp1 - temp2 - temp3)!(m/n)^(m/2)*((m + n - 2)/2)!/(m/2 - 1)!(n/2 - 1)!
         temp1 = dble(i(1))/two - one                             !(m/2 - 1)
         temp2 = dble(i(1) + i(2))/two                            !(m + n)/2 
         temp3 = dble(i(1))/dble(i(2))                            !m/n 
      elseif (ndist.eq.11) then
c ndist = 11: t      
         dn = dble(i(1))
         arg = (dn - one)/two + one
         temp1 = s14abfg(arg, ifail)
         arg = dn/two
         temp2 = s14abfg(arg, ifail)
         arg = exp(temp1 - temp2)
         temp3 = sqrt(x01aafg(dummy)*dn)
         factor = arg/temp3
         temp1 = -(dn + one)/two
      elseif (ndist.eq.12) then
c ndist = 12: Beta      
         arg = a(1) + a(2)
         temp1 = s14abfg(arg, ifail)
         arg = a(1)
         temp2 = s14abfg(arg, ifail)
         arg = a(2)
         temp3 = s14abfg(arg, ifail)
         factor = exp(temp1 - temp2 - temp3)
         temp1 = a(1) - one
         temp2 = a(2) - one    
      endif 
c
c calculate pdf values
c      
      do loop = 1, npdf
         if (ndist.eq.1) then
c ndist = 1: Cauchy           
            factor = a(2)*pi*(one + ((t(loop) - a(1))/a(2))**2)
            pdf(loop) = one/factor
         elseif (ndist.eq.2) then
c ndist = 2: Chi-square         
            pdf(loop) =  t(loop)**temp2*exp(-t(loop)/two)/factor
         elseif (ndist.eq.3) then
c ndist = 3: Negative Exponential         
            pdf(loop) = factor*exp(-t(loop)/a(1))  
         elseif (ndist.eq.4) then
c ndist = 4: Gamma         
            pdf(loop) = factor*(t(loop)**temp3)*exp(-t(loop)/a(2))
         elseif (ndist.eq.5) then
c ndist = 5: Logistic          
            temp1 = exp((t(loop) - a(1))/a(2))
            temp2 = (one + temp1)**2
            pdf(loop) = temp1/(a(2)*temp2)
         elseif (ndist.eq.6) then
c ndist = 6: Lognormal         
            factor = one/(t(loop)*temp1)
            arg = - (log(t(loop)) - a(1))**2/temp2
            pdf(loop) = factor*exp(arg)   
         elseif (ndist.eq.7) then
c ndist = 7: Normal         
            arg = - ((t(loop) - a(1))/a(2))**2/two 
            pdf(loop) = factor*exp(arg) 
         elseif (ndist.eq.8) then
c ndist = 8: Uniform      
            pdf(loop) = factor  
         elseif (ndist.eq.9) then
c ndist = 9: Weibull         
            pdf(loop) = temp1*t(loop)**temp2*exp(- t(loop)**a(1)/a(2))  
         elseif (ndist.eq.10) then
c ndist = 10: F         
            pdf(loop) =  factor*t(loop)**temp1/
     +                  (one + temp3*t(loop))**temp2  
         elseif (ndist.eq.11) then
c ndist = 11: t         
            pdf(loop) = factor*(one + t(loop)**2/dn)**temp1 
         elseif (ndist.eq.12) then
c ndist = 12: Beta         
            pdf(loop) = factor*(t(loop)**temp1)*(one - t(loop))**temp2 
         endif   
      enddo
      abort = .false.
      end
c
c--------------------------------------------------------------------------------------
c    
      subroutine pdfplt (npdf, nsamp,
     +                   pdf, t, sample,
     +                   title)
c
c action: simplified front end to pdplot
c author: w.g.bardsley, university of manchester, u.k. 11/03/2019
c
c   npdf: number of pdf values
c  nsamp: sample size 
c    pdf: exact pdf values
c sample: random sample or observations
c  
      implicit none
c
c arguments
c      
      integer,             intent (in) :: npdf, nsamp
      double precision,    intent (in) :: pdf(npdf), t(npdf),
     +                                    sample(nsamp)
      character (len = *), intent (in) :: title
c      
c allocatable
c      
      double precision, allocatable :: x(:), y(:)
c
c local
c      
      integer    ifail, nbins, nrmax, ntemp
      integer    nbot, ntop
      integer    ntxt, numdec, numopt, numsta, numtxt
      parameter (ntxt = 26, numsta = 8, numopt = 8,
     +           numtxt = numsta + numopt - 1) 
      integer    numbld(30)
      character (len = 100) text(30)
      character (len = 80 ) ptitle, qtitle, xtitle, ytitle
      logical    again, repeet, suppress
      data       suppress / .false. /
      data       xtitle, ytitle / 'Values', 'Normalised Frequencies' /
      data       numbld / 30*0 /
      data       nbins  / 10 /
      external   pdplot, lstbox, getjm1, getstr, patch2
      intrinsic  min
c
c initialise
c      
      ptitle = title
      ntemp = min(120,nsamp,npdf)
      if (nbins.gt.ntemp) nbins = ntemp
      numdec = numopt - 1
      
c
c outer loop
c      
      repeet = .true.
      do while (repeet)
         nrmax = 3*(nbins + 1)
         allocate (x(nrmax), stat = ifail)
         if (ifail.ne.0) return
         allocate (y(nrmax), stat = ifail)
         if (ifail.ne.0) return   
         call pdplot (nbins, npdf, nrmax, nsamp, 
     +                pdf, sample, t,x, y,
     +                ptitle, xtitle, ytitle)  
         deallocate (x, stat = ifail)
         deallocate (y, stat = ifail)
c
c inner loop
c         
         if (suppress) then
            return
         else   
            again = .true.
         endif   
         do while (again)
            write (text,100) nbins, ptitle, xtitle, ytitle
            numbld(1) = 4
            call lstbox (numbld, numdec, numopt, numsta, numtxt,
     +                   text) 
            numbld(1) = 0 
            if (numdec.eq.1) then
               nbot = 2
               ntop = min(120,nsamp,npdf)
               call getjm1 (nbot, nbins, ntop,
     +                     'Number of bins required')  
            elseif (numdec.eq.2) then
               qtitle = ptitle
               call getstr (qtitle, ptitle) 
            elseif (numdec.eq.3) then
               qtitle = xtitle
               call getstr (qtitle, xtitle)
            elseif (numdec.eq.4) then
               qtitle = ytitle
               call getstr (qtitle, ytitle)   
            elseif (numdec.eq.5) then
               again = .false. 
            elseif (numdec.eq.6) then
               again = .false.
               repeet = .false.   
               suppress = .true. 
            elseif (numdec.eq.7) then  
               write (text,200) 
               numbld(1) = 4
               numbld(3) = 1
               numbld(10) = 1
               numbld(18) = 1 
               call patch2 (numbld, ntxt,
     +                      text)
               numbld(1) = 0
               numbld(3) = 0
               numbld(10) = 0
               numbld(18) = 0                       
            elseif (numdec.eq.numopt) then
               again =.false.
               repeet = .false.
            endif
         enddo         
      enddo
c
c format statements
c      
  100 format ( 
     + 'Options for editing the histogram'
     +/
     +/'Number of bins =',i4 
     +/'Title =',1x,a
     +/'X-legend =',1x,a
     +/'Y-legend =',1x,a
     +/
     +/'Change the number of bins'
     +/'Edit the title'
     +/'Edit the X-legend'
     +/'Edit the Y-legend'
     +/'Display the histogram'
     +/'Suppress these options for this session'
     +/'Help'
     +/'Quit ... Exit pdf-histogram plot')  
  200 format (
     + 'Options available for editing this pdf-histogram plot'
     +/
     +/'Definition of a histogram'
     +/'A histogram is a special type of barchart where there is a'
     +/'sample of N observations, x, that all lie in a fixed range of'
     +/'values, say X_min =< x =< X_max, which can be divided into'
     +/'equal width intervals called bins, so that the frequency of'          
     +/'observations collected in these bins can be recorded.'
     +/
     +/'Definition of a pdf'
     +/'It is usually assumed that there is an underlying probability'
     +/'distribution, such as a chi-square or F distribution, that has'
     +/'a known probability distribution function (pdf), depending on'
     +/'supposed parameters, such as the mean and standard deviation'
     +/'in the case of a normal distribution. So, the aim of plotting'
     +/'a pdf on a histogram is to visualise goodness of fit.'
     +/
     +/'Scaling and shape considerations'
     +/'As pdfs integrate to 1 over their range of definition it can be'
     +/'useful to adjust the area under the histogram to 1 so that, if'
     +/'there are observations near X_min and X_max, goodness of fit'
     +/'can be judged. As histogram shape depends on the number of bins'
     +/'this can be changed. However, the variance of the frequency in'
     +/'any bin is of the order of the square root of the frequency in'
     +/'that bin, and this should be considered when choosing the'
     +/'number of bins chosen, given the sample size N.')
      end
c
c----------------------------------------------------------------------------------   
c
      subroutine pdplot (nbins, npdf, nrmax, nsamp,
     +                   pdf, sample, t, x, y,
     +                   ptitle, xtitle, ytitle)
c
c action: plot histogram and best-fit pdf
c author: w.g.bardsley, university of manchester, u.k., 24/08/2005
c         20/04/2007 added intents
c         19/10/2012 added ybin as allocatable for call to x2ybin 
c                    removed need for sample to be in increasing order
c                    a = xmin and b = xmax calculated from the sample 
c
c         nbins: (input/unchanged) number of bins required
c          npdf: (input/unchanged) number of pdf values
c         nrmax: (input/unchanged) dimension of workspace 
c         nsamp: (input/unchanged) sample size  
c           pdf: (input/unchanged) pdf values at t values (below)
c        sample: (input/unchanged) sample values in nondecreasing order
c             t: (input/unchanged) t values for pdf(t) in nondecreasing order
c             x:  workspace for plotting
c             y:  workspace for plotting
c        ptitle: (input/unchanged) plot title
c        xtitle: (input/unchanged) x-axis title
c        ytitle: (input/unchanged) y-axis title
c
      implicit   none
c
c arguments
c
      integer,             intent (in)    :: nbins, npdf, nrmax, nsamp
      double precision,    intent (in)    :: pdf(npdf), sample(nsamp),
     +                                       t(npdf)
      double precision,    intent (inout) :: x(nrmax), y(nrmax)
      character (len = *), intent (in)    :: ptitle, xtitle, ytitle
c
c allocatable
c      
      double precision, allocatable :: ybin(:)  
c
c locals
c
      integer    i, icount, ierr, jcount
      integer    l1, l2, l3, l4, m1, m2, m3, m4, n1, n2, n3, n4
      parameter (l1 = 1, l2 = 2, l3 = 0, l4 = 0, m1 = 0, m2 = 0, m3 = 0,
     +           m4 = 0, n3 = 2, n4 = 2)
      double precision a, b, delta, xmax, xmin
      double precision x3(n3), x4(n4), y3(n3), y4(n4)
      double precision dmin, zero
      parameter (dmin = 1.0d-100, zero = 0.0d+00)
      character  line*100
      logical    abort
      logical    axes, gsave
      parameter (axes = .true., gsave = .true.)
      external   putfat, gks004, x2ybin$
      intrinsic  dble
c
c histogram part 1: check
c
      if (nbins.lt.2 .or. npdf.lt.10 .or. nsamp.lt.5) then
         write (line,100)
         call putfat (line)
         return
      endif
      xmin = sample(1)
      xmax = sample(1)
      do i = 2, nsamp
         if (sample(i).lt.xmin) then
            xmin = sample(i)
         elseif (sample(i).gt.xmax) then
            xmax = sample(i)
         endif      
      enddo
      if (xmin.ge.xmax) then
         write (line,200)
         call putfat (line)
         return
      endif   
      do i = 2, npdf
         if (t(i).lt.t(i - 1)) then
            write (line,300)
            call putfat (line)
            return
         endif
      enddo
      if (nrmax.lt.3*nbins + 2) then
         write (line,400)
         call putfat (line)
         return
      endif
      delta = (xmax - xmin)/dble(nbins)
      if (delta.le.dmin) then
         write (line,500)
         call putfat (line)
         return
      endif
c
c histogram part 2: initialise
c
      ierr = 0
      if (allocated(ybin)) deallocate(ybin, stat = ierr)
      if (ierr.ne.0) return
      i = nbins
      allocate(ybin(i), stat = ierr)
      if (ierr.ne.0) return
        
      a = xmin
      b = xmax
      call x2ybin$(nbins, nsamp,
     +             a, b, sample, ybin,
     +             abort)
      if (abort) return         
c
c histogram part 3: set up arrays x and y with wrap round at the end
c
      icount = 0
      jcount = 0
      do i = 1, nbins
         icount = icount + 1
         jcount = jcount + 1
         if (i.eq.1) then
            x(icount) = a
         else
            x(icount) = x(icount - 1)
         endif
         y(icount) = zero
         icount = icount + 1
         x(icount) = x(icount - 1)
         y(icount) = ybin(jcount)
         icount = icount + 1
         x(icount) = x(icount - 1) + delta
         y(icount) = ybin(jcount)
      enddo
      x(icount) = b
      icount = icount + 1
      x(icount) = b
      y(icount) = zero
      icount = icount + 1
      x(icount) = a
      y(icount) = zero
c
c histogram part 4: plot then deallocate
c
      n1 = 3*nbins + 2
      n2 = npdf
      call gks004 (l1, l2, l3, l4, m1, m2, m3, m4, n1, n2, n3, n4,
     +             x,   t, x3, x4,
     +             y, pdf, y3, y4,
     +             ptitle, xtitle, ytitle,
     +             axes, gsave)   
      deallocate (ybin, stat = ierr)
c
c format statements
c     
  100 format ('Must have bins > 2, sample size > 5, pdf size > 5')
  200 format ('Sample values are identical')
  300 format ('Must have pdf in nondecreasing order')
  400 format ('Insufficient array space to plot all histogram')
  500 format ('Bin width too small to plot')
      end
c
c------------------------------------------------------------------------
c



