C
C G02GLIM
C GLIMCV
C
C...................................................................
C The main GLIM interface for G02GAF$, G02GBF$, G02GCF$, G02GDF$.
C This version uses SVD (LAPACK, DGELSS) for calculations not QR.
C It uses locally defined workspace for calculations instead of the
C workspace provided in the extra columns of V as in the NAG routines.
C V is returned as in the NAG routines except that columns 8 to IP + 7
C always contain the right singular vectors stored row-wise.
C When the no. of free variables = 1 a best fit curve is calculated
C and stored as NPTS, XMIN, XDELTA, then best-fit Y(i), i = 1, NPTS in
C column 8 of V starting at row IP + 1, i.e. after the singular vectors.
C GLIMCV is used to calculate the covariance matrix and leverages.
C..................................................................
C
      SUBROUTINE G02GLIM(IDF, IFAIL, IP, IPRINT, IRANK, ISX, LDV, LDX,
     +                   LTYPE, M, MAXIT, N, NTYPE,
     +                   A, B, COV, DEV, EPS, RSS, S, SE, T, TOL, V,
     +                   WK, WT, X, Y,
     +                   MEAN, OFFSET, WEIGHT)
C
C ACTION: Create temporary transformed data sets for GLIM then solve
C AUTHOR: W.G.Bardsley, University of Manchester, U.K., 13/04/2000
C         15/05/2002 reworked the code to return some information
C                    after nonzero IFAIL exits
C         14/06/2002 replaced IRANK by IP to fill in all V on return from
C                    DGELSS and to start plot data at position IP + 1
C                    Also corrected output of B(I) to agree with NAG
C                    V always has  D-inverseP_1P_0 as SVD is always used
C                    so it only agrees with NAG when rank-deficient
C         12/07/2002 altered convergence mechanism to prevent the objective
C                    function diverging too much
C         15/01/2006 introduced allocatable workspaces
C         13/04/2004 suppressed extra code to calculate best-fit curves
C         23/05/2013 introduced OK to make sure WT(i) is not referenced when WEIGHT = .FALSE.
C         28/12/2014 introduced MAXDIV to allow exit with IFAIL = 14 if diverging
C
C         Supplied parameters are:
C         ========================
C         LTYPE: link type
C         NTYPE: error type
C
C         Derived parameters are:
C         =======================
C         MFREE = no. of free parameters
C         NFREE = no. of free data points
C
C         IFAIL parameters are:
C         =====================
C         IFAIL = 10: failure to allocate workspace
C         IFAIL = 11: NTYPE out of bounds
C         IFAIL = 12: LTYPE out of bounds
C         IFAIL = 13: MFREE/NFREE out of bounds
C         IFAIL = 14: objective function diverged MAXDIV times in succession 
C 
C         Iteration details are:
C         ======================
C         IPRINT = no. of iterations before intermediate output
C
C NOTE: This routine does not use the workspace WK supplied, nor does it
C       use V except for output. It defines temporary local workspaces
C       to perform the calculations.
C       This routine should be re-written to use the workspace provided
C       in V and WRK when I find time.
C
      IMPLICIT NONE
C
C Arguments supplied
C
      INTEGER LDV, LDX, M
      INTEGER IDF, IFAIL, IP, IPRINT, IRANK, ISX(M), LTYPE, MAXIT, N,
     +        NTYPE
      DOUBLE PRECISION A, B(IP), COV(IP*(IP + 1)/2), DEV, EPS, RSS,
     +                 S, SE(IP), T(*),
     +                 TOL, V(LDV,IP + 7), WK((IP*IP + 3*IP + 22)/2),
     +                 WT(*), X(LDX,M), Y(N)
      LOGICAL  MEAN, OFFSET, WEIGHT
C
C Local allocatable workspaces
C
      DOUBLE PRECISION, ALLOCATABLE :: ANEW(:,:), ASAV(:,:), ATEMP(:,:),
     +                                 BNEW(:), BTEMP(:,:), HATETA(:),
     +                                 HATMU(:), OFFNEW(:), RESID(:),
     +                                 SVD(:), TAU(:), TNEW(:), U(:,:),
     +                                 WNEW(:), WORK(:), WRKWT(:),
     +                                 YNEW(:)
C
C Local workspace and variables
C
      INTEGER    LDA, LDB, LDU, LWORK, NCMAX, NDIVER, NRMAX
      INTEGER    LWMIN, NCADD, NCMIN, NRADD, NRMIN, NRHS
      PARAMETER (LWMIN = 5000, NCADD = 5, NCMIN = 20, NRADD = 5,
     +           NRMIN = 200, NRHS = 1)
      INTEGER    I, ICOUNT, IERR, INFO, INVERT, J, K,
     +           MFREE, NFREE, NPAR, NPTS
      INTEGER    MAXDIV
      DOUBLE PRECISION VAREST
      DOUBLE PRECISION ZERO, ONE, SMALL, RTOL, RTOL2
      PARAMETER (ZERO = 0.0D+00, ONE = 1.0D+00, SMALL = 1.0D-10,
     +           RTOL = 1.0D-100, RTOL2 = 2.0D+00*RTOL)
      DOUBLE PRECISION FCN1, FCN2, RCOND, SIGMA, SSQDEV, SSQTOL
      LOGICAL    OUTPUT
      LOGICAL    OK  
      EXTERNAL   DGELSS
      EXTERNAL   GLIMN1, GLIMN2, GLIMB1, GLIMB2, GLIMP1, GLIMP2,
     +           GLIMG1, GLIMG2, GLIMCV
      INTRINSIC  ABS, SQRT, DBLE, MOD
C
C Check NTYPE/LTYPE
C
      IF (NTYPE.LT.1 .OR. NTYPE.GT.4) THEN
         IFAIL = 11
         RETURN
      ENDIF
      IF (NTYPE.EQ.2) THEN
         ICOUNT = 3
      ELSE
         ICOUNT = 5
      ENDIF
      IF (LTYPE.LT.1 .OR. LTYPE.GT.ICOUNT) THEN
         IFAIL = 12
         RETURN
      ENDIF
C
C Allocate space
C
      IFAIL = 10
      IERR = 0
      NCMAX = M + NCADD
      IF (NCMAX.LT.NCMIN) NCMAX = NCMIN
      NRMAX = N + NRADD
      IF (NRMAX.LT.NRMIN) NRMAX = NRMIN
      LDA = NRMAX
      LDB = NRMAX
      LDU = NRMAX
      LWORK = 10*NRMAX + 10*NCMAX
      IF (LWORK.LT.LWMIN) LWORK = LWMIN
      IF (ALLOCATED(ANEW)) DEALLOCATE(ANEW, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(ASAV)) DEALLOCATE(ASAV, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(ATEMP)) DEALLOCATE(ATEMP, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(BNEW)) DEALLOCATE(BNEW, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(BTEMP)) DEALLOCATE(BTEMP, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(HATETA)) DEALLOCATE(HATETA, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(HATMU)) DEALLOCATE(HATMU, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(OFFNEW)) DEALLOCATE(OFFNEW, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(RESID)) DEALLOCATE(RESID, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(SVD)) DEALLOCATE(SVD, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(TAU)) DEALLOCATE(TAU, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(TNEW)) DEALLOCATE(TNEW, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(U)) DEALLOCATE(U, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(WNEW)) DEALLOCATE(WNEW, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(WORK)) DEALLOCATE(WORK, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(WRKWT)) DEALLOCATE(WRKWT, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(YNEW)) DEALLOCATE(YNEW, STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(ANEW(LDA,NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(ASAV(LDA,NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(ATEMP(LDA,NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(BNEW(LDB), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(BTEMP(LDB,NRHS), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(HATETA(LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(HATMU(LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(OFFNEW(LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(RESID(LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(SVD(NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(TAU(LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(TNEW(LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(U(LDU,NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(WNEW(LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(WORK(LWORK), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(WRKWT(LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(YNEW(LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN
C
C Initialise and set IFAIL = 0
C
      IDF = 0
      IFAIL = 0
      IF (IPRINT.LE.0) THEN
         OUTPUT = .FALSE.
      ELSE
         OUTPUT = .TRUE.
      ENDIF
      INVERT = 1
      DEV = ZERO
      RSS = ZERO
      WK(1) = ZERO
      DO I = 1, IP*(IP + 1)/2
         COV(I) = ONE
      ENDDO
      DO I = 1, IP
         B(I) = ONE
         BNEW(I) = ONE
         BTEMP(I,1) = ONE
         SE(I) = ONE
      ENDDO
      DO J = 1, IP + 7
         IF (J.NE.7) THEN
            DO I = 1, N
               V(I,J) = ONE
            ENDDO
         ENDIF
      ENDDO

C
C Generate MFREE, NFREE, ANEW, WNEW, YNEW, etc. as follows:
C =========================================================
C MFREE = no. of free variables
C NFREE = no. of free data points
C ANEW  = matrix of free X-data
C ATEMP = temporary A matrix (weighted) for LAPACK
C BNEW  = best fit parameters
C BTEMP = temporary B matrix (weighted) for LAPACK
C WNEW  = vector of free weights
C YNEW  = vector of free Y-data
C TNEW  = vector of free T-data (binomial only)
C
      NFREE = 0
      DO I = 1, N
         IF (WEIGHT) THEN
            IF (WT(I).GT.ZERO) THEN
               OK = .TRUE.
            ELSE
               OK = .FALSE.
            ENDIF
         ELSE
            OK = .TRUE.
         ENDIF            
         IF (OK) THEN
            NFREE = NFREE + 1
            IF (WEIGHT) THEN
               WNEW(NFREE) = WT(I)
            ELSE
               WNEW(NFREE) = ONE
            ENDIF
            IF (OFFSET) THEN
               OFFNEW(NFREE) = V(I,7)
            ELSE
               OFFNEW(NFREE) = ZERO
            ENDIF
            YNEW(NFREE) = Y(I)
            IF (MEAN) THEN
               K = 1
               ANEW(NFREE,K) = ONE
            ELSE
               K = 0
            ENDIF
            DO J = 1, M
               IF (ISX(J).GT.0) THEN
                  K = K + 1
                  ANEW(NFREE,K) = X(I,J)
               ENDIF
            ENDDO
            WRKWT(I) = ONE
            IF (NTYPE.EQ.2) THEN
               IF (T(I).GT.RTOL2) THEN
                  TNEW(NFREE) = T(I)
               ELSE
                  NFREE = NFREE - 1
               ENDIF
            ENDIF
         ENDIF
      ENDDO
      MFREE = K
C
C Check before proceeding
C
      IF (MFREE.GT.NFREE .OR. NFREE.LT.2 .OR. MFREE.EQ.0) THEN
         IFAIL = 13
         DEALLOCATE(ANEW, STAT = IERR)
         DEALLOCATE(ASAV, STAT = IERR)
         DEALLOCATE(ATEMP, STAT = IERR)
         DEALLOCATE(BNEW, STAT = IERR)
         DEALLOCATE(BTEMP, STAT = IERR)
         DEALLOCATE(HATETA, STAT = IERR)
         DEALLOCATE(HATMU, STAT = IERR)
         DEALLOCATE(OFFNEW, STAT = IERR)
         DEALLOCATE(RESID, STAT = IERR)
         DEALLOCATE(SVD, STAT = IERR)
         DEALLOCATE(TAU, STAT = IERR)
         DEALLOCATE(TNEW, STAT = IERR)
         DEALLOCATE(U, STAT = IERR)
         DEALLOCATE(WNEW, STAT = IERR)
         DEALLOCATE(WORK, STAT = IERR)
         DEALLOCATE(WRKWT, STAT = IERR)
         DEALLOCATE(YNEW, STAT = IERR)
         RETURN
      ENDIF

C
C Temporary print statement to check iteration details
C ----------------------------------------------------
         IF (OUTPUT) WRITE (*,100)
C ----------------------------------------------------
C

C
C Define NDIVER, NPAR and NPTS then start the iteration
C
      MAXDIV = MIN(MAXIT/2,30)
      NDIVER = 0
      NPAR = MFREE
      NPTS = NFREE
      RCOND = EPS
      ICOUNT = 0
      FCN1 = ONE
      FCN2 = ZERO
      SSQDEV = ABS(FCN2 - FCN1)
      SSQTOL = TOL*(ONE + FCN2)
      DO WHILE (IFAIL.EQ.0       .AND. ICOUNT.LT.MAXIT .AND.
     +          SSQDEV.GT.SSQTOL .AND. NDIVER.LT.MAXDIV)
         ICOUNT = ICOUNT + 1
         FCN1 = FCN2
C
C Generate ATEMP and BTEMP (BNEW used if ICOUNT > 1)
C
         IF (NTYPE.EQ.1) THEN
            CALL GLIMN1 (ICOUNT, IFAIL, LTYPE, MFREE, NFREE, NRHS, LDA,
     +                   LDB,
     +                   A, ANEW, ATEMP, BNEW, BTEMP, OFFNEW, WNEW,
     +                   WRKWT, YNEW)
         ELSEIF (NTYPE.EQ.2) THEN
            CALL GLIMB1 (ICOUNT, IFAIL, LTYPE, MFREE, NFREE, NRHS, LDA,
     +                   LDB,
     +                   ANEW, ATEMP, BNEW, BTEMP, OFFNEW, TAU, TNEW,
     +                   WNEW, WRKWT, YNEW)
         ELSEIF (NTYPE.EQ.3) THEN
            CALL GLIMP1 (ICOUNT, IFAIL, LTYPE, MFREE, NFREE, NRHS, LDA,
     +                   LDB,
     +                   A, ANEW, ATEMP, BNEW, BTEMP, OFFNEW, TAU, WNEW,
     +                   WRKWT, YNEW)
         ELSEIF (NTYPE.EQ.4) THEN
            CALL GLIMG1 (ICOUNT, IFAIL, LTYPE, MFREE, NFREE, NRHS, LDA,
     +                   LDB,
     +                   A, ANEW, ATEMP, BNEW, BTEMP, OFFNEW, TAU, WNEW,
     +                   WRKWT, YNEW)
         ENDIF
C
C Check the current value of IFAIL before any more calculations
C
         IF (IFAIL.EQ.0) THEN
C
C Copy the current matrix ATEMP into the temporary workspace ASAV
C
            DO J = 1, NPAR
               DO I = 1, NPTS
                  ASAV(I,J) = ATEMP(I,J)
               ENDDO
            ENDDO
C
C Fit the weighted least squares system ATEMP*X = BTEMP using LAPACK
C
            CALL DGELSS (NPTS, NPAR, NRHS, ATEMP, LDA, BTEMP, LDB,
     +                   SVD, RCOND, IRANK, WORK, LWORK, INFO)
C
C Copy the best fit parameters into BNEW
C
            DO I = 1, NPAR
               BNEW(I) = BTEMP(I,1)
            ENDDO
C
C Calculate the residuals and deviance, etc.
C
            IF (NTYPE.EQ.1) THEN
               CALL GLIMN2 (IFAIL, LTYPE, MFREE, NFREE, LDA,
     +                      A, ANEW, BNEW, HATETA, HATMU, OFFNEW, RESID,
     +                      RSS, YNEW)
               FCN2 = RSS
            ELSEIF (NTYPE.EQ.2) THEN
               CALL GLIMB2 (IFAIL, LTYPE, MFREE, NFREE, LDA,
     +                      ANEW, BNEW, DEV, HATETA, HATMU, OFFNEW,
     +                      RESID, TNEW, YNEW)
               FCN2 = DEV
            ELSEIF (NTYPE.EQ.3) THEN
               CALL GLIMP2 (IFAIL, LTYPE, MFREE, NFREE, LDA,
     +                      A, ANEW, BNEW, DEV, HATETA, HATMU, OFFNEW,
     +                      RESID, YNEW)
               FCN2 = DEV
            ELSEIF (NTYPE.EQ.4) THEN
               CALL GLIMG2 (IFAIL, LTYPE, MFREE, NFREE, LDA,
     +                      A, ANEW, BNEW, DEV, HATETA, HATMU, OFFNEW,
     +                      RESID, YNEW)
               FCN2 = DEV
            ENDIF
         ELSE
            RETURN   
         ENDIF

C
C Temporary print statement to check iteration details
C ------------------------------------------------------
         IF (OUTPUT) THEN
            IF (MOD(ICOUNT,IPRINT).EQ.0) WRITE (*,200) ICOUNT, INFO,
     +                                                 FCN2
         ENDIF
C ------------------------------------------------------
C

         IF (ICOUNT.GT.1) THEN
C
C Check if FCN is diverging ... 28/12/2014 does not now change ifail
C
            IF (FCN2.LE.FCN1) THEN
               NDIVER = 0
            ELSE
               NDIVER = NDIVER + 1
            ENDIF
         ENDIF
         SSQDEV = ABS(FCN2 - FCN1)
         SSQTOL = TOL*(ONE + FCN2)
      ENDDO
C
C Has the iteration converged ?
C
      IF (NDIVER.EQ.MAXDIV .AND. ICOUNT.LT.MAXIT) IFAIL = 14
      IF (IFAIL.EQ.0 .AND. ICOUNT.GE.MAXIT) THEN
         IF (NTYPE.EQ.1) THEN
            IFAIL = 6
         ELSEIF (NTYPE.EQ.2) THEN
            IFAIL = 8
         ELSE
            IFAIL = 7
         ENDIF
      ENDIF
C
C Assign S if NTYPE = 1 or 4 and the starting S supplied = 0.0
C
      IF (NTYPE.EQ.1 .AND. NPTS.GT.IRANK) THEN
         IF (ABS(S - ZERO).LT.SMALL) S = RSS/DBLE(NPTS - IRANK)
      ELSEIF (NTYPE.EQ.4 .AND. NPTS.GT.IRANK) THEN
         IF (ABS(S - ZERO).LT.SMALL) THEN
            S = ZERO
            DO I = 1, NPTS
               IF (HATMU(I).GT.RTOL) S = S +
     +            ((YNEW(I) - HATMU(I))/HATMU(I))**2
            ENDDO
            S = S/DBLE(NPTS - IRANK)
         ENDIF
      ENDIF
C
C Assign VAREST then SIGMA
C
      IF (NPTS.LE.IRANK) THEN
         VAREST = ZERO
      ELSEIF (NTYPE.EQ.1) THEN
         VAREST = S
      ELSEIF (NTYPE.EQ.2) THEN
         VAREST = ONE
      ELSEIF (NTYPE.EQ.3) THEN
         VAREST = ONE
      ELSEIF (NTYPE.EQ.4) THEN
         VAREST = S
      ENDIF
      SIGMA = SQRT(VAREST)
C
C Calculate covariance matrix, parameter standard errors and leverages
C
      IF (NPTS.GT.NPAR) THEN
C
C Copy the right singular vectors from ATEMP into V
C
         DO J = 1, NPAR
            DO I = 1, NPAR
               V(I,J + 7) = ATEMP(I,J)
            ENDDO
         ENDDO
C
C Generate U = cov. matrix and V(i,6) = leverages using ASAV = original
C A to DGELSS and V with singular vectors
C
         CALL GLIMCV (INVERT, IRANK, LDA, LDU, LDV, NPAR, NPTS,
     +                ASAV, SVD, U, V)
         IF (INVERT.EQ.0) THEN
            DO I = 1, NPAR
               SE(I) = SIGMA*SQRT(U(I,I))
            ENDDO
C
C Load the covariance matrix
C
            K = 0
            DO J = 1, NPAR
               DO I = 1, J
                  K = K + 1
                  COV(K) = VAREST*U(I,J)
               ENDDO
            ENDDO
         ENDIF
      ENDIF
C
C Define the best fit parameters
C
      IF (MEAN) THEN
         J = 1
         B(J) = BNEW(J)
      ELSE
         J = 0
      ENDIF
      DO I = 1, M
         IF (ISX(I).GT.0) THEN
            J = J + 1
            B(J) = BNEW(J)
         ENDIF
      ENDDO
C
C Assign the rest of V
C
      J = 0
      DO I = 1, N
         IF (WEIGHT) THEN
            IF (WT(I).GT.ZERO) THEN
               OK = .TRUE.
            ELSE
               OK = .FALSE.
            ENDIF
         ELSE
            OK = .TRUE.
         ENDIF            
         IF (OK) THEN
            J = J + 1
            V(I,1) = HATETA(J)
            V(I,2) = HATMU(J)
            IF (NTYPE.EQ.1) THEN
               V(I,3) = ZERO
            ELSE
               V(I,3) = TAU(J)
            ENDIF
            V(I,4) = WRKWT(J)
            V(I,5) = RESID(J)
         ENDIF
      ENDDO
C
C Assign IDF
C
      IDF = NPTS - IRANK
C 
C Note: added at 19/04/2006                                                                
C The following code is suppressed to maintain equality with the NAG routines
C Generate plot data to store in V if mu = mu(x), i.e. only 1 variable
C
C      IF (OFFSET) THEN
C         PLOT = .FALSE.
C         NFREE = 0
C      ELSE
C         IF (MEAN.AND.MFREE.EQ.2) THEN
C            PLOT = .TRUE.
C         ELSEIF (.NOT.MEAN .AND. MFREE.EQ.1) THEN
C            PLOT = .TRUE.
C         ELSE
C            PLOT = .FALSE.
C            NFREE = 0
C         ENDIF
C      ENDIF
C      IF (PLOT) THEN
C
C Find the range of X
C
C         IF (MEAN) THEN
C            XMIN = ANEW(1,2)
C            XMAX = ANEW(1,2)
C            DO I = 2, NFREE
C               IF (ANEW(I,2).LT.XMIN) XMIN = ANEW(I,2)
C               IF (ANEW(I,2).GT.XMAX) XMAX = ANEW(I,2)
C            ENDDO
C         ELSE
C            XMIN = ANEW(1,1)
C            XMAX = ANEW(1,1)
C            DO I = 2, NFREE
C               IF (ANEW(I,1).LT.XMIN) XMIN = ANEW(I,1)
C               IF (ANEW(I,1).GT.XMAX) XMAX = ANEW(I,1)
C            ENDDO
C         ENDIF
C
C Re-define NFREE then calculate evenly spaced x-values
C
C         NFREE = MIN(100,NRMIN,LDV - IP - 3)
C         XDELTA = (XMAX - XMIN)/DBLE(NFREE - 1)
C         IF (MEAN) THEN
C            ANEW(1,1) = ONE
C            ANEW(1,2) = XMIN
C         ELSE
C            ANEW(1,1) = XMIN
C         ENDIF
C         OFFNEW(1) = ZERO
C         YNEW(1) = ZERO
C         DO I = 2, NFREE - 1
C            IF (MEAN) THEN
C               ANEW(I,1) = ONE
C               ANEW(I,2) = ANEW(I - 1,2) + XDELTA
C            ELSE
C               ANEW(I,1) = ANEW(I - 1,1) + XDELTA
C            ENDIF
C            OFFNEW(I) = ZERO
C            YNEW(I) = ZERO
C         ENDDO
C         IF (MEAN) THEN
C            ANEW(NFREE,1) = ONE
C            ANEW(NFREE,2) = XMAX
C         ELSE
C            ANEW(NFREE,1) = XMAX
C         ENDIF
C         OFFNEW(NFREE) = ZERO
C         YNEW(NFREE) = ZERO
C
C Calculate HATMU = best fit y using IFAIL1, RSS1 and DEV1 so that
C IFAIL, RSS and DEV from the main fitting are unchanged
C
C         IFAIL1 = 0
C         IF (NTYPE.EQ.1) THEN
C            CALL GLIMN2 (IFAIL1, LTYPE, MFREE, NFREE, LDA,
C     +                   A, ANEW, BNEW, HATETA, HATMU, OFFNEW, RESID,
C     +                   RSS1, YNEW)
C         ELSEIF (NTYPE.EQ.2) THEN
C
C Note the use of TFIX to make sure 0 < y < 1
C
C            DO I = 1, NFREE
C               TNEW(I) = TFIX
C            ENDDO
C            CALL GLIMB2 (IFAIL1, LTYPE, MFREE, NFREE, LDA,
C     +                   ANEW, BNEW, DEV1, HATETA, HATMU, OFFNEW,
C     +                   RESID, TNEW, YNEW)
C            DO I = 1, NFREE
C               HATMU(I) = HATMU(I)/TNEW(I)
C            ENDDO
C         ELSEIF (NTYPE.EQ.3) THEN
C            CALL GLIMP2 (IFAIL1, LTYPE, MFREE, NFREE, LDA,
C     +                   A, ANEW, BNEW, DEV1, HATETA, HATMU, OFFNEW,
C     +                   RESID, YNEW)
C         ELSEIF (NTYPE.EQ.4) THEN
C            CALL GLIMG2 (IFAIL1, LTYPE, MFREE, NFREE, LDA,
C     +                   A, ANEW, BNEW, DEV1, HATETA, HATMU, OFFNEW,
C     +                   RESID, YNEW)
C         ENDIF
C
C Load the best-fit values if all is well
C
C         IF (IFAIL1.EQ.0) THEN
C            J = IP + 2
C            V(J,8) = XMIN
C            J = J + 1
C            V(J,8) = XDELTA
C            DO I = 1, NFREE
C               J = J + 1
C               V(J,8) = HATMU(I)
C            ENDDO
C         ELSE
C            NFREE = 0
C         ENDIF
C      ENDIF
C
C Load NFREE into position V(IP + 1,8)
C
      J = IP + 1
C      V(J,8) = DBLE(NFREE)
      V(J,8) = 0
C
C Deallocate workspaces
C
      DEALLOCATE(ANEW, STAT = IERR)
      DEALLOCATE(ASAV, STAT = IERR)
      DEALLOCATE(ATEMP, STAT = IERR)
      DEALLOCATE(BNEW, STAT = IERR)
      DEALLOCATE(BTEMP, STAT = IERR)
      DEALLOCATE(HATETA, STAT = IERR)
      DEALLOCATE(HATMU, STAT = IERR)
      DEALLOCATE(OFFNEW, STAT = IERR)
      DEALLOCATE(RESID, STAT = IERR)
      DEALLOCATE(SVD, STAT = IERR)
      DEALLOCATE(TAU, STAT = IERR)
      DEALLOCATE(TNEW, STAT = IERR)
      DEALLOCATE(U, STAT = IERR)
      DEALLOCATE(WNEW, STAT = IERR)
      DEALLOCATE(WORK, STAT = IERR)
      DEALLOCATE(WRKWT, STAT = IERR)
      DEALLOCATE(YNEW, STAT = IERR)
  100 FORMAT ('Iterate  INFO(dgelss)     Obj. Fun.')
  200 FORMAT (I7,I14,1P,E14.5)
      END
C
C
      SUBROUTINE GLIMCV (IFAIL, IRANK, LDA, LDU, LDW, NCOLS, NROWS,
     +                   A, SIGMA, U, W)
C
C ACTION: Return leverages and covariance matrix after a call to DGELSS.
C         This routine must be called immediately after a call to DGELSS
C         A = the A matrix originally supplied to DGELSS
C         IRANK = IRANK returned by DGELSS
C         SIGMA = SVD returned by DGELSS
C         W = A returned from DGELSS with a column offset of 7, i.e.
C         W contains V(transpose) in columns 8 to NPAR + 7
C AUTHOR: W.G.Bardsley, University of Manchester, U.K.
C         Helped by T.L.Freeman 01/07/2000
C
C Conventions:
C ============
C A = U*Sigma*V(transpose)
C U = A*V*Sigma(inverse)
C A*(A(transpose)*A)(inverse)*A(transpose) = U*U(transpose)
C (A(transpose)*A)(inverse) = V*Sigma(squared-inverse)*V(transpose)
C
C Arguments
C =========
C IFAIL: [output] returned as 0 if no errors, o/w > 0
C IRANK: [input] rank from DGELSS SVD
C LDA  : [input] leading dimension of A
C LDU  : [input] leading dimension of U
C LDW  : [input] leading dimension of W
C NCOLS: [input] no. of columns of A = no. parameters
C NROWS: [input] no. of rows of A = no. of observations
C A    : [input] matrix originally supplied to DGELSS
C SIGMA: [input] singular values returned by DGELSS
C U    : [output] used as workspace then returned with (A(transpose)*A)(inverse)
C W    : [input/output]
C        This is the V-matrix as formatted for the NAG G02 GLIM routine calls
C        On input the right singular vectors from DGELSS are stored row-wise
C        in columns 8 to NPAR + 7 (i.e. W actually contains V(transpose))
C        Unchanged on output except that column 6 returns the leverages
C
      IMPLICIT NONE
C
C Arguments supplied
C
      INTEGER  IFAIL, IRANK, LDA, LDU, LDW, NCOLS, NROWS
      DOUBLE PRECISION A(LDA,NCOLS), SIGMA(IRANK), U(LDU,NCOLS),
     +                 W(LDW,NCOLS + 7)
C
C Local variables
C
      INTEGER I, J, K
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO = 0.0D+00)
C
C Is it safe
C
      IFAIL = 0
      IF (IRANK.LE.0 .OR. IRANK.GT.NCOLS .OR. NCOLS.LT.1) THEN
         IFAIL = 1
         RETURN
      ENDIF
      IF (NROWS.LE.NCOLS) THEN
         IFAIL = 2
         RETURN
      ENDIF
      IF (NROWS.GT.LDA .OR. NROWS.GT.LDU .OR. NROWS.GT.LDW) THEN
         IFAIL = 3
         RETURN
      ENDIF
C
C Form U = A*V*Sigma(inverse)
C
      DO I = 1, NROWS
         DO J = 1, IRANK
            U(I,J) = ZERO
            DO K = 1, NCOLS
               U(I,J) = U(I,J) + A(I,K)*W(J,K + 7)/SIGMA(J)
            ENDDO
         ENDDO
      ENDDO
C
C Form W(i,6) = H = Diag(U*U(transpose))
C
      DO I = 1, NROWS
         W(I,6) = ZERO
         DO J = 1, IRANK
            W(I,6) = W(I,6) + U(I,J)*U(I,J)
         ENDDO
      ENDDO
C
C Form U = V*Sigma(inverse-squared)*V(transpose)
C
      DO I = 1, NCOLS
         DO J = 1, NCOLS
            U(I,J) = ZERO
            DO K = 1, IRANK
               U(I,J) = U(I,J) + W(K,I + 7)*W(K,J + 7)/(SIGMA(K)**2)
            ENDDO
         ENDDO
      ENDDO
C
C Multiply IRANK rows of W
C
      DO I = 1, IRANK
         DO J = 1, NCOLS
            W(I, J + 7) = W(I,J + 7)/SIGMA(I)
         ENDDO
      ENDDO
      END
C
C
