C
C------------------------------------
C Simfit version of G03CAF 24/02/2013 
C------------------------------------
C
      SUBROUTINE G03CAF$(MATRIX, WEIGHT, N, M, X, LDX, NVAR, ISX,
     +                   NFAC, WT, E, STAT, COM, PSI, RES, FL, LDFL,
     +                   IOP, IWK, WK, LWK, IFAIL)
C
C Allocations: Some of these could be provided by more careful partitioning of WK when time permits 
C              but in this version the following are allocated:
C              R(NVAR,NVAR) in main subroutine to hold the correlation matrix
C              WORK in the model subroutine for LAPACK workspace
C              W1, W2 in the fitting program for LBFGSB workspace
C              W, SSP and FREQ in the data transformation subroutine (will only be required when X is raw data)
C              A in G03CAF_FIT is used for starting estimates only
C Additional workspace are as supplied by partitioning IWK and WK
C If IOPT(1) > 0 iterations are written to unit = NF which must be opened by the
C calling program. The value of IPRINT is not used and f(psi) is printed at each iteration
C which in this version uses NF = 8. 
C                   
      IMPLICIT NONE
C
C arguments
C      
      INTEGER N, M, LDX, NVAR, ISX(M), NFAC, LDFL, IOP(5),
     +        IWK(4*NVAR + 2), LWK, IFAIL
      DOUBLE PRECISION X(LDX,M), WT(*), E(NVAR), STAT(4), COM(NVAR),
     +                 PSI(NVAR), RES(NVAR*(NVAR - 1)/2),
     +                 FL(LDFL,NFAC), WK(LWK)
      CHARACTER MATRIX*1, WEIGHT*1
C
C allocatable
C  
      DOUBLE PRECISION, ALLOCATABLE :: R(:,:)
C
C locals
C      
      INTEGER    I, I1, I2, IADD1, J, J1, J2, J3, J4, J5, K, N2_USE
      INTEGER    IERR, IR
      INTEGER    IPRINT, MAXFUN, NF
      DOUBLE PRECISION G01ECF$
      DOUBLE PRECISION ACC, EPS, F, WTSUM, WTMIN
      DOUBLE PRECISION ZERO, ONE, TWO, THREE, FIVE, SIX, TEN, MACHEPS
      PARAMETER (ZERO = 0.0D+00, ONE = 1.0D+00, TWO = 2.0D+00,
     +           THREE = 3.0D+00, FIVE = 5.0D+00, SIX = 6.0D+00,
     +           TEN = 10.0D+00, MACHEPS = 1.111307226798D-016)
      CHARACTER  MATRIX_COPY*1, WEIGHT_COPY*1
      EXTERNAL   G03CAF_DATA, G03CAF_FIT
      EXTERNAL   G01ECF$
      INTRINSIC  MAX, NINT, DBLE, SQRT
C
C PART 1: initialise IFAIL and arrays then check input details
C      
      IFAIL = 0
      DO I = 1, NVAR
         E(I) = ZERO
         COM(I) = ZERO
         PSI(I) = ZERO
      ENDDO
      DO I = 1, 4
         STAT(I) = ZERO
      ENDDO 
      DO I = 1, NVAR*(NVAR - 1)/2
         RES(I) = ZERO
      ENDDO
      DO J = 1, NFAC
         DO I = 1, NVAR
            FL(I,J) = ZERO
         ENDDO
      ENDDO               
C
C check 1:  MATRIX
C
      IF (MATRIX.EQ.'D' .OR. MATRIX.EQ.'d') THEN
         IF (LDX.LT.N) THEN
            IFAIL = 1
            RETURN
         ENDIF   
         MATRIX_COPY = 'D'
      ELSEIF (MATRIX.EQ.'S' .OR. MATRIX.EQ.'s') THEN 
         IF (LDX.LT.N) THEN
            IFAIL = 1
            RETURN
         ENDIF   
         MATRIX_COPY = 'S'
      ELSEIF (MATRIX.EQ.'C' .OR. MATRIX.EQ.'c') THEN
         MATRIX_COPY = 'C'
         IF (LDX.LT.M) THEN
            IFAIL = 1
            RETURN
         ENDIF   
      ELSE
         IFAIL = 1
         RETURN
      ENDIF 
C
C check 2: WEIGHT and LWK
C      
      IF (MATRIX_COPY.EQ.'D' .OR. MATRIX_COPY.EQ.'S') THEN
         IF (WEIGHT.EQ.'U' .OR. WEIGHT.Eq.'u') THEN
            WEIGHT_COPY = 'U'
         ELSEIF (WEIGHT.EQ.'W' .OR. WEIGHT.EQ.'w') THEN
            WEIGHT_COPY = 'W'
         ELSE
            IFAIL = 1
            RETURN
         ENDIF
         IF (LWK.LT.MAX((5*NVAR*NVAR + 33*NVAR - 4)/2,
     +                   N*NVAR + 7*NVAR + NVAR*(NVAR - 1)/2)) THEN
            IFAIL = 1
            RETURN
         ENDIF  
      ELSE
         IF (LWK.LT.(5*NVAR*NVAR + 33*NVAR - 4)/2) THEN
            IFAIL = 1
            RETURN
         ENDIF                 
         WEIGHT_COPY = 'U'
      ENDIF             
C
C check 3: input parameters
C     
      IF (LDFL.LT.NVAR .OR.
     +    NVAR.LT.2    .OR.
     +    N.LE.NVAR    .OR.
     +    NFAC.LT.1    .OR.
     +    NVAR.LT.NFAC .OR.
     +    M.LT.NVAR) THEN
          IFAIL = 1
          RETURN 
      ENDIF   
C
C check 4: IOP
C     
      IF (IOP(1).NE.0) THEN
         IPRINT = IOP(2)
         IF (IOP(4).GE.2 .AND. IOP(4).LE.7) THEN
            I = -IOP(4)
         ELSE
            I = -2
         ENDIF      
         ACC = TEN**I
         IF (IOP(5).GE.3 .AND. IOP(5).LE.10) THEN
            I = -IOP(5)
         ELSE
            I = -5
         ENDIF      
         EPS = TEN**I
         IF (IOP(3).LT.1    .OR.
     +       ACC.GE.ONE     .OR. 
     +       ACC.LT.MACHEPS .OR.
     +       EPS.GE.ONE     .OR.
     +       EPS.LT.MACHEPS) THEN
             IFAIL = 1
             RETURN
         ENDIF  
         MAXFUN = IOP(3)
      ELSE
         IPRINT = -1
         MAXFUN = 100*NVAR
         ACC = TEN**(-2) 
         EPS = TEN**(-5)   
      ENDIF 
      IF (MAXFUN.LT.1000) MAXFUN = 1000
C
C check 5: WEIGHT
C      
      IF (WEIGHT_COPY.EQ.'W') THEN
         WTMIN = SQRT(MACHEPS)
         WTSUM = ZERO
         DO I = 1, N
            IF (WT(I).LT.ZERO) THEN
               IFAIL = 1
               RETURN
            ELSEIF (WT(I).GT.WTMIN) THEN   
               WTSUM = WTSUM + WT(I)
            ENDIF
        ENDDO 
        N2_USE = NINT(WTSUM)
        IF (N2_USE.LT.NVAR) THEN
            IFAIL = 1
            RETURN
         ENDIF 
      ELSE
         WTMIN = ZERO
         WTSUM = DBLE(N)
         N2_USE = N           
      ENDIF   
C
C check 6: ISX
C     
      IADD1 = 0
      DO I = 1, M
         IF (ISX(I).GT.0) IADD1 = IADD1 + 1
      ENDDO
      IF (IADD1.NE.NVAR) THEN
         IFAIL = 3
         RETURN
      ENDIF
C
C PART 2: allocate R and calculate the correlation matrix if required
C
      IR = NVAR
      IERR = 0
      IF (ALLOCATED(R)) DEALLOCATE(R, STAT = IERR)
      IF (IERR.NE.0) THEN
         IFAIL = 100
         RETURN
      ENDIF
      ALLOCATE(R(IR,IR), STAT = IERR)
      IF (IERR.NE.0) THEN
         IFAIL = 101
         RETURN
      ENDIF  
C
C manipulate the data
C          
      J1 = 1
      J2 = NVAR + 1
C WK(1)        ... WK(NVAR) = STD   
C WK(NVAR + 1) ... WK(2*NVAR) = XBAR
      IFAIL = 0  
      CALL G03CAF_DATA (IFAIL, ISX, LDX, N, NVAR, M,
     +                  X, R, WK(J1), WT, WK(J2), WTMIN,
     +                  MATRIX_COPY, WEIGHT_COPY)
      IF (IFAIL.NE.0) RETURN
C
C PART 3: partition workspaces then fit the model
C
      I1 = NVAR + 1
      I2 = 1
C     IWK(1)        ... IWK(NVAR) = NBD(NVAR)
C     IWK(NVAR + 1) ... IWK(4*NVAR + 1) = IW(3*NVAR)
      J1 = NVAR + 1
      J2 = 2*NVAR + 1
      J3 = 3*NVAR + 1
      J4 = NVAR*NVAR + 4*NVAR + 1
      J5 = NVAR*NVAR + 5*NVAR + 1
C     WK(1)                      ... WK(NVAR) = SCALING FACTORS
C     WK(NVAR + 1)               ... WK(2*NVAR) = BL
C     WK(2*NVAR + 1)             ... WK(3*NVAR) = BU
C     WK(3*NVAR + 1)             ... WK(NVAR*NVAR + 4*NVAR) = EIGVEC
C     WK(NVAR*NVAR + 4*NVAR + 1) ... WK(NVAR*NVAR + 5*NVAR)) = G
C     WK(NVAR*NVAR + 5*NVAR + 1) ... WK(NVAR*NVAR + 6*NVAR)) = PSI_INV
      
      IF (IPRINT.GT.0) THEN
         NF = 8
      ELSE
         NF = 0
      ENDIF       
      
      CALL G03CAF_FIT (IFAIL, IWK(I1), NFAC, LDFL, MAXFUN, NVAR,
     +                 IWK(I2), NF, 
     +                 ACC, WK(J1), WK(J2), WK(J3), EPS, F, FL, WK(J4),
     +                 WK(J5), R, E, PSI)
C
C PART 4: calculate communalities and other parameters
C   
      DO I = 1, NVAR
         DO J = 1, NFAC
            COM(I) = COM(I) + FL(I,J)**2
         ENDDO   
      ENDDO
C
C calculate residual correlations in column major order
C      
      IADD1 = 0
      DO J = 2, NVAR
         DO I = 1, J - 1
            IADD1 = IADD1 + 1
            WTSUM = ZERO
            DO K = 1, NFAC
               WTSUM = WTSUM + FL(I,K)*FL(J,K)
            ENDDO  
            RES(IADD1) = R(I,J) - WTSUM 
         ENDDO  
      ENDDO               
C
C fill in STAT
C      
      STAT(1) = F
      STAT(2) = (DBLE(N2_USE) - ONE - (TWO*DBLE(NVAR) + FIVE)/SIX - 
     +          TWO*DBLE(NFAC)/THREE)*F
      STAT(3) = DBLE((NVAR - NFAC)**2 - NVAR - NFAC)/TWO
      IF (STAT(3).GT.ZERO) THEN
         STAT(4) = G01ECF$('U', STAT(2), STAT(3), IFAIL)
      ELSE
         STAT(3) = ZERO
         STAT(4) = ZERO
      ENDIF      
      IFAIL = 0!in case G01ECF returns nonzero IFAIL
C
C rescale PSI and FL values if MATRIX = 'S'
C      
      IF (MATRIX_COPY.EQ.'S' .OR. MATRIX_COPY.EQ.'C') THEN
         DO I = 1, NVAR
            PSI(I) = PSI(I)*WK(I)**2
            DO J = 1, NFAC
               FL(I,J) = FL(I,J)*WK(I)
            ENDDO  
         ENDDO
      ENDIF   

C
C deallocate R
C
      DEALLOCATE (R, STAT = IERR)

      END
C
C--------------------------------------------------------------------
C
      SUBROUTINE G03CAF_DATA (IFAIL, ISX, LDX, N, NVAR, M,
     +                        X, R, STD, WT, XBAR, WTMIN,
     +                        MATRIX, WEIGHT)
C
C ACTION: return correlation matrix R and W(i,1) containing standard errors
C AUTHOR: w.g.bardsley, university of manchester, u.k., 24/02/2013
C   
   
C
C arguments
C     
      INTEGER,             INTENT (INOUT) :: IFAIL
      INTEGER,             INTENT (IN)    :: LDX, N, NVAR, M
      INTEGER,             INTENT (IN)    :: ISX(M)
      DOUBLE PRECISION,    INTENT (IN)    :: X(LDX,M), WT(*), WTMIN
      DOUBLE PRECISION,    INTENT (OUT)   :: R(NVAR,NVAR), STD(NVAR),
     +                                       XBAR(NVAR) 
      CHARACTER (LEN = 1), INTENT (IN)    :: MATRIX, WEIGHT
C
C allocatable
C      
      DOUBLE PRECISION, ALLOCATABLE :: SSP(:,:), FREQ(:), W(:,:)
C
C locals
C      
      INTEGER    I, IADD1, ICOL, IERR, IFREQ, IR, IX, J, JADD1 
      DOUBLE PRECISION ONE, EPSI, RLOW, RHIGH 
      PARAMETER (ONE = 1.0D+00, EPSI = 1.0D-07,
     +           RLOW = ONE - EPSI, RHIGH = ONE + EPSI)
      LOGICAL    USE_WEIGHTS
      EXTERNAL   G02BAF$, G02BAF_WEIGHT 
      INTRINSIC  SQRT, DBLE
      
      IF (MATRIX.EQ.'D' .OR. MATRIX.EQ.'S') THEN
C
C allocate in order to calculate R 
C        
         IERR = 0
         
         IF (ALLOCATED(SSP)) DEALLOCATE(SSP, STAT = IERR)
         IF (IERR.NE.0) THEN
             IFAIL = 102
             RETURN
         ENDIF
         IR = NVAR
         ALLOCATE (SSP(IR,IR), STAT = IERR)
         IF (IERR.NE.0) THEN
            IFAIL = 103
            RETURN
         ENDIF

         IF (ALLOCATED(FREQ)) DEALLOCATE(FREQ, STAT = IERR)
         IF (IERR.NE.0) THEN
             IFAIL = 104
             RETURN
         ENDIF
         IFREQ = N
         ALLOCATE (FREQ(IFREQ), STAT = IERR)
         IF (IERR.NE.0) THEN
            IFAIL = 105
            RETURN
         ENDIF

         IF (ALLOCATED(W)) DEALLOCATE(W, STAT = IERR)
         IF (IERR.NE.0) THEN
             IFAIL = 106
             RETURN
         ENDIF
         IFREQ = N
         ICOL = NVAR
         ALLOCATE (W(IFREQ,ICOL), STAT = IERR)
         IF (IERR.NE.0) THEN
            IFAIL = 107
            RETURN
         ENDIF
c
c define W
c         
         IF (WEIGHT.EQ.'W') THEN
            IADD1 = 0 
            DO I = 1, N
               IF (WT(I).GT.WTMIN) THEN
                  IADD1 = IADD1 + 1
                  FREQ(IADD1) = WT(I)
                  JADD1 = 0
                  DO J = 1, M
                     IF (ISX(J).GT.0) THEN
                        JADD1 = JADD1 + 1
                        W(IADD1,JADD1) = X(I,J)
                     ENDIF   
                  ENDDO
               ENDIF      
            ENDDO  
            USE_WEIGHTS = .TRUE.
         ELSE
            USE_WEIGHTS = .FALSE.
            IADD1 = N
            JADD1 = 0
            DO J = 1, M
               IF (ISX(J).GT.0) THEN
                  JADD1 = JADD1 + 1
                  DO I = 1, N
                     W(I,JADD1) = X(I,J)
                  ENDDO  
               ENDIF
            ENDDO      
         ENDIF   
         
         IF (USE_WEIGHTS) THEN
            IX = N
            IFAIL = 0
            CALL G02BAF_WEIGHT (IADD1, NVAR, W, IX, XBAR, STD, SSP, IR,
     +                          R, IR, FREQ, IFAIL)
         ELSE  
            IX = N
            IFAIL = 0
            CALL G02BAF$(IADD1, NVAR, W, IX, XBAR, STD, SSP, IR, R, IR,
     +                   IFAIL)
         ENDIF  
         IF (IFAIL.NE.0) THEN
            IFAIL = 200 + I
            RETURN
         ENDIF 
C
C deallocate
C     
         DEALLOCATE (SSP, STAT = IERR) 
         DEALLOCATE (FREQ, STAT = IERR) 
         DEALLOCATE (W, STAT = IERR) 
      ELSE 
c
c fill in R from X supplied
c        
         IADD1 = 0
         DO I = 1, M
            IF (ISX(I).GT.0) THEN
               IADD1 = IADD1 + 1
               JADD1 = IADD1 - 1
               DO J = I, M
                  IF (ISX(J).GT.0) THEN
                     JADD1 = JADD1 + 1
                     R(IADD1,JADD1) = X(I,J)
                  ENDIF    
               ENDDO
            ENDIF     
         ENDDO    
         DO I = 2, NVAR
            DO J = 1, I - 1
               R(I,J) = R(J,I) 
            ENDDO  
         ENDDO
c
c check if R is already a correlation matrix then scale if required
c           
         IADD1 = 0
         DO I = 1, NVAR
            IF (R(I,I).GE.RLOW .AND. R(I,I).LE.RHIGH) IADD1 = IADD1 + 1
         ENDDO 
         IF (IADD1.EQ.NVAR) THEN
            DO I = 1, NVAR
               STD(I) = ONE
            ENDDO  
         ELSE       
            DO I = 1, NVAR
               STD(I) = SQRT(R(I,I))
            ENDDO
            DO J = 1, NVAR
               DO I = 1, NVAR
                  R(I,J) = R(I,J)/(STD(I)*STD(J))
               ENDDO  
            ENDDO  
         ENDIF   
      ENDIF
      END
C
c-----------------------------------------------------------------------
c
      subroutine g03caf_fit (ifail, iw, k, ldfl, maxfun, n, nbd, nf, 
     +                       acc, bl, bu, eigvec, eps, f, fl, g, 
     +                       psi_inv, s, theta, x)
c
c action : call setulb for minimisation
c author : w.g.bardsley, university of manchester, u.k.
c          25/02/2013 derived from qnfit0 
c
c Note: There are two possible model evaluation subroutines
c       g03caf_func1 uses spectral decomposition and lwrk at least 64*n 
c       g03caf_func2 uses cholesky/svd and lwrk at least 5*n
c
c ifail: should be 0 on entry, o/w error exit
c     k: number of factors
c     n: number of variables
c    nf: unit to write output
c     s: sample correlation matrix
c theta: eigenvalues
c     x: psi estimates
c   

c
c advice : this is a replacement for e04jaf in the simfit quasi newton
c          programs
c dimensions :
c          note: liw >= 3*n
c                lw1 >= 2*(2*m*n + 4*n + 11*m*m + 8*m)
c                lw2 >= 3*n
c arguments : deriv = subroutine to calculate function g (may need w2)
c             funct = subroutine to calculate gradient f
c             iw    = integer workspace
c             liw   = size of iw
c             lw1   = size of w1
c             lw2   = size of w2
c             n     = size of bl, bu, g, x
c             nbd   = type of bounds (0 = unbounded, 1 = only lower,
c                     2 = both, 3 = only upper)
c             nf    = output unit
c             bl    = lower bounds
c             bu    = upper bounds
c             g     = gradient
c             w1    = double precision workspace for optimiser
c             w2    = double precision workspace for gradient
c             x     = variables
c
c extra parameters : m = no. corrections in limited memory matrix 3=< m =< 20
c                    iprint = frequency and type of output < 0 => no output
c                    factr = tolerance 1.d+7 for moderate accuracy
c                    pgtol = tolerance on projected gradient
c
      implicit         none
c
c arguments
c      
      integer,             intent (in)    :: k, ldfl, maxfun, n, nf
      integer,             intent (inout) :: ifail, iw(3*n), nbd(n)
      double precision,    intent (in)    :: acc, eps, s(n,n)   
      double precision,    intent (out)   :: bl(n), bu(n),
     +                                       eigvec(n,n + 1), f,
     +                                       fl(ldfl,k), g(n), 
     +                                       psi_inv(n), theta(n), x(n)
c
c allocatable
c     
      double precision, allocatable :: a(:,:), work(:), w1(:), w2(:) 
c
c locals
c      
      integer          ia, ierr, ios, lwrk, lw1, lw2
      integer          i, j, m, mtry, iprint, isave(44)
      parameter       (m = 10, iprint = - 1)
      integer          maxfev, numfev
      double precision col_fac, row_fac
      double precision factr, pgtol, dsave(29)
      double precision one, two, ftol
      parameter (one = 1.0d+00, two = 2.0d+00,
     +           ftol = 1.0d-10)
      double precision xmax
      parameter (xmax = 1.0d+03)
      character        csave*60, task*60
      logical          lsave(4), op
      external         f01adf$, g03caf_deriv, g03caf_func2, setulb
      intrinsic        abs, dble, sqrt
            
      if (ifail.ne.0) return
      
      ifail = 300
      ierr = 0
c
c calculate starting estimates and check for pos. def.
c
      ia = n + 1
      if (allocated(a)) deallocate(a, stat = ierr) 
      if (ierr.ne.0) return   
      allocate(a(ia,ia), stat = ierr)
      if (ierr.ne.0) return 

      do i = 1, n
         do j = i,n
            a(i,j) = s(i,j) 
         enddo
      enddo  
       
      ifail = 0     
      call f01adf$(n, a, ia, ifail)
      if (ifail.ne.0) then
         ifail = 4 
         return
      endif
      
      j = 0
      do i = 1, n
         j = j + 1
         bu(i) = a(i + 1,j)
      enddo
      
      ifail = 300
      
      deallocate(a, stat = ierr)
      if (ierr.ne.0) return

      f = dble(k)/dble(n)
      f = one - f/two
      do i = 1, n
         x(i) = f/bu(i)
      enddo   
c
c allocate workspace
c      
      lw1 = 2*(2*m*n + 4*n + 11*m*m + 8*m)
      lw2 = 3*n
c      lwrk = 64*n!value if g03caf_func_1 is used
      lwrk = 10*n
      ifail = 300
      ierr = 0

      if (allocated(work)) deallocate(work, stat = ierr) 
      if (ierr.ne.0) return   
      allocate(work(lwrk), stat = ierr)
      if (ierr.ne.0) return    

      if (allocated(w1)) deallocate(w1, stat = ierr) 
      if (ierr.ne.0) return   
      allocate(w1(lw1), stat = ierr)
      if (ierr.ne.0) return  

      if (allocated(w2)) deallocate(w2, stat = ierr) 
      if (ierr.ne.0) return   
      allocate(w2(lw2), stat = ierr)
      if (ierr.ne.0) return  

      ifail = 0

c
c define type, limits, and check starting estimates 
c      
      do i = 1, n
         nbd(i) = 2
         bl(i) = eps
         bu(i) = xmax 
         if (x(i).lt.bl(i) .or. x(i).gt.bu(i)) x(i) = two*bl(i)
      enddo    
c
c Calculate starting f value
c
      call g03caf_func2 (ifail, k, lwrk, n,
     +                   eigvec, f, x, psi_inv, s, theta, work)
      if (ifail.ne.0) return
c
c We start the iteration by initializing task.
c
      task = 'START'
c
c Now we decide on the precision ... low, medium or high
c
c      if (type2(1:1).eq.'l' .or. type2(1:1).eq.'L') then
         mtry = m - 4
         factr = 1.0d+10
         pgtol = acc
c         pgtol = 1.0d-5
c      elseif (type2(1:1).eq.'h' .or. type2(1:1).eq.'H') then
c         mtry = m
c         factr = 1.0d+1
c         pgtol = 1.0d-7
c      else
c         mtry = m - 2
c         factr = 1.0d+5
c         pgtol = 1.0d-6
c      endif
c
c ------- the beginning of the loop ----------
c
      ios = -1
      op = .false.
      if (nf.gt.0) inquire (unit = nf, opened = op, iostat = ios)
      maxfev = maxfun
      numfev = 0
      if (ios.eq.0 .and. op) write (nf,'(i8,1p,e14.4)') numfev, f
      ifail = 0  
  20  continue
c
c This is the call to the L-BFGS-B code.
c
      call setulb (n, mtry, x, bl, bu, nbd, f, g, factr, pgtol, w1,
     +             iw, task, iprint, csave, lsave, isave, dsave)

      if (task(1:2) .eq. 'FG') then
c
c the minimization routine has returned to request the
c function f and gradient g values at the current x.
c

c
c Increment the number of function evaluations
c
         numfev = numfev + 1
c
c Check numfev and see if fitting has been interupted, i.e. nopen = 0
c
         if (numfev.eq.maxfev) then
            ifail = 6
            goto 40
         endif
         if (ios.eq.0 .and. op) write (nf,'(i8,1p,e14.4)') numfev, f
         call g03caf_func2 (ifail, k, lwrk, n,
     +                      eigvec, f, x, psi_inv, s, theta,
     +                      work)
         if (ifail.ne.0) return
         call g03caf_deriv (k, n,
     +                      g, eigvec, psi_inv, theta)
c
c go back to the minimization routine.
c
         goto 20
      endif
c
c
      if (task(1:5) .eq. 'NEW_X')  then
c
c the minimization routine has returned with a new iterate,
c

         if (numfev.lt.2 .or. dsave(13).gt.ftol*(one + abs(f))) then
            goto 20
         else
            task =
     +     'CONVERGENCE: projected gradient < 1.d-10*(1 + abs(f))'
         endif
      endif
c           ---------- the end of the loop -------------
c
c If task is neither FG nor NEW_X we terminate execution.
c
   40 continue
      if (ifail.eq.6) then
         ifail = 6!to silence ftn95
      elseif (task(1:4) .eq. 'CONV' .or.
     +        task .eq. 'ABNORMAL_TERMINATION_IN_LNSRCH') then
         ifail = 0
      else
         ifail = 1
      endif
      call g03caf_func2 (ifail, k, lwrk, n,
     +                   eigvec, f, x, psi_inv, s, theta, work)
      if (ifail.ne.0) return
c
c calculate fl
c 
      do j = 1, k
         col_fac = sqrt(theta(j) - one)
         do i = 1, n
            fl(i,j) = eigvec(i,j)*col_fac
         enddo  
      enddo 
      do i = 1, n
         row_fac = sqrt(x(i))
         do j = 1, k
           fl(i,j) = fl(i,j)*row_fac
         enddo
      enddo         
c
c deallocate workspaces
c
      deallocate(work, stat = ierr)
      deallocate(w1, stat = ierr)
      deallocate(w2, stat = ierr)
      end
c
c-------------------------------------------------------------------- 
c
      subroutine g03caf_deriv (k, p,
     +                         deriv, eigvec, psi_inv, theta)
c
c action: calculate the derivative function 
c author: w.g.bardsley, 25/02/2013
c  
c The method used is M.R.B.Clarke, Br.J.Math.Statist.Psych, 23, 1970, 43-52
c   
      implicit none
c
c arguments
c      
      integer,          intent (in)    :: k, p
      double precision, intent (in)    :: eigvec(p,p + 1), psi_inv(p),
     +                                    theta(p) 
      double precision, intent (out)   :: deriv(p)
c
c locals
c
      integer i, j
      double precision zero, one
      parameter (zero = 0.0d+00, one = 1.0d+00)
      do i = 1, p
         deriv(i) = zero
         do j = k + 1, p
            deriv(i) = deriv(i) + (one - theta(j))*eigvec(i,j)**2
         enddo
         deriv(i) = psi_inv(i)*deriv(i)  
      enddo  
      end
c
c
c
c---------------------------------------------------------------------
c
      subroutine g03caf_func1 (ifail, k, lwrk, p,
     +                         eigvec, obj_func, psi, psi_inv, s, theta,
     +                         work)
c
c action: calculate the factor analysis objective function using spectral decomposition
c author: w.g.bardsley, 25/02/2013
c 
c Note as follows: 1) the method used is M.R.B.Clarke, Br.J.Math.Statist.Psych, 23, 1970, 43-52
c                  2) eigvec is dimensioned p, p + 1 to be on the safe side
c                  3) the spectral decomposition (dsyev) can be replaced by cholesky/svd
c                  4) ifail = 5 is returned if problems are encountered
c                  5) lwrk should be at least 64*p 
c                  6) the parameter one used in defining obf_func is not actually necessary
c   
      implicit none
c
c arguments
c      
      integer,          intent (inout) :: ifail
      integer,          intent (in)    :: k, lwrk, p
      double precision, intent (in)    :: psi(p), s(p,p) 
      double precision, intent (inout) :: work(lwrk)
      double precision, intent (out)   :: eigvec(p,p + 1), obj_func,
     +                                    psi_inv(p), theta(p) 
c
c locals
c     
      integer    i, info, j, lda, n
      double precision zero, epsi, one
      parameter (zero = 0.0d+00, epsi = 1.0d-10, one = 1.0d+00)
      character (len = 1) job, uplo
      parameter (job = 'V', uplo = 'U')
      external   dsyev
      intrinsic  sqrt, log
c
c initialise obj_func
c      
      obj_func = zero
c
c calculate psi_inv
c       
      do i = 1, p
         if (psi(i).ge.epsi) then
            psi_inv(i) = one/sqrt(psi(i)) 
         else
            ifail = 5
            return
         endif      
      enddo  
c
c calculate star matrix
c      
      do j = 1, p
         do i = 1, j
            eigvec(i,j) = s(i,j)*psi_inv(i)*psi_inv(j)
         enddo  
      enddo   
c
c spectral decomposition ... Note: eigenvalues are returned in INCREASING ORDER
c      
      lda = p
      n = p
      call dsyev (job, uplo, n, eigvec, lda, theta, work, lwrk, info)
      if (info.ne.0) then
         ifail = 5
         return
      endif
c
c reverse the order of the eigenvalues
c      
      do i = 1, n
         work(i) = theta(i)
      enddo
      do i = 1, n
         theta(i) = work(n - i + 1)
      enddo
c
c reverse the columns of eigenvectors
c 
      do i = 1, n
         do j = 1, n
            work(j) = eigvec(i,j)
         enddo
         do j = 1, n
            eigvec(i,j) = work(n - j + 1)
         enddo      
      enddo       
c
c define obj_func
c
      do i = k + 1, p
         if (theta(i).gt.epsi) then
            obj_func = obj_func + theta(i) - log(theta(i)) - one
         else
            ifail = 5
            return   
         endif   
      enddo
      end

c
c---------------------------------------------------------------------
c
      subroutine g03caf_func2 (ifail, k, lwrk, p,
     +                         eigvec, obj_func, psi, psi_inv, s, theta,
     +                         work)
c
c action: calculate the factor anakysis objective function using cholesky/svd
c author: w.g.bardsley, 25/02/2013
c 
c Note as follows: 1) the method used is M.R.B.Clarke, Br.J.Math.Statist.Psych, 23, 1970, 43-52
c                  2) eigvec is dimensioned p, p + 1 as SVD is used
c                  3) cholesky/svd (dpotrf/dgesvd) is used
c                  4) ifail = 5 is returned if problems are encountered
c                  5) lwrk should be at least 5*p
c                  6) the parameter one used in defining obf_func is not actually necessary
c   
      implicit none
c
c arguments
c      
      integer,          intent (inout) :: ifail
      integer,          intent (in)    :: k, lwrk, p
      double precision, intent (in)    :: psi(p), s(p,p) 
      double precision, intent (inout) :: work(lwrk)
      double precision, intent (out)   :: eigvec(p,p + 1), obj_func,
     +                                    psi_inv(p), theta(p) 
c
c allocatables
c
      double precision, allocatable :: a(:,:)
c     
c
c locals
c     
      integer    i, ierr, info, j, lda, ldu, ldvt, m, n
      double precision u(1,1)
      double precision zero, epsi, one
      parameter (zero = 0.0d+00, epsi = 1.0d-10, one = 1.0d+00)
      character (len = 1) jobu, jobvt, uplo
      parameter (jobu = 'N', jobvt = 'A', uplo = 'U')
      external   dpotrf, dgesvd
      intrinsic  sqrt, log
c
c initialise obj_func
c      
      obj_func = zero
c
c calculate psi_inv
c       
      do i = 1, p
         if (psi(i).ge.epsi) then
            psi_inv(i) = one/sqrt(psi(i)) 
         else
            ifail = 5
            return
         endif      
      enddo  
c
c calculate cholesky factors
c      
      ifail = 300
      lda = p
      ierr = 0
      if (allocated(a)) deallocate(a, stat = ierr)
      if (ierr.ne.0) return 
      allocate (a(lda,lda + 1), stat = ierr)
      if (ierr.ne.0) return 
      
      ifail = 0   
      do i = 1, p
         do j = i, p
            a(i,j) = s(i,j)
         enddo  
      enddo  
       
      lda = p
      n = p
      call dpotrf (uplo, n, a, lda, info)
      if (info.ne.0) then
         ifail = 5
         return
      endif
      
      do j = 1, p
         do i = 1, j
            a(i,j) = a(i,j)*psi_inv(j)
         enddo  
      enddo  
      do i = 2, p
         do j = 1, i - 1
            a(i,j) = zero
         enddo   
      enddo   
c
c svd ... Note: singular values are returned in decreasing order
c     
      ldu = 1 
      ldvt = p
      m = p
      n = p
      call dgesvd (jobu, jobvt, m, n, a, lda, theta, u, ldu, eigvec,
     +             ldvt, work, lwrk, info) 
      if (info.ne.0) then
         ifail = 5
         return
      endif
c
c square the singular values to make eigenvalues
c      
      do i = 1, n
         theta(i) = theta(i)**2
      enddo
c
c transpose eigvec
c      
      do j = 1, p
         do i = 1, p
            a(i,j) = eigvec(i,j)
         enddo  
      enddo  
      do j = 1, p
         do i = 1, p
            eigvec(i,j) = a(j,i)
         enddo
      enddo  
        
      deallocate(a, stat = ierr)
c
c define obj_func
c
      do i = k + 1, p
         if (theta(i).gt.epsi) then
            obj_func = obj_func + theta(i) - log(theta(i)) - one
         else
            ifail = 5
            return   
         endif   
      enddo
      end      
c
c-------------------------------------------------------------------------
C
C
      SUBROUTINE G02BAF_WEIGHT (N, M, X, IX, XBAR, STD, SSP, ISSP, R,
     +                          IR, WT, IFAIL)
C
C ACTION : Correlation coefficients using weighted data
C AUTHOR : W.G.Bardsley, University of Manchester, U.K.
C          02/03/2013 adapted from G02BAF$
C
C          IFAIL is not checked on entry so it is like IFAIL = 1
C
C          This routine assumes that WT(i) = 0 are frequencies that have been used to eliminate 
C          X(I,*) for any cases with WT(i) =< 0 so that the arguments are as follows: 
C          X(I,J) = original NOT weighted X(I,J)
C          WT(i) = nonzero weights i.e. frequencies which must all be >= 0
C
      IMPLICIT NONE
      INTEGER,          INTENT (INOUT) :: IFAIL
      INTEGER,          INTENT (IN)    :: IR, ISSP, IX, M, N
      DOUBLE PRECISION, INTENT (IN)    :: X(IX,M), WT(N)
      DOUBLE PRECISION, INTENT (OUT)   :: R(IR,M), SSP(ISSP,M), STD(M),
     +                                    XBAR(M)
C
C Local variables
C
      INTEGER    I, J, K
      DOUBLE PRECISION WTSUM
      DOUBLE PRECISION ZERO, ONE, TWO, EPSI
      PARAMETER (ZERO = 0.0D+00, ONE = 1.0D+00, TWO = 2.0D+00, 
     +           EPSI = 1.0D-300)
      DOUBLE PRECISION DN, DNM1, TEMP, XB, XJ, XK
      INTRINSIC  DBLE, SQRT
C
C Is it safe ?
C
      IFAIL = 0
      IF (N.LT.2) THEN
         IFAIL = 1
         RETURN
      ELSEIF (M.LT.2) THEN
         IFAIL = 2
         RETURN
      ELSEIF (IX.LT.N .OR. ISSP.LT.M .OR. IR.LT.M) THEN
         IFAIL = 3
         RETURN
      ENDIF
C
C Calculate WTSUM
C      
      WTSUM = ZERO
      DO I = 1, N
         IF (WT(I).LT.ZERO) THEN
            IFAIL = 4
            RETURN
         ENDIF   
         WTSUM = WTSUM + WT(I)
      ENDDO  
      IF (WTSUM.LT.TWO) THEN
         IFAIL = 5
         RETURN
      ENDIF          
C
C Initialise all variables to be calculated as sums
C
      DO J = 1, M
         STD(J) = ZERO
         XBAR(J) = ZERO
         DO I = 1, M
            R(I,J) = ZERO
            SSP(I,J) = ZERO
         ENDDO
      ENDDO
C
C XBAR
C
      DN = WTSUM
      DNM1 = DN - ONE
      DO J = 1, M
         DO I = 1, N
            XBAR(J) = XBAR(J) + WT(I)*X(I,J)
         ENDDO
         XBAR(J) = XBAR(J)/DN
      ENDDO
C
C STD
C
      DO J = 1, M
         XB = XBAR(J)
         DO I = 1, N
            STD(J) = STD(J) + WT(I)*(X(I,J) - XB)**2
         ENDDO
         STD(J) = SQRT(STD(J)/DNM1)
      ENDDO
C
C SSP
C
      DO J = 1, M
         DO K = J, M
            XJ = XBAR(J)
            XK = XBAR(K)
            DO I = 1, N
               SSP(J,K) = SSP(J,K) + WT(I)*(X(I,J) - XJ)*(X(I,K) - XK)
            ENDDO
            IF (J.NE.K) SSP(K,J) = SSP(J,K)
         ENDDO
      ENDDO
C
C R
C
      DO J = 1, M
         R(J,J) = ONE
         DO K = J + 1, M
            TEMP = SQRT(SSP(J,J)*SSP(K,K))
            IF (TEMP.GT.EPSI) R(J,K) = SSP(J,K)/TEMP
            R(K,J) = R(J,K)
         ENDDO
      ENDDO
      END
C
C
                  
