     
C
C SIMFIT replacement for NAG routine G08EAF which requires the
C auxiliary routines G08EAF1, G08EAF2, G08EAF3, and G08EAF4 and ZTRANZ
C
      SUBROUTINE G08EAF$(CL, N, X, M, MAXR, NRUNS, NCOUNT, EX, COV,
     +                   LDCOV, CHI, DF, PROB, WRK, LWRK, IFAIL)
C 
C Important notes added by W.G.Bardsley at 15/08/2012 
C    
C 1) This replacement for NAG routine uses SAVE to save the internal variables
C    NTOTAL, XPREV, and IU between intermediate calls when CL is not 'S' or 's'.
C 2) It uses the NAG limit for LWRK >= MAXR*(MAXR + 5)/2 + 1 but it also
C    checks for LWRK < 2*MAXR + 5 and LWRK < MAXR*(MAXR + 1) and allocates extra
C    workspace if this happens.
C 3) The check for covariance matrix inverse requires IFAIL non zero on exit
C    from ZTRANZ which may not be quite the same as the NAG check.
C 4) The output is very slightly different from the NAG output unless N is large.
C 5) It uses the Simfit method for IFAIL error reporting so the value for
C    IFAIL on input is ignored
C 6) The variables ADJUST_MAXR and ADJUST_N are there to try to do what the NAG
C    routines does as follows:
C    a) If the last run is ascending then discard it and do not count it
C    b) Adjust the value of NTOTAL to allow for the truncated value of N this causes  
C    c) If adjustment leads to K < MAXR then EX(i) and COV(i,j) are set to -1 for
C       i, j > MAXR and DF is returned equal to the truncated MAXR so the calling 
C       program gets the the correct chi-sqd. degrees of freedom.  
C        
    
      IMPLICIT NONE
C
C Arguments
C      
      INTEGER N, M, MAXR, NRUNS, NCOUNT(MAXR), LDCOV, LWRK, IFAIL
      DOUBLE PRECISION X(N), EX(MAXR), COV(LDCOV, MAXR), CHI, DF,
     +                 PROB, WRK(LWRK)
      CHARACTER (LEN = 1) CL
C
C Allocatable
C 
      DOUBLE PRECISION, ALLOCATABLE :: WORK(:)     
C
C Locals
C     
      INTEGER    I, IERR, ISEND, IU, J, K, LWORK, NTOTAL
      DOUBLE PRECISION XPREV
      DOUBLE PRECISION DFOLT, ZERO
      PARAMETER (DFOLT = -1.0D+00, ZERO = 0.0D+00)
      CHARACTER (LEN = 1) CL_COPY
      LOGICAL    ADJUST_MAXR, ADJUST_N
      EXTERNAL   G08EAF1, G08EAF2, G08EAF3, G08EAF4
      INTRINSIC  MAX
      SAVE       IU, NTOTAL, XPREV
      DATA       IU, NTOTAL, XPREV / 0, 0, 0.0D+00 /
      SAVE       ADJUST_MAXR, ADJUST_N
      DATA       ADJUST_MAXR, ADJUST_N / .TRUE., .TRUE. /
C
C Inititialise IFAIL then check the arguments supplied
C
      IFAIL = 0
      IF (CL.EQ.'S' .OR. CL.EQ.'s') THEN
         CL_COPY = 'S'
      ELSEIF (CL.EQ.'F' .OR. CL.EQ.'f') THEN
         CL_COPY = 'F'
      ELSEIF (CL.EQ.'I' .OR. CL.EQ.'i') THEN
         CL_COPY = 'I'
      ELSEIF (CL.EQ.'L' .OR. CL.EQ.'l') THEN
         CL_COPY = 'L'
      ELSE
         IFAIL = 1
         RETURN
      ENDIF
      IF (CL_COPY.EQ.'S'.AND.N.LT.3 .OR. N.LT.1) THEN
         IFAIL = 2
         RETURN
      ENDIF 
      IF (CL_COPY.EQ.'S' .AND. M.GT.N) THEN
         IFAIL = 3
         RETURN
      ENDIF 
      IF (CL_COPY.EQ.'S'.AND.MAXR.GE.N .OR. MAXR.LT.1) THEN
         IFAIL = 4
         RETURN
      ENDIF
      IF (LDCOV.LT.MAXR) THEN
         IFAIL = 5
         RETURN
      ENDIF 
      IF (LWRK.LT.MAXR*(MAXR + 5)/2 + 1) THEN
         IFAIL = 6
         RETURN
      ENDIF 
      
      IF (CL_COPY.EQ.'S' .OR. CL_COPY.EQ.'F') THEN
C
C This is a first or else the only call to G08EAF$ so initialise everything
C        
         CHI = ZERO
         DF = ZERO
         PROB = ZERO
         NRUNS = -1
         DO J = 1, MAXR
            EX(J) = DFOLT
            DO I = 1, MAXR
               COV(I,J) = DFOLT
            ENDDO  
         ENDDO  
         ISEND = 1
         CALL G08EAF1 (IFAIL, ISEND, IU, M, MAXR, N, NCOUNT, NRUNS,
     +                 NTOTAL,
     +                 X, XPREV)
         IF (IFAIL.EQ.7 .OR. CL_COPY.EQ.'F') RETURN
      ENDIF
      
      IF (CL_COPY.EQ.'I') THEN 
C
C This is an intermediate call so NTOTAL, XPREV, and IU must have been saved
C        
         ISEND = 2
         CALL G08EAF1 (IFAIL, ISEND, IU, M, MAXR, N, NCOUNT, NRUNS,
     +                 NTOTAL,
     +                 X, XPREV)
         RETURN 
      ENDIF 
      
      IF (CL_COPY.EQ.'L') THEN
C
C This is the last call so NTOTAL, XPREV, and IU must have been saved 
C        
         ISEND = 2
         CALL G08EAF1 (IFAIL, ISEND, IU, M, MAXR, N, NCOUNT, NRUNS,
     +                 NTOTAL,
     +                 X, XPREV)                         
         IF (IFAIL.EQ.7) RETURN
      ENDIF 
      
      IF (NRUNS.LT.MAXR) THEN
         IFAIL = 8
      ENDIF
      
      K = MAXR
      IF (ADJUST_MAXR) THEN
C
C Check for MAXR too large resulting in empty bins
C      
         DO I = MAXR, 1, -1
            IF (NCOUNT(I).LE.0) THEN
               K = K - 1
            ELSE
               EXIT
            ENDIF
         ENDDO
      ENDIF
      
      IF (ADJUST_N) THEN
C
C Check for last run ascending 
C     
         DO I = N, 1, -1
            IF (X(I - 1).LT.X(I)) THEN
               NTOTAL = NTOTAL - 1
            ELSE
               EXIT
            ENDIF
         ENDDO 
         NTOTAL = NTOTAL - 1
      ENDIF
C
C Check work space then finish the calculations
C
      LWORK = MAX(2*K + 5, K*(K + 1))
      
      IF (LWRK.GE.LWORK) THEN   
         CALL G08EAF2 (K, NTOTAL,
     +                 EX, WRK) 
         CALL G08EAF3 (LDCOV, K, NTOTAL,
     +                 COV, WRK) 
         CALL G08EAF4 (IFAIL, K, LDCOV, NCOUNT,
     +                 CHI, COV, DF, EX, PROB, WRK, WRK(K**2 + 1)) 
      ELSE
         IERR = 0
         IF (ALLOCATED(WORK)) DEALLOCATE (WORK, STAT = IERR)
         IF (IERR.NE.0) THEN
            IFAIL = -100
            RETURN
         ENDIF     
         ALLOCATE (WORK(LWORK), STAT = IERR) 
         CALL G08EAF2 (K, NTOTAL,
     +                 EX, WORK) 
         CALL G08EAF3 (LDCOV, K, NTOTAL,
     +                 COV, WORK) 
         CALL G08EAF4 (IFAIL, K, LDCOV, NCOUNT,
     +                 CHI, COV, DF, EX, PROB, WORK, WORK(K**2 + 1)) 
         DEALLOCATE (WORK, STAT = IERR) 
      ENDIF
C
C Check if K < MAXR then make sure the undefined EX and COV values are set
C to -1 to allow the calling program to detect chi-sqd. degrees of freedom
C   
      IF (K.LT.MAXR) THEN
         DO I = K + 1, MAXR
            EX(I) = DFOLT
            IF (I.LT.K) THEN
               DO J = K + 1, MAXR
                  COV(I,J) = DFOLT
               ENDDO   
            ELSE
               DO J = 1, MAXR
                  COV(I,J) = DFOLT
               ENDDO
            ENDIF         
         ENDDO
      ENDIF           
C
C Some final IFAIL checks on IFAIL from G08EAF4, M and NRUNS
C      
      IF (IFAIL.EQ.9) RETURN
      IF (M.GT.0 .AND. NRUNS.LT.M) THEN
         IFAIL = 10
         RETURN
      ENDIF   
      END
 
c
c----------------------------------------------------------------------------
c
      subroutine g08eaf1 (ifail, isend, iu, m, maxr, n, ncount, nruns,
     +                    ntotal,
     +                    x, xprev)
c
c action: work out the number of runs up in an X vector
c author: w.g.bardsley, university of manchester, uk, 13/08/2012
c
c       ifail: tied values cause an isend = 7 exit to agree with NAG g08eaf 
c       isend: action required as follows
c            : isend = 1: initialise ncount and ntotal
c            : isend = 2: add to existing run counts
c          iu: current counter
c           m: used as run limit if m > 0  
c        maxr: maximum run length required
c           n: dimension of x
c      ncount: runs up counter
c       nruns: number of runs recorded 
c      ntotal: running total for values examined
c           x: vector of random numbers  
c       xprev: previous last x-value if used sequentially         
c     
      implicit none
c
c arguments
c   
      integer,          intent (in)    :: isend, m, maxr, n
      integer,          intent (inout) :: ifail, ncount(maxr), nruns 
      integer,          intent (inout) :: iu, ntotal
      double precision, intent (in)    :: x(n)
      double precision, intent (inout) :: xprev
c
c locals
c       
      integer i
      
      if (n.lt.1 .or. maxr.lt.1) return
      if (m.gt.0 .and. nruns.ge.m) return
        
      if (isend.eq.1) then  
         do i = 1, maxr
            ncount(i) = 0
         enddo
         nruns = 0
         ntotal = 1
         iu = 1
         if (n.gt.1) then
            do i = 2, n
               if (x(i).lt.x(i - 1)) then 
                  nruns = nruns + 1
                  ncount(iu) = ncount(iu) + 1
                  iu = 1
               elseif (x(i).gt.x(i - 1)) then   
                  if (iu.lt.maxr) iu = iu + 1
               else
                  ifail = 7
                  return     
               endif
               xprev = x(i)
               ntotal = ntotal + 1
               if (m.gt.0 .and. nruns.ge.m) return
            enddo
         endif
c
c The next line is added by as157 but gives one more run than the 
c NAG routine which discards the last ascending run
c        
c         ncount(iu) = ncount(iu) + 1
c
      elseif (isend.eq.2) then 
         do i = 1, n
            if (x(i).lt.xprev) then 
               nruns = nruns + 1
               ncount(iu) = ncount(iu) + 1
               iu = 1
            elseif (x(i).gt.xprev) then   
               if (iu.lt.maxr) iu = iu + 1
            else
               ifail = 7
               return     
            endif
            xprev = x(i) 
            ntotal = ntotal + 1
            if (m.gt.0 .and. nruns.ge.m) return 
         enddo
      endif 
      end 
c
c-------------------------------------------------------------------------
c
      subroutine g08eaf2 (maxr, n,
     +                    e, f)
c
c action: calculate expected values and factorials for runs up or down
c author: w.g.bardsley, university of manchester, u.k. 12/08/2012
c
c This subroutine uses the Levene method [Ann Math Stat 23, 1 (1952) pp 34 - 56]
c but the results are in exact agreement with the Knuth method [Vol 2 p 67] for MAXR = 6
c
c maxr: maximum run length 
c    n: this should be ntotal = total numbers of values examined
c    e: expected values
c    f: factorials used here and then in g08eaf3
c
c Expected values are e(i) = expected in sample size n
c and the e(i) values are calculated for r = i - 1 as follows:
c
c      {(r^2 + 3r + 1)n - (r^3 + 3r^2 - r - 4)}/(r + 3)! for i < maxr
c
c      {r + 1)n - (r^2 + r - 1)}/(r + 2)!                for i = maxr 
c
c Note that r is one greater than in Levene Ann Math Stat 23, 1 (1952) pp 34 - 56       
c     
      implicit none
c
c arguments
c      
      integer,           intent (in)  :: maxr, n
      double precision , intent (out) :: e(maxr), f(2*maxr + 5)
c
c locals
c      
      integer    i
      double precision bot, dn, r, top
      double precision one, three, four
      parameter (one = 1.0d+00, three = 3.0d+00, four = 4.0d+00)
      intrinsic  dble
c
c check that n and maxr are >= 1 and maxr =< n
c     
      if (n.lt.1 .or. maxr.lt.1 .or. maxr.gt.n) return
c
c calculate factorials long-hand which will usually be for nrmax fairly small
c
      f(1) = one
      do i = 2, 2*maxr + 5
         f(i) = f(i - 1)*dble(i)
      enddo
c
c initialise dn then compute e(i)
c      
      dn = dble(n) 
      do i = 1, maxr
         r = dble(i - 1)
         if (i.lt.maxr) then
            top = (r*(r + three) + one)*dn - 
     +            (r*(r*(r + three) - one) - four)
            bot = f(i + 2)
         else
            top = (r + one)*dn - (r*(r + one) - one) 
            bot = f(i + 1)
         endif
         e(i) = top/bot 
      enddo
      end
c
c----------------------------------------------------------------
c      
      subroutine g08eaf3 (ldcov, maxr, n,
     +                    cov, f)
c
c action: calculate covariance matrix for runs up or down
c author: w.g.bardsley, university of manchester, u.k., 13/08/2012
c
c This subroutine uses the Levene method [Ann Math Stat 23, 1 (1952) pp 34 - 56]
c but the results are in exact agreement with the Knuth method [Vol 2 p 67] for MAXR = 6
c
c ldcov: leading dimension for cov
c  maxr: maximum run length
c     n: n = ntotal
c   cov: covariance matrix
c     f: factorials as returned by g08eaf2 
c
c Note: The formulas quoted by Levene are used 
c       Ann Math Stat 23, 1 (1952) pp 34 - 56 
c       except these are our runs + 1
c       I have written out the equations in full using monomials to assist checking 
c       but they can easily be compressed using Horner's method if required
c
      implicit none
c
c arguments
c
      integer,          intent (in)  :: ldcov, maxr, n
      double precision, intent (in)  :: f(2*maxr + 5)
      double precision, intent (out) :: cov(ldcov,maxr)
c
c locals
c
      integer    i, ip, ip_sav, j, jq, jq_sav
      double precision p, p_sav, q, q_sav, s
      double precision p2, p3, p4, p5, q2, q3, q4, s2, s3, s4
      double precision a, b, dn, fact(4)
      double precision one, two, three, four, five, six, seven,
     +                 eight, nine, ten 
      parameter (  one = 1.0d+00,   two = 2.0d+00,
     +           three = 3.0d+00, four = 4.0d+00,  five = 5.0d+00,
     +            six = 6.0d+00, seven = 7.0d+00, eight = 8.0d+00,
     +           nine = 9.0d+00,   ten = 10.0d+00) 
      double precision f11, f13, f14, f15, f16, f18, f19, f23, f24, f25,
     +                 f28, f29, f32, f36, f46, f50, f65, f80,
     +                 f101, f116
      parameter (f11 = 11.0d+00,  f13 = 13.0d+00,   f14 = 14.0d+00,
     +           f15 = 15.0d+00,  f16 = 16.0d+00,   f18 = 18.0d+00, 
     +           f19 = 19.0d+00,  f23 = 23.0d+00,   f24 = 24.0d+00, 
     +           f25 = 25.0d+00,  f28 = 28.0d+00,   f29 = 29.0d+00, 
     +           f32 = 32.0d+00,  f36 = 36.0d+00,   
     +           f46 = 46.0d+00,  f50 = 50.0d+00,   f65 = 65.0d+00,
     +           f80 = 80.0d+00, f101 = 101.0d+00, f116 = 116.0d+00) 
      intrinsic dble
c
c check arguments
c      
      if (ldcov.lt.1 .or. ldcov.lt.maxr .or. maxr.lt.1 .or.
     +        n.lt.1 .or. n.lt.maxr) return
c
c initialise dn then calculate the lower triangle      
c
      dn = dble(n)   
      do i = 1, maxr
         ip = i - 1
         p = dble(ip)
         
         do j = 1, i
            jq = j - 1
            q = dble(jq)
            
            if (i.eq.j) then
c
c The case i = j so calculate variances
c         
               if (i.lt.maxr) then
c
c i < maxr so use sigma**2(s_p)
c                 
                  fact(1) = one/(f(ip + 3)*f(ip + 3))
                  fact(2) = two/f(2*ip + 5)
                  fact(3) = one/f(ip + 3)
                  p2 = p*p
                  p3 = p*p2
                  p4 = p*p3
                  p5 = p*p4                   

                  a = - fact(1)*
     +                 (two*p5 + f13*p4 + f24*p3 +
     +                 (three*p + one)*(p - seven))
     
     +                - fact(2)*
     +                 (eight*p3 + f36*p2 + f46*p + f14)
     
     +                + fact(3)*
     +                 (p2 + three*p + one)
     
                  b = fact(1)*                           !Levene does not have ) after three on the next line
     +               (p**4*(three*p + f11)*(p + three) - !I added this ) here as it seems wrong o/w in Levene
     +                p*(f28*p2 + f101*p + f50) + four)
     
     +              + fact(2)*
     +               (f16*p4 + f80*p3 + f116*p2 + f32*p - f19)
     
     +              - fact(3)*
     +               (p3 + three*p2 - p - four)
     
                  cov(i,i) = dn*a + b                     
               else 
c
c i = maxr so use sigma**2(s_p')
c                 
                  fact(1) = one/(f(ip + 2)*f(ip + 2))       
                  fact(2) = four/f(2*ip + 3)
                  fact(3) = one/f(ip + 2)
                  fact(4) = two/f(2*ip + 3)
                  p2 = p*p
                  p3 = p*p2
                  p4 = p*p3
                  
                  a = - fact(1)*
     +                 (p + one)*(two*p2 + three*p - one)
     
     +                - fact(2)*
     +                 (p + one)
     
     +                + fact(3)*
     +                 (p + one)
     
                  b = fact(1)*
     +               (three*p4 + eight*p3  + p2 - eight*p - three)
     
     +              + fact(4)*
     +               (four*p2 + six*p + one)
     
     +              - fact(3)*
     +               (p2 + p - one)
     
                  cov(i,i) = dn*a + b                                 
               endif
            elseif (i.lt.maxr) then 
c
c calculate covariance for i < maxr
c                
               fact(1) = one/(f(ip + 3)*f(jq + 3)) 
               fact(2) = two/f(ip + jq + 5)
               s = p + q
               p2 = p*p
               p3 = p*p2
               p4 = p*p3
               q2 = q*q
               q3 = q*q2
               q4 = q*q3
               s2 = s*s
               s3 = s*s2
               s4 = s*s3

               a = - fact(1)*
     +             (p3*(q2 + three*q + one) +
     +              p2*q*(q2 + seven*q + f11) +
     +              p*(three*q3 + f11*q2 + three*q - ten) 
     +            + q3 - ten*q - seven)
     
     +            - fact(2)*
     +             (s3 + nine*s2 + f23*s + f14)
     
               b = fact(1)*
     +            (p4*(q2 + three*q + one) +
     +             p3*q*(q2 + seven*q + f11) +
     +             p2*(q4 + seven*q3  + nine*q2 - f14*q - f18) +
     +             p*(three*q4 + f11*q3 - f14*q2 - f65*q - f25) +
     +             q4 - f18*q2 - f25*q + four)
     
     +             + fact(2)*
     +             (s4 + ten*s3 + f29*s2 + f16*s - f19)
      
               cov(i,j) = dn*a + b  
            else
c
c calculate covariance for i = maxr
c Note that ip, p jq, and q are first reversed so that q > p for the calculation but
c they are restored afterwards although this is only really required for ip and p
c                
               ip_sav = ip
               jq_sav = jq
               p_sav = p
               q_sav = q
               ip = jq_sav
               jq = ip_sav 
               p = q_sav 
               q = p_sav
               s = p + q
               p2 = p*p
               p3 = p*p2
               p4 = p*p3
               q2 = q*q
               q3 = q*q2
               s2 = s*s
               s3 = s*s2

               fact(1) = one/(f(ip + 3)*f(jq + 2))
               fact(2) = two/f(ip + jq + 4)

               a = - fact(1)*
     +             (p3*(q + one) + p2*(q2 + five*q + three) +
     +              p*(three*q2 + five*q - one) + q2 - two*q - four)
     
     +             - fact(2)*
     +              (s2 + five*s + five)

               b = fact(1)*
     +            (p4*(q + one) + p3*(q2 + five*q + three) +
     +             p2*(q3 + five*q2 + two*q - five) +
     +             p*(three*q3 + five*q2 - f15*q - f16) +
     +             q3 - two*q2 - f11*q - four)
     
     +           + fact(2)*
     +            (s3 + six*s2 + eight*s - one)                 

               cov(i,j) = dn*a + b
                
               ip = ip_sav
               jq = jq_sav
               p = p_sav
               q = q_sav
            endif
            
         enddo
         
      enddo 
c
c fill in the upper triangle to agree with NAG
c      
      do i = 1, maxr - 1
         do j = i + 1, maxr
           cov(i,j) = cov(j,i)
         enddo  
      enddo  
      end
c
c---------------------------------------------------------------------------
c       
      subroutine g08eaf4 (ifail, maxr, ldcov, ncount,
     +                    chi, cov, df, ex, prob, w, w1)
c
c action: calculate the chi-squared statistic using ztranz
c author: w.g.bardsley, university of manchester, u.k., 14/08/2012
c
      implicit none
c
c arguments
c          
      integer,          intent (in)    :: maxr, ldcov, ncount(maxr) 
      integer,          intent (inout) :: ifail
      double precision, intent (in)    :: cov(ldcov,maxr), ex(maxr)
      double precision, intent (out)   :: chi, df, prob
      double precision, intent (inout) :: w(maxr,maxr), w1(maxr) 
c
c locals
c       
      integer    i, j, n
      double precision g01ecf$
      external   ztranz
      external   g01ecf$
      intrinsic  dble
c
c load w1 with the difference vector and cov into upper triangle of w 
c      
      do i = 1, maxr
         w1(i) = dble(ncount(i)) - ex(i)
         do j = i, maxr
            w(i,j) = cov(i,j)
         enddo   
      enddo
c
c calculate chi
c      
      ifail = 0 
      n = maxr
      call ztranz (i, maxr, n,
     +             w, w1, chi)      
      if (i.ne.0) then
c
c test for satifactory calculation of chi
c        
        ifail = 9
        return
      endif
c
c calculate df and prob
c      
      df = dble(maxr)
      prob = g01ecf$('U', chi, df, ifail)    
      end
c
c            
      


      