C
C Subroutines used to create and use Simfit library files
C
C LIBFIL
C MAKLIB_DRIVER
C MAKLIB_ADVISE
C SVGLIB
C
      SUBROUTINE LIBFIL (NCOL, NFILES, NIN, NROW, NSMALL,
     +                   FSAV, TSAV,
     +                   ABORT, FIXCOL, FIXROW)
C
C ACTION: Use a library file to read in a set of matrix files
C         Returns matrix files FSAV(NFILES), TSAV(NFILES)
C AUTHOR: W. G. Bardsley, University of Manchester, U.K.
C         23/08/1999 derived from VECFIL
C         29/11/2001 added VU2CHK and FNLIB
C         06/01/2007 added calls to SIMDIR and INFOFL  
C         19/01/2007 made sure Simfit folder exists before looking there
C         19/07/2007 changed SIMDIR to DEMDIR
C         02/08/2008 added call to ISITSF
C 
C   NCOL: (input/output) depending on FIXCOL
C NFILES: (output) number of files located
C    NIN: (input/unchanged) unconnected unit for reading files
C   NROW: (input/output) depending on FIXROW
C NSMALL: (input/unchanged) maximum dimension of FSAV and TSAV
C   FSAV: (input/output) files selected
C   TSAV: (input/output) titles of files selected
C  ABORT: (output) .true. if all is well o/w .false.
C FIXCOL: (input) if .true. then all columns must be of the same length
C FIXROW: (input) if .true. then all files must have the same number
C                 of rows 
C
      IMPLICIT   NONE  
C
C Arguments
C          
      INTEGER,             INTENT (IN)    :: NIN, NSMALL          
      INTEGER,             INTENT (OUT)   :: NFILES
      INTEGER,             INTENT (INOUT) :: NCOL, NROW
      CHARACTER (LEN = *), INTENT (INOUT) :: FSAV(NSMALL), TSAV(NSMALL) 
      LOGICAL,             INTENT (OUT)   :: ABORT 
      LOGICAL,             INTENT (IN)    :: FIXCOL, FIXROW
C
C Locals
C      
      INTEGER    N0, N1, N2, N3, N4
      PARAMETER (N0 = 0, N1 = 1, N2 = 2, N3 = 3, N4 = 4)
      INTEGER    I, IADD1, IOS, ISEND, J, K, L, L1, L2, LEN200
      CHARACTER  LINE*100
      CHARACTER  FNAME*1024, FNLIB*1024, SIMFIT*1024, TEMP*1024,
     +           TITLE*80
      CHARACTER  BLANK*1, BSLASH*1
      PARAMETER (BLANK = ' ', BSLASH = '\')
      LOGICAL    YES
      EXTERNAL   PSWARN, CHECKF, OFILES, PUTADV, PUTFAT, TXT2FN, VU2CHK,
     +           DEMDIR, LEN200, TRIML1, INFOFL, ISITSF  
      INTRINSIC  INDEX
C
C Part 1: First initialise ABORT, NFILES, FSAV, and TSAV
C ======
C
      ABORT = .TRUE.
      NFILES = N0 
      IF (NSMALL.LE.N0) THEN    
         RETURN
      ELSE   
         DO I = N1, NSMALL
            FSAV(I) = BLANK
            TSAV(I) = BLANK
         ENDDO 
      ENDIF     
C
C Part 2: Check that the file seems to be a library file
C ======
C
      ISEND = N3
      CLOSE (UNIT = NIN)
      CALL OFILES (ISEND, NIN, 
     +             FNAME,
     +             ABORT)
      CLOSE (UNIT = NIN)
      IF (ABORT) RETURN
      FNLIB = FNAME
      OPEN (UNIT = NIN, FILE = FNLIB)
      READ (NIN,'(A)',IOSTAT=IOS) TITLE
      IF (IOS.NE.N0) THEN 
C
C Format 100: cannot read TITLE from FNLIB = FNAME
C      
         WRITE (LINE,100)
         CALL PUTFAT (LINE)
         CLOSE (UNIT = NIN)
         CALL VU2CHK (FNLIB)
         RETURN
      ENDIF
      CALL PSWARN (TITLE)
C
C Disconnect then check, re-connect and wind on
C
      CLOSE (UNIT = NIN)
      CALL CHECKF (FNAME, TITLE, 
     +             ABORT)
      IF (ABORT) THEN
         RETURN
      ELSE
         OPEN (UNIT = NIN, FILE = FNAME, IOSTAT = IOS)
         READ (NIN,'(A)',IOSTAT=IOS) TITLE
      ENDIF        
C
C Read in the file names
C
      IADD1 = N0
      IOS = N0
      DO WHILE (IOS.EQ.N0 .AND. IADD1.LT.NSMALL)
         ABORT = .TRUE.
         READ (NIN,'(A)',IOSTAT=IOS) FNAME
         IF (IOS.EQ.N0 .AND. FNAME.EQ.BLANK) IOS = N1
         IF (IOS.EQ.N0) THEN
            CALL TXT2FN (FNAME,
     +                   ABORT)
            IF (ABORT) THEN
C
C Format 200: check file ... not a valid filename
C            
               WRITE (LINE,200) IADD1 + N1
               CALL PUTFAT (LINE)
               CLOSE (UNIT = NIN)
               CALL VU2CHK (FNLIB)
               RETURN
            ELSE      
C
C Store the possible file
C
               IADD1 = IADD1 + N1
               FSAV(IADD1) = FNAME
            ENDIF
         ENDIF
      ENDDO
      CLOSE (UNIT = NIN)
      ABORT = .TRUE.
C
C Part 3: Check for files existing and correct dimensions
C ======
C
      IF (IADD1.LE.N0) THEN    
C
C Format 200: check file ... no files specified
C      
         WRITE (LINE,200) N2
         CALL PUTFAT (LINE)
         CALL VU2CHK (FNLIB)
         RETURN
      ENDIF        
C
C Get the Simfit folder
C  
      CALL DEMDIR (L,
     +             SIMFIT)    
C
C Check each file
C     
      DO I = N1, IADD1
         INQUIRE (FILE = FSAV(I), EXIST = YES, IOSTAT = IOS) 
         IF (IOS.EQ.0 .AND. .NOT.YES) THEN
            IF (L.GT.N0 .AND. INDEX(FSAV(I),BSLASH).LE.N0) THEN
C
C File missing so try simfit(1:l)//fsav but only if the Dem folder exists
C             
               TEMP = FSAV(I)
               CALL TRIML1 (TEMP)
               L1 = N1
               L2 = LEN200(TEMP)
               FSAV(I) = SIMFIT(L1:L)//TEMP(L1:L2) 
               INQUIRE (FILE = FSAV(I), EXIST = YES, IOSTAT = IOS)
            ENDIF 
         ENDIF  
         IF (YES .AND. IOS.EQ.N0) THEN    
C
C Check it is a simfit data file
C           
            CALL ISITSF (J, K,
     +                   FSAV(I))
            IF (J.LE.N0 .OR. K.LE.N0) THEN
               IOS = N1
               YES = .FALSE.
            ENDIF   
         ENDIF                  
         IF (YES .AND. IOS.EQ.N0) THEN    
C
C File located so check details
C         
           CLOSE (UNIT = NIN)
           OPEN (UNIT = NIN, FILE = FSAV(I))
           READ (NIN,'(A)',IOSTAT=IOS) TITLE
           IF (IOS.NE.N0) THEN
C
C Format 300: cannot read TITLE at FSAV(I)
C           
              WRITE (LINE,300) I
              CALL PUTFAT (LINE)
              CLOSE (UNIT = NIN)
              CALL VU2CHK (FSAV(I))
              RETURN
           ENDIF
           READ (NIN,*,IOSTAT=IOS) J, K
           IF (IOS.NE.N0) THEN   
C
C Format 400: cannot read J, K from FSAV(I)
C           
              WRITE (LINE,400) I
              CALL PUTFAT (LINE)
              CLOSE (UNIT = NIN)
              CALL VU2CHK (FSAV(I))
              RETURN
           ENDIF
           IF (FIXCOL) THEN
              IF (K.NE.NCOL) THEN
C
C Format 500: incorrect column dimension at FSAV(I) 
C              
                 WRITE (LINE,500) I
                 CALL PUTFAT (LINE)
                 CLOSE (UNIT = NIN)
                 CALL VU2CHK (FSAV(I))
                 RETURN
              ENDIF
           ENDIF
           IF (FIXROW) THEN
              IF (J.NE.NROW) THEN 
C
C Format 600: incorrect row dimension at FSAV(I) 
C              
                 WRITE (LINE,600) I
                 CALL PUTFAT (LINE)
                 CLOSE (UNIT = NIN)
                 CALL VU2CHK (FSAV(I))
                 RETURN
              ENDIF
           ENDIF
           IF (J.GT.N0 .AND. K.GT.N0) THEN
              NFILES = NFILES + N1
              FSAV(NFILES) = FSAV(I)
              TSAV(NFILES) = TITLE
           ELSE   
C
C illegal file specified
C           
              IOS = N1    
              ISEND = N4
              CALL INFOFL (ISEND,
     +                     FSAV(I)) 
              CALL VU2CHK (FSAV(I))              
           ENDIF
         ELSE
            IOS = N1 
C
C file not found
C                        
            ISEND = N1
            CALL INFOFL (ISEND,
     +                   FSAV(I))            
         ENDIF
         IF (IOS.NE.N0) THEN 
C
C check line I + 1
C         
            WRITE (LINE,200) I + N1
            CALL PUTADV (LINE)
            ABORT = .TRUE.
            CLOSE (UNIT = NIN)
            CALL VU2CHK (FNLIB)
            RETURN
         ENDIF
         CLOSE (UNIT = NIN)
      ENDDO
      CLOSE (UNIT = NIN)
C
C Set ABORT = .FALSE. if at least one file is satisfactory
C
      IF (NFILES.GT.N0) THEN
         ABORT = .FALSE.  
      ELSE
         ABORT = .TRUE.
      ENDIF      
C
C Format statements
C      
  100 FORMAT ('Title not defined in supposed library file')
  200 FORMAT ('Check your library file at line',I4)
  300 FORMAT ('Failure to read title at library file',I4)
  400 FORMAT ('Failure to read header details at library file',I4)
  500 FORMAT ('Incorrect column dimension at library file',I4)
  600 FORMAT ('Incorrect row dimension at library file',I4)
      END
C
C
C
C MAKLIB: dll VERSION
C =====================
C
      SUBROUTINE MAKLIB_DRIVER
C
C VERSION : details from SIMVER/DLLCHK
C FORTRAN : 95
C ACTION  : Make a library file
C AUTHOR  : W. G. Bardsley, University of Manchester, U.K.,12/12/96
C           10/02/1997 Win32 version
C           13/09/1999 added calls to WINDOW
C           07/02/2000 Substantial revision
C           14/02/2000 added SIMVER
C           16/02/2001 added MULTFS
C           24/02/2001 added PFILES, NOUT10, removed ISITPS
C           30/07/2005 increased DVER to *30 and added to call to ADVISE 
C           14/03/2007 initialised FILES
C           25/03/2008 edited for version 6 
C           28/12/2018 dll version  
C
      IMPLICIT   NONE
      INTEGER    NIN, NMAX, NOUT, NOUT10, N0, N1
      PARAMETER (NIN = 3, NMAX = 200, NOUT = 4, NOUT10 = 10, N0 = 0,
     +           N1 = 1)
      INTEGER    NUMBLD(30), NUMDEC, NUMOPT, NUMSTA, NUMTXT
      PARAMETER (NUMOPT = 10, NUMSTA = 10, NUMTXT = NUMSTA + NUMOPT - 1)
      INTEGER    I, IPREV, ISEND, ITYPE, L, NFILES
      CHARACTER (LEN = 1024) FNAME, FILES(NMAX)
      CHARACTER (LEN = 100 ) TITLE, OPTS(30)
      CHARACTER (LEN = 45  ) QUERY
      PARAMETER (QUERY = 'A sensible short title for this library file')
      CHARACTER (LEN = 32  ) WORD32 
      CHARACTER (LEN = 6   ) PNAME
      PARAMETER (PNAME = 'MAKLIB')
      CHARACTER (LEN = 1   ) BLANK
      PARAMETER (BLANK = ' ')
      LOGICAL    FIRST, ISTOP
      LOGICAL    ACCEPT(NMAX)
      LOGICAL    ABORT, FAIL, REPEET, THERE
      LOGICAL    ASKIF
      PARAMETER (ASKIF = .FALSE.)
      EXTERNAL   STOPGO, OFILES, GETSTR, PUTADV, PFILES, DELEET, 
     +           YMDHMS, MAKLIB_ADVISE, LSTBOX
      INTRINSIC  LEN_TRIM
      DATA       NUMBLD / 30*0 /


C
C Advise the user
C
      FIRST = .TRUE.
      CALL MAKLIB_ADVISE (ABORT, FIRST)
      IF (.NOT.ABORT) THEN
C
C Action is required so Initialise
C
         IPREV = N0
         NFILES = N0
         FNAME = BLANK
         TITLE = BLANK 
         DO I = N1, NMAX
            FILES(I) = BLANK
            ACCEPT(I) = .FALSE.
         ENDDO   
C
C .....................................................................
C Start of main loop
C .....................................................................
C
         REPEET = .TRUE.
         DO WHILE (REPEET)
C
C Open the file to become the library file
C
            CLOSE (UNIT = NOUT)
            CALL PUTADV ('Now you have to name a Simfit library file')
            ISEND = N1
            CALL OFILES (ISEND, NOUT,
     +                   FNAME, 
     +                   ISTOP)
            IF (ISTOP) THEN
               FNAME = BLANK
            ELSE
C
C A file has been opened so get a title for the library file
C
               CALL YMDHMS (WORD32)
               TITLE = 'Library file: '//WORD32
               CALL GETSTR (QUERY, TITLE)
               WRITE (NOUT,100) TITLE
               WRITE (OPTS,200)
               NUMDEC = NUMOPT
               DO WHILE (NUMDEC.EQ.NUMOPT)
C
C Show the menu
C
                  NUMBLD(1) = 4
                  CALL LSTBOX (NUMBLD, NUMDEC, NUMOPT, NUMSTA, NUMTXT,
     +                         OPTS)
                  IF (NUMDEC.EQ.NUMOPT) THEN
                     CLOSE (UNIT = NIN)
                     CLOSE (UNIT = NOUT)
                     CLOSE (UNIT = NOUT10)
                     RETURN
                  ENDIF   
                  IF (NUMDEC.EQ.NUMOPT - 1) THEN
C
C Provide advice
C
                     FIRST = .FALSE.
                     CALL MAKLIB_ADVISE (FAIL, FIRST)
                  ENDIF
               ENDDO
C
C Define ITYPE
C
               ITYPE = NUMDEC
C
C Call PFILES for multiple file selection
C
               CALL PFILES (IPREV, ITYPE, NFILES, NIN, NMAX, NOUT10,
     +                      FILES,
     +                      ACCEPT)
C
C Write the files selected to the library file
C
               IF (NFILES.GT.N0) THEN
                  DO I = N1, NFILES
                      L = LEN_TRIM(FILES(I))
                      WRITE (NOUT,100) FILES(I)(1:L) 
                  ENDDO
               ENDIF
C
C Make sure all units are closed
C
               CLOSE (UNIT = NIN)
               CLOSE (UNIT = NOUT)
               CLOSE (UNIT = NOUT10)
               IF (NFILES.EQ.N0) THEN
                  CALL DELEET (FNAME, 
     +                         ASKIF, THERE)
                  FNAME = BLANK
               ENDIF
            ENDIF
C
C Another go ?
C
            CALL STOPGO (BLANK, FNAME, PNAME, 
     +                   ISTOP)
            IF (ISTOP) THEN
               REPEET = .FALSE.
            ELSE
               REPEET = .TRUE.
            ENDIF
C
C .....................................................................
C End of main loop
C .....................................................................
C
         ENDDO
      ENDIF


                  
C
C Format statements
C
  100 FORMAT (A)
C
C Note: this order MUST be consistent with PFILES and PFILE1
C ==========================================================
C a_recent.cfg = all
C v_recent.cfg = vectors
C m_recent.cfg = matrices
C g_recent.cfg = graph coordinates
C f_recent.cfg = fitting files
C p_recent.cfg = PostScript
C c_recent.cfg = covariance matrices
C s_recent.cfg = SVG files 
C
  200 FORMAT (
     + 'Choosing the file type for this library file'
     +/
     +/'Simfit library files consist of a list of files that are all'
     +/'of the same type, such as EPS or SVG graphics files.'
     +/
     +/'When populating such a list by selecting one file at a time'
     +/'or multiple files then the file type will be checked before'
     +/'any file is acceptable.'
     +/
     +/'Any files (*.*)'
     +/'Vector files (1 column)'
     +/'Matrix files (n columns)'
     +/'Plot files (2,3,4, or 6 columns)'
     +/'Curve fit files (2 or 3 columns)'
     +/'EPS files (Encapsulated PostScript files)'
     +/'Covariance matrix files'
     +/'SVG files (Scalable vector graphics)'
     +/'Help'
     +/'Quit ... Exit program MAKLIB')
      END
C
C-------------------------------------------------------------------
C
      SUBROUTINE MAKLIB_ADVISE (ABORT, FIRST)
C
C Advise user
C
      IMPLICIT   NONE
C
C Arguments
C
      LOGICAL,             INTENT (IN)  :: FIRST
      LOGICAL,             INTENT (OUT) :: ABORT
C
C locals
C
      INTEGER    ISEND
      INTEGER    ICOLOR, NUMHDR, NUMOPT
      PARAMETER (ICOLOR = 7, NUMHDR = 11, NUMOPT = 3)
      INTEGER    NUMBLD(NUMHDR), NUMPOS(NUMOPT)
      CHARACTER  HEADER(NUMHDR)*100, OPTION(NUMOPT)*100
      LOGICAL    REPEET
      EXTERNAL   TITLES, HELP_MAKLIB
      DATA       NUMBLD / NUMHDR*0 /
      DATA       NUMPOS / NUMOPT*1 /
      DATA       OPTION /
     +'Help           ',
     +'Run the program',
     +'Quit  ...  Exit' /
      ABORT = .FALSE.
      REPEET = .TRUE.
      DO WHILE (REPEET)
         WRITE (HEADER,100) 
         IF (FIRST) THEN
            ISEND = 1
            CALL TITLES (ICOLOR, NUMBLD, ISEND, NUMHDR, NUMOPT, NUMPOS,
     +                   HEADER, OPTION)
         ELSE
            ISEND = 1
         ENDIF
         IF (ISEND.EQ.1) THEN
            CALL HELP_MAKLIB ('maklib')
            IF (FIRST) THEN
               REPEET = .TRUE.
            ELSE
               REPEET = .FALSE.
            ENDIF
         ELSEIF (ISEND.EQ.2) THEN
            ABORT = .FALSE.
            REPEET = .FALSE.
         ELSEIF (ISEND.EQ.3) THEN
            ABORT = .TRUE.
            REPEET = .FALSE.
         ENDIF
      ENDDO        
C
C Format statement
C      
  100 FORMAT (
     + 'Package`SIMFIT'
     +/'       `      '
     +/'Program`MAKLIB'
     +/'       `      '
     +/'Action `Create a SIMFIT type library file to reference'
     +/'       `a set of individual data files, so they can be'
     +/'       `used for plotting, statistics or curve fitting.'
     +/'       `      '
     +/'Version`DLL' 
     +/'       `      '
     +/'Author `W.G.Bardsley, University of Manchester, U.K.')
      END
C
C-----------------------------------------------------------------------------------
C
      SUBROUTINE SVGLIB (NFILES, NIN, NSMALL, 
     +                   FSAV,
     +                   ABORT)
C
C ACTION: Use a library file to read in a set of SVG files
C         Returns SVG files in FSAV(NFILES)
C AUTHOR: W. G. Bardsley, University of Manchester, U.K.
C         29/12/2018 derived from EDITPS_PS2LIB
C
C         ABORT = .TRUE. if library file not usable
C         If NSMALL = 1 then just return a SVG file  
C
C NFILES: (output) number of svg files selected
C    NIN: (input/unchanged) unconnected unit for reading files
C NSMALL: (input/unchanged) dimension
C   FSAV: (output) svg files
C  ABORT: (output) error indicator
C
      IMPLICIT   NONE   
C
C Arguments
C      
      INTEGER,             INTENT (IN)  :: NIN, NSMALL  
      INTEGER,             INTENT (OUT) :: NFILES 
      CHARACTER (LEN = *), INTENT (OUT) :: FSAV(NSMALL)
      LOGICAL,             INTENT (OUT) :: ABORT
C
C Locals
C      
      INTEGER    N0, N1, N3
      PARAMETER (N0 = 0, N1 = 1, N3 = 3)
      INTEGER    I, IADD1, IOS, ISEND, K, L, LEN200
      INTEGER    IX, IY, IZ
      CHARACTER (LEN = 1024) FNAME, FNLIB, SIMFIT, TEMP, AUX256
      CHARACTER (LEN = 100 ) LINE, TITLE
      CHARACTER (LEN = 1   ) BLANK, BSLASH
      PARAMETER (BLANK = ' ', BSLASH = '\')
      LOGICAL    YES
      LOGICAL    ALLOW_BLANKS, ALLOW_ISVG, NOISY
      PARAMETER (ALLOW_BLANKS = .FALSE., ALLOW_ISVG = .FALSE.)
      EXTERNAL   CHECKF, OFILES, PUTADV, TXT2FN, VU2CHK,
     +           DEMDIR, TRIML1, LEN200, AUX256  
      EXTERNAL   W_SVGCHK    
      INTRINSIC  INDEX   
C
C Part 1: First initialise ABORT and NFILES
C ======
C
      ABORT = .TRUE.
      NOISY = .TRUE.
      NFILES = N0
      IF (NSMALL.LT.N1) RETURN
      DO I = N1, NSMALL
         FSAV(I) = BLANK
      ENDDO   
C
C Part 2: Check that the file seems to be a library file
C ======
C
      ISEND = N3
      CLOSE (UNIT = NIN)
      CALL OFILES (ISEND, NIN, 
     +             FNAME,
     +             ABORT)
      CLOSE (UNIT = NIN)
      IF (ABORT) RETURN
C
C Is it an svg file ? ... if so then return silently
C
      NOISY = .FALSE.
      CALL W_SVGCHK (IX, IY, IZ,
     +               FNAME,
     +               ALLOW_BLANKS, ALLOW_ISVG, NOISY) 
      IF (IZ.EQ.N1) THEN
         NFILES = N1
         FSAV(N1) = FNAME
         RETURN
      ENDIF
      IF (NSMALL.EQ.N1) RETURN
C
C Is it a library file ?
C
      OPEN (UNIT = NIN, FILE = FNAME)
      FNLIB = FNAME
      READ (NIN,'(A)',IOSTAT=IOS) TITLE
      IF (IOS.NE.N0) THEN  
         WRITE (LINE,100)
         CALL PUTADV (LINE) 
         CLOSE (UNIT = NIN)
         CALL VU2CHK (FNLIB)
         RETURN
      ENDIF
      CLOSE (UNIT = NIN)
      CALL CHECKF (FNAME, TITLE, 
     +             ABORT)
      IF (ABORT) THEN
         RETURN
      ELSE
         OPEN (UNIT = NIN, FILE = FNAME)
         READ (NIN,'(A)',IOSTAT=IOS) TITLE
      ENDIF
C
C Read in the file names
C
      IADD1 = N0
      IOS = N0
      DO WHILE (IOS.EQ.N0 .AND. IADD1.LT.NSMALL)
         ABORT = .TRUE.
         READ (NIN,'(A)',IOSTAT=IOS) FNAME
         IF (IOS.EQ.N0) THEN
            IF (FNAME.EQ.BLANK) THEN
               IOS = - N1
            ELSE   
               CALL TXT2FN (FNAME, 
     +                      ABORT)
               IF (ABORT) THEN                
                  WRITE (LINE,200)
                  CALL PUTADV (LINE)
                  WRITE (LINE,300) I + N1
                  CALL PUTADV (LINE)
                  CLOSE (UNIT = NIN)
                  CALL VU2CHK (FNLIB)
                  RETURN
               ELSE
                  IADD1 = IADD1 + N1
                  FSAV(IADD1) = FNAME
               ENDIF
            ENDIF   
         ENDIF
      ENDDO
      CLOSE (UNIT = NIN)
      ABORT = .TRUE.
C
C Part 3: Check for files existing and SVG format
C ======
C
      IF (IADD1.LE.N0) THEN    
         WRITE (LINE,400)
         CALL PUTADV (LINE) 
         CALL VU2CHK (FNLIB)
         RETURN
      ENDIF 
      CALL DEMDIR (L,
     +             SIMFIT)       
      DO I = N1, IADD1
         CLOSE (UNIT = NIN)
         INQUIRE (FILE = FSAV(I), EXIST = YES, IOSTAT = IOS)
         IF (L.GT.0 .AND. .NOT.YES .AND. 
     +       INDEX(FSAV(I),BSLASH).LT.N1) THEN
            TEMP = FSAV(I)
            CALL TRIML1 (TEMP)
            K = LEN200(TEMP)
            FSAV(I) = SIMFIT(N1:L)//TEMP(N1:K)
            INQUIRE (FILE = FSAV(I), EXIST = YES, IOSTAT = IOS)
         ELSEIF (.NOT.YES) THEN
            TEMP = AUX256 (FSAV(I)) 
            INQUIRE (FILE = TEMP, EXIST = YES, IOSTAT = IOS)
            IF (YES .AND. IOS.EQ.N0) FSAV(I) = TEMP   
         ENDIF   
         IF (YES .AND. IOS.EQ.N0) THEN
            NOISY = .TRUE.
            CALL W_SVGCHK (IX, IY, IZ,
     +                     FSAV(I),
     +                     ALLOW_BLANKS, ALLOW_ISVG, NOISY)            
            IF (IZ.NE.N1) THEN
               WRITE (LINE,300) I
               CALL PUTADV (LINE)
               CLOSE (UNIT = NIN)
               RETURN
            ENDIF
            NFILES = NFILES + N1
            FSAV(NFILES) = FSAV(I)
         ELSE
            IOS = N1 
            WRITE (LINE,500)
            CALL PUTADV (LINE)
         ENDIF
         IF (IOS.NE.N0) THEN
            WRITE (LINE,300) I + N1
            CALL PUTADV (LINE)
            ABORT = .TRUE.
            CLOSE (UNIT = NIN)
            CALL VU2CHK (FNLIB)
            RETURN
         ENDIF
         CLOSE (UNIT = NIN)
      ENDDO
      CLOSE (UNIT = NIN)
C
C Set ABORT = .FALSE. if at least one file is satisfactory
C
      IF (NFILES.GT.N0) ABORT = .FALSE.
C
C Format statements
C      
  100 FORMAT (
     +'SVGLIB error: Title not defined in library file')
  200 FORMAT (
     +'SVGLIB error: File inconsistent with Operating System')
  300 FORMAT (
     +'SVGLIB error: File missing or not Simfit-type *.SVG at line',I4)
  400 FORMAT (
     +'SVGLIB error: No files in library file')
  500 FORMAT (
     +'SVGLIB error: File referenced in library file not found')
      END
C
C
