C
C Clausen integral from MISC but with ERRPRN disabled
C ===================================================
C
C
      DOUBLE PRECISION FUNCTION CLAUSN(XVALUE)
C
C DESCRIPTION:
C
C   This program calculates Clausen's integral defined by
C
C          CLAUSN(x) = integral 0 to x of (-ln(2*sin(t/2))) dt
C
C   The code uses Chebyshev expansions with the coefficients
C   given to 20 decimal places.
C
C
C ERROR RETURNS:
C
C   If |x| is too large it is impossible to reduce the argument
C   to the range [0,2*pi] with any precision. An error message
C   is printed and the program returns the value 0.0
C
C
C MACHINE-DEPENDENT CONSTANTS:
C
C   NTERMS - INTEGER - the no. of terms of the array ACLAUS
C                      to be used. The recommended value is
C                      such that ABS(ACLAUS(NTERMS)) < EPS/100
C                      subject to 1 <= NTERMS <= 15
C
C   XSMALL - DOUBLE PRECISION - the value below which Cl(x) can be
C                   approximated by x (1-ln x). The recommended
C                   value is pi*sqrt(EPSNEG/2).
C
C   XHIGH - DOUBLE PRECISION - The value of |x| above which we cannot
C                  reliably reduce the argument to [0,2*pi].
C                  The recommended value is   1/EPS.
C
C     For values of EPS and EPSNEG refer to the file MACHCON.TXT
C
C     The machine-dependent constants are computed internally by
C     using the D1MACH subroutine.
C
C
C INTRINSIC FUNCTIONS USED:
C
C   AINT , LOG , SQRT
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
C          CHEVAL , ERRPRN, D1MACH
C
C
C AUTHOR:  Dr. Allan J. MacLeod,
C          Dept. of Mathematics and Statistics,
C          University of Paisley,
C          High St.
C          PAISLEY
C          SCOTLAND
C
C          ( e-mail: macl_ms0@paisley.ac.uk )
C
C
C LATEST MODIFICATION: 23 January, 1996
C
      INTEGER INDX,NTERMS
      DOUBLE PRECISION ACLAUS(0:15),CHEVAL,HALF,ONE,ONEHUN,PI,PISQ,T,
     &     TWOPI,TWOPIA,TWOPIB,X,XHIGH,XSMALL,XVALUE,ZERO,D1MACH
C
C INTRINSIC FUNCTIONS USED:
C
       INTRINSIC ABS, AINT, LOG , SQRT
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
       EXTERNAL CHEVAL, D1MACH
C
c*****CHARACTER FNNAME*6,ERRMSG*26
c*****DATA FNNAME/'CLAUSN'/
c*****DATA ERRMSG/'ARGUMENT TOO LARGE IN SIZE'/
      DATA ZERO,HALF,ONE/0.0 D 0 , 0.5 D 0 , 1.0 D 0/
      DATA ONEHUN/100.0 D 0/
      DATA PI/3.14159 26535 89793 2385 D 0/
      DATA PISQ/9.86960 44010 89358 6188 D 0/
      DATA TWOPI/6.28318 53071 79586 4769 D 0/
      DATA TWOPIA,TWOPIB/6.28125 D 0 , 0.19353 07179 58647 69253 D -2/
      DATA ACLAUS/2.14269 43637 66688 44709  D    0,
     1            0.72332 42812 21257 9245   D   -1,
     2            0.10164 24750 21151 164    D   -2,
     3            0.32452 50328 53164 5      D   -4,
     4            0.13331 51875 71472        D   -5,
     5            0.62132 40591 653          D   -7,
     6            0.31300 41353 37           D   -8,
     7            0.16635 72305 6            D   -9,
     8            0.91965 9293               D  -11,
     9            0.52400 462                D  -12,
     X            0.30580 40                 D  -13,
     1            0.18196 9                  D  -14,
     2            0.11004                    D  -15,
     3            0.675                      D  -17,
     4            0.42                       D  -18,
     5            0.3                        D  -19/
C
C  Start execution
C
      X = XVALUE
C
C   Compute the machine-dependent constants.
C
      T = D1MACH(3)
      XHIGH = ONE / T
C
C   Error test
C
      IF ( ABS(X) .GT. XHIGH ) THEN
c********CALL ERRPRN(FNNAME,ERRMSG)
         CLAUSN = ZERO
         RETURN
      ENDIF
C
C   Continue with machine-dependent constants
C
      XSMALL = PI * SQRT ( HALF * T )
      T = T / ONEHUN
      DO 10 NTERMS = 15 , 0 , -1
         IF ( ABS(ACLAUS(NTERMS)) .GT. T ) GOTO 19
 10   CONTINUE
C
C  Continue with computation
C
 19   INDX = 1
      IF ( X .LT. ZERO ) THEN
         X = -X
         INDX = -1
      ENDIF
C
C  Argument reduced using simulated extra precision
C
      IF ( X .GT. TWOPI ) THEN
         T = AINT( X / TWOPI )
         X =  ( X - T * TWOPIA ) - T * TWOPIB
      ENDIF
      IF ( X .GT. PI ) THEN
         X = ( TWOPIA - X ) + TWOPIB
         INDX = -INDX
      ENDIF
C
C  Set result to zero if X multiple of PI
C
      IF ( X .EQ. ZERO ) THEN
         CLAUSN = ZERO
         RETURN
      ENDIF
C
C  Code for X < XSMALL
C
      IF ( X .LT. XSMALL ) THEN
         CLAUSN = X * ( ONE - LOG( X ) )
      ELSE
C
C  Code for XSMALL < =  X < =  PI
C
         T =  ( X * X ) / PISQ - HALF
         T = T + T
         IF ( T .GT. ONE ) T = ONE
         CLAUSN = X * CHEVAL( NTERMS,ACLAUS,T ) - X * LOG( X )
      ENDIF
      IF ( INDX .LT. 0 ) CLAUSN = -CLAUSN
      RETURN
      END

