C
C Debye functions from the MISC library
C

       double precision function debye0 (x, n)
c
c action: integrate debye n using MISC
c author: w.g.bardsley, university of manchester, u.k., 19/12/2001
c         Note: the error messaging is turned off in the functions
c
      implicit none
      integer n
      double precision x
      double precision zero
      parameter (zero = 0.0d+00)
      double precision debye1, debye2, debye3, debye4
      external debye1, debye2, debye3, debye4
      debye0 = zero
      if (x.le.zero) return
      if (n.lt.1 .or. n.gt.4) return
      if (n.eq.1) then
         debye0 = debye1 (x)
      elseif (n.eq.2) then
         debye0 = debye2 (x)
      elseif (n.eq.3) then
         debye0 = debye3 (x)
      elseif (n.eq.4) then
         debye0 = debye4 (x)
      endif
      end
c
c

      DOUBLE PRECISION FUNCTION DEBYE1(XVALUE)
C
C
C   DEFINITION:
C
C      This program calculates the Debye function of order 1, defined as
C
C            DEBYE1(x) = [Integral {0 to x} t/(exp(t)-1) dt] / x
C
C      The code uses Chebyshev series whose coefficients
C      are given to 20 decimal places.
C
C
C   ERROR RETURNS:
C
C      If XVALUE < 0.0 an error message is printed and the
C      function returns the value 0.0
C
C
C   MACHINE-DEPENDENT PARAMETERS:
C
C      NTERMS - INTEGER - The no. of elements of the array ADEB1.
C                         The recommended value is such that
C                             ABS(ADEB1(NTERMS)) < EPS/100 , with
C                                   1 <= NTERMS <= 18
C
C      XLOW - DOUBLE PRECISION - The value below which
C                    DEBYE1 = 1 - x/4 + x*x/36 to machine precision.
C                    The recommended value is
C                        SQRT(8*EPSNEG)
C
C      XUPPER - DOUBLE PRECISION - The value above which
C                      DEBYE1 = (pi*pi/(6*x)) - exp(-x)(x+1)/x.
C                      The recommended value is
C                          -LOG(2*EPS)
C
C      XLIM - DOUBLE PRECISION - The value above which DEBYE1 = pi*pi/(6*x)
C                    The recommended value is
C                          -LOG(XMIN)
C
C      For values of EPS, EPSNEG, and XMIN see the file MACHCON.TXT
C
C      The machine-dependent constants are computed internally by
C      using the D1MACH subroutine.
C
C
C   INTRINSIC FUNCTIONS USED:
C
C      AINT , EXP , INT , LOG , SQRT
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
C          CHEVAL , ERRPRN, D1MACH
C
C
C   AUTHOR:
C          Dr. Allan J. MacLeod,
C          Dept. of Mathematics and Statistics,
C          University of Paisley
C          High St.
C          PAISLEY
C          SCOTLAND
C          PA1 2BE
C
C          (e-mail:  macl_ms0@paisley.ac.uk )
C
C
C   LATEST UPDATE:  23 january, 1996
C
      INTEGER I,NEXP,NTERMS
      DOUBLE PRECISION ADEB1(0:18),CHEVAL,DEBINF,EIGHT,EXPMX,FOUR,HALF,
     &     NINE,ONE,ONEHUN,QUART,RK,SUM,T,THIRT6,X,XK,XLIM,XLOW,
     &     XUPPER,XVALUE,ZERO,D1MACH
C
C   INTRINSIC FUNCTIONS USED:
C
      intrinsic AINT , EXP , INT , LOG , SQRT, abs
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
       external CHEVAL, D1MACH
C
c*****CHARACTER FNNAME*6,ERRMSG*17
c*****DATA FNNAME/'DEBYE1'/
c*****DATA ERRMSG/'ARGUMENT NEGATIVE'/
      DATA ZERO,QUART/0.0 D 0 , 0.25 D 0/
      DATA HALF,ONE/0.5 D 0 , 1.0 D 0/
      DATA FOUR,EIGHT/4.0 D 0 , 8.0 D 0/
      DATA NINE,THIRT6,ONEHUN/9.0 D 0 , 36.0 D 0 , 100.0 D 0/
      DATA DEBINF/0.60792 71018 54026 62866 D 0/
      DATA ADEB1/2.40065 97190 38141 01941  D    0,
     1           0.19372 13042 18936 00885  D    0,
     2          -0.62329 12455 48957 703    D   -2,
     3           0.35111 74770 20648 00     D   -3,
     4          -0.22822 24667 01231 0      D   -4,
     5           0.15805 46787 50300        D   -5,
     6          -0.11353 78197 0719         D   -6,
     7           0.83583 36118 75           D   -8,
     8          -0.62644 24787 2            D   -9,
     9           0.47603 34890              D  -10,
     X          -0.36574 1540               D  -11,
     1           0.28354 310                D  -12,
     2          -0.22147 29                 D  -13,
     3           0.17409 2                  D  -14,
     4          -0.13759                    D  -15,
     5           0.1093                     D  -16,
     6          -0.87                       D  -18,
     7           0.7                        D  -19,
     8          -0.1                        D  -19/
C
C   Start computation
C
      X = XVALUE
C
C   Check XVALUE >= 0.0
C
      IF ( X .LT. ZERO ) THEN
c********CALL ERRPRN(FNNAME,ERRMSG)
         DEBYE1 = ZERO
         RETURN
      ENDIF
C
C   Compute the machine-dependent constants.
C
      T = D1MACH(3)
      XLOW = SQRT ( T * EIGHT )
      XUPPER = - LOG( T + T )
      XLIM = - LOG( D1MACH(1) )
      T = T / ONEHUN
      DO 10 NTERMS = 18 , 0 , -1
         IF ( ABS(ADEB1(NTERMS)) .GT. T ) GOTO 19
 10   CONTINUE
C
C   Code for x <= 4.0
C
 19   IF ( X .LE. FOUR ) THEN
         IF ( X .LT. XLOW ) THEN
            DEBYE1 = ( ( X - NINE ) * X + THIRT6 ) / THIRT6
         ELSE
            T = ( ( X * X / EIGHT ) - HALF ) - HALF
            DEBYE1 = CHEVAL( NTERMS , ADEB1 , T ) - QUART * X
         ENDIF
      ELSE
C
C   Code for x > 4.0
C
         DEBYE1 = ONE / ( X * DEBINF )
         IF ( X .LT. XLIM ) THEN
            EXPMX = EXP( -X )
            IF ( X .GT. XUPPER ) THEN
               DEBYE1 = DEBYE1 - EXPMX * ( ONE + ONE / X )
            ELSE
               SUM = ZERO
               RK = AINT( XLIM / X )
               NEXP = INT( RK )
               XK = RK * X
               DO 100 I = NEXP,1,-1
                  T =  ( ONE + ONE / XK ) / RK
                  SUM = SUM * EXPMX + T
                  RK = RK - ONE
                  XK = XK - X
 100           CONTINUE
               DEBYE1 = DEBYE1 - SUM * EXPMX
            ENDIF
         ENDIF
      ENDIF
      RETURN
      END

      DOUBLE PRECISION FUNCTION DEBYE2(XVALUE)
C
C
C   DEFINITION:
C
C      This program calculates the Debye function of order 1, defined as
C
C            DEBYE2(x) = 2*[Integral {0 to x} t*t/(exp(t)-1) dt] / (x*x)
C
C      The code uses Chebyshev series whose coefficients
C      are given to 20 decimal places.
C
C
C   ERROR RETURNS:
C
C      If XVALUE < 0.0 an error message is printed and the
C      function returns the value 0.0
C
C
C   MACHINE-DEPENDENT PARAMETERS:
C
C      NTERMS - INTEGER - The no. of elements of the array ADEB2.
C                         The recommended value is such that
C                             ABS(ADEB2(NTERMS)) < EPS/100,
C                         subject to 1 <= NTERMS <= 18.
C
C      XLOW - DOUBLE PRECISION - The value below which
C                    DEBYE2 = 1 - x/3 + x*x/24 to machine precision.
C                    The recommended value is
C                        SQRT(8*EPSNEG)
C
C      XUPPER - DOUBLE PRECISION - The value above which
C                      DEBYE2 = (4*zeta(3)/x^2) - 2*exp(-x)(x^2+2x+1)/x^2.
C                      The recommended value is
C                          -LOG(2*EPS)
C
C      XLIM1 - DOUBLE PRECISION - The value above which DEBYE2 = 4*zeta(3)/x^2
C                     The recommended value is
C                          -LOG(XMIN)
C
C      XLIM2 - DOUBLE PRECISION - The value above which DEBYE2 = 0.0 to machine
C                     precision. The recommended value is
C                           SQRT(4.8/XMIN)
C
C      For values of EPS, EPSNEG, and XMIN see the file MACHCON.TXT
C
C
C      The machine-dependent constants are computed internally by
C      using the D1MACH subroutine.
C
C
C   INTRINSIC FUNCTIONS USED:
C
C      AINT , EXP , INT , LOG , SQRT
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
C          CHEVAL , ERRPRN, D1MACH
C
C
C   AUTHOR:
C          Dr. Allan J. MacLeod,
C          Dept. of Mathematics and Statistics,
C          University of Paisley
C          High St.
C          PAISLEY
C          SCOTLAND
C          PA1 2BE
C
C          (e-mail:  macl_ms0@paisley.ac.uk )
C
C
C   LATEST UPDATE:  23 January, 1996
C
      INTEGER I,NEXP,NTERMS
      DOUBLE PRECISION ADEB2(0:18),CHEVAL,DEBINF,EIGHT,EXPMX,FOUR,
     &     HALF,ONE,ONEHUN,RK,SUM,T,THREE,TWENT4,TWO,X,XK,XLIM1,
     &     XLIM2,XLOW,XUPPER,XVALUE,ZERO,D1MACH
C   INTRINSIC FUNCTIONS USED:
C
       intrinsic AINT , EXP , INT , LOG , SQRT, abs
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
       external CHEVAL, D1MACH
C
c*****CHARACTER FNNAME*6,ERRMSG*17
c*****DATA FNNAME/'DEBYE2'/
c*****DATA ERRMSG/'ARGUMENT NEGATIVE'/
      DATA ZERO,HALF/0.0 D 0 , 0.5 D 0/
      DATA ONE,TWO,THREE/1.0 D 0 , 2.0 D 0 , 3.0 D 0/
      DATA FOUR,EIGHT,TWENT4/4.0 D 0 , 8.0 D 0 , 24.0 D 0/
      DATA ONEHUN/100.0 D 0/
      DATA DEBINF/4.80822 76126 38377 14160 D 0/
      DATA ADEB2/2.59438 10232 57077 02826  D    0,
     1           0.28633 57204 53071 98337  D    0,
     2          -0.10206 26561 58046 7129   D   -1,
     3           0.60491 09775 34684 35     D   -3,
     4          -0.40525 76589 50210 4      D   -4,
     5           0.28633 82632 88107        D   -5,
     6          -0.20863 94303 0651         D   -6,
     7           0.15523 78758 264          D   -7,
     8          -0.11731 28008 66           D   -8,
     9           0.89735 85888              D  -10,
     X          -0.69317 6137               D  -11,
     1           0.53980 568                D  -12,
     2          -0.42324 05                 D  -13,
     3           0.33377 8                  D  -14,
     4          -0.26455                    D  -15,
     5           0.2106                     D  -16,
     6          -0.168                      D  -17,
     7           0.13                       D  -18,
     8          -0.1                        D  -19/
C
C   Start computation
C
      X = XVALUE
C
C   Check XVALUE >= 0.0
C
      IF ( X .LT. ZERO ) THEN
c********CALL ERRPRN(FNNAME,ERRMSG)
         DEBYE2 = ZERO
         RETURN
      ENDIF
C
C   Compute the machine-dependent constants.
C
      T = D1MACH(1)
      XLIM1 = - LOG( T )
      XLIM2 = SQRT( DEBINF ) / SQRT( T )
      T = D1MACH(3)
      XLOW = SQRT ( T * EIGHT )
      XUPPER = - LOG( T + T )
      T = T / ONEHUN
      DO 10 NTERMS = 18 , 0 , -1
         IF ( ABS(ADEB2(NTERMS)) .GT. T ) GOTO 19
 10   CONTINUE
C
C   Code for x <= 4.0
C
 19   IF ( X .LE. FOUR ) THEN
         IF ( X .LT. XLOW ) THEN
            DEBYE2 = ( ( X - EIGHT ) * X + TWENT4 ) / TWENT4
         ELSE
            T = ( ( X * X / EIGHT ) - HALF ) - HALF
            DEBYE2 = CHEVAL ( NTERMS , ADEB2 , T ) - X / THREE
         ENDIF
      ELSE
C
C   Code for x > 4.0
C
         IF ( X .GT. XLIM2 ) THEN
            DEBYE2 = ZERO
         ELSE
            DEBYE2 = DEBINF / ( X * X )
            IF ( X .LT. XLIM1 ) THEN
               EXPMX = EXP ( -X )
               IF ( X .GT. XUPPER ) THEN
                  SUM = ( ( X + TWO ) * X + TWO ) / ( X * X )
               ELSE
                  SUM = ZERO
                  RK = AINT ( XLIM1 / X )
                  NEXP = INT ( RK )
                  XK = RK * X
                  DO 100 I = NEXP,1,-1
                     T =  ( ONE + TWO / XK + TWO / ( XK*XK ) ) / RK
                     SUM = SUM * EXPMX + T
                     RK = RK - ONE
                     XK = XK - X
 100              CONTINUE
               ENDIF
               DEBYE2 = DEBYE2 - TWO * SUM * EXPMX
            ENDIF
         ENDIF
      ENDIF
      RETURN
      END

      DOUBLE PRECISION FUNCTION DEBYE3(XVALUE)
C
C
C   DEFINITION:
C
C      This program calculates the Debye function of order 3, defined as
C
C            DEBYE3(x) = 3*[Integral {0 to x} t^3/(exp(t)-1) dt] / (x^3)
C
C      The code uses Chebyshev series whose coefficients
C      are given to 20 decimal places.
C
C
C   ERROR RETURNS:
C
C      If XVALUE < 0.0 an error message is printed and the
C      function returns the value 0.0
C
C
C   MACHINE-DEPENDENT PARAMETERS:
C
C      NTERMS - INTEGER - The no. of elements of the array ADEB3.
C                         The recommended value is such that
C                             ABS(ADEB3(NTERMS)) < EPS/100,
C                         subject to 1 <= NTERMS <= 18
C
C      XLOW - DOUBLE PRECISION - The value below which
C                    DEBYE3 = 1 - 3x/8 + x*x/20 to machine precision.
C                    The recommended value is
C                        SQRT(8*EPSNEG)
C
C      XUPPER - DOUBLE PRECISION - The value above which
C               DEBYE3 = (18*zeta(4)/x^3) - 3*exp(-x)(x^3+3x^2+6x+6)/x^3.
C                      The recommended value is
C                          -LOG(2*EPS)
C
C      XLIM1 - DOUBLE PRECISION - The value above which DEBYE3 = 18*zeta(4)/x^3
C                     The recommended value is
C                          -LOG(XMIN)
C
C      XLIM2 - DOUBLE PRECISION - The value above which DEBYE3 = 0.0 to machine
C                     precision. The recommended value is
C                          CUBE ROOT(19/XMIN)
C
C      For values of EPS, EPSNEG, and XMIN see the file MACHCON.TXT
C
C      The machine-dependent constants are computed internally by
C      using the D1MACH subroutine.
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
C          CHEVAL , ERRPRN, D1MACH
C
C
C   INTRINSIC FUNCTIONS USED:
C
C      AINT , EXP , INT , LOG , SQRT
C
C
C   AUTHOR:
C          Dr. Allan J. MacLeod,
C          Dept. of Mathematics and Statistics,
C          University of Paisley
C          High St.
C          PAISLEY
C          SCOTLAND
C          PA1 2BE
C
C          (e-mail:  macl_ms0@paisley.ac.uk )
C
C
C   LATEST UPDATE:  23 January, 1996
C
      INTEGER I,NEXP,NTERMS
      DOUBLE PRECISION ADEB3(0:18),CHEVAL,DEBINF,EIGHT,EXPMX,FOUR,
     &     HALF,ONE,ONEHUN,PT375,RK,SEVP5,SIX,SUM,T,THREE,TWENTY,X,
     &     XK,XKI,XLIM1,XLIM2,XLOW,XUPPER,XVALUE,ZERO,D1MACH
C
C   OTHER MISCFUN SUBROUTINES USED:
C
       external CHEVAL, D1MACH
C
C
C   INTRINSIC FUNCTIONS USED:
C
       intrinsic  AINT , EXP , INT , LOG , SQRT, abs
C
c*****CHARACTER FNNAME*6,ERRMSG*17
c*****DATA FNNAME/'DEBYE3'/
c*****DATA ERRMSG/'ARGUMENT NEGATIVE'/
      DATA ZERO,PT375/0.0 D 0 , 0.375 D 0/
      DATA HALF,ONE/0.5 D 0 , 1.0 D 0/
      DATA THREE,FOUR,SIX/3.0 D 0 , 4.0 D 0 , 6.0 D 0/
      DATA SEVP5,EIGHT,TWENTY/7.5 D 0 , 8.0 D 0 , 20.0 D 0/
      DATA ONEHUN/100.0 D 0/
      DATA DEBINF/0.51329 91127 34216 75946 D -1/
      DATA ADEB3/2.70773 70683 27440 94526  D    0,
     1           0.34006 81352 11091 75100  D    0,
     2          -0.12945 15018 44408 6863   D   -1,
     3           0.79637 55380 17381 64     D   -3,
     4          -0.54636 00095 90823 8      D   -4,
     5           0.39243 01959 88049        D   -5,
     6          -0.28940 32823 5386         D   -6,
     7           0.21731 76139 625          D   -7,
     8          -0.16542 09994 98           D   -8,
     9           0.12727 96189 2            D   -9,
     X          -0.98796 3459               D  -11,
     1           0.77250 740                D  -12,
     2          -0.60779 72                 D  -13,
     3           0.48075 9                  D  -14,
     4          -0.38204                    D  -15,
     5           0.3048                     D  -16,
     6          -0.244                      D  -17,
     7           0.20                       D  -18,
     8          -0.2                        D  -19/
C
C   Start computation
C
      X = XVALUE
C
C   Error test
C
      IF ( X .LT. ZERO ) THEN
c********CALL ERRPRN(FNNAME,ERRMSG)
         DEBYE3 = ZERO
         RETURN
      ENDIF
C
C   Compute the machine-dependent constants.
C
      T = D1MACH(1)
      XLIM1 = - LOG( T )
      XK = ONE / THREE
      XKI = (ONE/DEBINF) ** XK
      RK = T ** XK
      XLIM2 = XKI / RK
      T = D1MACH(3)
      XLOW = SQRT ( T * EIGHT )
      XUPPER = - LOG( T + T )
      T = T / ONEHUN
      DO 10 NTERMS = 18 , 0 , -1
         IF ( ABS(ADEB3(NTERMS)) .GT. T ) GOTO 19
 10   CONTINUE
C
C   Code for x <= 4.0
C
 19   IF ( X .LE. FOUR ) THEN
         IF ( X .LT. XLOW ) THEN
            DEBYE3 = ( ( X - SEVP5 ) * X + TWENTY ) / TWENTY
         ELSE
            T = ( ( X * X / EIGHT ) - HALF ) - HALF
            DEBYE3 = CHEVAL ( NTERMS , ADEB3 , T ) - PT375 * X
         ENDIF
      ELSE
C
C   Code for x > 4.0
C
         IF ( X .GT. XLIM2 ) THEN
            DEBYE3 = ZERO
         ELSE
            DEBYE3 = ONE / ( DEBINF * X * X * X )
            IF ( X .LT. XLIM1 ) THEN
               EXPMX = EXP ( -X )
               IF ( X .GT. XUPPER ) THEN
                  SUM = (((X+THREE)*X+SIX)*X+SIX) / (X*X*X)
               ELSE
                  SUM = ZERO
                  RK = AINT ( XLIM1 / X )
                  NEXP = INT ( RK )
                  XK = RK * X
                  DO 100 I = NEXP,1,-1
                     XKI = ONE / XK
                     T =  (((SIX*XKI+SIX)*XKI+THREE)*XKI+ONE) / RK
                     SUM = SUM * EXPMX + T
                     RK = RK - ONE
                     XK = XK - X
 100              CONTINUE
               ENDIF
               DEBYE3 = DEBYE3 - THREE * SUM * EXPMX
            ENDIF
         ENDIF
      ENDIF
      RETURN
      END

      DOUBLE PRECISION FUNCTION DEBYE4(XVALUE)
C
C
C   DEFINITION:
C
C      This program calculates the Debye function of order 4, defined as
C
C            DEBYE4(x) = 4*[Integral {0 to x} t^4/(exp(t)-1) dt] / (x^4)
C
C      The code uses Chebyshev series whose coefficients
C      are given to 20 decimal places.
C
C
C   ERROR RETURNS:
C
C      If XVALUE < 0.0 an error message is printed and the
C      function returns the value 0.0
C
C
C   MACHINE-DEPENDENT PARAMETERS:
C
C      NTERMS - INTEGER - The no. of elements of the array ADEB4.
C                         The recommended value is such that
C                             ABS(ADEB4(NTERMS)) < EPS/100,
C                         subject to 1 <= NTERMS <= 18
C
C      XLOW - DOUBLE PRECISION - The value below which
C                    DEBYE4 = 1 - 4x/10 + x*x/18 to machine precision.
C                    The recommended value is
C                        SQRT(8*EPSNEG)
C
C      XUPPER - DOUBLE PRECISION - The value above which
C               DEBYE4=(96*zeta(5)/x^4)-4*exp(-x)(x^4+4x^2+12x^2+24x+24)/x^4.
C                      The recommended value is
C                          -LOG(2*EPS)
C
C      XLIM1 - DOUBLE PRECISION - The value above which DEBYE4 = 96*zeta(5)/x^4
C                     The recommended value is
C                          -LOG(XMIN)
C
C      XLIM2 - DOUBLE PRECISION - The value above which DEBYE4 = 0.0 to machine
C                     precision. The recommended value is
C                          FOURTH ROOT(99/XMIN)
C
C      For values of EPS, EPSNEG, and XMIN see the file MACHCON.TXT
C
C
C      The machine-dependent constants are computed internally by
C      using the D1MACH subroutine.
C
C
C   INTRINSIC FUNCTIONS USED:
C
C      AINT , EXP , INT , LOG , SQRT
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
C          CHEVAL , ERRPRN, D1MACH
C
C
C   AUTHOR:
C          Dr. Allan J. MacLeod,
C          Dept. of Mathematics and Statistics,
C          University of Paisley
C          High St.
C          PAISLEY
C          SCOTLAND
C          PA1 2BE
C
C          (e-mail:  macl_ms0@paisley.ac.uk )
C
C
C   LATEST UPDATE:  23 January, 1996
C
      INTEGER I,NEXP,NTERMS
      DOUBLE PRECISION ADEB4(0:18),CHEVAL,DEBINF,EIGHT,EIGHTN,EXPMX,
     1     FIVE,FOUR,FORTY5,HALF,ONE,ONEHUN,RK,SUM,T,TWELVE,TWENT4,
     2     TWOPT5,X,XK,XKI,XLIM1,XLIM2,XLOW,XUPPER,XVALUE,ZERO,D1MACH
C
C   INTRINSIC FUNCTIONS USED:
C
       intrinsic AINT , EXP , INT , LOG , SQRT, abs
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
       external CHEVAL, D1MACH
C
c*****CHARACTER FNNAME*6,ERRMSG*17
c*****DATA FNNAME/'DEBYE4'/
c*****DATA ERRMSG/'ARGUMENT NEGATIVE'/
      DATA ZERO,HALF,ONE/0.0 D 0 , 0.5 D 0 , 1.0 D 0/
      DATA TWOPT5,FOUR,FIVE/2.5 D 0 , 4.0 D 0 , 5.0 D 0/
      DATA EIGHT,TWELVE,EIGHTN/8.0 D 0 , 12.0 D 0 , 18.0 D 0/
      DATA TWENT4,FORTY5,ONEHUN/24.0 D 0 , 45.0 D 0 , 100.0 D 0/
      DATA DEBINF/99.54506 44937 63512 92781 D 0/
      DATA ADEB4/2.78186 94150 20523 46008  D    0,
     1           0.37497 67835 26892 86364  D    0,
     2          -0.14940 90739 90315 8326   D   -1,
     3           0.94567 98114 37042 74     D   -3,
     4          -0.66132 91613 89325 5      D   -4,
     5           0.48156 32982 14449        D   -5,
     6          -0.35880 83958 7593         D   -6,
     7           0.27160 11874 160          D   -7,
     8          -0.20807 09912 23           D   -8,
     9           0.16093 83869 2            D   -9,
     X          -0.12547 09791              D  -10,
     1           0.98472 647                D  -12,
     2          -0.77723 69                 D  -13,
     3           0.61648 3                  D  -14,
     4          -0.49107                    D  -15,
     5           0.3927                     D  -16,
     6          -0.315                      D  -17,
     7           0.25                       D  -18,
     8          -0.2                        D  -19/
C
C   Start computation
C
      X = XVALUE
C
C   Check XVALUE >= 0.0
C
      IF ( X .LT. ZERO ) THEN
c********CALL ERRPRN(FNNAME,ERRMSG)
         DEBYE4 = ZERO
         RETURN
      ENDIF
C
C   Compute the machine-dependent constants.
C
      T = D1MACH(1)
      XLIM1 = - LOG( T )
      RK = ONE / FOUR
      XK = DEBINF ** RK
      XKI = T ** RK
      XLIM2 = XK / XKI
      T = D1MACH(3)
      XLOW = SQRT ( T * EIGHT )
      XUPPER = - LOG( T + T )
      T = T / ONEHUN
      DO 10 NTERMS = 18 , 0 , -1
         IF ( ABS(ADEB4(NTERMS)) .GT. T ) GOTO 19
 10   CONTINUE
C
C   Code for x <= 4.0
C
 19   IF ( X .LE. FOUR ) THEN
         IF ( X .LT. XLOW ) THEN
            DEBYE4 = ( ( TWOPT5 * X - EIGHTN ) * X + FORTY5 ) / FORTY5
         ELSE
            T = ( ( X * X / EIGHT ) - HALF ) - HALF
            DEBYE4 = CHEVAL ( NTERMS , ADEB4 , T ) - ( X + X ) / FIVE
         ENDIF
      ELSE
C
C   Code for x > 4.0
C
         IF ( X .GT. XLIM2 ) THEN
            DEBYE4 = ZERO
         ELSE
            T = X * X
            DEBYE4 = ( DEBINF / T ) / T
            IF ( X .LT. XLIM1 ) THEN
               EXPMX = EXP ( -X )
               IF ( X .GT. XUPPER ) THEN
                  SUM = ( ( ( ( X + FOUR ) * X + TWELVE ) * X +
     &                  TWENT4 ) * X + TWENT4 ) / ( X * X * X * X )
               ELSE
                  SUM = ZERO
                  RK = AINT ( XLIM1 / X )
                  NEXP = INT ( RK )
                  XK = RK * X
                  DO 100 I = NEXP,1,-1
                     XKI = ONE / XK
                     T =  ( ( ( ( TWENT4 * XKI + TWENT4 ) * XKI +
     &                    TWELVE ) * XKI + FOUR ) * XKI + ONE ) / RK
                     SUM = SUM * EXPMX + T
                     RK = RK - ONE
                     XK = XK - X
 100              CONTINUE
               ENDIF
               DEBYE4 = DEBYE4 - FOUR * SUM * EXPMX
            ENDIF
         ENDIF
      ENDIF
      RETURN
      END

