C
C Added LOGGAM = .FALSE. after each CALL GAMMAC 20/12/2001
C FERMID
C FERINC
C FDNINT
C FDNEG
C FDPOS
C FDETA
C FDASYM
C FERRAR
C GAMMAC
C WHIZ
C
C      ALGORITHM 745, COLLECTED ALGORITHMS FROM ACM.
C      THIS WORK PUBLISHED IN TRANSACTIONS ON MATHEMATICAL SOFTWARE,
C      VOL. 21, NO. 3, September, 1995, P.  221-232.
C
* **********************************************************************
*
        SUBROUTINE FERMID(ORD, X, RELERR, FD, IERR)
*
* **********************************************************************
* FERMID returns in FD the value of the Fermi-Dirac integral of real
*        order ORD and real argument X, approximated with a relative
*        error RELERR.  FERMID is a driver routine that selects FDNINT
*        for integer ORD .LE. 0, FDNEG for X .LE. 0, and FDETA, FDPOS or
*        FDASYM for X .GT. 0.  A nonzero value is assigned to the error
*        flag IERR when an error condition occurs:
*           IERR = 1:  on input, the requested relative error RELERR is
*                      smaller than the machine precision;
*           IERR = 3:  an integral of large negative integer order could
*                      not be evaluated:  increase the parameter NMAX
*                      in subroutine FDNINT.
*           IERR = 4:  an integral (probably of small argument and large
*                      negative order) could not be evaluated with the
*                      requested accuracy after the inclusion of ITMAX
*                      terms of the series expansion:  increase the
*                      parameter ITMAX in the routine which produced the
*                      error message and in its subroutines.
*        When an error occurs, a message is also printed on the standard
*        output unit by the subroutine FERERR, and the execution of the
*        program is not interrupted; to change/suppress the output unit
*        or to stop the program when an error occurs, only FERERR should
*        be modified.
*
* References:
*
*   [1] M. Goano, "Series expansion of the Fermi-Dirac integral F_j(x)
*       over the entire domain of real j and x", Solid-State
*       Electronics, vol. 36, no. 2, pp. 217-221, 1993.
*
*   [2] J. S. Blakemore, "Approximation for Fermi-Dirac integrals,
*       especially the function F_1/2(eta) used to describe electron
*       density in a semiconductor", Solid-State Electronics, vol. 25,
*       no. 11, pp. 1067-1076, 1982.
*
* If a single precision version is desired, change all occurrences of
* *SP in columns 1 to 3 to blanks and comment the corresponding double
* precision statements.
*
* Michele Goano, Politecnico di Torino  (goano@polito.it).
* Latest revision:  March 23, 1994.
* **********************************************************************
*   Parameters
*SP     REAL             ONE, TEN, THREE, TWO, ZERO
        DOUBLE PRECISION ONE, TEN, THREE, TWO, ZERO
*SP     PARAMETER (ONE = 1.0E+0, TEN = 10.0E+0, THREE = 3.0E+0,
*SP  &             TWO = 2.0E+0, ZERO = 0.0E+0)
        PARAMETER (ONE = 1.0D+0, TEN = 10.0D+0, THREE = 3.0D+0,
     &             TWO = 2.0D+0, ZERO = 0.0D+0)
*   Scalar arguments
        INTEGER IERR
*SP     REAL             ORD, X, RELERR, FD
        DOUBLE PRECISION ORD, X, RELERR, FD
*   Local scalars
        LOGICAL INTORD, TRYASY
        INTEGER NORD
*SP     REAL             RKDIV, XASYMP
        DOUBLE PRECISION RKDIV, XASYMP
*   External subroutines
        EXTERNAL FDASYM, FDETA, FDNEG, FDNINT, FDPOS, FERERR
*   Intrinsic functions
*SP     INTRINSIC ABS, ANINT, EXP, LOG, LOG10, MAX, NINT, SQRT
        INTRINSIC ABS, ANINT, LOG10, MAX, NINT, SQRT
* ----------------------------------------------------------------------
*   Parameters of the floating-point arithmetic system.  Only the values
*   for very common machines are provided:  the subroutine MACHAR [1]
*   can be used to determine the machine constants of any other system.
*
*   [1] W. J. Cody,"Algorithm 665. MACHAR: A subroutine to dynamically
*       determine machine parameters", ACM Transactions on Mathematical
*       Software, vol. 14, no. 4, pp. 303-311, 1988.
*
        INTEGER MACHEP, MINEXP, MAXEXP, NEGEXP
*   ANSI/IEEE standard 745-1985:  IBM RISC 6000, DEC Alpha (S_floating
*   and T_floating), Apple Macintosh, SunSparc, most IBM PC compilers...
*SP     PARAMETER (MACHEP = -23, MINEXP = -126, MAXEXP = 128,
*SP  &             NEGEXP = -24)
        PARAMETER (MACHEP = -52, MINEXP = -1022, MAXEXP = 1024,
     &             NEGEXP = -53)
*   DEC VAX (F_floating and D_floating)
*SP     PARAMETER (MACHEP = -24, MINEXP = -128, MAXEXP = 127,
*SP  &             NEGEXP = -24)
*DP     PARAMETER (MACHEP = -56, MINEXP = -128, MAXEXP = 127,
*DP  &             NEGEXP = -56)
*   CRAY
*SP     PARAMETER (MACHEP = -47, MINEXP = -8193, MAXEXP = 8191,
*SP  &             NEGEXP = -47)
*DP     PARAMETER (MACHEP = -95, MINEXP = -8193, MAXEXP = 8191,
*DP  &             NEGEXP = -95)
*
*SP     REAL             BETA, EPS, XBIG, XMIN, XMAX
        DOUBLE PRECISION BETA, EPS, XMIN, XMAX
        PARAMETER (BETA = TWO, EPS = BETA**MACHEP, XMIN = BETA**MINEXP,
     &             XMAX = (BETA**(MAXEXP-1) -
     &                                    BETA**(MAXEXP+NEGEXP-1))*BETA)
C*******XBIG = LOG(XMAX)
* ----------------------------------------------------------------------
        IERR = 0
        FD = ZERO
        INTORD = ABS(ORD - ANINT(ORD)).LE.ABS(ORD)*EPS
        IF (RELERR.LT.EPS) THEN
*   Test on the accuracy requested by the user
          IERR = 1
          CALL FERERR(' FERMID:  Input error: ' //
     &                ' RELERR is smaller than the machine precision')
        ELSE IF (INTORD .AND. ORD.LE.ZERO) THEN
*   Analytic expression for integer ORD .le. 0
          NORD = NINT(ORD)
          CALL FDNINT(NORD, X, FD, IERR)
        ELSE IF (X.LE.ZERO) THEN
*   Series expansion for negative argument
          CALL FDNEG(ORD, X, XMIN, RELERR, FD, IERR)
        ELSE
*   Positive argument:  approximations for k_div - 1 and x_min (RKDIV
*   and XASYMP)
          RKDIV = -LOG10(RELERR)
          XASYMP = MAX(ORD - ONE,
     &                 TWO*RKDIV - ORD*(TWO+RKDIV/TEN),
     &                 SQRT(ABS((2*RKDIV-ONE-ORD)*(2*RKDIV-ORD))))
          IF (X.GT.XASYMP .OR. INTORD) THEN
*   Asymptotic expansion, used also for positive integer order
            TRYASY = .TRUE.
            CALL FDASYM(ORD, X, EPS, XMAX, XMIN, RELERR, FD, IERR)
          ELSE
            TRYASY = .FALSE.
          END IF
          IF (.NOT.TRYASY .OR. IERR.NE.0) THEN
            IF (ORD.GT.-TWO .AND. X.LT.TWO/THREE) THEN
*   Taylor series expansion, involving eta function
              CALL FDETA(ORD, X, EPS, XMAX, RELERR, FD, IERR)
            ELSE
*   Series expansion for positive argument, involving confluent
*   hypergeometric functions
              CALL FDPOS(ORD, X, EPS, XMAX, RELERR, FD, IERR)
            END IF
          END IF
        END IF
        RETURN
        END

* **********************************************************************
*
        SUBROUTINE FERINC(ORD, X, B, RELERR, FDI, IERR)
*
* **********************************************************************
* FERINC returns in FDI the value of the incomplete Fermi-Dirac integral
*        of real order ORD and real arguments X and B, approximated with
*        a relative error RELERR.  Levin's u transform [2] is used to
*        sum the alternating series (21) of [1].  A nonzero value is
*        assigned to the error flag IERR when an error condition occurs:
*           IERR = 1:  on input, the requested relative error RELERR is
*                      smaller than the machine precision;
*           IERR = 2:  on input, the lower bound B of the incomplete
*                      integral is lower than zero;
*           IERR = 3:  a complete integral of very large negative
*                      integer order could not be evaluated:  increase
*                      the parameter NMAX in subroutine FDNINT.
*           IERR = 4:  an integral (probably of small argument and large
*                      negative order) could not be evaluated with the
*                      requested accuracy after the inclusion of ITMAX
*                      terms of the series expansion:  increase the
*                      parameter ITMAX in the routine which produced the
*                      error message and in its subroutines.
*        When an error occurs, a message is also printed on the standard
*        output unit by the subroutine FERERR, and the execution of the
*        program is not interrupted; to change/suppress the output unit
*        and/or to stop the program when an error occurs, only FERERR
*        should be modified.
*
* References:
*
*   [1] M. Goano, "Series expansion of the Fermi-Dirac integral F_j(x)
*       over the entire domain of real j and x", Solid-State
*       Electronics, vol. 36, no. 2, pp. 217-221, 1993.
*
*   [2] T. Fessler, W. F. Ford, D. A. Smith, "ALGORITHM 602. HURRY: An
*       acceleration algorithm for scalar sequences and series", ACM
*       Transactions on Mathematical Software, vol. 9, no. 3,
*       pp. 355-357, September  1983.
*
* If a single precision version is desired, change all occurrences of
* *SP in columns 1 to 3 to blanks and comment the corresponding double
* precision statements.
*
* Michele Goano, Politecnico di Torino  (goano@polito.it).
* Latest revision:  March 22, 1994.
* **********************************************************************
*   Parameters
        INTEGER ITMAX
        PARAMETER (ITMAX = 100)
*SP     REAL             ONE, TWO, ZERO
        DOUBLE PRECISION ONE, TWO, ZERO
*SP     PARAMETER (ONE = 1.0E+0, TWO = 2.0E+0, ZERO = 0.0E+0)
        PARAMETER (ONE = 1.0D+0, TWO = 2.0D+0, ZERO = 0.0D+0)
*   Scalar arguments
        INTEGER IERR
*SP     REAL             ORD, X, B, RELERR, FDI
        DOUBLE PRECISION ORD, X, B, RELERR, FDI
*   Local scalars
        LOGICAL LOGGAM
        INTEGER JTERM
*SP     REAL             BMX, BMXN, BN, EBMX, ENBMX, ENXMB,
*SP  &                   EXMB, FD, FDOLD, GAMMA, M, S, TERM, U,
*SP  &                   XMB, XMBN
        DOUBLE PRECISION BMX, BMXN, BN, EBMX, ENBMX, ENXMB,
     &                   EXMB, FD, FDOLD, GAMMA, M, S, TERM, U,
     &                   XMB, XMBN
*   Local arrays
*SP     REAL             QNUM(ITMAX), QDEN(ITMAX)
        DOUBLE PRECISION QNUM(ITMAX), QDEN(ITMAX)
*   External subroutines
        EXTERNAL FERERR, FERMID, GAMMAC, M1KUMM, U1KUMM, WHIZ
*   Intrinsic functions
        INTRINSIC ABS, ANINT, EXP, LOG
* ----------------------------------------------------------------------
*   Parameters of the floating-point arithmetic system.  Only the values
*   for very common machines are provided:  the subroutine MACHAR [1]
*   can be used to determine the machine constants of any other system.
*
*   [1] W. J. Cody,"Algorithm 665. MACHAR: A subroutine to dynamically
*       determine machine parameters", ACM Transactions on Mathematical
*       Software, vol. 14, no. 4, pp. 303-311, 1988.
*
        INTEGER MACHEP, MINEXP, MAXEXP, NEGEXP
*   ANSI/IEEE standard 745-1985:  IBM RISC 6000, DEC Alpha (S_floating
*   and T_floating), Apple Macintosh, SunSparc, most IBM PC compilers...
*SP     PARAMETER (MACHEP = -23, MINEXP = -126, MAXEXP = 128,
*SP  &             NEGEXP = -24)
        PARAMETER (MACHEP = -52, MINEXP = -1022, MAXEXP = 1024,
     &             NEGEXP = -53)
*   DEC VAX (F_floating and D_floating)
*SP     PARAMETER (MACHEP = -24, MINEXP = -128, MAXEXP = 127,
*SP  &             NEGEXP = -24)
*DP     PARAMETER (MACHEP = -56, MINEXP = -128, MAXEXP = 127,
*DP  &             NEGEXP = -56)
*   CRAY
*SP     PARAMETER (MACHEP = -47, MINEXP = -8193, MAXEXP = 8191,
*SP  &             NEGEXP = -47)
*DP     PARAMETER (MACHEP = -95, MINEXP = -8193, MAXEXP = 8191,
*DP  &             NEGEXP = -95)
*
*SP     REAL             EPS, XMIN, XMAX, XTINY
        DOUBLE PRECISION EPS, XMIN, XMAX, XTINY
        PARAMETER (EPS = TWO**MACHEP, XMIN = TWO**MINEXP, XMAX =
     &                   (TWO**(MAXEXP-1) - TWO**(MAXEXP+NEGEXP-1))*TWO)
        intrinsic nint
        XTINY = LOG(XMIN)
* ----------------------------------------------------------------------
        IERR = 0
        FDI = ZERO
        IF (RELERR.LT.EPS) THEN
*   Test on the accuracy requested by the user
          IERR = 1
          CALL FERERR(
     &  ' FERINC:  Input error:  RELERR smaller than machine precision')
        ELSE IF (B.LT.ZERO) THEN
*   Error in the argument B
          IERR = 2
          CALL FERERR(' FERINC:  Input error:  B is lower than zero')
        ELSE IF (B.EQ.ZERO) THEN
*   Complete integral
          CALL FERMID(ORD, X, RELERR, FDI, IERR)
        ELSE IF (ORD.LE.ZERO .AND.
     &                         ABS(ORD-ANINT(ORD)).LE.ABS(ORD)*EPS) THEN
*   Analytic expression for integer ORD .le. 0
          IF (NINT(ORD).EQ.0) THEN
            XMB = X - B
            IF (XMB.GE.ZERO) THEN
              FDI = XMB + LOG(ONE + EXP(-XMB))
            ELSE
              FDI = LOG(ONE + EXP(XMB))
            END IF
          ELSE
            FDI = ZERO
          END IF
        ELSE IF (B.LT.X) THEN
*   Series involving Kummer's function M
          CALL FERMID(ORD, X, RELERR, FD, IERR)
          CALL GAMMAC(ORD + TWO, EPS, XMAX, GAMMA, IERR)
          LOGGAM = .FALSE.!added by w.g.bardsley, 20/12/2001
          IF (IERR.EQ.-1) THEN
            LOGGAM = .TRUE.
            IERR = 0
          END IF
          BMX = B - X
          BMXN = BMX
          EBMX = -EXP(BMX)
          ENBMX = -EBMX
          BN = B
          FDI = XMAX
          DO 10 JTERM = 1, ITMAX
            FDOLD = FDI
            CALL M1KUMM(ORD, BN, EPS, XMAX, RELERR, M)
            TERM = ENBMX*M
            CALL WHIZ(TERM, JTERM, QNUM, QDEN, FDI, S)
*   Check truncation error and convergence
            BMXN = BMXN + BMX
            IF (ABS(FDI-FDOLD).LE.ABS(ONE-FDI)*RELERR .OR.
     &                                           BMXN.LT.XTINY) GO TO 20
            ENBMX = ENBMX*EBMX
            BN = BN + B
   10     CONTINUE
          IERR = 4
          CALL FERERR(
     &       ' FERINC:  RELERR not achieved:  increase parameter ITMAX')
   20     CONTINUE
          IF (LOGGAM) THEN
            FDI = FD - EXP((ORD+ONE)*LOG(B) - GAMMA)*(ONE - FDI)
          ELSE
            FDI = FD - B**(ORD + ONE)/GAMMA*(ONE - FDI)
          END IF
        ELSE
*   Series involving Kummer's function U
          CALL GAMMAC(ORD + ONE, EPS, XMAX, GAMMA, IERR)
          LOGGAM = .FALSE.!added by w.g.bardsley, 20/12/2001
          IF (IERR.EQ.-1) THEN
            LOGGAM = .TRUE.
            IERR = 0
          END IF
          XMB = X - B
          XMBN = XMB
          EXMB = -EXP(XMB)
          ENXMB = -EXMB
          BN = B
          FDI = XMAX
          DO 30 JTERM = 1, ITMAX
            FDOLD = FDI
            CALL U1KUMM(ORD + ONE, BN, EPS, XMAX, RELERR, U)
            TERM = ENXMB*U
            CALL WHIZ(TERM, JTERM, QNUM, QDEN, FDI, S)
*   Check truncation error and convergence
            XMBN = XMBN + XMB
            IF (ABS(FDI-FDOLD).LE.ABS(FDI)*RELERR .OR.
     &                                           XMBN.LT.XTINY) GO TO 40
            ENXMB = ENXMB*EXMB
            BN = BN + B
   30     CONTINUE
          IERR = 4
          CALL FERERR(
     &       ' FERINC:  RELERR not achieved:  increase parameter ITMAX')
   40     CONTINUE
          IF (LOGGAM) THEN
            FDI = EXP((ORD+ONE)*LOG(B) - GAMMA)*FDI
          ELSE
            FDI = B**(ORD + ONE)/GAMMA*FDI
          END IF
        END IF
        RETURN
        END

* **********************************************************************
*
        SUBROUTINE FDNINT(NORD, X, FD, IERR)
*
* **********************************************************************
* FDNINT returns in FD the value of the Fermi-Dirac integral of integer
*        order NORD (-NMAX-1 .LE. NORD .LE. 0) and argument X, for
*        which an analytical expression is available.  A nonzero value
*        is assigned to the error flag IERR when ABS(NORD).GT.NMAX+1:
*        to remedy, increase the parameter NMAX.
*
* If a single precision version is desired, change all occurrences of
* *SP in columns 1 to 3 to blanks and comment the corresponding double
* precision statements.
*
* Michele Goano, Politecnico di Torino  (goano@polito.it).
* Latest revision:  March 22, 1994.
* **********************************************************************
*   Parameters
        INTEGER NMAX
        PARAMETER (NMAX = 100)
*SP     REAL             ONE
        DOUBLE PRECISION ONE, ZERO
*SP     PARAMETER (ONE = 1.0E+0, ZERO = 0.0E+0)
        PARAMETER (ONE = 1.0D+0, ZERO = 0.0D+0)
*   Scalar arguments
        INTEGER NORD, IERR
*SP     REAL             X, FD
        DOUBLE PRECISION X, FD
*   Local scalars
        INTEGER I, K, N
*SP     REAL             A
        DOUBLE PRECISION A
*   Local arrays
*SP     REAL             QCOEF(NMAX)
        DOUBLE PRECISION QCOEF(NMAX)
*   External subroutines
        EXTERNAL FERERR
*   Intrinsic functions
*SP     INTRINSIC EXP, LOG, REAL
        INTRINSIC DBLE, EXP, LOG
* ----------------------------------------------------------------------
        IERR = 0
        FD = ZERO
*   Test on the order, whose absolute value must be lower or equal than
*   NMAX+1
        IF (NORD.LT.-NMAX - 1) THEN
          IERR = 3
          CALL FERERR(
     &            ' FDNINT:  order too large:  increase parameter NMAX')
        ELSE IF (NORD.EQ.0) THEN
*   Analytic expression for NORD .eq. 0
          IF (X.GE.ZERO) THEN
            FD = X + LOG(ONE + EXP(-X))
          ELSE
            FD = LOG(ONE + EXP(X))
          END IF
        ELSE IF (NORD.EQ.-1) THEN
*   Analytic expression for NORD .eq. -1
          IF (X.GE.ZERO) THEN
            FD = ONE/(ONE + EXP(-X))
          ELSE
            A = EXP(X)
            FD = A/(ONE + A)
          END IF
        ELSE
*   Evaluation of the coefficients of the polynomial P(a), having degree
*   (-NORD - 2), appearing at the numerator of the analytic expression
*   for NORD .le. -2
          N = -NORD - 1
          QCOEF(1) = ONE
          DO 20 K = 2, N
            QCOEF(K) = -QCOEF(K - 1)
            DO 10 I = K - 1, 2, -1
*SP           QCOEF(I) = REAL(I)*QCOEF(I) - REAL(K - (I-1))*QCOEF(I - 1)
              QCOEF(I) = DBLE(I)*QCOEF(I) - DBLE(K - (I-1))*QCOEF(I - 1)
   10       CONTINUE
   20     CONTINUE
*   Computation of P(a)
          IF (X.GE.ZERO) THEN
            A = EXP(-X)
            FD = QCOEF(1)
            DO 30 I = 2, N
              FD = FD*A + QCOEF(I)
   30       CONTINUE
          ELSE
            A = EXP(X)
            FD = QCOEF(N)
            DO 40 I = N - 1, 1, -1
              FD = FD*A + QCOEF(I)
   40       CONTINUE
          END IF
*   Evaluation of the Fermi-Dirac integral
          FD = FD*A*(ONE + A)**NORD
        END IF
        RETURN
        END


* **********************************************************************
*
        SUBROUTINE FDNEG(ORD, X, XMIN, RELERR, FD, IERR)
*
* **********************************************************************
* FDNEG returns in FD the value of the Fermi-Dirac integral of real
*       order ORD and negative argument X, approximated with a relative
*       error RELERR.  XMIN represent the smallest non-vanishing
*       floating-point number.  Levin's u transform [2] is used to sum
*       the alternating series (13) of [1].
*
* References:
*
*   [1] J. S. Blakemore, "Approximation for Fermi-Dirac integrals,
*       especially the function F_1/2(eta) used to describe electron
*       density in a semiconductor", Solid-State Electronics, vol. 25,
*       no. 11, pp. 1067-1076, 1982.
*
*   [2] T. Fessler, W. F. Ford, D. A. Smith, "ALGORITHM 602. HURRY: An
*       acceleration algorithm for scalar sequences and series", ACM
*       Transactions on Mathematical Software, vol. 9, no. 3,
*       pp. 355-357, September  1983.
*
* If a single precision version is desired, change all occurrences of
* *SP in columns 1 to 3 to blanks and comment the corresponding double
* precision statements.
*
* Michele Goano, Politecnico di Torino  (goano@polito.it).
* Latest revision:  February 5, 1996.
* **********************************************************************
*   Parameters
        INTEGER ITMAX
        PARAMETER (ITMAX = 100)
*SP     REAL             ONE, ZERO
        DOUBLE PRECISION ONE, ZERO
*SP     PARAMETER (ONE = 1.0E+0, ZERO = 0.0E+0)
        PARAMETER (ONE = 1.0D+0, ZERO = 0.0D+0)
*   Scalar arguments
        INTEGER IERR
*SP     REAL             ORD, X, XMIN, RELERR, FD
        DOUBLE PRECISION ORD, X, XMIN, RELERR, FD
*   Local scalars
        INTEGER JTERM
*SP     REAL             EX, ENX, FDOLD, S, TERM, XN, XTINY
        DOUBLE PRECISION EX, ENX, FDOLD, S, TERM, XN, XTINY
*   Local arrays
*SP     REAL             QNUM(ITMAX), QDEN(ITMAX)
        DOUBLE PRECISION QNUM(ITMAX), QDEN(ITMAX)
*   External subroutines
        EXTERNAL FERERR, WHIZ
*   Intrinsic functions
*SP     INTRINSIC ABS, EXP, LOG, REAL
        INTRINSIC ABS, DBLE, EXP, LOG
* ----------------------------------------------------------------------
        IERR = 0
        FD = ZERO
        XTINY = LOG(XMIN)
*
        IF (X.GT.XTINY) THEN
          XN = X
          EX = -EXP(X)
          ENX = -EX
          DO 10 JTERM = 1, ITMAX
            FDOLD = FD
*SP         TERM = ENX/REAL(JTERM)**(ORD + ONE)
            TERM = ENX/DBLE(JTERM)**(ORD + ONE)
            CALL WHIZ(TERM, JTERM, QNUM, QDEN, FD, S)
*   Check truncation error and convergence
            XN = XN + X
            IF (ABS(FD-FDOLD).LE.ABS(FD)*RELERR .OR. XN.LT.XTINY) RETURN
            ENX = ENX*EX
   10     CONTINUE
          IERR = 4
          CALL FERERR(
     &        ' FDNEG:  RELERR not achieved:  increase parameter ITMAX')
        END IF
        RETURN
        END
* **********************************************************************
*
        SUBROUTINE FDPOS(ORD, X, EPS, XMAX, RELERR, FD, IERR)
*
* **********************************************************************
* FDPOS returns in FD the value of the Fermi-Dirac integral of real
*       order ORD and argument X .GT. 0, approximated with a relative
*       error RELERR.  EPS and XMAX represent the smallest positive
*       floating-point number such that 1.0+EPS .NE. 1.0, and the
*       largest finite floating-point number, respectively.  Levin's u
*       transform [2] is used to sum the alternating series (11) of [1].
*
* References:
*
*   [1] M. Goano, "Series expansion of the Fermi-Dirac integral F_j(x)
*       over the entire domain of real j and x", Solid-State
*       Electronics, vol. 36, no. 2, pp. 217-221, 1993.
*
*   [2] T. Fessler, W. F. Ford, D. A. Smith, "ALGORITHM 602. HURRY: An
*       acceleration algorithm for scalar sequences and series", ACM
*       Transactions on Mathematical Software, vol. 9, no. 3,
*       pp. 355-357, September  1983.
*
* If a single precision version is desired, change all occurrences of
* *SP in columns 1 to 3 to blanks and comment the corresponding double
* precision statements.
*
* Michele Goano, Politecnico di Torino  (goano@polito.it).
* Latest revision:  March 22, 1994.
* **********************************************************************
*   Parameters
        INTEGER ITMAX
        PARAMETER (ITMAX = 100)
*SP     REAL             ONE, TWO, ZERO
        DOUBLE PRECISION ONE, TWO, ZERO
*SP     PARAMETER (ONE = 1.0E+0, TWO = 2.0E+0, ZERO = 0.0E+0)
        PARAMETER (ONE = 1.0D+0, TWO = 2.0D+0, ZERO = 0.0D+0)
*   Scalar arguments
        INTEGER IERR
*SP     REAL             ORD, X, EPS, XMAX, RELERR, FD
        DOUBLE PRECISION ORD, X, EPS, XMAX, RELERR, FD
*   Local scalars
        LOGICAL LOGGAM
        INTEGER JTERM
*SP     REAL             FDOLD, GAMMA, M, S, SEGN, TERM, U, XN
        DOUBLE PRECISION FDOLD, GAMMA, M, S, SEGN, TERM, U, XN
*   Local arrays
*SP     REAL             QNUM(ITMAX), QDEN(ITMAX)
        DOUBLE PRECISION QNUM(ITMAX), QDEN(ITMAX)
*   External subroutines
        EXTERNAL FERERR, GAMMAC, M1KUMM, U1KUMM, WHIZ
*   Intrinsic functions
        INTRINSIC ABS, EXP, LOG
* ----------------------------------------------------------------------
        IERR = 0
        FD = ZERO
*
        CALL GAMMAC(ORD + TWO, EPS, XMAX, GAMMA, IERR)
        LOGGAM = .FALSE.!added by w.g.bardsley, 20/12/2001
        IF (IERR.EQ.-1) LOGGAM = .TRUE.
        SEGN = ONE
        XN = X
        FD = XMAX
        DO 10 JTERM = 1, ITMAX
          FDOLD = FD
          CALL U1KUMM(ORD + ONE, XN, EPS, XMAX, RELERR, U)
          CALL M1KUMM(ORD, XN, EPS, XMAX, RELERR, M)
          TERM = SEGN*((ORD+ONE)*U - M)
          CALL WHIZ(TERM, JTERM, QNUM, QDEN, FD, S)
*   Check truncation error and convergence
          IF (ABS(FD-FDOLD).LE.ABS(FD+ONE)*RELERR) GO TO 20
          SEGN = -SEGN
          XN = XN + X
   10   CONTINUE
        IERR = 4
        CALL FERERR(
     &        ' FDPOS:  RELERR not achieved:  increase parameter ITMAX')
   20   CONTINUE
        IF (LOGGAM) THEN
          FD = EXP((ORD+ONE)*LOG(X) - GAMMA)*(ONE + FD)
        ELSE
          FD = X**(ORD + ONE)/GAMMA*(ONE + FD)
        END IF
        RETURN
        END

* **********************************************************************
*
        SUBROUTINE FDETA(ORD, X, EPS, XMAX, RELERR, FD, IERR)
*
* **********************************************************************
* FDETA returns in FD the value of the Fermi-Dirac integral of real
*       order ORD and argument X such that ABS(X) .LE. PI, approximated
*       with a relative error RELERR.  EPS and XMAX represent the
*       smallest positive floating-point number such that
*       1.0+EPS .NE. 1.0, and the largest finite floating-point number,
*       respectively.  Taylor series expansion (4) of [1] is used,
*       involving eta function defined in (23.2.19) of [2].
*
*
* References:
*
*   [1] W. J. Cody and H. C. Thacher, Jr., "Rational Chebyshev
*       approximations for Fermi-Dirac integrals of orders -1/2, 1/2 and
*       3/2", Mathematics of Computation, vol. 21, no. 97, pp. 30-40,
*       1967.
*
*   [2] E. V. Haynsworth and K. Goldberg, "Bernoulli and Euler
*       Polynomials - Riemann Zeta Function", in "Handbook of
*       Mathematical Functions with Formulas, Graphs and Mathematical
*       Tables" (M. Abramowitz and I. A. Stegun, eds.), no. 55 in
*       National Bureau of Standards Applied Mathematics Series, ch. 23,
*       pp. 803-819, Washington, D.C.:  U.S. Government Printing Office,
*       1964.
*
* If a single precision version is desired, change all occurrences of
* *SP in columns 1 to 3 to blanks and comment the corresponding double
* precision statements.
*
* Michele Goano, Politecnico di Torino  (goano@polito.it).
* Latest revision:  March 22, 1994.
* **********************************************************************
*   Parameters
        INTEGER ITMAX
        PARAMETER (ITMAX = 100)
*SP     REAL             ONE, PI, TWO, ZERO
        DOUBLE PRECISION ONE,     TWO, ZERO
*SP     PARAMETER (ONE = 1.0E+0, PI = 3.141592653589793238462643E+0,
*SP  &             TWO = 2.0E+0, ZERO = 0.0E+0)
        PARAMETER (ONE = 1.0D+0,
     &             TWO = 2.0D+0, ZERO = 0.0D+0)
*   Scalar arguments
        INTEGER IERR
*SP     REAL             ORD, X, EPS, XMAX, RELERR, FD
        DOUBLE PRECISION ORD, X, EPS, XMAX, RELERR, FD
*   Local scalars
        LOGICAL OKJM1, OKJM2
        INTEGER JTERM
*SP     REAL             ETA, RJTERM, TERM, XNOFAC
        DOUBLE PRECISION ETA, RJTERM, TERM, XNOFAC
*   External subroutines
        EXTERNAL ETARIE, FERERR
*   Intrinsic functions
*SP     INTRINSIC ABS, REAL
        INTRINSIC ABS, DBLE
* ----------------------------------------------------------------------
        IERR = 0
        FD = ZERO
*
        OKJM1 = .FALSE.
        OKJM2 = .FALSE.
        XNOFAC = ONE
        DO 10 JTERM = 1, ITMAX
*SP       RJTERM = REAL(JTERM)
          RJTERM = DBLE(JTERM)
          CALL ETARIE(ORD + TWO - RJTERM, EPS, XMAX, RELERR, ETA)
          TERM = ETA*XNOFAC
          FD = FD + TERM
*   Check truncation error and convergence.  The summation is terminated
*   when three consecutive terms of the series satisfy the bound on the
*   relative error
          IF (ABS(TERM).GT.ABS(FD)*RELERR) THEN
            OKJM1 = .FALSE.
            OKJM2 = .FALSE.
          ELSE IF (.NOT.OKJM1) THEN
            OKJM1 = .TRUE.
          ELSE IF (OKJM2) THEN
            RETURN
          ELSE
            OKJM2 = .TRUE.
          END IF
          XNOFAC = XNOFAC*X/RJTERM
   10   CONTINUE
        IERR = 4
        CALL FERERR(
     &        ' FDETA:  RELERR not achieved:  increase parameter ITMAX')
        RETURN
        END

* **********************************************************************
*
        SUBROUTINE FDASYM(ORD, X, EPS, XMAX, XMIN, RELERR, FD, IERR)
*
* **********************************************************************
* FDASYM returns in FD the value of the Fermi-Dirac integral of real
*        order ORD and argument X .GT. 0, approximated with a relative
*        error RELERR by means of an asymptotic expansion.  EPS, XMAX
*        and XMIN represent the smallest positive floating-point number
*        such that 1.0+EPS .NE. 1.0, the largest finite floating-point
*        number, and the smallest non-vanishing floating-point number,
*        respectively.  A nonzero value is assigned to the error flag
*        IERR when the series does not converge.  The expansion always
*        terminates after a finite number of steps in case of integer
*        ORD.
*
* References:
*
*   [1] P. Rhodes, "Fermi-Dirac function of integral order", Proceedings
*       of the Royal Society of London. Series A - Mathematical and
*       Physical Sciences, vol. 204, pp. 396-405, 1950.
*
*   [2] R. B. Dingle, "Asymptotic Expansions: Their Derivation and
*       Interpretation", London and New York:  Academic Press, 1973.
*
* If a single precision version is desired, change all occurrences of
* *SP in columns 1 to 3 to blanks and comment the corresponding double
* precision statements.
*
* Michele Goano, Politecnico di Torino  (goano@polito.it).
* Latest revision:  March 22, 1994.
* **********************************************************************
*   Parameters
        INTEGER ITMAX
        PARAMETER (ITMAX = 100)
*SP     REAL             HALF, ONE, PI, TWO, ZERO
        DOUBLE PRECISION HALF, ONE, PI, TWO, ZERO
*SP     PARAMETER (HALF = 0.5E+0, ONE = 1.0E+0,
*SP  &             PI = 3.141592653589793238462643E+0, TWO = 2.0E+0,
*SP  &             ZERO = 0.0E+0)
        PARAMETER (HALF = 0.5D+0, ONE = 1.0D+0,
     &             PI = 3.141592653589793238462643D+0, TWO = 2.0D+0,
     &             ZERO = 0.0D+0)
*   Scalar arguments
        INTEGER IERR
*SP     REAL             ORD, X, EPS, XMAX, XMIN, RELERR, FD
        DOUBLE PRECISION ORD, X, EPS, XMAX, XMIN, RELERR, FD
*   Local scalars
        LOGICAL LOGGAM
        INTEGER N
*SP     REAL             ADD, ADDOLD, ETA, GAMMA, SEQN, XGAM, XM2
        DOUBLE PRECISION ADD, ADDOLD, ETA, GAMMA, SEQN, XGAM, XM2
*   External subroutines
        EXTERNAL ETAN, FDNEG, FERERR, GAMMAC
*   Intrinsic functions
*SP     INTRINSIC ABS, ANINT, COS, EXP, LOG, REAL
        INTRINSIC ABS, ANINT, COS, DBLE, EXP, LOG
* ----------------------------------------------------------------------
        IERR = 0
        FD = ZERO
*
        CALL GAMMAC(ORD + TWO, EPS, XMAX, GAMMA, IERR)
        LOGGAM = .FALSE.!added by w.g.bardsley 20/12/2001
        IF (IERR.EQ.-1) THEN
          LOGGAM = .TRUE.
          IERR = 0
        END IF
        SEQN = HALF
        XM2 = X**(-2)
        XGAM = ONE
        ADD = XMAX
        DO 10 N = 1, ITMAX
          ADDOLD = ADD
*SP       XGAM = XGAM*XM2*(ORD + ONE - REAL(2*N-2))*
*SP  &                    (ORD + ONE - REAL(2*N-1))
          XGAM = XGAM*XM2*(ORD + ONE - DBLE(2*N-2))*
     &                    (ORD + ONE - DBLE(2*N-1))
          CALL ETAN(2*N, ETA)
          ADD = ETA*XGAM
          IF (ABS(ADD).GE.ABS(ADDOLD) .AND.
     &                       ABS(ORD - ANINT(ORD)).GT.ABS(ORD)*EPS) THEN
*   Asymptotic series is diverging
            IERR = 1
            RETURN
          END IF
          SEQN = SEQN + ADD
*   Check truncation error and convergence
          IF (ABS(ADD).LE.ABS(SEQN)*RELERR) GO TO 20
   10   CONTINUE
        IERR = 4
        CALL FERERR(
     &       ' FDASYM:  RELERR not achieved:  increase parameter ITMAX')
   20   CONTINUE
        CALL FDNEG(ORD, -X, XMIN, RELERR, FD, IERR)
        IF (LOGGAM) THEN
          FD = COS(ORD*PI)*FD + TWO*SEQN*EXP((ORD + ONE)*LOG(X) - GAMMA)
        ELSE
          FD = COS(ORD*PI)*FD + X**(ORD + ONE)*TWO*SEQN/GAMMA
        END IF
        RETURN
        END

* **********************************************************************
*
        SUBROUTINE FERERR(ERRMSG)
*
* **********************************************************************
* FERERR prints on the standard output unit an explanatory message of
*        the error condition occured in the package which approximates
*        the complete and incomplete Fermi-Dirac integral.
*
* Michele Goano, Politecnico di Torino  (goano@polito.it).
* Latest revision:  March 22, 1994.
* **********************************************************************
*   Scalar arguments
        CHARACTER*(*) ERRMSG
* ----------------------------------------------------------------------
        WRITE (*, FMT = 99999) ERRMSG
*   If you want to interrupt the execution after an error has occurred,
*   replace the RETURN statement with a STOP
        RETURN
99999   FORMAT (A)
        END
*
* **********************************************************************
*
        SUBROUTINE GAMMAC(X, EPS, XINF, GAMMA, IERR)
C-----------------------------------------------------------------------
C This routine calculates the gamma function for a real argument X.  The
C logarithm of the gamma function is computed, and the error flag IERR
C is set to -1, whenever the result would be too large to be represented
C on the floating-point arithmetic system.  Computation is based on an
C algorithm outlined in W. J. Cody, 'An overview of software development
C for special functions', Lecture Notes in Mathematics, 506, Numerical
C Analysis Dundee, 1975, G. A. Watson (ed.), Springer Verlag, Berlin,
C 1976.  The program uses rational functions that approximate the gamma
C function to at least 20 significant decimal digits.  Coefficients for
C the approximation over the interval (1,2) are unpublished.  Those for
C the approximation for X .GE. 12 are from Hart et al., Computer
C Approximations, Wiley and Sons, New York, 1968.
C
C If a single precision version is desired, change all occurrences of CS
C in columns 1 and 2 to blanks and comment the corresponding double
C precision statements.
C
C Explanation of machine-dependent variables
C
C EPS    - the smallest positive floating-point number such that
C          1.0 + EPS .GT. 1.0
C XINF   - the largest machine representable floating-point number.
C XBIG   - the largest floating-point number such that EXP(XBIG) is
C          machine representable.
C
C Error returns
C
C  The program returns LOG(GAMMA) and sets IERR = -1 when overflow would
C  occur.
C
C Author: W. J. Cody
C         Argonne National Laboratory
C
C Revised by M. Goano, Politecnico di Torino, to take advantage of
C Fortran 77 control structures.
C
C Latest modification of the original version: May 18, 1982
C                     of the revised version:  March 21, 1994
C-----------------------------------------------------------------------
        INTEGER I, IERR, J, N
CS      REAL             C, EPS, FACT, GAMMA, HALF, ONE, P, PI, Q, RES,
CS   &                   SQRTPI, SUM, TWELVE, X, XBIG, XDEN, XINF,
CS   &                   XNUM, Y, Y1, YSQ, Z, ZERO
        DOUBLE PRECISION C, EPS, FACT, GAMMA, HALF, ONE, P, PI, Q, RES,
     &                   SQRTPI, SUM, TWELVE, X, XBIG, XDEN, XINF,
     &                   XNUM, Y, Y1, YSQ, Z, ZERO
        LOGICAL PARITY
        DIMENSION C(7), P(8), Q(8)
CS      INTRINSIC ALOG, EXP, FLOAT, IFIX, SIN
        INTRINSIC DBLE, DEXP, DLOG, DSIN, FLOAT, IFIX, SNGL
C-----------------------------------------------------------------------
C Mathematical constants
C-----------------------------------------------------------------------
CS      PARAMETER (ONE = 1.0E+0, HALF = 0.5E+0, TWELVE = 12.0E+0,
CS   &             ZERO = 0.0E+0, PI = 3.1415926535897932384626434E+0,
CS   &             SQRTPI = 0.9189385332046727417803297E+0)
        PARAMETER (ONE = 1.0D+0, HALF = 0.5D+0, TWELVE = 12.0D+0,
     &             ZERO = 0.0D+0, PI = 3.1415926535897932384626434D+0,
     &             SQRTPI = 0.9189385332046727417803297D+0)
C-----------------------------------------------------------------------
C SAVE declaration for the arrays of the coefficients
C-----------------------------------------------------------------------
        SAVE C, P, Q
C-----------------------------------------------------------------------
C Numerator and denominator coefficients for rational minimax
C approximation over (1,2)
C-----------------------------------------------------------------------
CS      DATA P /-1.71618513886549492533811E+0,
CS   &           2.47656508055759199108314E+1,
CS   &          -3.79804256470945635097577E+2,
CS   &           6.29331155312818442661052E+2,
CS   &           8.66966202790413211295064E+2,
CS   &          -3.14512729688483675254357E+4,
CS   &          -3.61444134186911729807069E+4,
CS   &           6.64561438202405440627855E+4/
        DATA P /-1.71618513886549492533811D+0,
     &           2.47656508055759199108314D+1,
     &          -3.79804256470945635097577D+2,
     &           6.29331155312818442661052D+2,
     &           8.66966202790413211295064D+2,
     &          -3.14512729688483675254357D+4,
     &          -3.61444134186911729807069D+4,
     &           6.64561438202405440627855D+4/
CS      DATA Q /-3.08402300119738975254353E+1,
CS   &           3.15350626979604161529144E+2,
CS   &          -1.01515636749021914166146E+3,
CS   &          -3.10777167157231109440444E+3,
CS   &           2.25381184209801510330112E+4,
CS   &           4.75584627752788110767815E+3,
CS   &          -1.34659959864969306392456E+5,
CS   &          -1.15132259675553483497211E+5/
        DATA Q /-3.08402300119738975254353D+1,
     &           3.15350626979604161529144D+2,
     &          -1.01515636749021914166146D+3,
     &          -3.10777167157231109440444D+3,
     &           2.25381184209801510330112D+4,
     &           4.75584627752788110767815D+3,
     &          -1.34659959864969306392456D+5,
     &          -1.15132259675553483497211D+5/
C-----------------------------------------------------------------------
C Coefficients for minimax approximation over (12, INF)
C-----------------------------------------------------------------------
CS      DATA C /-1.910444077728E-03,
CS   &           8.4171387781295E-04,
CS   &          -5.952379913043012E-04,
CS   &           7.93650793500350248E-04,
CS   &          -2.777777777777681622553E-03,
CS   &           8.333333333333333331554247E-02,
CS   &           5.7083835261E-03/
        DATA C /-1.910444077728D-03,
     &           8.4171387781295D-04,
     &          -5.952379913043012D-04,
     &           7.93650793500350248D-04,
     &          -2.777777777777681622553D-03,
     &           8.333333333333333331554247D-02,
     &           5.7083835261D-03/
C-----------------------------------------------------------------------
C Machine dependent local variables
C-----------------------------------------------------------------------
CS      XBIG = ALOG(XINF)
        XBIG = DLOG(XINF)
C-----------------------------------------------------------------------
        IERR = 0
        PARITY = .FALSE.
        FACT = ONE
        N = 0
        Y = X
        IF (Y.LE.ZERO) THEN
C-----------------------------------------------------------------------
C Argument is negative
C-----------------------------------------------------------------------
          Y = -X
CS        J = IFIX(Y)
          J = IFIX(SNGL(Y))
CS        RES = Y - FLOAT(J)
          RES = Y - DBLE(J)
          IF (J.NE.(J/2)*2) PARITY = .TRUE.
CS        FACT = -PI/SIN(PI*RES)
          FACT = -PI/DSIN(PI*RES)
          Y = Y + ONE
        END IF
C-----------------------------------------------------------------------
C Argument is positive
C-----------------------------------------------------------------------
        IF (Y.LT.EPS) THEN
C-----------------------------------------------------------------------
C Argument .LT. EPS
C-----------------------------------------------------------------------
          RES = ONE/Y
        ELSE IF (Y.GE.TWELVE) THEN
C-----------------------------------------------------------------------
C Evaluate for argument .GE. 12.0
C-----------------------------------------------------------------------
          YSQ = Y*Y
          SUM = C(7)
          DO 10 I = 1, 6
            SUM = SUM/YSQ + C(I)
   10     CONTINUE
CS        SUM = SUM/Y + (Y - HALF)*ALOG(Y) - Y + SQRTPI
          SUM = SUM/Y + (Y - HALF)*DLOG(Y) - Y + SQRTPI
          IF (SUM.GT.XBIG) THEN
C-----------------------------------------------------------------------
C Return the logarithm to avoid overflow
C-----------------------------------------------------------------------
            RES = SUM
            IERR = -1
          ELSE
CS          RES = EXP(SUM)
            RES = DEXP(SUM)
          END IF
        ELSE
          Y1 = Y
          IF (Y.GE.ONE) THEN
C-----------------------------------------------------------------------
C 1.0 .LT. argument .LT. 12.0, reduce argument if necessary
C-----------------------------------------------------------------------
CS          N = IFIX(Y) - 1
            N = IFIX(SNGL(Y)) - 1
CS          Y = Y - FLOAT(N)
            Y = Y - DBLE(N)
            Z = Y - ONE
          ELSE
C-----------------------------------------------------------------------
C 0.0 .LT. argument .LT. 1.0
C-----------------------------------------------------------------------
            Z = Y
            Y = Y + ONE
          END IF
C-----------------------------------------------------------------------
C Evaluate approximation for 1.0 .LT. argument .LT. 2.0
C-----------------------------------------------------------------------
          XNUM = ZERO
          XDEN = ONE
          DO 20 I = 1, 8
            XNUM = (XNUM + P(I))*Z
            XDEN = XDEN*Z + Q(I)
   20     CONTINUE
          RES = XNUM/XDEN + ONE
          IF (Y.NE.Y1) THEN
            IF (Y1.GT.Y) THEN
C-----------------------------------------------------------------------
C Adjust result for case  2.0 .LT. argument .LT. 12.0
C-----------------------------------------------------------------------
              DO 30 I = 1, N
                RES = RES*Y
                Y = Y + ONE
   30         CONTINUE
            ELSE
C-----------------------------------------------------------------------
C Adjust result for case  0.0 .LT. argument .LT. 1.0
C-----------------------------------------------------------------------
              RES = RES/Y1
            END IF
          END IF
        END IF
C-----------------------------------------------------------------------
C Final adjustments and return
C-----------------------------------------------------------------------
        IF (PARITY) RES = -RES
        IF (FACT.NE.ONE) RES = FACT/RES
        GAMMA = RES
C**40   CONTINUE
        RETURN
        END
*
* **********************************************************************
*
        SUBROUTINE WHIZ(TERM, ITERM, QNUM, QDEN, RESULT, S)
************************************************************************
*     ALGORITHM 602, COLLECTED ALGORITHMS FROM ACM.
*     ALGORITHM APPEARED IN ACM-TRANS. MATH. SOFTWARE, VOL.9, NO. 3,
*     SEP., 1983, P. 355-357.
*
* The u algorithm for accelerating a series.
*
* Arguments:
*    TERM   = last element of series
*    ITERM   = order of TERM in the series = number of calls to WHIZ
*    QNUM   = backward diagonal of numerator array, at least N long
*    QDEN   = backward diagonal of denominator array, at least N long
*    RESULT = accelerated value of the sum
*    S      = simple sum of the series
*
* Inputs:  TERM, ITERM
*
* Outputs:  RESULT, S
*
* If a single precision version is desired, change all occurrences of
* *SP in columns 1 to 3 to blanks and comment the corresponding double
* precision statements.
*
* Revised by M. Goano, Politecnico di Torino.
* Latest modification of the revised version: April 12, 1993
************************************************************************
*   Parameters
*SP     REAL             ONE, ZERO
        DOUBLE PRECISION ONE, ZERO
*SP     PARAMETER (ONE = 1.0E+0, ZERO = 0.0E+0)
        PARAMETER (ONE = 1.0D+0, ZERO = 0.0D+0)
*   Scalar arguments
        INTEGER ITERM
*SP     REAL             RESULT, S, TERM
        DOUBLE PRECISION RESULT, S, TERM
*   Array arguments
*SP     REAL             QNUM(*), QDEN(*)
        DOUBLE PRECISION QNUM(*), QDEN(*)
*   Local scalars
        INTEGER J, K, L
*SP     REAL             C, FACTOR, FJ, FL, FTERM, RATIO
        DOUBLE PRECISION C, FACTOR, FJ, FL, FTERM, RATIO
*   Intrinsic functions
*SP     INTRINSIC REAL
        INTRINSIC DBLE
* ----------------------------------------------------------------------
        IF (ITERM.EQ.1) S = ZERO
* Get ITERM diagonal
        S = TERM + S
        L = ITERM - 1
*SP     FTERM = REAL(ITERM)
        FTERM = DBLE(ITERM)
        QDEN(ITERM) = ONE/(TERM*FTERM**2)
        QNUM(ITERM) = S*QDEN(ITERM)
        IF (ITERM.GT.1) THEN
          FACTOR = ONE
*SP       FL = REAL(L)
          FL = DBLE(L)
          RATIO = FL/FTERM
          DO 10 K = 1, L
            J = ITERM - K
*SP         FJ = REAL(J)
            FJ = DBLE(J)
            C = FACTOR*FJ/FTERM
            FACTOR = FACTOR*RATIO
            QDEN(J) = QDEN(J + 1) - C*QDEN(J)
            QNUM(J) = QNUM(J + 1) - C*QNUM(J)
   10     CONTINUE
        END IF
        RESULT = QNUM(1)/QDEN(1)
        RETURN
        END
