c
c******************************************************************
c Kummer's confluent hypergeometric functions M(a,b,x) and U(a,b,x)
c******************************************************************
c 17/01/2012 replaced dcmplx by cmplx (., ., kind = 2) w.g.b
c
c KUMMER
c M1KUMM
c U1KUMM
c
      double precision function kummer (isend, a, b, x)
c
c action: calculate Kummer M(a,b,x) or U(a,b,x) for real a, b, x
c author: w.g.bardsley, university of manchester, u.k, 23/12/2001
c
c         isend = 1: M(a,b,x)
c         isend = 2: U(a,b,x) logarithmic case
c
      implicit   none
      integer    isend
      integer    ip, lnchf
      parameter (ip = 10, lnchf = 0)
      double precision a, b, x
      double precision dchu
      double precision zero
      parameter (zero = 0.0d+00)
      complex*16 ca, cb, chf, conhyp, z
      external   conhyp, dchu
      intrinsic  dble, cmplx
      if (isend.eq.1) then
c
c M(a,b,x): generate complex variables
c
         ca = cmplx(a, zero, kind = 2)
         cb = cmplx(b, zero, kind = 2)
         z = cmplx(x, zero, kind = 2)
c
c calculate the complex Kummer function
c
         chf = conhyp(ca, cb, z, lnchf, ip)
c
c just return the real part
c
         kummer = dble(chf)
      elseif (isend.eq.2) then
c
c U(a,b,x)
c
         kummer = dchu (a, b, x)
      else
c
c Error
c
         kummer = zero
      endif
      end
C
C From ACMTOMS 745 ..Kummer functions for Fermi-Dirac integrals
C
C Added LOGGAM = .FALSE. after CALL GAMMAC, 20/12/2001 to force
C initialisation of LOGGAM
C
C M1KUMM
C U1KUMM
C
* **********************************************************************
*
        SUBROUTINE M1KUMM(A, X, EPS, XMAX, RELERR, M)
*
* **********************************************************************
* M1KUMM returns in M the value of Kummer's confluent hypergeometric
*        function M(1,2+A,-X), defined in (13.1.2) of [1], for real
*        arguments A and X, approximated with a relative error RELERR.
*        EPS and XMAX represent the smallest positive floating-point
*        number such that 1.0+EPS .NE. 1.0, and the largest finite
*        floating-point number, respectively.  Asymptotic expansion [1]
*        or continued fraction representation [2] is used.
*        Renormalization is carried out as proposed in [3].
*
* References:
*
*   [1] L. J. Slater, "Confluent Hypergeometric Functions", in "Handbook
*       of Mathematical Functions with Formulas, Graphs and Mathematical
*       Tables" (M. Abramowitz and I. A. Stegun, eds.), no. 55 in
*       National Bureau of Standards Applied Mathematics Series, ch. 13,
*       pp. 503-535, Washington, D.C.:  U.S. Government Printing Office,
*       1964.
*
*   [2] P. Henrici, "Applied and Computational Complex Analysis.
*       Volume 2.  Special Functions-Integral Transforms-Asymptotics-
*       Continued Fractions", New York:  John Wiley & Sons, 1977.
*
*   [3] W. H. Press, B. P. Flannery, S. A. Teukolsky, W. T. Vetterling,
*       "Numerical Recipes. The Art of Scientific Computing", Cambridge:
*       Cambridge University Press, 1986.
*
* If a single precision version is desired, change all occurrences of
* *SP in columns 1 to 3 to blanks and comment the corresponding double
* precision statements.
*
* Michele Goano, Politecnico di Torino  (goano@polito.it).
* Latest revision:  March 22, 1994.
* **********************************************************************
*   Parameters
        INTEGER ITMAX
        PARAMETER (ITMAX = 100)
*SP     REAL             ONE, PI, TEN, THREE, TWO, ZERO
        DOUBLE PRECISION ONE, PI, TEN, THREE, TWO, ZERO
*SP     PARAMETER (ONE = 1.0E+0, PI = 3.141592653589793238462643E+0,
*SP  &             TEN = 10.0E+0, THREE = 3.0E+0, TWO = 2.0E+0,
*SP  &             ZERO = 0.0E+0)
        PARAMETER (ONE = 1.0D+0, PI = 3.141592653589793238462643D+0,
     &             TEN = 10.0D+0, THREE = 3.0D+0, TWO = 2.0D+0,
     &             ZERO = 0.0D+0)
*   Scalar arguments
*SP     REAL             A, X, EPS, XMAX, RELERR, M
        DOUBLE PRECISION A, X, EPS, XMAX, RELERR, M
*   Local scalars
C*******LOGICAL OKASYM
        INTEGER IERR, N
*SP     REAL             AA, ADD, ADDOLD, BB, FAC, GAMMA, GOLD, MLOG,
*SP  &                   P1, P2, Q1, Q2, RKDIV, RN, XASYMP, XBIG
        DOUBLE PRECISION AA, ADD, ADDOLD, BB, FAC, GAMMA, GOLD, MLOG,
     &                   P1, P2, Q1, Q2, RKDIV, RN, XASYMP, XBIG
*   External subroutines
        EXTERNAL GAMMAC
*   Intrinsic functions
*SP     INTRINSIC ABS, COS, EXP, LOG, LOG10, REAL
        INTRINSIC ABS, COS, EXP, LOG, LOG10, DBLE
        intrinsic max
* ----------------------------------------------------------------------
        XBIG = LOG(XMAX)
C*******OKASYM = .TRUE.
        M = ZERO
*   Special cases
        IF (X.EQ.ZERO) THEN
          M = ONE
        ELSE
*   Approximations for k_div - 1 and x_min (RKDIV and XASYMP)
          RKDIV = -TWO/THREE*LOG10(RELERR)
          XASYMP = MAX(A - ONE,
     &                 TWO + RKDIV - A*(TWO+RKDIV/TEN),
     &                 ABS(RKDIV-A))
          IF (X.GT.XASYMP) THEN
*   Asymptotic expansion
            CALL GAMMAC(A+ONE, EPS, XMAX, GAMMA, IERR)
            IF (IERR.EQ.-1) THEN
*   Handling of the logarithm of the gamma function to avoid overflow
              MLOG = GAMMA - X - A*LOG(X)
              IF (MLOG.LT.XBIG) THEN
                M = ONE - COS(PI*A)*EXP(MLOG)
              ELSE
C***************OKASYM = .FALSE.
                GO TO 20
              END IF
            ELSE
              M = ONE - COS(PI*A)*GAMMA*EXP(-X)/X**A
            END IF
            ADDOLD = XMAX
            ADD = -A/X
            DO 10 N = 1, ITMAX
*   Divergence
              IF (ABS(ADD).GE.ABS(ADDOLD)) THEN
C***************OKASYM = .FALSE.
                GO TO 20
              END IF
              M = M + ADD
*   Check truncation error and convergence
              IF (ABS(ADD).LE.ABS(M)*RELERR) THEN
                M = M*(A + ONE)/X
                RETURN
              END IF
              ADDOLD = ADD
*SP           ADD = -ADD*(A - REAL(N))/X
              ADD = -ADD*(A - DBLE(N))/X
   10       CONTINUE
          END IF
*   Continued fraction:  initial conditions
   20     CONTINUE
          GOLD = ZERO
          P1 = ONE
          Q1 = ONE
          P2 = A + TWO
          Q2 = X + A + TWO
          BB = A + TWO
*   Initial value of the normalization factor
          FAC = ONE
          DO 30 N = 1, ITMAX
*   Evaluation of a_(2N+1) and b_(2N+1)
*SP         RN = REAL(N)
            RN = DBLE(N)
            AA = -RN*X
            BB = BB + ONE
            P1 = (AA*P1 + BB*P2)*FAC
            Q1 = (AA*Q1 + BB*Q2)*FAC
*   Evaluation of a_(2N+2) and b_(2N+2)
            AA = (A + RN + ONE)*X
            BB = BB + ONE
            P2 = BB*P1 + AA*P2*FAC
            Q2 = BB*Q1 + AA*Q2*FAC
            IF (Q2.NE.ZERO) THEN
*   Renormalization and evaluation of w_(2N+2)
              FAC = ONE/Q2
              M = P2*FAC
*   Check truncation error and convergence
              IF (ABS(M-GOLD).LT.ABS(M)*RELERR) RETURN
              GOLD = M
            END IF
   30     CONTINUE
        END IF
        RETURN
        END

* **********************************************************************
*
        SUBROUTINE U1KUMM(A, X, EPS, XMAX, RELERR, U)
*
* **********************************************************************
* U1KUMM returns in U the value of Kummer's confluent hypergeometric
*        function U(1,1+A,X), defined in (13.1.3) of [1], for real
*        arguments A and X, approximated with a relative error RELERR.
*        EPS and XMAX represent the smallest positive floating-point
*        number such that 1.0+EPS .NE. 1.0, and the largest finite
*        floating-point number, respectively.  The relation with the
*        incomplete gamma function is exploited, by means of (13.6.28)
*        and (13.1.29) of [1].  For A .LE. 0 an expansion in terms of
*        Laguerre polynomials is used [3].  Otherwise the recipe of [4]
*        is followed: series expansion (6.5.29) of [2] if X .LT. A+1,
*        continued fraction (6.5.31) of [2] if X .GE. A+1.
*
* References:
*
*   [1] L. J. Slater, "Confluent Hypergeometric Functions", ch. 13 in
*       [5], pp. 503-535.
*
*   [2] P. J. Davis, "Gamma Function and Related Functions", ch. 6 in
*       [5], pp. 253-293.
*
*   [3] P. Henrici, "Computational Analysis with the HP-25 Pocket
*       Calculator", New York:  John Wiley & Sons, 1977.
*
*   [4] W. H. Press, B. P. Flannery, S. A. Teukolsky, W. T. Vetterling,
*       "Numerical Recipes. The Art of Scientific Computing", Cambridge:
*       Cambridge University Press, 1986.
*
*   [5] M. Abramowitz and I. A. Stegun (eds.), "Handbook of Mathematical
*       Functions with Formulas, Graphs and Mathematical Tables", no. 55
*       in National Bureau of Standards Applied Mathematics Series,
*       Washington, D.C.:  U.S. Government Printing Office, 1964.
*
* If a single precision version is desired, change all occurrences of
* *SP in columns 1 to 3 to blanks and comment the corresponding double
* precision statements.
*
* Michele Goano, Politecnico di Torino  (goano@polito.it).
* Latest revision:  March 22, 1994.
* **********************************************************************
*   Parameters
        INTEGER ITMAX
        PARAMETER (ITMAX = 100)
*SP     REAL             ONE, ZERO
        DOUBLE PRECISION ONE, ZERO
*SP     PARAMETER (ONE = 1.0E+0, ZERO = 0.0E+0)
        PARAMETER (ONE = 1.0D+0, ZERO = 0.0D+0)
*   Scalar arguments
*SP     REAL             A, X, EPS, XMAX, RELERR, U
        DOUBLE PRECISION A, X, EPS, XMAX, RELERR, U
*   Local scalars
        LOGICAL LOGGAM
        INTEGER IERR, N
*SP     REAL             A0, A1, ANA, ANF, AP, B0, B1, DEL, FAC, G,
*SP  &                   GAMMA, GOLD, PLAGN, PLAGN1, PLAGN2, RN, T,
*SP  &                   ULOG, XBIG
        DOUBLE PRECISION A0, A1, ANA, ANF, AP, B0, B1, DEL, FAC, G,
     &                   GAMMA, GOLD, PLAGN, PLAGN1, PLAGN2, RN, T,
     &                   ULOG
*   External subroutines
        EXTERNAL GAMMAC
*   Intrinsic functions
*SP     INTRINSIC ABS, EXP, LOG, REAL
        INTRINSIC ABS, DBLE, EXP, LOG
* ----------------------------------------------------------------------
C*******XBIG = LOG(XMAX)
        U = ZERO
*   Special cases
        IF (X.EQ.ZERO) THEN
          U = -ONE/A
*   Laguerre polynomials
        ELSE IF (A.LE.ZERO) THEN
          U = ZERO
          PLAGN2 = ZERO
          PLAGN1 = ONE
          G = ONE
          DO 10 N = 1, ITMAX
*SP         RN = REAL(N)
            RN = DBLE(N)
            PLAGN = ((RN-A-ONE)*(PLAGN1-PLAGN2) + (RN+X)*PLAGN1)/RN
            T = G/(PLAGN1*PLAGN)
            U = U + T
            IF (ABS(T).LT.ABS(U)*RELERR) RETURN
            G = G*(RN - A)/(RN + ONE)
            PLAGN2 = PLAGN1
            PLAGN1 = PLAGN
   10     CONTINUE
*   Series expansion
        ELSE IF (X.LT.A+ONE) THEN
          CALL GAMMAC(A, EPS, XMAX, GAMMA, IERR)
          LOGGAM = .FALSE.!added by w.g.bardsley, 20/12/2001
          IF (IERR.EQ.-1) LOGGAM = .TRUE.
          AP = A
          U = ONE/A
          DEL = U
          DO 20 N = 1, ITMAX
            AP = AP + ONE
            DEL = DEL*X/AP
            U = U + DEL
            IF (ABS(DEL).LT.ABS(U)*RELERR) THEN
              IF (LOGGAM) THEN
                ULOG = GAMMA + X + A*LOG(X)
                U = EXP(ULOG) - U
              ELSE
                U = GAMMA*EXP(X)/X**A - U
              END IF
              RETURN
            END IF
   20     CONTINUE
*   Continued fraction
        ELSE
          GOLD = ZERO
          A0 = ONE
          A1 = X
          B0 = ZERO
          B1 = ONE
          FAC = ONE
          DO 30 N = 1, ITMAX
*SP         RN = REAL(N)
            RN = DBLE(N)
            ANA = RN - A
            A0 = (A1 + A0*ANA)*FAC
            B0 = (B1 + B0*ANA)*FAC
            ANF = RN*FAC
            A1 = X*A0 + ANF*A1
            B1 = X*B0 + ANF*B1
            IF (A1.NE.ZERO) THEN
              FAC = ONE/A1
              U = B1*FAC
              IF (ABS(U-GOLD).LT.ABS(U)*RELERR) RETURN
              GOLD = U
            END IF
   30     CONTINUE
        END IF
        RETURN
        END

