c
c ===============================================================
c AMTOMS 757: Struve functions from MISC but with ERRPRN disabled
c ===============================================================
c
      double precision function struve (x, m, n)
c
c m = 1, n = 0: H0
c m = 1, n = 1: H1
c m = 2, n = 0: L0
c m = 2, n = 1: L1
c
      implicit none
      integer  m, n
      double precision x
      double precision zero
      parameter (zero = 0.0d+00)
      double precision strvh0, strvh1, strvl0, strvl1
      external strvh0, strvh1, strvl0, strvl1
      if (m.eq.1 .and. n.eq.0) then
         struve = strvh0 (x)
      elseif (m.eq.1 .and. n.eq.1) then
         struve = strvh1 (x)
      elseif (m.eq.2 .and. n.eq.0) then
         struve = strvl0 (x)
      elseif (m.eq.2 .and. n.eq.1) then
         struve = strvl1 (x)
      else
         struve = zero
      endif
      end
C
C
      DOUBLE PRECISION FUNCTION STRVH0(XVALUE)
C
C
C   DESCRIPTION:
C
C      This function calculates the value of the Struve function
C      of order 0, denoted H0(x), for the argument XVALUE, defined
C
C         STRVHO(x) = (2/pi) integral{0 to pi/2} sin(x cos(t)) dt
C
C      H0 also satisfies the second-order equation
C
C                 x*D(Df) + Df + x*f = 2x/pi
C
C      The code uses Chebyshev expansions whose coefficients are
C      given to 20D.
C
C
C   ERROR RETURNS:
C
C      As the asymptotic expansion of H0 involves the Bessel function
C      of the second kind Y0, there is a problem for large x, since
C      we cannot accurately calculate the value of Y0. An error message
C      is printed and STRVH0 returns the value 0.0.
C
C
C   MACHINE-DEPENDENT CONSTANTS:
C
C      NTERM1 - The no. of terms to be used in the array ARRH0. The
C               recommended value is such that
C                      ABS(ARRH0(NTERM1)) < EPS/100.
C
C      NTERM2 - The no. of terms to be used in the array ARRH0A. The
C               recommended value is such that
C                      ABS(ARRH0A(NTERM2)) < EPS/100.
C
C      NTERM3 - The no. of terms to be used in the array AY0ASP. The
C               recommended value is such that
C                      ABS(AY0ASP(NTERM3)) < EPS/100.
C
C      NTERM4 - The no. of terms to be used in the array AY0ASQ. The
C               recommended value is such that
C                      ABS(AY0ASQ(NTERM4)) < EPS/100.
C
C      XLOW - The value for which H0(x) = 2*x/pi to machine precision, if
C             abs(x) < XLOW. The recommended value is
C                      XLOW = 3 * SQRT(EPSNEG)
C
C      XHIGH - The value above which we are unable to calculate Y0 with
C              any reasonable accuracy. An error message is printed and
C              STRVH0 returns the value 0.0. The recommended value is
C                      XHIGH = 1/EPS.
C
C      For values of EPS and EPSNEG refer to the file MACHCON.TXT.
C
C      The machine-dependent constants are computed internally by
C      using the D1MACH subroutine.
C
C
C   INTRINSIC FUNCTIONS USED:
C
C      ABS, COS, SIN, SQRT.
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
C          CHEVAL , ERRPRN, D1MACH
C
C
C   AUTHOR:
C          ALLAN J. MACLEOD
C          DEPT. OF MATHEMATICS AND STATISTICS
C          UNIVERSITY OF PAISLEY
C          HIGH ST.
C          PAISLEY
C          SCOTLAND
C          PA1 2BE
C
C          (e-mail: macl_ms0@paisley.ac.uk )
C
C
C   LATEST REVISION:
C                   23 January, 1996
C
C
      INTEGER INDSGN,NTERM1,NTERM2,NTERM3,NTERM4
      DOUBLE PRECISION ARRH0(0:19),ARRH0A(0:20),AY0ASP(0:12),
     1     AY0ASQ(0:13),CHEVAL,EIGHT,ELEVEN,HALF,H0AS,
     2     ONEHUN,ONE,PIBY4,RT2BPI,SIXTP5,T,THR2P5,TWENTY,
     3     TWOBPI,TWO62,X,XHIGH,XLOW,XMP4,XSQ,XVALUE,
     4     Y0P,Y0Q,Y0VAL,ZERO,D1MACH
C
C   INTRINSIC FUNCTIONS USED:
C
       intrinsic ABS, COS, SIN, SQRT
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
       external CHEVAL, D1MACH
C
c*****CHARACTER FNNAME*6,ERRMSG*26
c*****DATA FNNAME/'STRVH0'/
c*****DATA ERRMSG/'ARGUMENT TOO LARGE IN SIZE'/
      DATA ZERO,HALF,ONE/0.0 D 0 , 0.5 D 0 , 1.0 D 0/
      DATA EIGHT,ELEVEN/8.0 D 0 , 11.0 D 0/
      DATA TWENTY,ONEHUN/20.0 D 0 , 100.0 D 0/
      DATA SIXTP5,TWO62,THR2P5/60.5 D 0 , 262.0 D 0 , 302.5 D 0/
      DATA PIBY4/0.78539 81633 97448 30962 D 0/
      DATA RT2BPI/0.79788 45608 02865 35588 D 0/
      DATA TWOBPI/0.63661 97723 67581 34308 D 0/
      DATA ARRH0(0)/  0.28696 48739 90132 25740  D    0/
      DATA ARRH0(1)/ -0.25405 33268 16183 52305  D    0/
      DATA ARRH0(2)/  0.20774 02673 93238 94439  D    0/
      DATA ARRH0(3)/ -0.20364 02956 03865 85140  D    0/
      DATA ARRH0(4)/  0.12888 46908 68661 86016  D    0/
      DATA ARRH0(5)/ -0.48256 32815 62226 1202   D   -1/
      DATA ARRH0(6)/  0.11686 29347 56900 1242   D   -1/
      DATA ARRH0(7)/ -0.19811 81356 42418 416    D   -2/
      DATA ARRH0(8)/  0.24899 13851 24212 86     D   -3/
      DATA ARRH0(9)/ -0.24188 27913 78595 0      D   -4/
      DATA ARRH0(10)/ 0.18743 75479 93431        D   -5/
      DATA ARRH0(11)/-0.11873 34607 4362         D   -6/
      DATA ARRH0(12)/ 0.62698 49433 46           D   -8/
      DATA ARRH0(13)/-0.28045 54679 3            D   -9/
      DATA ARRH0(14)/ 0.10769 41205              D  -10/
      DATA ARRH0(15)/-0.35904 793                D  -12/
      DATA ARRH0(16)/ 0.10494 47                 D  -13/
      DATA ARRH0(17)/-0.27119                    D  -15/
      DATA ARRH0(18)/ 0.624                      D  -17/
      DATA ARRH0(19)/-0.13                       D  -18/
      DATA ARRH0A(0)/  1.99291 88575 19923 05515  D    0/
      DATA ARRH0A(1)/ -0.38423 26687 01456 887    D   -2/
      DATA ARRH0A(2)/ -0.32871 99371 23530 50     D   -3/
      DATA ARRH0A(3)/ -0.29411 81203 70340 9      D   -4/
      DATA ARRH0A(4)/ -0.26731 53519 87066        D   -5/
      DATA ARRH0A(5)/ -0.24681 03107 5013         D   -6/
      DATA ARRH0A(6)/ -0.22950 14861 143          D   -7/
      DATA ARRH0A(7)/ -0.21568 22318 33           D   -8/
      DATA ARRH0A(8)/ -0.20303 50648 3            D   -9/
      DATA ARRH0A(9)/ -0.19345 75509              D  -10/
      DATA ARRH0A(10)/-0.18277 3144               D  -11/
      DATA ARRH0A(11)/-0.17768 424                D  -12/
      DATA ARRH0A(12)/-0.16432 96                 D  -13/
      DATA ARRH0A(13)/-0.17156 9                  D  -14/
      DATA ARRH0A(14)/-0.13368                    D  -15/
      DATA ARRH0A(15)/-0.2077                     D  -16/
      DATA ARRH0A(16)/ 0.2                        D  -19/
      DATA ARRH0A(17)/-0.55                       D  -18/
      DATA ARRH0A(18)/ 0.10                       D  -18/
      DATA ARRH0A(19)/-0.4                        D  -19/
      DATA ARRH0A(20)/ 0.1                        D  -19/
      DATA AY0ASP/1.99944 63940 23982 71568  D    0,
     1           -0.28650 77864 70319 58     D   -3,
     2           -0.10050 72797 43762 0      D   -4,
     3           -0.35835 94100 2463         D   -6,
     4           -0.12879 65120 531          D   -7,
     5           -0.46609 48663 6            D   -9,
     6           -0.16937 69454              D  -10,
     7           -0.61852 269                D  -12,
     8           -0.22618 41                 D  -13,
     9           -0.83268                    D  -15,
     X           -0.3042                     D  -16,
     1           -0.115                      D  -17,
     2           -0.4                        D  -19/
      DATA AY0ASQ/1.99542 68138 68286 04092  D    0,
     1           -0.23601 31928 67514 472    D   -2,
     2           -0.76015 38908 50296 6      D   -4,
     3           -0.25610 88714 56343        D   -5,
     4           -0.87502 92185 106          D   -7,
     5           -0.30430 42121 59           D   -8,
     6           -0.10621 42831 4            D   -9,
     7           -0.37737 1479               D  -11,
     8           -0.13213 687                D  -12,
     9           -0.48862 1                  D  -14,
     X           -0.15809                    D  -15,
     1           -0.762                      D  -17,
     2           -0.3                        D  -19,
     3           -0.3                        D  -19/
C
C   Start computation
C
      X = XVALUE
      INDSGN = 1
      IF ( X .LT. ZERO ) THEN
         X = -X
         INDSGN = -1
      ENDIF
C
C   Compute the machine-dependent constants.
C
      H0AS = D1MACH(3)
      XHIGH = ONE / D1MACH(4)
C
C   Error test
C
      IF ( ABS(XVALUE) .GT. XHIGH ) THEN
C********CALL ERRPRN(FNNAME,ERRMSG)
         STRVH0 = ZERO
         RETURN
      ENDIF
C
C   continue with machine constants
C
      T = H0AS / ONEHUN
      IF ( X .LE. ELEVEN ) THEN
         DO 10 NTERM1 = 19 , 0 , -1
            IF ( ABS(ARRH0(NTERM1)) .GT. T ) GOTO 19
 10      CONTINUE
 19      Y0P = SQRT ( H0AS )
         XLOW = Y0P + Y0P + Y0P
      ELSE
         DO 40 NTERM2 = 20 , 0 , -1
            IF ( ABS(ARRH0A(NTERM2)) .GT. T ) GOTO 49
 40      CONTINUE
 49      DO 50 NTERM3 = 12 , 0 , -1
            IF ( ABS(AY0ASP(NTERM3)) .GT. T ) GOTO 59
 50      CONTINUE
 59      DO 60 NTERM4 = 13 , 0 , -1
            IF ( ABS(AY0ASQ(NTERM4)) .GT. T ) GOTO 69
 60      CONTINUE
C69   ENDIF
 69      CONTINUE
      ENDIF
C
C   Code for abs(x) <= 11
C
      IF ( X .LE. ELEVEN ) THEN
         IF ( X .LT. XLOW ) THEN
            STRVH0 = TWOBPI * X
         ELSE
            T = ( ( X * X ) / SIXTP5 - HALF ) - HALF
            STRVH0 = TWOBPI * X * CHEVAL ( NTERM1 , ARRH0 , T )
         ENDIF
      ELSE
C
C   Code for abs(x) > 11
C
         XSQ = X * X
         T = ( TWO62 - XSQ ) / ( TWENTY + XSQ )
         Y0P = CHEVAL ( NTERM3 , AY0ASP , T )
         Y0Q = CHEVAL ( NTERM4 , AY0ASQ , T ) / ( EIGHT * X )
         XMP4 = X - PIBY4
         Y0VAL = Y0P * SIN ( XMP4 ) - Y0Q * COS ( XMP4 )
         Y0VAL = Y0VAL * RT2BPI / SQRT ( X )
         T = ( THR2P5 - XSQ ) / ( SIXTP5 + XSQ )
         H0AS = TWOBPI * CHEVAL ( NTERM2 , ARRH0A , T ) / X
         STRVH0 = Y0VAL + H0AS
      ENDIF
      IF ( INDSGN .EQ. -1 ) STRVH0 = -STRVH0
      RETURN
      END

      DOUBLE PRECISION FUNCTION STRVH1(XVALUE)
C
C
C   DESCRIPTION:
C      This function calculates the value of the Struve function
C      of order 1, denoted H1(x), for the argument XVALUE, defined as
C
C                                                                  2
C        STRVH1(x) = (2x/pi) integral{0 to pi/2} sin( x cos(t))*sin t dt
C
C      H1 also satisfies the second-order differential equation
C
C                    2   2                   2            2
C                   x * D f  +  x * Df  +  (x - 1)f  =  2x / pi
C
C      The code uses Chebyshev expansions with the coefficients
C      given to 20D.
C
C
C   ERROR RETURNS:
C      As the asymptotic expansion of H1 involves the Bessel function
C      of the second kind Y1, there is a problem for large x, since
C      we cannot accurately calculate the value of Y1. An error message
C      is printed and STRVH1 returns the value 0.0.
C
C
C   MACHINE-DEPENDENT CONSTANTS:
C
C      NTERM1 - The no. of terms to be used in the array ARRH1. The
C               recommended value is such that
C                      ABS(ARRH1(NTERM1)) < EPS/100.
C
C      NTERM2 - The no. of terms to be used in the array ARRH1A. The
C               recommended value is such that
C                      ABS(ARRH1A(NTERM2)) < EPS/100.
C
C      NTERM3 - The no. of terms to be used in the array AY1ASP. The
C               recommended value is such that
C                      ABS(AY1ASP(NTERM3)) < EPS/100.
C
C      NTERM4 - The no. of terms to be used in the array AY1ASQ. The
C               recommended value is such that
C                      ABS(AY1ASQ(NTERM4)) < EPS/100.
C
C      XLOW1 - The value of x, below which H1(x) set to zero, if
C              abs(x)<XLOW1. The recommended value is
C                      XLOW1 = 1.5 * SQRT(XMIN)
C
C      XLOW2 - The value for which H1(x) = 2*x*x/pi to machine precision, if
C              abs(x) < XLOW2. The recommended value is
C                      XLOW2 = SQRT(15*EPSNEG)
C
C      XHIGH - The value above which we are unable to calculate Y1 with
C              any reasonable accuracy. An error message is printed and
C              STRVH1 returns the value 0.0. The recommended value is
C                      XHIGH = 1/EPS.
C
C      For values of EPS, EPSNEG and XMIN refer to the file MACHCON.TXT.
C
C      The machine-dependent constants are computed internally by
C      using the D1MACH subroutine.
C
C
C   INTRINSIC FUNCTIONS USED:
C
C      ABS, COS, SIN, SQRT.
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
C          CHEVAL , ERRPRN, D1MACH
C
C
C   AUTHOR:
C          ALLAN J. MACLEOD
C          DEPT. OF MATHEMATICS AND STATISTICS
C          UNIVERSITY OF PAISLEY
C          HIGH ST.
C          PAISLEY
C          SCOTLAND
C          PA1 2BE
C
C          (e-mail: macl_ms0@paisley.ac.uk)
C
C
C   LATEST REVISION:
C                   23 January, 1996
C
C
      INTEGER NTERM1,NTERM2,NTERM3,NTERM4
      DOUBLE PRECISION ARRH1(0:17),ARRH1A(0:21),AY1ASP(0:14),
     1     AY1ASQ(0:15),CHEVAL,EIGHT,FIFTEN,FORTP5,HALF,
     2     H1AS,NINE,ONEHUN,ONE82,RT2BPI,T,THPBY4,
     3     TWENTY,TWOBPI,TW02P5,X,XHIGH,XLOW1,XLOW2,
     4     XM3P4,XSQ,XVALUE,Y1P,Y1Q,Y1VAL,ZERO,D1MACH
C
C   INTRINSIC FUNCTIONS USED:
C
      intrinsic ABS, COS, SIN, SQRT
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
      external CHEVAL, D1MACH
C
c*****CHARACTER FNNAME*6,ERRMSG*26
c*****DATA FNNAME/'STRVH1'/
c*****DATA ERRMSG/'ARGUMENT TOO LARGE IN SIZE'/
      DATA ZERO,HALF,EIGHT/0.0 D 0 , 0.5 D 0 , 8.0 D 0/
      DATA NINE,FIFTEN/ 9.0 D 0 , 15.0 D 0 /
      DATA TWENTY,ONEHUN/ 20.0 D 0 , 100.0 D 0/
      DATA FORTP5,ONE82,TW02P5/40.5 D 0 , 182.0 D 0 , 202.5 D 0/
      DATA RT2BPI/0.79788 45608 02865 35588 D 0/
      DATA THPBY4/2.35619 44901 92344 92885 D 0/
      DATA TWOBPI/0.63661 97723 67581 34308 D 0/
      DATA ARRH1/0.17319 06108 36754 39319  D    0,
     1          -0.12606 91759 13526 72005  D    0,
     2           0.79085 76160 49535 7500   D   -1,
     3          -0.31964 93222 32187 0820   D   -1,
     4           0.80804 05814 04918 834    D   -2,
     5          -0.13600 08206 93074 148    D   -2,
     6           0.16227 14861 98894 71     D   -3,
     7          -0.14423 52451 48592 9      D   -4,
     8           0.99219 52573 4072         D   -6,
     9          -0.54416 28049 180          D   -7,
     X           0.24363 16625 63           D   -8,
     1          -0.90770 71338              D  -10,
     2           0.28592 6585               D  -11,
     3          -0.77169 75                 D  -13,
     4           0.18048 9                  D  -14,
     5          -0.3694                     D  -16,
     6           0.67                       D  -18,
     7          -0.1                        D  -19/
      DATA ARRH1A(0)/  2.01083 50495 14733 79407  D    0/
      DATA ARRH1A(1)/  0.59221 86100 36099 903    D   -2/
      DATA ARRH1A(2)/  0.55274 32269 84141 30     D   -3/
      DATA ARRH1A(3)/  0.52698 73856 31103 6      D   -4/
      DATA ARRH1A(4)/  0.50637 45221 40969        D   -5/
      DATA ARRH1A(5)/  0.49028 73642 0678         D   -6/
      DATA ARRH1A(6)/  0.47635 40023 525          D   -7/
      DATA ARRH1A(7)/  0.46525 86522 83           D   -8/
      DATA ARRH1A(8)/  0.45465 16608 1            D   -9/
      DATA ARRH1A(9)/  0.44724 62193              D  -10/
      DATA ARRH1A(10)/ 0.43730 8292               D  -11/
      DATA ARRH1A(11)/ 0.43568 368                D  -12/
      DATA ARRH1A(12)/ 0.41821 90                 D  -13/
      DATA ARRH1A(13)/ 0.44104 4                  D  -14/
      DATA ARRH1A(14)/ 0.36391                    D  -15/
      DATA ARRH1A(15)/ 0.5558                     D  -16/
      DATA ARRH1A(16)/-0.4                        D  -19/
      DATA ARRH1A(17)/ 0.163                      D  -17/
      DATA ARRH1A(18)/-0.34                       D  -18/
      DATA ARRH1A(19)/ 0.13                       D  -18/
      DATA ARRH1A(20)/-0.4                        D  -19/
      DATA ARRH1A(21)/ 0.1                        D  -19/
      DATA AY1ASP/2.00135 24004 58893 96402 D   0,
     1            0.71104 24159 64619 38    D  -3,
     2            0.36659 77028 23244 9     D  -4,
     3            0.19130 15686 57728       D  -5,
     4            0.10046 91138 9777        D  -6,
     5            0.53040 17425 38          D  -8,
     6            0.28100 88617 6           D  -9,
     7            0.14938 86051             D -10,
     8            0.79578 420               D -12,
     9            0.42523 63                D -13,
     X            0.22719 5                 D -14,
     1            0.12216                   D -15,
     2            0.650                     D -17,
     3            0.36                      D -18,
     4            0.2                       D -19/
      DATA AY1ASQ/5.99065 10947 78881 89116 D   0,
     1           -0.48959 32623 36579 635   D  -2,
     2           -0.23238 32130 70706 26    D  -3,
     3           -0.11447 34723 85767 9     D  -4,
     4           -0.57169 92618 9106        D  -6,
     5           -0.28955 16716 917         D  -7,
     6           -0.14751 33456 36          D  -8,
     7           -0.75965 37378             D -10,
     8           -0.39065 8184              D -11,
     9           -0.20464 654               D -12,
     X           -0.10426 36                D -13,
     1           -0.57702                   D -15,
     2           -0.2550                    D -16,
     3           -0.210                     D -17,
     4            0.2                       D -19,
     5           -0.2                       D -19/
C
C   Start computation
C
      X = ABS ( XVALUE )
C
C   Compute the machine-dependent constants.
C
      XHIGH = ( HALF + HALF ) / D1MACH(4)
C
C   Error test
C
      IF ( X .GT. XHIGH ) THEN
C********CALL ERRPRN(FNNAME,ERRMSG)
         STRVH1 = ZERO
         RETURN
      ENDIF
C
C   continue with machine constants
C
      H1AS = D1MACH(3)
      T = H1AS / ONEHUN
      IF ( X .LE. NINE ) THEN
         DO 10 NTERM1 = 17 , 0 , -1
            IF ( ABS(ARRH1(NTERM1)) .GT. T ) GOTO 19
 10      CONTINUE
 19      XLOW1 = HALF * SQRT(D1MACH(1))
         XLOW1 = XLOW1 + XLOW1 + XLOW1
         XLOW2 = SQRT ( FIFTEN * H1AS )
      ELSE
         DO 40 NTERM2 = 21 , 0 , -1
            IF ( ABS(ARRH1A(NTERM2)) .GT. T ) GOTO 49
 40      CONTINUE
 49      DO 50 NTERM3 = 14 , 0 , -1
            IF ( ABS(AY1ASP(NTERM3)) .GT. T ) GOTO 59
 50      CONTINUE
 59      DO 60 NTERM4 = 15 , 0 , -1
            IF ( ABS(AY1ASQ(NTERM4)) .GT. T ) GOTO 69
 60      CONTINUE
C69   ENDIF
 69      CONTINUE
      ENDIF
C
C   Code for abs(x) <= 9
C
      IF ( X .LE. NINE ) THEN
         IF ( X .LT. XLOW1 ) THEN
            STRVH1 = ZERO
         ELSE
            XSQ = X * X
            IF ( X .LT. XLOW2 ) THEN
               STRVH1 = TWOBPI * XSQ
            ELSE
               T = ( XSQ / FORTP5 - HALF ) - HALF
               STRVH1 = TWOBPI * XSQ * CHEVAL ( NTERM1 , ARRH1 , T )
            ENDIF
         ENDIF
      ELSE
C
C   Code for abs(x) > 9
C
         XSQ = X * X
         T = ( ONE82 - XSQ ) / ( TWENTY + XSQ )
         Y1P = CHEVAL ( NTERM3 , AY1ASP , T )
         Y1Q = CHEVAL ( NTERM4 , AY1ASQ , T ) / ( EIGHT * X)
         XM3P4 = X - THPBY4
         Y1VAL = Y1P * SIN ( XM3P4 ) + Y1Q * COS ( XM3P4 )
         Y1VAL = Y1VAL * RT2BPI / SQRT ( X )
         T = ( TW02P5 - XSQ ) / ( FORTP5 + XSQ )
         H1AS = TWOBPI * CHEVAL ( NTERM2 , ARRH1A , T )
         STRVH1 = Y1VAL + H1AS
      ENDIF
      RETURN
      END

      DOUBLE PRECISION FUNCTION STRVL0(XVALUE)
C
C   DESCRIPTION:
C
C      This function calculates the modified Struve function of
C      order 0, denoted L0(x), defined as the solution of the
C      second-order equation
C
C                  x*D(Df) + Df - x*f  =  2x/pi
C
C
C   ERROR RETURNS:
C
C      If the value of |XVALUE| is too large, the result
C      would cause an floating-pt overflow. An error message
C      is printed and the function returns the value of
C      sign(XVALUE)*XMAX where XMAX is the largest possible
C      floating-pt argument.
C
C
C   MACHINE-DEPENDENT PARAMETERS:
C
C      NTERM1 - INTEGER - The no. of terms for the array ARL0.
C                         The recommended value is such that
C                             ABS(ARL0(NTERM1)) < EPS/100
C
C      NTERM2 - INTEGER - The no. of terms for the array ARL0AS.
C                         The recommended value is such that
C                             ABS(ARL0AS(NTERM2)) < EPS/100
C
C      NTERM3 - INTEGER - The no. of terms for the array AI0ML0.
C                         The recommended value is such that
C                             ABS(AI0ML0(NTERM3)) < EPS/100
C
C      XLOW - DOUBLE PRECISION - The value of x below which L0(x) = 2*x/pi
C                    to machine precision. The recommended value is
C                             3*SQRT(EPS)
C
C      XHIGH1 - DOUBLE PRECISION - The value beyond which the Chebyshev series
C                      in the asymptotic expansion of I0 - L0 gives
C                      1.0 to machine precision. The recommended value
C                      is   SQRT( 30/EPSNEG )
C
C      XHIGH2 - DOUBLE PRECISION - The value beyond which the Chebyshev series
C                      in the asymptotic expansion of I0 gives 1.0
C                      to machine precision. The recommended value
C                      is   28 / EPSNEG
C
C      XMAX - DOUBLE PRECISION - The value of XMAX, where XMAX is the
C                    largest possible floating-pt argument.
C                    This is used to prevent overflow.
C
C      For values of EPS, EPSNEG and XMAX the user should refer
C      to the file MACHCON.TXT
C
C      The machine-dependent constants are computed internally by
C      using the D1MACH subroutine.
C
C
C   INTRINSIC FUNCTIONS USED:
C
C      EXP , LOG , SQRT
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
C          CHEVAL , ERRPRN, D1MACH
C
C
C   AUTHOR:
C          DR. ALLAN J. MACLEOD
C          DEPT. OF MATHEMATICS AND STATISTICS
C          UNIVERSITY OF PAISLEY
C          HIGH ST.
C          PAISLEY
C          SCOTLAND
C          PA1 2BE
C
C      (e-mail: macl_ms0@paisley.ac.uk )
C
C
C   LATEST REVISION:
C                   23 January, 1996
C
C
      INTEGER INDSGN,NTERM1,NTERM2,NTERM3
      DOUBLE PRECISION ARL0(0:27),ARL0AS(0:15),AI0ML0(0:23),
     1     ATEHUN,CHEVAL,CH1,CH2,FOUR,LNR2PI,ONE,ONEHUN,
     2     SIXTEN,T,TEST,TWENT4,TWENT8,TWO,TWOBPI,TWO88,
     3     X,XHIGH1,XHIGH2,XLOW,XMAX,XVALUE,XSQ,ZERO,D1MACH
C
C   INTRINSIC FUNCTIONS USED:
C
       intrinsic EXP , LOG , SQRT, abs
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
       external CHEVAL, D1MACH
C
c*****CHARACTER FNNAME*6,ERRMSG*24
c*****DATA FNNAME/'STRVL0'/
c*****DATA ERRMSG/'ARGUMENT CAUSES OVERFLOW'/
      DATA ZERO,ONE,TWO/0.0 D 0 , 1.0 D 0 , 2.0 D 0/
      DATA FOUR,SIXTEN/4.0 D 0 , 16.0 D 0/
      DATA TWENT4,TWENT8,ONEHUN/24.0 D 0 , 28.0 D 0 , 100.0 D 0/
      DATA TWO88,ATEHUN/288.0 D 0 , 800.0 D 0/
      DATA LNR2PI/0.91893 85332 04672 74178 D 0/
      DATA TWOBPI/0.63661 97723 67581 34308 D 0/
      DATA ARL0(0)/  0.42127 45834 99799 24863  D    0/
      DATA ARL0(1)/ -0.33859 53639 12206 12188  D    0/
      DATA ARL0(2)/  0.21898 99481 27107 16064  D    0/
      DATA ARL0(3)/ -0.12349 48282 07131 85712  D    0/
      DATA ARL0(4)/  0.62142 09793 86695 8440   D   -1/
      DATA ARL0(5)/ -0.28178 06028 10954 7545   D   -1/
      DATA ARL0(6)/  0.11574 19676 63809 1209   D   -1/
      DATA ARL0(7)/ -0.43165 85743 06921 179    D   -2/
      DATA ARL0(8)/  0.14614 23499 07298 329    D   -2/
      DATA ARL0(9)/ -0.44794 21180 54614 78     D   -3/
      DATA ARL0(10)/ 0.12364 74610 59437 61     D   -3/
      DATA ARL0(11)/-0.30490 28334 79704 4      D   -4/
      DATA ARL0(12)/ 0.66394 14015 21146        D   -5/
      DATA ARL0(13)/-0.12553 83577 03889        D   -5/
      DATA ARL0(14)/ 0.20073 44645 1228         D   -6/
      DATA ARL0(15)/-0.25882 60170 637          D   -7/
      DATA ARL0(16)/ 0.24114 37427 58           D   -8/
      DATA ARL0(17)/-0.10159 67435 2            D   -9/
      DATA ARL0(18)/-0.12024 30736              D  -10/
      DATA ARL0(19)/ 0.26290 6137               D  -11/
      DATA ARL0(20)/-0.15313 190                D  -12/
      DATA ARL0(21)/-0.15747 60                 D  -13/
      DATA ARL0(22)/ 0.31563 5                  D  -14/
      DATA ARL0(23)/-0.4096                     D  -16/
      DATA ARL0(24)/-0.3620                     D  -16/
      DATA ARL0(25)/ 0.239                      D  -17/
      DATA ARL0(26)/ 0.36                       D  -18/
      DATA ARL0(27)/-0.4                        D  -19/
      DATA ARL0AS(0)/  2.00861 30823 56058 88600  D   0/
      DATA ARL0AS(1)/  0.40373 79665 00438 470    D  -2/
      DATA ARL0AS(2)/ -0.25199 48028 65802 67     D  -3/
      DATA ARL0AS(3)/  0.16057 36682 81117 6      D  -4/
      DATA ARL0AS(4)/ -0.10369 21824 73444        D  -5/
      DATA ARL0AS(5)/  0.67655 78876 305          D  -7/
      DATA ARL0AS(6)/ -0.44499 99067 56           D  -8/
      DATA ARL0AS(7)/  0.29468 88922 8            D  -9/
      DATA ARL0AS(8)/ -0.19621 80522              D -10/
      DATA ARL0AS(9)/  0.13133 0306               D -11/
      DATA ARL0AS(10)/-0.88191 90                 D -13/
      DATA ARL0AS(11)/ 0.59537 6                  D -14/
      DATA ARL0AS(12)/-0.40389                    D -15/
      DATA ARL0AS(13)/ 0.2651                     D -16/
      DATA ARL0AS(14)/-0.208                      D -17/
      DATA ARL0AS(15)/ 0.11                       D -18/
      DATA AI0ML0(0)/ 2.00326 51024 11606 43125  D    0/
      DATA AI0ML0(1)/ 0.19520 68515 76492 081    D   -2/
      DATA AI0ML0(2)/ 0.38239 52356 99083 28     D   -3/
      DATA AI0ML0(3)/ 0.75342 80817 05443 6      D   -4/
      DATA AI0ML0(4)/ 0.14959 57655 89707 8      D   -4/
      DATA AI0ML0(5)/ 0.29994 05312 10557        D   -5/
      DATA AI0ML0(6)/ 0.60769 60482 2459         D   -6/
      DATA AI0ML0(7)/ 0.12399 49554 4506         D   -6/
      DATA AI0ML0(8)/ 0.25232 62552 649          D   -7/
      DATA AI0ML0(9)/ 0.50463 48573 32           D   -8/
      DATA AI0ML0(10)/0.97913 23623 0            D   -9/
      DATA AI0ML0(11)/0.18389 11524 1            D   -9/
      DATA AI0ML0(12)/0.33763 09278              D  -10/
      DATA AI0ML0(13)/0.61117 9703               D  -11/
      DATA AI0ML0(14)/0.10847 2972               D  -11/
      DATA AI0ML0(15)/0.18861 271                D  -12/
      DATA AI0ML0(16)/0.32803 45                 D  -13/
      DATA AI0ML0(17)/0.56564 7                  D  -14/
      DATA AI0ML0(18)/0.93300                    D  -15/
      DATA AI0ML0(19)/0.15881                    D  -15/
      DATA AI0ML0(20)/0.2791                     D  -16/
      DATA AI0ML0(21)/0.389                      D  -17/
      DATA AI0ML0(22)/0.70                       D  -18/
      DATA AI0ML0(23)/0.16                       D  -18/
C
C   Start computation
C
      X = XVALUE
      INDSGN = 1
      IF ( X .LT. ZERO ) THEN
         X = -X
         INDSGN = -1
      ENDIF
C
C   Compute the machine-dependent constants.
C
      TEST = D1MACH(3)
      T = TEST / ONEHUN
      IF ( X .LE. SIXTEN ) THEN
         DO 10 NTERM1 = 27 , 0 , -1
            IF ( ABS(ARL0(NTERM1)) .GT. T ) GOTO 19
 10      CONTINUE
 19      XLOW = ( ONE + TWO ) * SQRT ( TEST )
      ELSE
         DO 40 NTERM2 = 15 , 0 , -1
            IF ( ABS(ARL0AS(NTERM2)) .GT. T ) GOTO 49
 40      CONTINUE
 49      DO 50 NTERM3 = 23 , 0 , -1
            IF ( ABS(AI0ML0(NTERM3)) .GT. T ) GOTO 59
 50      CONTINUE
 59      XMAX = D1MACH(2)
         XHIGH1 = SQRT ( ( TWENT8 + TWO ) / TEST )
         XHIGH2 = TWENT8 / TEST
      ENDIF
C
C   Code for |xvalue| <= 16
C
      IF ( X .LE. SIXTEN ) THEN
         IF ( X .LT. XLOW ) THEN
            STRVL0 = TWOBPI * X
         ELSE
            T = ( FOUR * X - TWENT4 ) / ( X + TWENT4 )
            STRVL0 = TWOBPI * X * CHEVAL(NTERM1,ARL0,T) * EXP(X)
         ENDIF
      ELSE
C
C   Code for |xvalue| > 16
C
         IF ( X .GT. XHIGH2 ) THEN
            CH1 = ONE
         ELSE
            T = ( X - TWENT8 ) / ( FOUR - X )
            CH1 = CHEVAL(NTERM2,ARL0AS,T)
         ENDIF
         IF ( X .GT. XHIGH1 ) THEN
            CH2 = ONE
         ELSE
            XSQ = X * X
            T = ( ATEHUN - XSQ ) / ( TWO88 + XSQ )
            CH2 = CHEVAL(NTERM3,AI0ML0,T)
         ENDIF
         TEST = LOG(CH1) - LNR2PI - LOG(X)/TWO + X
         IF ( TEST .GT. LOG(XMAX) ) THEN
C***********CALL ERRPRN(FNNAME,ERRMSG)
            STRVL0 = XMAX
         ELSE
            STRVL0 = EXP(TEST) - TWOBPI * CH2 / X
         ENDIF
      ENDIF
      IF ( INDSGN .EQ. -1 ) STRVL0 = -STRVL0
      RETURN
      END

      DOUBLE PRECISION FUNCTION STRVL1(XVALUE)
C
C   DESCRIPTION:
C
C      This function calculates the modified Struve function of
C      order 1, denoted L1(x), defined as the solution of
C
C               x*x*D(Df) + x*Df - (x*x+1)f = 2*x*x/pi
C
C
C   ERROR RETURNS:
C
C      If the value of |XVALUE| is too large, the result
C      would cause an floating-pt overflow. An error message
C      is printed and the function returns the value of
C      sign(XVALUE)*XMAX where XMAX is the largest possible
C      floating-pt argument.
C
C
C   MACHINE-DEPENDENT PARAMETERS:
C
C      NTERM1 - INTEGER - The no. of terms for the array ARL1.
C                         The recommended value is such that
C                             ABS(ARL1(NTERM1)) < EPS/100
C
C      NTERM2 - INTEGER - The no. of terms for the array ARL1AS.
C                         The recommended value is such that
C                             ABS(ARL1AS(NTERM2)) < EPS/100
C
C      NTERM3 - INTEGER - The no. of terms for the array AI1ML1.
C                         The recommended value is such that
C                             ABS(AI1ML1(NTERM3)) < EPS/100
C
C      XLOW1 - DOUBLE PRECISION - The value of x below which L1(x) = 2*x*x/(3*pi)
C                     to machine precision. The recommended value is
C                              SQRT(15*EPS)
C
C      XLOW2 - DOUBLE PRECISION - The value of x below which L1(x) set to 0.0.
C                     This is used to prevent underflow. The
C                     recommended value is
C                              SQRT(5*XMIN)
C
C      XHIGH1 - DOUBLE PRECISION - The value of |x| above which the Chebyshev
C                      series in the asymptotic expansion of I1
C                      equals 1.0 to machine precision. The
C                      recommended value is  SQRT( 30 / EPSNEG ).
C
C      XHIGH2 - DOUBLE PRECISION - The value of |x| above which the Chebyshev
C                      series in the asymptotic expansion of I1 - L1
C                      equals 1.0 to machine precision. The recommended
C                      value is   30 / EPSNEG.
C
C      XMAX - DOUBLE PRECISION - The value of XMAX, where XMAX is the
C                    largest possible floating-pt argument.
C                    This is used to prevent overflow.
C
C      For values of EPS, EPSNEG, XMIN, and XMAX the user should refer
C      to the file MACHCON.TXT
C
C      The machine-dependent constants are computed internally by
C      using the D1MACH subroutine.
C
C
C   INTRINSIC FUNCTIONS USED:
C
C      EXP , LOG , SQRT
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
C          CHEVAL , ERRPRN, D1MACH
C
C
C   AUTHOR:
C          DR. ALLAN J. MACLEOD
C          DEPT. OF MATHEMATICS AND STATISTICS
C          UNIVERSITY OF PAISLEY
C          HIGH ST.
C          PAISLEY
C          SCOTLAND
C          PA1 2BE
C
C          (e-mail: macl_ms0@paisley.ac.uk )
C
C
C   LATEST UPDATE:
C                 23 January, 1996
C
C
      INTEGER NTERM1,NTERM2,NTERM3
      DOUBLE PRECISION ARL1(0:26),ARL1AS(0:16),AI1ML1(0:25),
     1     ATEHUN,CHEVAL,CH1,CH2,FOUR,LNR2PI,
     2     ONE,ONEHUN,PI3BY2,SIXTEN,T,TEST,THIRTY,TWENT4,
     3     TWO,TWOBPI,TWO88,X,XHIGH1,XHIGH2,XLOW1,XLOW2,
     4     XMAX,XVALUE,XSQ,ZERO,D1MACH
C
C   INTRINSIC FUNCTIONS USED:
C
       intrinsic EXP , LOG , SQRT, abs
C
C
C   OTHER MISCFUN SUBROUTINES USED:
C
       external CHEVAL, D1MACH
C
c*****CHARACTER FNNAME*6,ERRMSG*24
c*****DATA FNNAME/'STRVL1'/
c*****DATA ERRMSG/'ARGUMENT CAUSES OVERFLOW'/
      DATA ZERO,ONE,TWO/0.0 D 0 , 1.0 D 0 , 2.0 D 0/
      DATA FOUR,SIXTEN/4.0 D 0 , 16.0 D 0/
      DATA TWENT4,THIRTY/24.0 D 0 , 30.0 D 0/
      DATA ONEHUN/100.0 D 0/
      DATA TWO88,ATEHUN/288.0 D 0 , 800.0 D 0/
      DATA LNR2PI/0.91893 85332 04672 74178 D 0/
      DATA PI3BY2/4.71238 89803 84689 85769 D 0/
      DATA TWOBPI/0.63661 97723 67581 34308 D 0/
      DATA ARL1(0)/  0.38996 02735 12295 38208  D    0/
      DATA ARL1(1)/ -0.33658 09610 19757 49366  D    0/
      DATA ARL1(2)/  0.23012 46791 25016 45616  D    0/
      DATA ARL1(3)/ -0.13121 59400 79608 32327  D    0/
      DATA ARL1(4)/  0.64259 22289 91284 6518   D   -1/
      DATA ARL1(5)/ -0.27500 32950 61663 5833   D   -1/
      DATA ARL1(6)/  0.10402 34148 63720 8871   D   -1/
      DATA ARL1(7)/ -0.35053 22949 36388 080    D   -2/
      DATA ARL1(8)/  0.10574 84984 21439 717    D   -2/
      DATA ARL1(9)/ -0.28609 42640 36665 58     D   -3/
      DATA ARL1(10)/ 0.69257 08785 94220 8      D   -4/
      DATA ARL1(11)/-0.14896 93951 12271 7      D   -4/
      DATA ARL1(12)/ 0.28103 55825 97128        D   -5/
      DATA ARL1(13)/-0.45503 87929 7776         D   -6/
      DATA ARL1(14)/ 0.60901 71561 770          D   -7/
      DATA ARL1(15)/-0.62354 37248 08           D   -8/
      DATA ARL1(16)/ 0.38430 01206 7            D   -9/
      DATA ARL1(17)/ 0.79054 3916               D  -11/
      DATA ARL1(18)/-0.48982 4083               D  -11/
      DATA ARL1(19)/ 0.46356 884                D  -12/
      DATA ARL1(20)/ 0.68420 5                  D  -14/
      DATA ARL1(21)/-0.56974 8                  D  -14/
      DATA ARL1(22)/ 0.35324                    D  -15/
      DATA ARL1(23)/ 0.4244                     D  -16/
      DATA ARL1(24)/-0.644                      D  -17/
      DATA ARL1(25)/-0.21                       D  -18/
      DATA ARL1(26)/ 0.9                        D  -19/
      DATA ARL1AS(0)/  1.97540 37844 16523 56868  D    0/
      DATA ARL1AS(1)/ -0.11951 30555 08829 4181   D   -1/
      DATA ARL1AS(2)/  0.33639 48526 91960 46     D   -3/
      DATA ARL1AS(3)/ -0.10091 15655 48154 9      D   -4/
      DATA ARL1AS(4)/  0.30638 95132 1998         D   -6/
      DATA ARL1AS(5)/ -0.95370 43703 96           D   -8/
      DATA ARL1AS(6)/  0.29524 73555 8            D   -9/
      DATA ARL1AS(7)/ -0.95107 8318               D  -11/
      DATA ARL1AS(8)/  0.28203 667                D  -12/
      DATA ARL1AS(9)/ -0.11341 75                 D  -13/
      DATA ARL1AS(10)/ 0.147                      D  -17/
      DATA ARL1AS(11)/-0.6232                     D  -16/
      DATA ARL1AS(12)/-0.751                      D  -17/
      DATA ARL1AS(13)/-0.17                       D  -18/
      DATA ARL1AS(14)/ 0.51                       D  -18/
      DATA ARL1AS(15)/ 0.23                       D  -18/
      DATA ARL1AS(16)/ 0.5                        D  -19/
      DATA AI1ML1(0)/  1.99679 36189 67891 36501  D    0/
      DATA AI1ML1(1)/ -0.19066 32614 09686 132    D   -2/
      DATA AI1ML1(2)/ -0.36094 62241 01744 81     D   -3/
      DATA AI1ML1(3)/ -0.68418 47304 59982 0      D   -4/
      DATA AI1ML1(4)/ -0.12990 08228 50942 6      D   -4/
      DATA AI1ML1(5)/ -0.24715 21887 05765        D   -5/
      DATA AI1ML1(6)/ -0.47147 83969 1972         D   -6/
      DATA AI1ML1(7)/ -0.90208 19982 592          D   -7/
      DATA AI1ML1(8)/ -0.17304 58637 504          D   -7/
      DATA AI1ML1(9)/ -0.33232 36701 59           D   -8/
      DATA AI1ML1(10)/-0.63736 42173 5            D   -9/
      DATA AI1ML1(11)/-0.12180 23975 6            D   -9/
      DATA AI1ML1(12)/-0.23173 46832              D  -10/
      DATA AI1ML1(13)/-0.43906 8833               D  -11/
      DATA AI1ML1(14)/-0.82847 110                D  -12/
      DATA AI1ML1(15)/-0.15562 249                D  -12/
      DATA AI1ML1(16)/-0.29131 12                 D  -13/
      DATA AI1ML1(17)/-0.54396 5                  D  -14/
      DATA AI1ML1(18)/-0.10117 7                  D  -14/
      DATA AI1ML1(19)/-0.18767                    D  -15/
      DATA AI1ML1(20)/-0.3484                     D  -16/
      DATA AI1ML1(21)/-0.643                      D  -17/
      DATA AI1ML1(22)/-0.118                      D  -17/
      DATA AI1ML1(23)/-0.22                       D  -18/
      DATA AI1ML1(24)/-0.4                        D  -19/
      DATA AI1ML1(25)/-0.1                        D  -19/
C
C   START CALCULATION
C
      X = ABS ( XVALUE )
C
C   Compute the machine-dependent constants.
C
      TEST = D1MACH(3)
      T = TEST / ONEHUN
      IF ( X .LE. SIXTEN ) THEN
         DO 10 NTERM1 = 26 , 0 , -1
            IF ( ABS(ARL1(NTERM1)) .GT. T ) GOTO 19
 10      CONTINUE
 19      XLOW1 = SQRT ( THIRTY * TEST / TWO )
         XLOW2 = SQRT ( (FOUR + ONE) * D1MACH(1) )
      ELSE
         DO 40 NTERM2 = 16 , 0 , -1
            IF ( ABS(ARL1AS(NTERM2)) .GT. T ) GOTO 49
 40      CONTINUE
 49      DO 50 NTERM3 = 25 , 0 , -1
            IF ( ABS(AI1ML1(NTERM3)) .GT. T ) GOTO 59
 50      CONTINUE
 59      XMAX = D1MACH(2)
         XHIGH2 = THIRTY / TEST
         XHIGH1 = SQRT ( XHIGH2 )
      ENDIF
C
C   CODE FOR |XVALUE| <= 16
C
      IF ( X .LE. SIXTEN ) THEN
         IF ( X .LE. XLOW2 ) THEN
            STRVL1 = ZERO
         ELSE
            XSQ = X * X
            IF ( X .LT. XLOW1 ) THEN
               STRVL1 = XSQ / PI3BY2
            ELSE
               T = ( FOUR * X - TWENT4 ) / ( X + TWENT4 )
               STRVL1 = XSQ * CHEVAL(NTERM1,ARL1,T) * EXP(X) / PI3BY2
            ENDIF
         ENDIF
      ELSE
C
C   CODE FOR |XVALUE| > 16
C
         IF ( X .GT. XHIGH2 ) THEN
            CH1 = ONE
         ELSE
            T = ( X - THIRTY ) / ( TWO - X )
            CH1 = CHEVAL(NTERM2,ARL1AS,T)
         ENDIF
         IF ( X .GT. XHIGH1 ) THEN
            CH2 = ONE
         ELSE
            XSQ = X * X
            T = ( ATEHUN - XSQ ) / ( TWO88 + XSQ )
            CH2 = CHEVAL(NTERM3,AI1ML1,T)
         ENDIF
         TEST = LOG(CH1) - LNR2PI - LOG(X)/TWO + X
         IF ( TEST .GT. LOG(XMAX) ) THEN
C***********CALL ERRPRN(FNNAME,ERRMSG)
            STRVL1 = XMAX
         ELSE
            STRVL1 = EXP(TEST) - TWOBPI * CH2
         ENDIF
      ENDIF
      RETURN
      END
