C
C
      SUBROUTINE SVD001 (ISEND, NCMAX, NCOL, NRMAX, NROW, NSVD,
     +                   A, D, TOL,
     +                   ABORT)
C
C ACTION : Return U, D or VT from SVD using BLAS/LAPACK
C AUTHOR : W.G.Bardsley, University of Manchester, U.K.
C          08/01/2004 Derived from SVDVAL 
C
C  ISEND: (input/unchanged)
C         ISEND = 1: Overwrite A by U
C         ISEND = 2: Overwrite A by VT, i.e. V^T
C   NCOL: (input/unchanged) no. columns
C  NCMAX: (input/unchanged) max. no. columns for A
C  NRMAX: (input/unchanged) leading dimension of A
C   NROW: (input/unchanged) no. rows
C   NSVD: (output) no. singular values > TOL*largest
C      A: (input/output) matrix on input then overwritten by U or V^T
C      D: (output) singular values
C    TOL: (input/unchanged) tolerance for singular values
C  ABORT: (output) error indicator
C
C ADVICE:  A should be a(NRMAX,NCMAX) where NCMAX > NCOL
C          LWORK should really be >= NBLOCK*(NCOL + NROW) for optimum results
C          where NBLOCK = 64, i.e., blocksize
C          E = workspace
C          TAUP = workspace
C          TAUQ = workspace
C          VT = workspace and right singular vectors stored by rows
C          WORK = workspace
C
      IMPLICIT   NONE
C
C Arguments
C
      INTEGER    ISEND, NCMAX, NCOL, NRMAX, NROW, NSVD
      DOUBLE PRECISION A(NRMAX,*), D(*), TOL
      LOGICAL    ABORT
C
C Locals
C
      INTEGER    LDA, LDC, LDVT, LWORK
      INTEGER    I, IERR, INFO, J, K, M, N
      INTEGER    N0, N1, N2, NBLOCK, NCC
      PARAMETER (N0 = 0, N1 = 1, N2 = 2, NBLOCK = 64, NCC = 0)
      DOUBLE PRECISION C(N1,N1)
      DOUBLE PRECISION, ALLOCATABLE :: E(:), TAUP(:), TAUQ(:),
     +                                 VT(:,:), WORK(:)
      DOUBLE PRECISION TOL1
      EXTERNAL   DGEBRD, DORGBR, DBDSQR
      INTRINSIC  MIN
C
C Initialise then check
C
      NSVD = N0
      ABORT = .TRUE.
      IF (ISEND.LT.N1 .OR. ISEND.GT.N2) RETURN
      IF (NCOL.LT.N2 .OR. NROW.LT.N2) RETURN
      IF (NCOL.GT.NCMAX .OR. NROW.GT.NRMAX) RETURN
C
C Set the first dimensions
C
      LDA = NRMAX
      LDVT = NRMAX
      LDC = N1
      LWORK = NBLOCK*(NROW + NCOL)
C
C Define M, N and K
C
      M = NROW
      N = NCOL
      K = MIN(M,N)
C
C Create workspaces
C
      IERR = N0
      IF (ALLOCATED(E)) DEALLOCATE(E, STAT = IERR)
      IF (IERR.NE.N0) RETURN
      IF (ALLOCATED(TAUP)) DEALLOCATE(TAUP, STAT = IERR)
      IF (IERR.NE.N0) RETURN
      IF (ALLOCATED(TAUQ)) DEALLOCATE(TAUQ, STAT = IERR)
      IF (IERR.NE.N0) RETURN
      IF (ALLOCATED(VT)) DEALLOCATE(VT, STAT = IERR)
      IF (IERR.NE.N0) RETURN
      IF (ALLOCATED(WORK)) DEALLOCATE(WORK, STAT = IERR)
      IF (IERR.NE.N0) RETURN
      ALLOCATE (E(K), STAT = IERR)
      IF (IERR.NE.N0) RETURN
      ALLOCATE (TAUP(K), STAT = IERR)
      IF (IERR.NE.N0) RETURN
      ALLOCATE (TAUQ(K), STAT = IERR)
      IF (IERR.NE.N0) RETURN
      ALLOCATE (VT(LDVT,NCMAX), STAT = IERR)
      IF (IERR.NE.N0) RETURN
      ALLOCATE (WORK(LWORK), STAT = IERR)
      IF (IERR.NE.N0) RETURN
C
C Reduce A to bidiagonal form
C
      CALL DGEBRD (M, N, A, LDA, D, E, TAUQ, TAUP, WORK, LWORK, INFO)
      IF (INFO.NE.N0) THEN
         DEALLOCATE(E, STAT = IERR)
         DEALLOCATE(TAUP, STAT = IERR)
         DEALLOCATE(TAUQ, STAT = IERR)
         DEALLOCATE(VT, STAT = IERR)
         DEALLOCATE(WORK, STAT = IERR)
         RETURN
      ENDIF
C
C Copy A to VT
C
      DO J = N1, N
         DO I = N1, M
            VT(I,J) = A(I,J)
         ENDDO
      ENDDO
C
C The SVD
C
      IF (M.GE.N) THEN
          NSVD = N
          CALL DORGBR ('P', N, N, M, VT, LDVT, TAUP, WORK, LWORK,
     +                 INFO)
          CALL DORGBR ('Q', M, N, N, A, LDA, TAUQ, WORK, LWORK,
     +                 INFO)
          CALL DBDSQR ('Upper', N, N, M, NCC, D, E, VT, LDVT, A, LDA,
     +                 C, LDC, WORK, INFO)
      ELSE
          NSVD = M
          CALL DORGBR ('P', M, N, M, VT, LDVT, TAUP, WORK, LWORK,
     +                 INFO)
          CALL DORGBR ('Q', M, M, N, A, LDA, TAUQ, WORK, LWORK,
     +                 INFO)
          CALL DBDSQR ('Lower', M, N, M, NCC, D, E, VT, LDVT, A, LDA,
     +                 C, LDC, WORK, INFO)
      ENDIF
      IF (INFO.EQ.N0) THEN
C
C Success so work out the data required
C
         ABORT = .FALSE.
         IF (ISEND.EQ.N2) THEN
            IF (M.GE.N) THEN
               DO J = N1, N
                  DO I = 1, N
                     A(I,J) = VT(I,J)
                  ENDDO
               ENDDO
            ELSE
               DO J = N1, N
                  DO I = N1, M
                     A(I,J) = VT(I,J)
                  ENDDO
               ENDDO
            ENDIF
         ENDIF
         TOL1 = TOL*D(1)
         J = NSVD
         NSVD = N1
         DO I = N2, J
            IF (D(I).GT.TOL1) THEN
               NSVD = NSVD + N1
            ELSE
               RETURN
            ENDIF
         ENDDO
      ENDIF
C
C Destroy local workspaces
C
      DEALLOCATE(E, STAT = IERR)
      DEALLOCATE(TAUP, STAT = IERR)
      DEALLOCATE(TAUQ, STAT = IERR)
      DEALLOCATE(VT, STAT = IERR)
      DEALLOCATE(WORK, STAT = IERR)
      END
C
C
