C
C
      SUBROUTINE SVDV01 (ISEND, LV, M, N, NRANK,
     +                   V, TOL,
     +                   ABORT)
C
C ACTION: SVD of M by N matrix A stored in vector V
C AUTHOR: W.G.Bardsley, University of Manchester, U.K., 17/01/2004
C         03/01/2006 version using ALLOCATE/DEALLOCATE
C
C ISEND: (input/unchanged)
C         ISEND = 0: just calculate the rank and do not alter V
C         ISEND = 1: overwrite V by S (singular values)
C         ISEND = 2: overwrite V by U(short)
C         ISEND = 3: overwrite V by VT(short)
C         ISEND = 4: overwrite V by S then U(short)
C         ISEND = 5: overwrite V by S then VT(short)
C         ISEND = 6: overwrite V by S then U(short) then VT(short)
C         ISEND = 7: overwrite V by U(short) then D*VT(full)
C    LV: (input/unchanged) length of V
C     M: (input/unchanged) no. of rows
C     N: (input/unchanged) no. of columns
C NRANK: (output) rank
C     V: (input/output if ISEND > 0) the matrix by columns
C   TOL: (input/unchanged) tolerance for rank
C ABORT: (output) error indicator
C
C         The structure of V when ISEND > 0 with matrices stored by column is
C         ISEND = 1: S has min(M,N) singular values
C         ISEND = 2: U has M rows by min(M,N) columns
C         ISEND = 3: VT has min(M,N) rows by N columns, etc.
C         So the minimum dimension LV of V is
C         ISEND = 0: M*N
C         ISEND = 1: M*N
C         ISEND = 2: MAX(M*MIN(M,N), M*N)
C         ISEND = 3: MAX(N*MIN(M,N), M*N)
C         ISEND = 4: MAX((M + 1)*MIN(M,N), M*N)
C         ISEND = 5: MAX((N + 1)*MIN(M,N), M*N)
C         ISEND = 6: MAX((M + N + 1)*MIN(M,N), M*N)
C         ISEND = 7: MAX(M*MIN(M,N), M*N) + N*N
C
C         NCADD and NRADD add borders as required by other LAPACK routines
C
      IMPLICIT   NONE
C
C Arguments
C
      INTEGER    ISEND, LV, M, N, NRANK
      DOUBLE PRECISION V(LV), TOL
      LOGICAL    ABORT
C
C Locals
C
      INTEGER    I, IERR, J, JSEND, K, MN, MINMN
      INTEGER    NCMAX, NRMAX
      INTEGER    NCADD, NRADD
      PARAMETER (NCADD = 2, NRADD = 1)
      DOUBLE PRECISION, ALLOCATABLE :: A(:,:), U(:,:), VT(:,:), S(:)
      DOUBLE PRECISION SVDLIM, TOL1, DLAMCH
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO = 0.0D+00)
      LOGICAL    FIRST
      EXTERNAL   SVD000, DLAMCH
      INTRINSIC  SQRT, MAX, MIN
      SAVE       FIRST, TOL1
      DATA       FIRST / .TRUE. /
C
C Initialise ABORT then check LV, M and N
C
      ABORT = .TRUE.
      MN = M*N
      IF (M.LT.2 .OR. N.LT.2 .OR. LV.LT.MN) RETURN
C
C Check ISEND and LV
C
      MINMN = MIN(M,N)
      IF (ISEND.LT.0) THEN
         RETURN
      ELSEIF (ISEND.EQ.0) THEN
         K = MN
      ELSEIF (ISEND.EQ.1) THEN
         K = MN
      ELSEIF (ISEND.EQ.2) THEN
         K = M*MINMN
      ELSEIF (ISEND.EQ.3) THEN
         K = N*MINMN
      ELSEIF (ISEND.EQ.4) THEN
         K = (M + 1)*MINMN
      ELSEIF (ISEND.EQ.5) THEN
         K = (N + 1)*MINMN
      ELSEIF (ISEND.EQ.6) THEN
         K = (M + N + 1)*MINMN
      ELSEIF (ISEND.EQ.7) THEN
         K = M*MINMN + N*N
      ELSE
         RETURN
      ENDIF
      IF (LV.LT.K) RETURN
C
C Initialise then store TOL1
C
      IF (FIRST) THEN
         FIRST = .FALSE.
         TOL1 = SQRT(DLAMCH('E'))
      ENDIF
C
C Create workspaces
C
      IERR = 0
      IF (ALLOCATED(A)) DEALLOCATE(A, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(U)) DEALLOCATE(U, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(VT)) DEALLOCATE(VT, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(S)) DEALLOCATE(S, STAT = IERR)
      IF (IERR.NE.0) RETURN
C
C Make arrays larger than necessary to avoid crash in DGEMV
C
      NCMAX = N + NCADD
      NRMAX = M + NRADD
      ALLOCATE(A(NRMAX,NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(U(NRMAX,NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(VT(NRMAX,NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(S(NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
C
C Copy V into A
C
      K = 0
      DO J = 1, N
         DO I = 1, M
            K = K + 1
            A(I,J) = V(K)
         ENDDO
      ENDDO
C
C Initialise JSEND
C
      IF (ISEND.EQ.0) THEN
         JSEND = 1
      ELSEIF (ISEND.EQ.1) THEN
         JSEND = 1
      ELSEIF (ISEND.EQ.2) THEN
         JSEND = 2
      ELSEIF (ISEND.EQ.3) THEN
         JSEND = 3
      ELSEIF (ISEND.EQ.4) THEN
         JSEND = 2
      ELSEIF (ISEND.EQ.5) THEN
         JSEND = 3
      ELSEIF (ISEND.EQ.6) THEN
         JSEND = 4
      ELSEIF (ISEND.EQ.7) THEN
         JSEND = 6
      ENDIF
C
C Perform SVD
C
      CALL SVD000 (JSEND, NRMAX, NRMAX, NRMAX, M, N,
     +             A, S, U, VT)
      IF (JSEND.NE.0) RETURN
C
C Success so set ABORT = .FALSE.
C
      ABORT = .FALSE.
C
C Work out the rank
C
      NRANK = 0
      IF (S(1).GT.ZERO) THEN
         NRANK = 1
         IF (MINMN.GT.1) THEN
            SVDLIM = MAX(TOL,TOL1)*S(1)
            DO I = 2, MINMN
               IF (S(I).GE.SVDLIM) NRANK = NRANK + 1
            ENDDO
         ENDIF
      ENDIF
      IF (ISEND.EQ.1) THEN
C
C copy S into V
C
         DO I = 1, MINMN
            V(I) = S(I)
         ENDDO
      ELSEIF (ISEND.EQ.2) THEN
C
C Copy U into V
C
         K = 0
         DO J = 1, MINMN
            DO I = 1, M
               K = K + 1
               V(K) = U(I,J)
            ENDDO
         ENDDO
      ELSEIF (ISEND.EQ.3) THEN
C
C Copy VT into V
C
         K = 0
         DO J = 1, N
            DO I = 1, MINMN
               K = K + 1
               V(K) = VT(I,J)
            ENDDO
         ENDDO
      ELSEIF (ISEND.EQ.4) THEN
C
C copy S into V
C
         K = 0
         DO I = 1, MINMN
            K = K + 1
            V(I) = S(I)
         ENDDO
C
C Copy U into V
C
         DO J = 1, MINMN
            DO I = 1, M
               K = K + 1
               V(K) = U(I,J)
            ENDDO
         ENDDO
      ELSEIF (ISEND.EQ.5) THEN
C
C copy S into V
C
         K = 0
         DO I = 1, MINMN
            K = K + 1
            V(I) = S(I)
         ENDDO
C
C Copy VT into V
C
         DO J = 1, N
            DO I = 1, MINMN
               K = K + 1
               V(K) = VT(I,J)
            ENDDO
         ENDDO
      ELSEIF (ISEND.EQ.6) THEN
C
C copy S into V
C
         K = 0
         DO I = 1, MINMN
            K = K + 1
            V(I) = S(I)
         ENDDO
C
C Copy U into V
C
         DO J = 1, MINMN
            DO I = 1, M
               K = K + 1
               V(K) = U(I,J)
            ENDDO
         ENDDO
C
C Copy VT into V
C
         DO J = 1, N
            DO I = 1, MINMN
               K = K + 1
               V(K) = VT(I,J)
            ENDDO
         ENDDO
      ELSEIF (ISEND.EQ.7) THEN
C
C Copy U into V
C
         K = 0
         DO J = 1, MINMN
            DO I = 1, M
               K = K + 1
               V(K) = U(I,J)
            ENDDO
         ENDDO
C
C Work out D*V^T
C
         DO I = 1, N
            DO J = 1, N
               VT(I,J) = S(I)*VT(I,J)
            ENDDO
         ENDDO
C
C Store D*V^T
C
         DO J = 1, N
            DO I = 1, N
               K = K + 1
               V(K) = VT(I,J)
            ENDDO
         ENDDO
      ENDIF
C
C Empty the workspaces
C
      DEALLOCATE(A, STAT = IERR)
      DEALLOCATE(U, STAT = IERR)
      DEALLOCATE(VT, STAT = IERR)
      DEALLOCATE(S, STAT = IERR)
      END
C
C
