C
C
      SUBROUTINE ATAINV (IFAIL, IRANK, LDA, LDC, LDW, NCOLS, NROWS,
     +                   A, COV, H, SIGMA, W)
C
C ACTION: Return leverages and covariance matrix after a call to DGELSS.
C         This routine must be called immediately after a call to DGELSS
C         A = the A matrix originally supplied to DGELSS
C         IRANK = IRANK returned by DGELSS
C         SIGMA = SVD returned by DGELSS
C         W = A returned from DGELSS, i.e.
C         W contains V(transpose) in columns 1 to NPAR
C AUTHOR: W.G.Bardsley, University of Manchester, U.K.
C         Derived from GLIMCV 08/08/2000
C
C Conventions:
C ============
C A = U*Sigma*V(transpose)
C U = A*V*Sigma(inverse)
C A*(A(transpose)*A)(inverse)*A(transpose) = U*U(transpose)
C (A(transpose)*A)(inverse) = V*Sigma(squared-inverse)*V(transpose)
C
C Arguments
C =========
C IFAIL: [output] returned as 0 if no errors, o/w > 0
C IRANK: [input] rank from DGELSS SVD
C LDA  : [input] leading dimension of A
C LDC  : [input] leading dimension of COV
C LDW  : [input] leading dimension of W
C NCOLS: [input] no. of columns of A = no. parameters
C NROWS: [input] no. of rows of A = no. of observations
C A    : [input] matrix originally supplied to DGELSS
C COV  : [output] used as workspace then returned with (A(transpose)*A)(inverse)
C H    : [output] leverages
C SIGMA: [input] singular values returned by DGELSS
C W    : [input/output]
C        On input the right singular vectors from DGELSS are stored row-wise
C        in columns 1 to NPAR (i.e. W actually contains V(transpose))
C
      IMPLICIT NONE
C
C Arguments supplied
C
      INTEGER  IFAIL, IRANK, LDA, LDC, LDW, NCOLS, NROWS
      DOUBLE PRECISION A(LDA,NCOLS), COV(LDC,NCOLS), H(NROWS),
     +                 SIGMA(IRANK), W(LDW,NCOLS)
C
C Local variables
C
      INTEGER I, J, K
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO = 0.0D+00)
C
C Is it safe
C
      IFAIL = 0
      IF (IRANK.LE.0 .OR. IRANK.GT.NCOLS .OR. NCOLS.LT.1) THEN
         IFAIL = 1
         RETURN
      ENDIF
      IF (NROWS.LE.NCOLS) THEN
         IFAIL = 2
         RETURN
      ENDIF
      IF (NROWS.GT.LDA .OR. NROWS.GT.LDC .OR. NROWS.GT.LDW) THEN
         IFAIL = 3
         RETURN
      ENDIF
C
C Form U = A*V*Sigma(inverse)
C
      DO I = 1, NROWS
         DO J = 1, IRANK
            COV(I,J) = ZERO
            DO K = 1, NCOLS
               COV(I,J) = COV(I,J) + A(I,K)*W(J,K)/SIGMA(J)
            ENDDO
         ENDDO
      ENDDO
C
C Form H(i) = Diag(U*U(transpose))
C
      DO I = 1, NROWS
         H(I) = ZERO
         DO J = 1, IRANK
            H(I) = H(I) + COV(I,J)*COV(I,J)
         ENDDO
      ENDDO
C
C Form U = V*Sigma(inverse-squared)*V(transpose)
C
      DO I = 1, NCOLS
         DO J = 1, NCOLS
            COV(I,J) = ZERO
            DO K = 1, IRANK
               COV(I,J) = COV(I,J) + W(K,I)*W(K,J)/(SIGMA(K)**2)
            ENDDO
         ENDDO
      ENDDO
      END
C
C
