
C
C CDF_INVERT
C CDF_G05ECF$ 
C CDF_G05EDF$ 
C CDF_G05EYF$ 
C
      SUBROUTINE CDF_INVERT (ISEND, I1, I2, N, 
     +                       P, PVAL, T) 
C
C ACTION: Invert a binomial or Poisson distribution
C AUTHOR: w.g.bardsley, university of manchester, u.k., 01/12/2012
C
C ISEND: as follows 
C        ISEND = 1: set up a binomial reference vector
C        ISEND = 2: set up a Poisson reference vector
C              o/w: return I1 and I2 or -1, -1 if an error has happened
C    I1: integer locating highest level in CDF <= PVAL
C    I2: integer locating lowest  level in CDF >= PVAL
C     N: binomial parameter
C     P: binomial parameter
C  PVAL: specified CDF level
C     T: Poisson parameter  
C  
C This code is based on the Simfit G05 routines but it will work with
C the NAG library because it does not call the actual NAG G05 routines 
C   
      IMPLICIT NONE
C
C Arguments
C  
      INTEGER,          INTENT (IN)  :: ISEND, N
      INTEGER,          INTENT (OUT) :: I1, I2
      DOUBLE PRECISION, INTENT (IN)  :: P, PVAL, T  
C
C Locals
C      
      INTEGER    IFAIL, IVAL
      INTEGER    NR
      PARAMETER (NR = 1000)
      INTEGER    CDF_G05EYF$
      DOUBLE PRECISION R(NR)
      DOUBLE PRECISION ZERO, ONE
      PARAMETER (ZERO = 0.0D+00, ONE = 1.0D+00)
      LOGICAL    READY
      EXTERNAL   CDF_G05ECF$, CDF_G05EDF$, CDF_G05EYF$
      SAVE       R
      SAVE       READY
      DATA       READY / .FALSE. /
C
C Initialise I1 and I2
C      
      I1 = -1
      I2 = -1
      IF (ISEND.EQ.1) THEN
C
C ISEND = 1: Set up a binomial reference vector
C        
         IF (N.LT.1 .OR. P.LT.ZERO .OR. P.GT.ONE) THEN
            READY = .FALSE.
         ELSE    
            CALL CDF_G05EDF$(N, P, R, NR, IFAIL)
            IF (IFAIL.EQ.0) READY = .TRUE.
         ENDIF     
      ELSEIF (ISEND.EQ.2) THEN 
C
C ISEND = 2: Set up a Poisson reference vector
C      
         IF (T.LT.ZERO) THEN
            READY = .FALSE.
         ELSE     
            CALL CDF_G05ECF$(T, R, NR, IFAIL)
            IF (IFAIL.EQ.0) READY = .TRUE.
         ENDIF     
      ELSEIF (READY) THEN
C
C ISEND not 1 or 2: return I1 and I2 
C
      
         IF (PVAL.GE.ZERO .AND. PVAL.LE.ONE) THEN
            IVAL =  CDF_G05EYF$(PVAL, R, NR)
            IF (IVAL.EQ.0) THEN
               I1 = 0
               I2 = 1
            ELSE
               I1 = IVAL - 1
               I2 = IVAL
            ENDIF      
         ENDIF
      ENDIF
      END     
C
C--------------------------------------------------------------------
C
      SUBROUTINE CDF_G05ECF$(T, R, NR, IFAIL)
C
C ACTION : Set up a Poisson reference vector for the Simfit package
C          This does not correspond to the NAG method but it works
C          OK for small N
C AUTHOR : W.G.Bardsley, University of Manchester, U.K, 10/5/97
C
C Note: This version just sets up start/stop points and CDF values as follows:
C       R(1) = NSTART = lowest  reasonable value for a random Poisson variable
C       R(2) =  NSTOP = highest reasonable value for a random Poisson variable 
C       with R(3) to R(NSTOP - NSTART + 3) set to corresponding CDF values. 
C       This version will only work with the corresponding Simfit version of G05EYF$
C   
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)    :: NR
      INTEGER,          INTENT (INOUT) :: IFAIL
      DOUBLE PRECISION, INTENT (IN)    :: T
      DOUBLE PRECISION, INTENT (OUT)   :: R(NR)
C
C Locals
C      
      INTEGER    NMAX
      PARAMETER (NMAX = 100000)
      INTEGER    N0, N1, N2, N3, N4
      PARAMETER (N0 = 0, N1 = 1, N2 = 2, N3 = 3, N4 = 4)
      INTEGER    I, J, K, L, M, NSTART, NSTOP
      DOUBLE PRECISION ZERO, ONE, PBOT, PTOP, Q
      PARAMETER (ZERO = 0.0D+00, ONE = 1.0D+00, PBOT = 1.0D-06,
     +           PTOP = ONE - PBOT, Q = 7.0D+00)
      DOUBLE PRECISION TEMP, PEQK, PGTK, PLEK
      EXTERNAL   G01BKF$
      INTRINSIC  DBLE, NINT, SQRT, MAX
C
C Is it safe ?
C
      IFAIL = N0
      IF (T.LT.ZERO) THEN
         IFAIL = N1
         RETURN
       ENDIF
C
C Initialise
C
      TEMP = MAX(T,PBOT)
      NSTART = - N1
      NSTOP = - N1
      M = NINT(TEMP - Q*SQRT(TEMP))
      IF (M.LE.N0) THEN
         M = N0
         K = - N1
      ELSE   
        K = M - 1
      ENDIF
      L = - N1
C
C Loop until NSTART and NSTOP are located
C
      SEARCH_LOOP : DO I = M, NMAX
         J = N0
         K = K + N1
         IF (NSTART.LT.N0) THEN
C
C Executed only until NSTART is assigned
C
            CALL G01BKF$(TEMP, K, PLEK, PGTK, PEQK, J)
            IF (PLEK.GE.PBOT) THEN
               NSTART = K
               L = N3
               R(L) = PLEK
            ENDIF
         ELSEIF (NSTOP.LT.N0) THEN
C
C Executed until NSTOP is assigned
C
            CALL G01BKF$(TEMP, K, PLEK, PGTK, PEQK, J)
            L = L + N1
            IF (L.GT.NR - N4) THEN
               IFAIL = N2
               RETURN
            ENDIF   
            R(L) = PLEK
            IF (PLEK.GE.PTOP) THEN
               NSTOP = K 
               R(L) = ONE
               EXIT SEARCH_LOOP
            ENDIF
         ENDIF
      ENDDO SEARCH_LOOP
C
C Finally assign R(1) and R(2)
C
      IF (NSTART.LT.N0) NSTART = N0
      IF (NSTOP.LE.NSTART) NSTOP = NSTART + 1
      R(N1) = DBLE(NSTART)
      R(N2) = DBLE(NSTOP)
      END
C
C -------------------------------------------------------------------------
C
      SUBROUTINE CDF_G05EDF$(N, P, R, NR, IFAIL)
C
C ACTION : Set up a binomial reference vector for the Simfit package
C          This does not correspond to the NAG method but it works
C          OK for fairly small N as explained below.
C AUTHOR : W.G.Bardsley, University of Manchester, U.K, 10/5/97
C          28/09/2012 extensively revised          
C
C Note: this version does not exploit symmetry to allow for differing action if 
C       P =< 0.5 or P > 0.5 but sets up start/stop points and CDF values as follows:
C       R(1) = NSTART = lowest  reasonable value for a random binomial variable
C       R(2) =  NSTOP = highest reasonable value for a random binomial variable 
C       with R(3) to R(NSTOP - NSTART + 3) set to corresponding CDF values. 
C       This version will only work with the corresponding Simfit version of G05EYF$
C       Caution is required if I decide to restore symmetry without editing G05EYF$
C           
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)    :: N, NR
      INTEGER,          INTENT (INOUT) :: IFAIL
      DOUBLE PRECISION, INTENT (IN)    :: P
      DOUBLE PRECISION, INTENT (OUT)   :: R(NR)
C
C Locals
C      
      INTEGER    N0, N1, N2, N3, N4, N20
      PARAMETER (N0 = 0, N1 = 1, N2 = 2, N3 = 3, N4 = 4, N20 = 20)
      INTEGER    I, J, K, L, M, NSTART, NSTOP, NTEMP
      DOUBLE PRECISION DN
      DOUBLE PRECISION ZERO, ONE, PBOT, PTOP, Q
      PARAMETER (ZERO = 0.0D+00, ONE = 1.0D+00, PBOT = 1.0D-06,
     +           PTOP = ONE - PBOT, Q = 7.0D+00)
      DOUBLE PRECISION PTEMP, PEQK, PGTK, PLEK
      EXTERNAL   G01BJF$
      INTRINSIC  DBLE, NINT, SQRT
C
C Is it safe ?
C
      IFAIL = N0
      IF (N.LT.N1) THEN
         IFAIL = N1
         RETURN
       ENDIF
       IF (P.LT.ZERO .OR. P.GT.ONE) THEN
          IFAIL = N2
          RETURN
       ENDIF
       IF (NR.LT.N4) THEN
          IFAIL = N3
          RETURN
       ENDIF
C
C Initialise and define PTEMP and NTEMP
C
      PTEMP = P
      IF (PTEMP.LT.PBOT) THEN
         PTEMP = PBOT
      ELSEIF (PTEMP.GT.PTOP) THEN
         PTEMP = PTOP
      ENDIF  
      NTEMP = N    
      IF (N.LE.N20) THEN
C
C N =< 20 so set up a full vector of N + 1 values
C        
         IF (NR.LT.N + N3) THEN
            IFAIL = N3
            RETURN
         ENDIF
         NSTART = N0
         NSTOP = N
         R(1) = NSTART
         R(2) = NSTOP
         L = N2
         DO I = NSTART, NSTOP - N1
            J = N0
            K = I
            CALL G01BJF$(NTEMP, PTEMP, K, PLEK, PGTK, PEQK, J)
            L = L + N1
            R(L) = PLEK
         ENDDO    
         L = L + N1
         R(L) = ONE  
      ELSE  
C
C N > 20 so work out a reasonable span
C          
         NSTART = - N1
         NSTOP = - N1
         DN = DBLE(N)
C
C Use mean and variance to guess a sensible starting point
C         
         M = NINT(DN*P - Q*SQRT(DN*P*(ONE - P)))
         IF (M.LT.N0) THEN
            M = N0
         ELSEIF (M.GT.N - N20) THEN
            M = N - N20
         ENDIF        
         K = M - N1
C
C Loop until NSTART and NSTOP are located
C
         SETUP_LOOP : DO I = M, N
            J = N0
            K = K + N1
            IF (NSTART.LT.N0) THEN
C
C Executed only until NSTART is assigned
C
               CALL G01BJF$(NTEMP, PTEMP, K, PLEK, PGTK, PEQK, J)
               IF (PLEK.GE.PBOT) THEN
                  NSTART = K
                  L = N3
                  R(L) = PLEK
               ENDIF
            ELSEIF (NSTOP.LT.N0) THEN
C
C Executed until NSTOP is assigned
C
               CALL G01BJF$(NTEMP, PTEMP, K, PLEK, PGTK, PEQK, J)
               L = L + N1
               IF (NR.LT.L + N1) THEN
                  IFAIL = N3
                  RETURN
               ENDIF
               R(L) = PLEK
               IF (PLEK.GE.PTOP) THEN
                  NSTOP = K
                  R(L) = ONE
                  EXIT SETUP_LOOP
              ENDIF   
            ENDIF
         ENDDO SETUP_LOOP
C
C Finally assign R(1) and R(2) and make sure NSTART >= 0, NSTOP =< N
C
         IF (NSTART.LT.N0) NSTART = N0
         IF (NSTOP.GT.N) NSTOP = N
         R(N1) = DBLE(NSTART)
         R(N2) = DBLE(NSTOP)
      ENDIF
      END
C
C--------------------------------------------------------------------------
C
      INTEGER FUNCTION CDF_G05EYF$(P, R, NR)
C
C ACTION : Integer random variable using range and CDF values in R 
C AUTHOR : W.G.Bardsley, University of Manchester, U.K., 10/5/97
C          28/09/2012 revised to use bisection
C
C Note: This does not correspond to the NAG method but it works
C       OK for fairly small N as now explained.
C       It must be set up by G05ECF$ and G05EDF$ as follows. 
C       R(1) = starting value,
C       R(2) = stopping value
C       R(3) to R(NSTOP - NSTART + 3) the CDF values 
C       It simply uses bisection to locate the correct CDF values so
C       it could be improved for large N by using a guesswork start
C       routine instead of simple bisection. However, the way that
C       the Simfit routines G05ECF$ and G05EDF$ work means that the
C       table to be searched should never be much longer than about 20. 
C
      IMPLICIT  NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: NR
      DOUBLE PRECISION, INTENT (IN) :: P, R(NR)
C
C Locals
C      
      INTEGER   K, M, NSTART, NSTOP
      INTRINSIC NINT
      EXTERNAL  BISECT
C
C Retrieve NSTART and NSTOP
C
      NSTART = NINT(R(1))
      NSTOP = NINT(R(2))
C
C Search the available range of CDF values
C      
      K = NSTOP - NSTART + 1
      CALL BISECT (K, M,
     +             R(3), P) 
C
C Add the offset and make sure values cannot be negative
C         
      CDF_G05EYF$ = NSTART + M   
      IF (CDF_G05EYF$.LT.0) CDF_G05EYF$ = 0
      END
C
C



