C
C
      SUBROUTINE COCHRQ (LWORK, NCMAX, NCOL, NIN, NOUT, NRMAX, NROW,
     +                   A, B,
     +                   TITLE,
     +                   SUPPLY)
C
C ACTION: Cochran Q test. Based on Zar, Biostatistical Analysis 3rd edn. page 272-274
C AUTHOR: W.G.Bardsley, University of Manchester, U.K., 12/12/99
C         07/02/2001 added CHOP80
C         27/09/2002 replaced patch1 by table1
C         12/02/2006 added NCOL, NROW, TITLE, and SUPPLY to arguments
C         27/10/2013 added INTENTS, CIPHER_1, and CIPHER_2
C         28/07/2021 added E_NUMBERS and E_NUMBERS etc.
C
      IMPLICIT   NONE
C
C Arguments
C
      INTEGER,             INTENT (IN)    :: NCMAX, NIN, NOUT, NRMAX
      INTEGER,             INTENT (INOUT) :: NCOL, NROW
      INTEGER,             INTENT (OUT)   :: LWORK(NRMAX,NCMAX)
      DOUBLE PRECISION,    INTENT (INOUT) :: A(NRMAX,NCMAX), B(NRMAX)
      CHARACTER (LEN = *), INTENT (INOUT) :: TITLE
      LOGICAL,             INTENT (IN)    :: SUPPLY
C
C Locals
C
      INTEGER    ICOUNT, IFAIL
      INTEGER    I, J, K, L, M
      INTEGER    NA, NB, NBJ, NBJ2, NGI, NGI2, NSUM
      INTEGER    ISEND, N0, N1, N3, N4, N24
      PARAMETER (ISEND = 0, N0 = 0, N1 = 1, N3 = 3, N4 = 4, N24 = 24)
      INTEGER    ICOLOR, NUMTXT
      PARAMETER (NUMTXT = 11)
      DOUBLE PRECISION BOT, DF, P, P95, P99, Q, TOP
      DOUBLE PRECISION G01ECF$, G01FCF$
      DOUBLE PRECISION PNT95, PNT99
      PARAMETER (PNT95 = 0.95D+00, PNT99 = 0.99D+00)
      CHARACTER (LEN = 1024) FNAME
      CHARACTER (LEN = 100 ) LINE, TEXT(NUMTXT)
      CHARACTER (LEN = 80  ) CHOP80
      CHARACTER (LEN = 40  ) CIPHER_1, CIPHER_2
      CHARACTER (LEN = 25  ) SYMBOL
      CHARACTER (LEN = 13  ) D13(3), SHOWLJ
      CHARACTER (LEN = 12  ) I12(2), FORM12 
      CHARACTER (LEN = 1   ) BLANK, TAIL
      PARAMETER (BLANK = ' ', TAIL = 'U')
      LOGICAL    E_FORMATS, E_NUMBERS
      LOGICAL    ABORT, CHECK
      LOGICAL    FIXCOL, FIXROW, LABEL
      PARAMETER (FIXCOL = .FALSE., FIXROW = .FALSE., LABEL = .TRUE.)
      EXTERNAL   E_FORMATS, FORM12, SHOWLJ
      EXTERNAL   MATTIN, PUTFAT, PUTWAR, TABLE1, PUTIFA, PLEVEL, CHOP80,
     +           PUTADV
      EXTERNAL   G01ECF$, G01FCF$
      INTRINSIC  NINT, DBLE
      SAVE       ICOUNT
      DATA       ICOUNT / 0 /
C
C Get data
C
      IF (.NOT.SUPPLY) THEN
         CALL PUTADV ('Now input data formatted like cochranq.tf1')
         CLOSE (UNIT = NIN)
         CALL MATTIN (ISEND, NCMAX, NCOL, NIN, NRMAX, NROW,
     +                A, B,
     +                FNAME, TITLE,
     +                ABORT, FIXCOL, FIXROW, LABEL)
         CLOSE (UNIT = NIN)
         IF (ABORT) RETURN
      ENDIF
C
C Check that the matrix is not too large (1 extra row and column are required)
C
      IF (NROW.EQ.NRMAX .OR. NCOL.EQ.NCMAX) THEN
         CALL PUTFAT ('Max. dimension exceeded in call to COCHRQ')
         RETURN
      ENDIF
C
C Copy matrix into the integer workspace
C
      E_NUMBERS = E_FORMATS()
      DO I = N1, NCOL
         DO J = N1, NROW
            LWORK(J,I) = NINT(A(J,I))
         ENDDO
      ENDDO
C
C Check if column 1 = succesive integers
C
      CHECK = .TRUE.
      I = N0
      DO WHILE (CHECK .AND. I.LT.NROW)
         I = I + N1
         IF (LWORK(I,N1).NE.I) CHECK = .FALSE.
      ENDDO
C
C If so then shift columns to left
C
      IF (CHECK) THEN
         NCOL = NCOL - N1
         DO I = N1, NCOL
            K = I + N1
            DO J = N1, NROW
               LWORK(J,I) = LWORK(J,K)
            ENDDO
         ENDDO
      ENDIF
C
C Check that only 0 or 1 occurs in the integer matrix
C
      DO I = N1, NCOL
         DO J = N1, NROW
            K = LWORK(J,I)
            IF (K.LT.N0 .OR. K.GT.N1) THEN
               WRITE (LINE,100) J, I
               CALL PUTFAT (LINE)
               RETURN
            ENDIF
         ENDDO
      ENDDO
C
C Check that sample size is not too small
C
      IF (NCOL.LT.N3 .OR. NROW.LT.N3) THEN
         CALL PUTFAT ('Sample size too small for a meaningful test')
         RETURN
      ENDIF
C
C Work out row sums
C
      K = NCOL + N1
      DO I = N1, NROW
         NSUM = N0
         DO J = N1, NCOL
            NSUM = NSUM + LWORK(I,J)
         ENDDO
         LWORK(I,K) = NSUM
      ENDDO
C
C Calculate NB, NBJ and NBJ2
C
      NB = N0
      NBJ = N0
      NBJ2 = N0
      K = NCOL + N1
      L = NROW + N1
      DO I = N1, NCOL
         LWORK(L,I) = N0
      ENDDO
      DO I = N1, NROW
         J = LWORK(I,K)
         IF (J.NE.N0 .AND. J.NE.NCOL) THEN
            NB = NB + N1
            NBJ = NBJ + J
            NBJ2 = NBJ2 + J**2
            DO M = N1, NCOL
               LWORK(L,M) = LWORK(L,M) + LWORK(I,M)
            ENDDO
         ENDIF
      ENDDO
C
C Calculate NA, NGI and NGI2
C
      NA = NCOL
      NGI = N0
      NGI2 = N0
      L = NROW + N1
      DO I = N1, NCOL
         NGI = NGI + LWORK(L,I)
         NGI2 = NGI2 + LWORK(L,I)**2
      ENDDO
C
C Check for equal row and column sums
C
      IF (NGI.NE.NBJ) THEN
         CALL PUTFAT ('Row and column sums unequal in COCHRQ')
         RETURN
      ENDIF
C
C Check on sample size
C
      IF (NA.LT.N4 .OR. NA*NB.LT.N24) THEN
         CALL PUTWAR ('Should have no.rows >= 4, no.rows*no.cols >= 24')
      ENDIF
      CIPHER_1 = BLANK
      CIPHER_2 = BLANK
      IF (NB.LT.NROW) WRITE (CIPHER_1,200) NROW - NB
      IF (CHECK) WRITE (CIPHER_2,300)
C
C Increment ICOUNT then calculate Q
C
      ICOUNT = ICOUNT + N1
      TOP = DBLE(NA - N1)*(DBLE(NGI2) - DBLE(NGI)**2/DBLE(NA))
      BOT = DBLE(NBJ) - DBLE(NBJ2)/DBLE(NA)
      Q = TOP/BOT
      IFAIL = N0
      DF = DBLE(NA - N1)
      P = G01ECF$(TAIL, Q, DF, IFAIL)
      CALL PUTIFA (IFAIL, NOUT, 'G01ECF/COCHRQ')
      CALL PLEVEL (P, SYMBOL)
      IFAIL = 0
      P95 = G01FCF$(PNT95, DF, IFAIL)
      CALL PUTIFA (IFAIL, NOUT, 'G01FCF/COCHRQ')
      IFAIL = 0
      P99 = G01FCF$(PNT99, DF, IFAIL)
      CALL PUTIFA (IFAIL, NOUT, 'G01FCF/COCHRQ')
      IF (E_NUMBERS) THEN
         WRITE (TEXT,400) ICOUNT, CHOP80(TITLE), NB, CIPHER_1, NA,
     +                    CIPHER_2, Q, P, SYMBOL, P95, P99
         WRITE (NOUT,500) ICOUNT, TITLE, NB, CIPHER_1, NA,
     +                    CIPHER_2, Q, P, SYMBOL, P95, P99
      ELSE
         I12(1) = FORM12(NB)
         I12(2) = FORM12(NA)
         D13(1) = SHOWLJ(Q)
         D13(2) = SHOWLJ(P95)
         D13(3) = SHOWLJ(P99)
         WRITE (TEXT,450) ICOUNT, CHOP80(TITLE), TRIM(I12(1)), CIPHER_1,
     +                    TRIM(I12(2)), CIPHER_2, D13(1), P, SYMBOL, 
     +                    D13(2), D13(3)
         WRITE (NOUT,550) ICOUNT, TITLE, TRIM(I12(1)), CIPHER_1,
     +                    TRIM(I12(2)), CIPHER_2, D13(1), P, SYMBOL, 
     +                    D13(2), D13(3) 
      ENDIF  
      ICOLOR = 15
      CALL TABLE1 (ICOLOR, 'OPEN')
      DO I = 1, NUMTXT
         IF (I.EQ.1 .OR. I.EQ.4) THEN
            ICOLOR = 4
         ELSE
            ICOLOR = 0
         ENDIF
         CALL TABLE1 (ICOLOR, TEXT(I))
      ENDDO
      CALL TABLE1 (ICOLOR, 'CLOSE')
C
C Format statements
C      
  100 FORMAT ('Data value at row',I4,', column',I4,' not 0 or 1')
  200 FORMAT (', Rows suppressed:',I3,' (all 0 or all 1)')
  300 FORMAT (', Cols suppressed:  1 (not data)') 
  400 FORMAT (
     + 'Cochran Q test',I4
     +/
     +/'Data:'
     +/A
     +/
     +/'Number of blocks (rows) =',I4,A
     +/'Number of groups (cols) =',I4,A
     +/'Cochran Q value         =',1P,E12.5
     +/'p = P(chi-sq. >= Q)     =',0P,F7.4,1X,A
     +/'95% chi-sq. point       =',1P,E12.5
     +/'99% chi-sq. point       =',   E12.5)
  450 FORMAT (
     + 'Cochran Q test',I4
     +/
     +/'Data:'
     +/A
     +/
     +/'Number of blocks (rows) =',1X,A,A
     +/'Number of groups (cols) =',1X,A,A
     +/'Cochran Q value         =',1X,A
     +/'p = P(chi-sq. >= Q)     =',F7.4,1X,A
     +/'95% chi-sq. point       =',1X,A
     +/'99% chi-sq. point       =',1X,A)     
  500 FORMAT (
     +/
     +/'***'
     +/
     +/' Cochran Q test',I4
     +/' ------------------'
     +/' Data: ',A
     +/
     +/' Number of blocks (rows) =',I4,A
     +/' Number of groups (cols) =',I4,A
     +/' Cochran Q value         =',1P,E12.5
     +/' p = P(chi-sq. >= Q)     =',0P,F7.4,1X,A
     +/' 95% chi-sq. point       =',1P,E12.5
     +/' 99% chi-sq. point       =',   E12.5)
  550 FORMAT (
     +/
     +/'***'
     +/
     +/' Cochran Q test',I4
     +/' ------------------'
     +/' Data: ',A
     +/
     +/' Number of blocks (rows) =',1X,A,1X,A
     +/' Number of groups (cols) =',1X,A,1X,A
     +/' Cochran Q value         =',1X,A
     +/' p = P(chi-sq. >= Q)     =',F7.4,1X,A
     +/' 95% chi-sq. point       =',1X,A
     +/' 99% chi-sq. point       =',1X,A)     
      END

C
C
