C
C
      SUBROUTINE LINEAR (ISX, NCMAX, NCOLS, NF, NRMAX, NROWS,
     +                   A1, A2, B, H, RES, S, SE, THEORY,
     +                   WK, WT, Y,
     +                   FNAME, TITLE)
C
C
C ACTION : Multilinear regression
C AUTHOR : W.G.Bardsley, University of Manchester, U.K., 1/7/96
C          18/12/1996 added UWTSSQ to allow for weights
C          20/10/1997 win32 version calling AXEQB2 not nag G02DAF
C                     The original matrix in A1 is scanned and rows are
C                     only included if S > 0. If all S = 1 regression
C                     is unweighted.
C          30/10/1998 removed COV, P
C          29/01/1999 corrected selection of x for correlation
C          09/08/2000 added S, THEORY and H = leverages
C          16/10/2000 added FTEST1
C          07/02/2001 added CHOP80 and TRIM80
C          12/02/2003 revised and added TTEST2, HNPLOT, PCVTST, replaced
C                     CHOP80 and TRIM80 by CHOP60 and TRIM60
C          15/04/2004 added ISXEDI, ISXTYP, and ISXVEC
C          09/01/2006 moved A3 and A4 from argument to allocatable array
C          18/07/2007 introduced NISX, LDA, ISXSAV, XSAV and made CV and X allocatable
C          25/01/2015 introduced DONE0 to output CP_0 for just fitting the constant term   
C          29/06/2018 added INTENTS and introduced XCORR and YCORR to correct error when plotting y = f(x_i)  
C          22/09/2021 added E_NUMBERS and E_FORMATS, etc.       
C
C    ISX: (workspace) variables out (0) or in (1)
C  NCMAX: (input/unchanged) max. no. of columns
C  NCOLS: (input/unchanged) actual no. of columns
C     NF: (input/unchanged) preconnected unit for results
C  NRMAX: (input/unchanged) max. no. of rows
C  NROWS: (input/unchanged) actual no. of rows
C     A1: (input/unchanged) original data matrix ... this is never altered
C     A2: (workspace) current active copy of A1 made as required
C          Note that this is weighted if required (S .ne. 1)
C     A3: local allocated workspace
C     A4: local allocated workspace
C      B: (output) parameters
C      H: (output) leverages
C    RES: (output) residuals
C      S: (output) sigular values from SVD
C     SE: (output) parameter standard errors
C THEORY: (output) theoretical fit
C     WK: workspace, NWMAX >= NRMAX + 5*NCMAX
C     WT: (output) weights (active) calculated on demand
C      Y: (output) original (active) Y-values
C          Each time the data set is altered a new active set is calculated
C  FNAME
C  TITLE
C
      IMPLICIT   NONE
C
C Arguments
C
      INTEGER,              INTENT (IN)    :: NCMAX, NF, NRMAX
      INTEGER,              INTENT (IN)    :: NCOLS, NROWS
      INTEGER,              INTENT (INOUT) :: ISX(NCMAX)
      DOUBLE PRECISION,     INTENT (IN)    :: A1(NRMAX,NCMAX) 
      DOUBLE PRECISION,     INTENT (OUT)   :: A2(NRMAX,NCMAX), B(NCMAX),
     +                                        H(NRMAX), RES(NRMAX), 
     +                                        S(NCMAX), SE(NCMAX),
     +                                        THEORY(NRMAX),
     +                                        WK(NRMAX + 5*NCMAX),
     +                                        WT(NRMAX), Y(NRMAX)
      CHARACTER (LEN = *), INTENT (IN)     :: FNAME, TITLE
C
C Allocatable local arrays
C
      DOUBLE PRECISION, ALLOCATABLE :: A3(:,:), A4(:,:), CV(:,:), X(:,:)
      DOUBLE PRECISION, ALLOCATABLE :: XCORR(:), YCORR(:)
C
C Locals
C
      INTEGER    I, IADD1, IDF, IFAIL, IP, IRANK, J, K, M, N, NVAR,
     +           NDF0, NDF1, NDF2
      INTEGER    ICOUNT, IERR, LWORK, NCOL1, NIN, NROW1
      INTEGER    COLOUR
      INTEGER    N0, N1, N2, N3, N4, N5, N6, N7, N8, NXMIN
      PARAMETER (N0 = 0, N1 = 1, N2 = 2, N3 = 3, N4 = 4, N5 = 5, N6 = 6,
     +           N7 = 7, N8 = 8, NXMIN = 1)
      INTEGER    ICOLOR, IXL, IYL, LSHADE, NUMDEC, NUMOPT, NSTART, NTEXT
      PARAMETER (ICOLOR = 3, IXL = 4, IYL = 4, LSHADE = 1, NUMOPT = 10,
     +           NSTART = 15, NTEXT = NSTART + NUMOPT - 1)
      INTEGER    NUMBLD(NTEXT), NUMPOS(NUMOPT)
      INTEGER    NPAR1, NPTS1
      INTEGER    ISEND, ITYPE, LDA, MODE, NPAR, NPTS
      PARAMETER (ISEND = 2, ITYPE = 1)
      INTEGER    NISX
      PARAMETER (NISX = 1000)
      INTEGER    ISXSAV(NISX)
      DOUBLE PRECISION TEMP
      DOUBLE PRECISION CP_0, YTEMP
      DOUBLE PRECISION XSAV(1,NISX), YPRED
      DOUBLE PRECISION CP, DENOM, DF, FSTAT, PVAL, REGSSQ, RSQD, RTOL, 
     +                 SIGEST, SIGMA2, TNU, TOL, TSS, TVAL, UWTSSQ,
     +                 VALUE, VAREST, WSSQ, YBAR
      DOUBLE PRECISION WSSQ1
      DOUBLE PRECISION ONE, TWO, ZERO
      PARAMETER (ONE = 1.0D+00, TWO = 2.0D+00, ZERO = 0.0D+00)
      DOUBLE PRECISION PNT025, PNT05, PNT1, PNT2
      PARAMETER (PNT025 = 0.025D+00, PNT05 = 0.05D+00, PNT1 = 0.1D+00,
     +           PNT2 = 0.2D+00)
      DOUBLE PRECISION X02AJF$, G01EBF$, X02AMF$, G01EDF$, G01FBF$
      DOUBLE PRECISION X2(N2), X3(N2), X4(N2)
      DOUBLE PRECISION Y2(N2), Y3(N2), Y4(N2)
      CHARACTER (LEN = 12) I12(4), FORM12
      CHARACTER (LEN = 13) D13(6), SHOWLJ, SHOWRJ
      CHARACTER  CHOP60*60, TITLE1*60, TRIM60*60
      CHARACTER  T60*60, C60*60, TXT*100
      CHARACTER  TEXT(30)*100, LINE*100, LINE2(2)*100
      CHARACTER  CIPHER*4, LABELM*5, LABELW*5, MEAN*1, WEIGHT*1
      CHARACTER  BLANK*1, TAIL*1
      PARAMETER (BLANK = ' ', TAIL = 'U')
      CHARACTER  PTITLE*14, XTITLE*10, YTITLE*12
      PARAMETER (PTITLE = 'Residuals Plot', XTITLE = 'Best-Fit Y')
      LOGICAL    DISPLY, FILE1, SUPPLY
      PARAMETER (DISPLY = .FALSE., FILE1 = .FALSE., SUPPLY = .TRUE.)
      LOGICAL    BORDER, FLASH, HIGH
      PARAMETER (BORDER = .FALSE., FLASH = .FALSE., HIGH = .TRUE.)
      LOGICAL    E_NUMBERS, E_FORMATS
      LOGICAL    AXES, FILE, PRINT1
      PARAMETER (AXES = .TRUE., FILE = .TRUE., PRINT1 = .TRUE.)
      LOGICAL    CURVE, FIXCOL, FIXROW, LABEL, ORDER, WTD
      PARAMETER (CURVE = .FALSE., FIXCOL = .TRUE., FIXROW = .TRUE.,
     +           LABEL = .TRUE., ORDER = .FALSE., WTD = .FALSE.)
      LOGICAL    ABORT, FITNOW, READY, SHOWIT
      LOGICAL    AGAIN1, AGAIN2, DONE0, DONE1, DONE2
      EXTERNAL   PUTIFA, LBOX01, PUTFAT, TABLE1, PUTADV, PUTWAR, LBOX02,
     +           GETIM1, LINFIT, AXEQB2, ATAINV, GKS004, REVPRO, FTEST1,
     +           TRIM60, CHOP60, TTEST2, HNPLOT, PCVTST, EDITOR, ISXEDI,
     +           ISXTYP, ISXVEC, FORM12, E_FORMATS, SHOWLJ, SHOWRJ
      EXTERNAL   X02AJF$, G01EBF$, X02AMF$, G01EDF$, G01FBF$
      INTRINSIC  ABS, DBLE, SQRT, MAX, MIN, TRIM
      SAVE       ICOUNT, ISXSAV
      SAVE       XSAV 
      DATA       ICOUNT / 0 /    
      DATA       ISXSAV / NISX*1 / 
      DATA       XSAV / NISX*ONE /
      DATA       NUMBLD / NTEXT*0 /
      DATA       NUMPOS / NUMOPT*1 /
      
C
C Part 1: Check input data
C ========================
C
      IF (NCOLS.GT.NCMAX) THEN
         CALL PUTFAT ('Column dimension exceeded in call to LINEAR')
         RETURN
      ENDIF
      IF (NCOLS.LT.N3) THEN
         CALL PUTFAT ('Must have at least 3 columns: e.g. x, y, and s')
         RETURN
      ENDIF
      IF (NROWS.GT.NRMAX) THEN
         CALL PUTFAT ('Row dimension exceeded in call to LINEAR')
         RETURN
      ENDIF
      IF (NROWS.LE.NCOLS - 2) THEN
         CALL PUTFAT ('Must have more y-values than variables')
         RETURN
      ENDIF
C
C Allocate workspace
C
      IERR = 0
      IF (ALLOCATED(A3)) DEALLOCATE(A3, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(A4)) DEALLOCATE(A4, STAT = IERR)
      IF (IERR.NE.0) RETURN 
      IF (ALLOCATED(CV)) DEALLOCATE(CV, STAT = IERR)
      IF (IERR.NE.0) RETURN    
      IF (ALLOCATED(X)) DEALLOCATE(X, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(XCORR)) DEALLOCATE(XCORR, STAT = IERR)
      IF (IERR.NE.0) RETURN  
      IF (ALLOCATED(YCORR)) DEALLOCATE(YCORR, STAT = IERR)
      IF (IERR.NE.0) RETURN   
      ALLOCATE(A3(NRMAX,NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(A4(NRMAX,NCMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN 
      LDA = NCMAX
      ALLOCATE(CV(LDA,LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN 
      ALLOCATE(X(N1,LDA), STAT = IERR)
      IF (IERR.NE.0) RETURN 
      ALLOCATE(XCORR(NROWS), STAT = IERR)
      IF (IERR.NE.0) RETURN 
      ALLOCATE(YCORR(NROWS), STAT = IERR)
      IF (IERR.NE.0) RETURN     
C
C Initialise then check if weights should be calculated
C
      E_NUMBERS = E_FORMATS()
      TOL = SQRT(X02AJF$())
      RTOL = 1.0D+09*X02AMF$()
      MODE = N3
      NPAR = N0
      NPAR1 = N0
      NPTS = N0
      NPTS1 = N0
      WSSQ1 = - ONE
      WEIGHT = 'U'
      I = 0
      DO WHILE (I.LT.NROWS .AND.WEIGHT.EQ.'U')
         I = I + 1
         VALUE = A1(I,NCOLS)
         IF (VALUE.GT.ZERO .AND. ABS(VALUE - ONE).GT.TOL) WEIGHT = 'W'
      ENDDO
      IF (WEIGHT.EQ.'W') THEN
         LABELW = '[Yes]'
      ELSE
         LABELW = ' [No]'
      ENDIF
C
C Initialise elements of ISX ... M = total number of variables
C
      M = NCOLS - N2
      DO I = N1, M 
         IF (I.LE.NISX) THEN
            ISX(I) = ISXSAV(I)
         ELSE   
            ISX(I) = N1
         ENDIF   
      ENDDO
      ISX(M + N1) = N0 
C
C Initialise elements of X
C                         
      DO I = N1, M
         IF (I.LE.NISX) THEN
            X(N1,I) = XSAV(N1,I)
         ELSE
            X(N1,I) = ONE
         ENDIF      
      ENDDO 
C
C Define N and assign data to array A2 ... include a first column of 1
C for the constant term X = 1 and then calculate the weights
C
      N = N0
      K = NCOLS - N1
      DO I = N1, NROWS
         VALUE = A1(I,NCOLS)
         IF (VALUE.GT.TOL) THEN
            N = N + N1
            IF (WEIGHT.EQ.'W') THEN
               WT(N) = ONE/VALUE
            ELSE
               WT(N) = ONE
            ENDIF
            A2(N,N1) = WT(N)
            DO J = N1, M
               A2(N,J + N1) = A1(I,J)*WT(N)!Note that x values are weighted
            ENDDO
            Y(N) = A1(I,K)                 !but y values are not weighted
         ENDIF
      ENDDO
C
C Calculate YBAR ... Y is always unweighted
C
      YBAR = ZERO
      DO I = N1, N
         YBAR = YBAR + Y(I)
      ENDDO
      YBAR = YBAR/DBLE(N)
C
C Now a full analysis to calculate SIGMA2 for CP evaluation
C
      MEAN = 'M'
      IP = M + N1
      ISX(IP) = N0
C
C Assign Y temporarily to the residual vector RES then call AXEQB2
C In this call A4 is workspace and A3 holds the solution on return
C
      DO I = N1, N
         RES(I) = Y(I)*WT(I)
      ENDDO
      LWORK = NRMAX + 5*NCMAX
      NCOL1 = IP
      NROW1 = N
      CALL AXEQB2 (IRANK, LWORK, NCMAX, NCOL1, NIN, NF, NRMAX, NROW1,
     +             A2, A4, A3, S, RES, WK,
     +             TITLE, TITLE1,
     +             ABORT, DISPLY, FILE1, SUPPLY)
C
C Current degrees of freedom and define solution vector
C
      IDF = N - IRANK
      DO I = 1, IP
         B(I) = A3(I,N1)
      ENDDO
C
C Calculate the unweighted sum of squares and SIGMA2 for Mallows Cp, etc.
C
      UWTSSQ = ZERO
      DO I = N1, N
         VALUE = B(1)*WT(I)
         DO J = N1, M
            VALUE = VALUE + A2(I,J + N1)*B(J + N1)
         ENDDO
         UWTSSQ = UWTSSQ + (Y(I) - VALUE/WT(I))**2! Note that VALUE has to be fully unweighted at this stage
      ENDDO
      DF = DBLE(N - IRANK)
      SIGMA2 = UWTSSQ/DF
C
C Calculate CP_0 for fitting with just the constant term = YBAR
C
      YTEMP = ZERO
      DO I = N1, N
        YTEMP = YTEMP + (Y(I) - YBAR)**2
      ENDDO
      CP_0 = YTEMP/SIGMA2 - DBLE(N - 2)
      DONE0 = .FALSE.
C
C Part 2: Main branch point for repeated analysis
C ===============================================
C                           
      ICOUNT = ICOUNT + N1
      T60 = TRIM60(FNAME)
      C60 = CHOP60(TITLE)
      FITNOW = .TRUE.
      READY = .FALSE.
      AGAIN1 = .TRUE.
      AGAIN2 = .FALSE.  
      DONE1 = .TRUE.
      DONE2 = .TRUE.
      DO WHILE (AGAIN1)
C
C First of all define IP, NVAR and LABELM then initialise the menu
C
         NUMDEC = N1
         CALL ISXVEC (ISX, M, NVAR, NXMIN)
         IF (MEAN.EQ.'M' .OR. MEAN.EQ.'m') THEN
            IP = NVAR + N1
            LABELM = '(Yes)'
         ELSE
            IP = NVAR
            LABELM = ' (No)'
         ENDIF
              
         WRITE (TEXT,100) ICOUNT, T60, C60, LABELW, LABELM, M, NVAR,
     +                    M - NVAR, IP
C
C Now refine details of the menu
C
         CALL ISXTYP (ISX, M, NVAR, NXMIN,
     +                LINE,
     +                SHOWIT)
         TEXT(N4) = LINE
         IF (SHOWIT) TEXT(N5) = '* = suppressed variable'
C
C The main menu
C                
         NUMBLD(1) = 1 
         NUMBLD(4) = 1
         NUMBLD(6) = 1
         NUMBLD(7) = 1
         IF (.NOT.READY) THEN
            NUMDEC = N1
         ELSE
            NUMDEC = NUMOPT - N1
         ENDIF
         FITNOW = .FALSE.
         CALL LBOX01 (ICOLOR, IXL, IYL, LSHADE, NUMBLD, NUMDEC, NUMOPT,
     +                NUMPOS, NSTART, NTEXT, 
     +                TEXT, 
     +                BORDER, FLASH, HIGH)
         IF (NUMDEC.EQ.N1) THEN
C
C Proceed to fitting
C
            WRITE (NF,'(A)') BLANK 
            WRITE (NF,'(A)') '...' 
            AGAIN1 = .TRUE.
            FITNOW = .TRUE.
         ELSEIF (NUMDEC.EQ.N2) THEN
C
C Change status of MEAN
C
            IF (MEAN.EQ.'M' .OR. MEAN.EQ.'m') THEN
               MEAN = 'Z'
               CALL PUTADV (
     +'Regression will now pass through the origin')
            ELSE
               MEAN = 'M'
               CALL PUTADV (
     +'A mean term, i.e. intercept or constant, will be in the model')
            ENDIF
            NPAR1 = N0
            NPTS1 = N0
            WSSQ = - ONE
            AGAIN1 = .TRUE.
            READY = .FALSE.
         ELSEIF (NUMDEC.EQ.N3) THEN
C
C Suppress/Restore variables
C
            CALL ISXEDI (ISX, M, NVAR, NXMIN)
            NPAR1 = N0
            NPTS1 = N0
            WSSQ1 = - ONE
            AGAIN1 = .TRUE.
            READY = .FALSE.
         ELSEIF (NUMDEC.EQ.N4) THEN
C
C Linear regression/correlation/plot using original data
C
             IF (M.GT.N1) THEN
                CALL GETIM1 (N1, J, M,
     +                      'Number of x-column for plotting')
             ELSE
                J = N1
             ENDIF
             IF (ISX(J).EQ.N0) THEN
                CALL PUTFAT (
     +              'This is not in the current regression set')
             ELSE
                IADD1 = N0
                DO I = N1, NROWS
                   IF (WT(I).GT.TOL) THEN
                      IADD1 = IADD1 + N1
                      XCORR(IADD1) = A1(I,J)           !Get XCORR and YCORR from the original unweighted data 
                      YCORR(IADD1) = A1(I,NCOLS - N1)
                   ENDIF  
                ENDDO
                IF (IADD1.EQ.N) THEN
                   CALL LINFIT (NF, N, XCORR, YCORR, FILE, PRINT1)
                ELSE
                   CALL PUTFAT ('IADD1 not equal N in call to LINFIT')
                ENDIF   
             ENDIF
             AGAIN1 = .TRUE.
         ELSEIF (NUMDEC.EQ.N5) THEN
C
C Output the residuals, etc.
C
            IF (READY) THEN
               NUMDEC = N1
               AGAIN2 = .TRUE.
               DO WHILE (AGAIN2)
                  WRITE (TEXT,700)
                  CALL LBOX02 (ICOLOR, IXL, IYL, NUMDEC, N8, NUMPOS,
     +                         TEXT)
                  IF (NUMDEC.EQ.N1) THEN
C
C Display a table
C
                     COLOUR = 15
                     CALL TABLE1 (COLOUR, 'OPEN')
                     IF (WEIGHT.EQ.'W' .OR. WEIGHT.EQ.'w') THEN
                        WRITE (LINE,800)
                     ELSE
                       WRITE (LINE,900)
                     ENDIF
                     COLOUR = 4
                     CALL TABLE1 (COLOUR, LINE)
                     COLOUR = 0
                     DO I = N1, N
                        DENOM = MAX(RTOL, ONE - H(I))
                        DENOM = MAX(RTOL, SIGEST*SQRT(DENOM))
                        IF (E_NUMBERS) THEN
                           WRITE (LINE,1000) I, Y(I), THEORY(I), RES(I),
     +                                       H(I), RES(I)/DENOM
                        ELSE
                           D13(1) = SHOWRJ(Y(I))
                           D13(2) = SHOWRJ(THEORY(I))
                           D13(3) = SHOWRJ(RES(I))
                           D13(4) = SHOWRJ(H(I))
                           TEMP = RES(I)/DENOM
                           D13(5) = SHOWRJ(TEMP)
                           WRITE (LINE,1050) I, D13(1), D13(2), D13(3), 
     +                                       D13(4), D13(5)
                        ENDIF   
                        CALL TABLE1 (COLOUR, LINE)
                     ENDDO
                     CALL TABLE1 (COLOUR, 'CLOSE')
                     NUMDEC = N2
                  ELSEIF (NUMDEC.EQ.N2) THEN
C
C Write out to a file
C                     
                     IF (DONE1) THEN
                        CALL PUTADV (
     +'Residuals have already been written to the results file')
                     ELSE
                        WRITE (NF,'(A)') ' '
                        IF (WEIGHT.EQ.'W' .OR. WEIGHT.EQ.'w') THEN
                           WRITE (NF,800)
                        ELSE
                           WRITE (NF,900)
                        ENDIF
                        IF (E_NUMBERS) THEN
                           DO I = N1, N
                              DENOM = MAX(RTOL, ONE - H(I))
                              DENOM = MAX(RTOL, SIGEST*SQRT(DENOM))
                              WRITE (NF,1000) I, Y(I), THEORY(I),
     +                                        RES(I), H(I), RES(I)/DENOM
                           ENDDO
                        ELSE
                           DO I = N1, N
                              DENOM = MAX(RTOL, ONE - H(I))
                              DENOM = MAX(RTOL, SIGEST*SQRT(DENOM))
                              D13(1) = SHOWRJ(Y(I))
                              D13(2) = SHOWRJ(THEORY(I))
                              D13(3) = SHOWRJ(RES(I))
                              D13(4) = SHOWRJ(H(I))
                              TEMP = RES(I)/DENOM
                              D13(5) = SHOWRJ(TEMP)
                              WRITE (NF,1050) I, D13(1), D13(2), D13(3), 
     +                                        D13(4), D13(5)
                           ENDDO
                        ENDIF  
                        DONE1 = .TRUE.
                        CALL PUTADV (
     +'Residuals have now been written to the results file')
                        DONE1 = .TRUE.
                     ENDIF                            
                     NUMDEC = N3
                  ELSEIF (NUMDEC.EQ.N3) THEN
C
C ANOVA
C
                     IFAIL = N1
                     PVAL = G01EDF$(TAIL, FSTAT, DBLE(NDF1), DBLE(NDF2),
     +                              IFAIL)
                     CALL PUTIFA (IFAIL, NF, 'G01EDF/LINEAR')
                     IF (E_NUMBERS) THEN
                        WRITE (TEXT,1100) NDF0, TSS, NDF1, REGSSQ,
     +                                    REGSSQ/DBLE(NDF1),
     +                                    FSTAT, PVAL, NDF2, UWTSSQ,
     +                                    UWTSSQ/DBLE(NDF2)
                        IF (.NOT.DONE2)
     +                  WRITE (NF,1100) NDF0, TSS, NDF1, REGSSQ,
     +                                  REGSSQ/DBLE(NDF1),
     +                                  FSTAT, PVAL, NDF2, UWTSSQ,
     +                                  UWTSSQ/DBLE(NDF2)
                     ELSE
                        D13(1) = SHOWRJ(TSS)
                        D13(2) = SHOWRJ(REGSSQ)
                        TEMP = REGSSQ/DBLE(NDF1)
                        D13(3) = SHOWRJ(TEMP)
                        D13(4) = SHOWRJ(FSTAT)
                        D13(5) = SHOWRJ(UWTSSQ)
                        TEMP = UWTSSQ/DBLE(NDF2)
                        D13(6) = SHOWRJ(TEMP)
                        WRITE (TEXT,1150) NDF0, D13(1), NDF1, D13(2), 
     +                                    D13(3), D13(4),
     +                                    PVAL, NDF2, D13(5), D13(6)
                        IF (.NOT.DONE2)
     +                  WRITE (NF,1150) NDF0, D13(1), NDF1, D13(2), 
     +                                  D13(3), D13(4),
     +                                  PVAL, NDF2, D13(5), D13(6)
                     ENDIF  
                     COLOUR = 15
                     CALL TABLE1 (COLOUR, 'OPEN')
                     DO I = N1, N6
                        IF (I.LE.N3) THEN
                           COLOUR = 4
                        ELSE
                           COLOUR = 0
                        ENDIF
                        CALL TABLE1 (COLOUR, TEXT(I))
                     ENDDO
                     CALL TABLE1 (COLOUR, 'CLOSE')
                     DONE2 = .TRUE.
                     NUMDEC = N4
                  ELSEIF (NUMDEC.EQ.N4) THEN
C
C Plot residuals
C
                     DO I = N1, N2
                        X2(I) = ZERO
                        X3(I) = ZERO
                        X4(I) = ZERO
                        Y2(I) = ZERO
                        Y3(I) = ZERO
                        Y4(I) = ZERO
                     ENDDO
                     IF (WEIGHT.EQ.'W' .OR. WEIGHT.EQ.'w') THEN
                        YTITLE = 'Wtd. Resids.'
                     ELSE
                       YTITLE = 'Residuals'
                     ENDIF
                     CALL GKS004 (N0, N0, N0, N0, N4, N0, N0, N0,
     +                            N, N2, N2, N2,
     +                            THEORY, X2, X3, X4,
     +                            RES, Y2, Y3, Y4,
     +                            PTITLE, XTITLE, YTITLE,
     +                            AXES, AXES)
                     NUMDEC = N5
                  ELSEIF (NUMDEC.EQ.N5) THEN
C
C Half normal plot
C
                     CALL HNPLOT (N1, N,
     +                            RES)
                     NUMDEC = N6
                  ELSEIF (NUMDEC.EQ.N6) THEN
C
C Normal plot
C
                     CALL HNPLOT (N2, N,
     +                            RES)
                     NUMDEC = N7
                  ELSEIF (NUMDEC.EQ.N7) THEN
C
C F tests
C
                     CALL FTEST1 (NF, NPAR1, NPTS1, 
     +                            WSSQ1)
                     NUMDEC = N8
                  ELSE
C
C Go back for further analysis
C
                     AGAIN2 = .FALSE.
                  ENDIF
               ENDDO
            ELSE
               CALL PUTFAT ('First fit the data')
            ENDIF
            AGAIN1 = .TRUE.
         ELSEIF (NUMDEC.EQ.N6) THEN
C
C Compare 2 parameters
C
            CALL TTEST2 (NF)
            AGAIN1 = .TRUE.
         ELSEIF (NUMDEC.EQ.N7) THEN
C
C Compare 2 sets of parameters
C
            IF (MODE.EQ.N2 .AND. NPAR.NE.M + N1) CALL PUTWAR (
     +'Data can only be saved for the parameters actually fitted')
            CALL PCVTST (MODE, NF, NPAR, NPTS, LDA,
     +                   CV, B)
            AGAIN1 = .TRUE.
         ELSEIF (NUMDEC.EQ.NUMOPT - N2) THEN
C
C Evaluate y = f(x)
C
            IF (READY .AND. MODE.EQ.N2) THEN
               WRITE (TXT,1200) M
               CALL EDITOR (ISEND, ITYPE, M, N1, N1,
     +                      X,
     +                      TXT,
     +                      CURVE, FIXCOL, FIXROW, LABEL, ORDER, WTD)
               IF (MEAN.EQ.'M') THEN
                  J = N1
                  YPRED = B(J)
               ELSE
                  J = N0
                  YPRED = ZERO
               ENDIF
               DO I = N1, M
                  IF (ISX(I).GT.N0) THEN
                     J = J + N1
                     YPRED = YPRED + B(J)*X(1,I)
                  ENDIF
               ENDDO
               COLOUR = 15
               CALL TABLE1 (COLOUR, 'OPEN')
               COLOUR = N0
               IF (MEAN.EQ.'M') THEN
                  J = N1
                  IF (E_NUMBERS) THEN
                     WRITE (TXT,1300) N0, ONE, B(J),
     +' (the constant term)'
                     WRITE (NF,'(A)') TXT
                  ELSE
                     D13(1) = SHOWLJ(ONE)
                     D13(2) = SHOWLJ(B(J))
                     WRITE (TXT,1350) N0, D13(1), D13(2),
     +' (the constant term)'
                     WRITE (NF,'(A)') TXT  
                  ENDIF  
                  CALL TABLE1 (COLOUR, TXT)
               ELSE
                  J = N0
               ENDIF
               DO I = N1, M
                  IF (ISX(I).GT.N0) THEN
                     J = J + N1
                     IF (E_NUMBERS) THEN
                        WRITE (TXT,1300) I, X(1,I), B(J), BLANK
                        WRITE (NF,'(A)') TXT
                     ELSE
                        D13(1) = SHOWLJ(X(1,I))
                        D13(2) = SHOWLJ(B(J))
                        WRITE (TXT,1350) I, D13(1), D13(2), BLANK
                        WRITE (NF,'(A)') TXT  
                     ENDIF  
                     CALL TABLE1 (COLOUR, TXT)
                  ENDIF
               ENDDO
               COLOUR = N4
               IF (E_NUMBERS) THEN
                  WRITE (TXT,1400) YPRED
                  WRITE (NF,'(A)') TXT
               ELSE
                  D13(1) = SHOWLJ(YPRED) 
                  WRITE (TXT,1450) D13(1)
                  WRITE (NF,'(A)') TXT 
               ENDIF  
               CALL TABLE1 (COLOUR, TXT)
               CALL TABLE1 (COLOUR, 'CLOSE')
            ELSE
               CALL PUTFAT ('Model not yet fitted')
            ENDIF
            AGAIN1 = .TRUE.
         ELSEIF (NUMDEC.EQ.NUMOPT - N1) THEN
C
C Results
C
            CALL REVPRO (NF)
            AGAIN1 = .TRUE.
         ELSEIF (NUMDEC.EQ.NUMOPT) THEN
C
C Terminate current data analysis
C
            AGAIN1 = .FALSE.
         ENDIF
C
C Part 3: the main fitting procedure
C ==================================
C Define N and assign data to array A2
C Include a first column of 1 for the constant term X = 1 if MEAN
C Then apply the weights if appropropriate
C
         IF (FITNOW) THEN
            FITNOW = .FALSE.
            READY = .TRUE.
            N = N0
            DO I = N1, NROWS
               VALUE = A1(I,NCOLS)
               IF (VALUE.GT.TOL) THEN
                  N = N + N1
                  K = N0
                  IF (MEAN.EQ.'M') THEN
                     K = K + N1
                     A2(N,K) = WT(N)
                  ENDIF
                  DO J = N1, M
                     IF (ISX(J).NE.N0) THEN
                        K = K + N1
                        A2(N,K) = A1(I,J)*WT(N)!Load the appropriate columns from the original data and weight
                     ENDIF
                  ENDDO
               ENDIF
            ENDDO
C
C Define Y = RES temporarily then call AXEQB2
C
            DO I = N1, N
               RES(I) = Y(I)*WT(I)
            ENDDO
            NCOL1 = IP
            NROW1 = N

            CALL AXEQB2 (IRANK, LWORK, NCMAX, NCOL1, NIN, NF, NRMAX,
     +                   NROW1,
     +                   A2, A4, A3, S, RES, WK,
     +                   TITLE, TITLE1,
     +                   ABORT, DISPLY, FILE1, SUPPLY)
C
C Is the problem of full rank ?
C
            IF (IRANK.LT.IP) THEN
               CALL PUTWAR (
     +'Less than full rank (SVD used) ... Ignore parameters/std.errors')
               MODE = N3
               NPAR = N0
               NPAR1 = N0
               NPTS = N0
               NPTS1 = N0
               WSSQ1 = - ONE
               DONE1 = .TRUE.
               DONE2 = .TRUE.
            ELSE
               MODE = N2
               NPAR = IP
               NPAR1 = IP
               NPTS = N
               NPTS1 = N  
               DONE1 = .FALSE.
               DONE2 = .FALSE. 
            ENDIF
C
C Define degrees of freedom then the parameters
C
            IDF = N - IRANK
            DO I = N1, IP
               B(I) = A3(I,N1)
            ENDDO
C
C Calculate UWTSSQ, RES, RSS, TSS
C
            UWTSSQ = ZERO
            IF (MEAN.EQ.'M') THEN
               DO I = N1, N
                  VALUE = B(1)*WT(I)
                  K = N1
                  DO J = N1, IP - N1
                     K = K + N1
                     VALUE = VALUE + A2(I,K)*B(K)
                  ENDDO
                  THEORY(I) = VALUE/WT(I)
                  RES(I) = Y(I) - VALUE/WT(I)
                  UWTSSQ = UWTSSQ + RES(I)**2
               ENDDO
            ELSE
               DO I = N1, N
                  VALUE = ZERO
                  DO J = N1, IP
                     VALUE = VALUE + A2(I,J)*B(J)
                  ENDDO
                  THEORY(I) = VALUE/WT(I)
                  RES(I) = Y(I) - VALUE/WT(I)
                  UWTSSQ = UWTSSQ + RES(I)**2
               ENDDO
            ENDIF
C
C Calculate statistics CP, RSQD and F
C
            TSS = ZERO
            IF (MEAN.EQ.'M' .OR. MEAN.EQ.'m') THEN
               DO I = N1, N
                  TSS = TSS + (Y(I) - YBAR)**2
               ENDDO
            ELSE
               DO I = N1, N
                  TSS = TSS + Y(I)**2
               ENDDO
            ENDIF
            REGSSQ = TSS - UWTSSQ
            RSQD = REGSSQ/TSS
            IF (RSQD.LT.ZERO) RSQD = ZERO
            IF (RSQD.GT.ONE) RSQD = ONE
            CP = UWTSSQ/SIGMA2 - DBLE(N - TWO*IP)
            IF (MEAN.EQ.'M' .OR. MEAN.EQ.'m') THEN
               NDF0 =  N - N1
            ELSE
               NDF0 = N
            ENDIF
            NDF2 = N - IP
            NDF1 = NDF0 - NDF2
            FSTAT = (REGSSQ/DBLE(NDF1))/(UWTSSQ/DBLE(NDF2))
C
C Weight the residuals then calculate WSSQ
C
            WSSQ = ZERO
            DO  I = 1, N
               RES(I) = WT(I)*RES(I)
               WSSQ = WSSQ + RES(I)**2
            ENDDO
            IF (NPAR1.GT.N0 .AND. NPTS1.GT.N0) THEN
               WSSQ1 = WSSQ
            ELSE
               WSSQ1 = - ONE
            ENDIF
C
C Parameter standard errors
C
            NCOL1 = IP
            NROW1 = N
            CALL ATAINV (IFAIL, IRANK, NRMAX, NRMAX, NRMAX, NCOL1,
     +                   NROW1,
     +                   A2, A3, H, S, A4)
            CALL PUTIFA (IFAIL, NF, 'ATAINV/LINEAR')
            VAREST = WSSQ/DBLE(N - IRANK)
            SIGEST = SQRT(VAREST)
            DO I = 1, IP
               SE(I) = SIGEST*SQRT(A3(I,I))
            ENDDO
C
C Store the covariance matrix if satisfactory fit
C
            IF (IFAIL.NE.N0) MODE = N3
            IF (MODE.EQ.N2) THEN
               NPTS = N
               NPAR = IP
               DO J = N1, NPAR
                  DO I = N1, NPAR
                     CV(I,J) = VAREST*A3(I,J)
                  ENDDO
               ENDDO
            ELSE
               NPTS = N0
               NPAR = N0
            ENDIF
C
C Output the best-fit parameters
C
            WRITE (NF,'(A)') BLANK 
            DO I = N1, N7
               IF (I.NE.2) WRITE (NF,'(A)') TEXT(I)
            ENDDO
            WRITE (NF,'(A)') BLANK
            COLOUR = 15
            CALL TABLE1 (COLOUR, 'OPEN')
            COLOUR = 0
C
C First time only: output CP_0
C
            IF (.NOT.DONE0) THEN
               DONE0 = .TRUE.
               IF (E_NUMBERS) THEN
                  WRITE (LINE2,150) YBAR, YTEMP, CP_0
               ELSE
                  D13(1) = SHOWLJ(YBAR)
                  D13(2) = SHOWLJ(YTEMP)
                  D13(3) = SHOWLJ(CP_0)
                  WRITE (LINE2,155) TRIM(D13(1)), TRIM(D13(2)),
     +                              TRIM(D13(3)) 
               ENDIF  
               CALL TABLE1 (COLOUR, LINE2(1))
               CALL TABLE1 (COLOUR, LINE2(2))
               CALL TABLE1 (COLOUR, BLANK)
               WRITE (NF,'(A)') LINE2(1)
               WRITE (NF,'(A)') LINE2(2)
               WRITE (NF,'(A)') BLANK
            ENDIF
            I12(1) = FORM12(IP)
            I12(2) = FORM12(IRANK)
            I12(3) = FORM12(N)
            I12(4) = FORM12(IDF)
            WRITE (LINE,200) TRIM(I12(1)), TRIM(I12(2)), TRIM(I12(3)),
     +                       TRIM(I12(4))
            WRITE (NF,200) TRIM(I12(1)), TRIM(I12(2)), TRIM(I12(3)),
     +                     TRIM(I12(4))
            CALL TABLE1 (COLOUR, LINE)
            IF (E_NUMBERS) THEN
               WRITE (LINE,300) UWTSSQ, CP, RSQD
               WRITE (NF,300) UWTSSQ, CP, RSQD
            ELSE
               D13(1) = SHOWLJ(UWTSSQ)
               D13(2) = SHOWLJ(CP)
               WRITE (LINE,305) TRIM(D13(1)), TRIM(D13(2)), RSQD
               WRITE (NF,305) TRIM(D13(1)), TRIM(D13(2)), RSQD               
            ENDIF  
            CALL TABLE1 (COLOUR, LINE)
            IF (WEIGHT.EQ.'W' .OR. WEIGHT.EQ.'w') THEN
               IF (E_NUMBERS) THEN
                  WRITE (LINE,350) WSSQ
                  WRITE (NF,350) WSSQ
               ELSE
                  D13(1) = SHOWLJ(WSSQ)  
                  WRITE (LINE,355) TRIM(D13(1))
                  WRITE (NF,355) TRIM(D13(1))
               ENDIF  
               CALL TABLE1 (COLOUR, LINE)
            ENDIF

            WRITE (LINE,400)
            WRITE (NF,400)
            COLOUR = 4
            CALL TABLE1 (COLOUR, LINE)
            COLOUR = 0
C
C Calculate t values and parameter p-values
C
            IFAIL = N1
            TNU = G01FBF$(TAIL, PNT025, DBLE(NDF2), IFAIL)
            CALL PUTIFA (IFAIL, NF, 'G01FBF/LINEAR')
            DF = DBLE(N - IRANK)
            IF (MEAN.EQ.'M' .OR. MEAN.EQ.'m') THEN
               J = N1
               IF (SE(J).GT.RTOL) THEN
                  TVAL = ABS(B(J)/SE(J))
                  IFAIL = N1
                  PVAL = TWO*G01EBF$(TAIL, TVAL, DF, IFAIL)
                  CALL PUTIFA (IFAIL, NF, 'G01EBF/LINEAR')
               ELSE
                  PVAL = ONE
               ENDIF
               IF (PVAL.GT.PNT2) THEN
                  CIPHER = ' ***'
               ELSEIF (PVAL.GT.PNT1) THEN
                  CIPHER = '  **'
               ELSEIF (PVAL.GT.PNT05) THEN
                  CIPHER = '   *'
               ELSE
                  CIPHER = '    '
               ENDIF
               IF (PVAL.LT.ZERO) THEN
                  PVAL = ZERO
               ELSEIF (PVAL.GT.ONE) THEN
                  PVAL = ONE
               ENDIF
               IF (E_NUMBERS) THEN
                  WRITE (LINE,500) B(J), B(J) - TNU*SE(J),
     +                             B(J) + TNU*SE(J),
     +                             SE(J), PVAL, CIPHER
                  WRITE (NF,500) B(J), B(J) - TNU*SE(J),
     +                           B(J) + TNU*SE(J),
     +                           SE(J), PVAL, CIPHER
               ELSE
                  D13(1) = SHOWRJ(B(J))
                  TEMP = B(J) - TNU*SE(J)
                  D13(2) = SHOWRJ(TEMP)
                  TEMP = B(J) + TNU*SE(J)
                  D13(3) = SHOWRJ(TEMP)
                  D13(4) = SHOWRJ(SE(J)) 
                  WRITE (LINE,505) D13(1), D13(2), D13(3), D13(4),
     +                             PVAL, CIPHER
                  WRITE (NF,505) D13(1), D13(2), D13(3), D13(4),
     +                           PVAL, CIPHER
               ENDIF  
               CALL TABLE1 (COLOUR, LINE)
            ELSE
               J = N0
            ENDIF
            DO I = N1, M
               IF (ISX(I).GT.N0) THEN
                  J = J + N1
                  IF (SE(J).GT.RTOL) THEN
                     TVAL = ABS(B(J)/SE(J))
                     IFAIL = N1
                     PVAL = TWO*G01EBF$(TAIL, TVAL, DF, IFAIL)
                     CALL PUTIFA (IFAIL, NF, 'G01EBF/LINEAR')
                  ELSE
                     PVAL = ONE
                  ENDIF
                  IF (PVAL.GT.PNT2) THEN
                     CIPHER = ' ***'
                  ELSEIF (PVAL.GT.PNT1) THEN
                     CIPHER = '  **'
                  ELSEIF (PVAL.GT.PNT05) THEN
                     CIPHER = '   *'
                  ELSE
                     CIPHER = '    '
                  ENDIF
                  IF (PVAL.LT.ZERO) PVAL = ZERO
                  IF (PVAL.GT.ONE) PVAL = ONE
                  IF (E_NUMBERS) THEN  
                     IF (J.LT.10) THEN  
                        WRITE (LINE,600) I, B(J), B(J) - TNU*SE(J),
     +                                   B(J) + TNU*SE(J), SE(J), PVAL,
     +                                   CIPHER
                        WRITE (NF,600) I, B(J), B(J) - TNU*SE(J),
     +                                 B(J) + TNU*SE(J),
     +                                 SE(J), PVAL, CIPHER
                     ELSEIF (J.LT.100) THEN  
                        WRITE (LINE,610) I, B(J), B(J) - TNU*SE(J),
     +                                   B(J) + TNU*SE(J), SE(J), PVAL,
     +                                   CIPHER
                        WRITE (NF,610) I, B(J), B(J) - TNU*SE(J),
     +                                 B(J) + TNU*SE(J),
     +                                 SE(J), PVAL, CIPHER
                     ELSE  
                        WRITE (LINE,620) I, B(J), B(J) - TNU*SE(J),
     +                                   B(J) + TNU*SE(J), SE(J), PVAL,
     +                                   CIPHER
                        WRITE (NF,620) I, B(J), B(J) - TNU*SE(J),
     +                                 B(J) + TNU*SE(J),
     +                                 SE(J), PVAL, CIPHER
                     ENDIF
                  ELSE
                     D13(1) = SHOWRJ(B(J))
                     TEMP = B(J) - TNU*SE(J)
                     D13(2) = SHOWRJ(TEMP)
                     TEMP = B(J) + TNU*SE(J)
                     D13(3) = SHOWRJ(TEMP)
                     D13(4) = SHOWRJ(SE(J))  
                     IF (J.LT.10) THEN  
                        WRITE (LINE,605) I, D13(1), D13(2),
     +                                   D13(3), D13(4), PVAL, CIPHER
                        WRITE (NF,605) I, D13(1), D13(2),
     +                                 D13(3), D13(4), PVAL, CIPHER
                     ELSEIF (J.LT.100) THEN  
                        WRITE (LINE,615) I, D13(1), D13(2),
     +                                   D13(3), D13(4), PVAL, CIPHER
                        WRITE (NF,615) I, D13(1), D13(2),
     +                                 D13(3), D13(4), PVAL, CIPHER
                     ELSE  
                        WRITE (LINE,625) I, D13(1), D13(2),
     +                                   D13(3), D13(4), PVAL, CIPHER
                        WRITE (NF,625) I, D13(1), D13(2), 
     +                                 D13(3), D13(4), PVAL, CIPHER
                     ENDIF  
                  ENDIF
                  CALL TABLE1 (COLOUR, LINE)
               ENDIF
            ENDDO
            CALL TABLE1 (COLOUR, 'CLOSE')
         ENDIF
      ENDDO    
C
C Store ISX and XSAV
C          
      DO I = N1, MIN(M, NISX)
         ISXSAV(I) = ISX(I)
         XSAV(N1,I) = X(N1,I)
      ENDDO
C
C Deallocate workspace
C
      DEALLOCATE(A3, STAT = IERR)
      DEALLOCATE(A4, STAT = IERR) 
      DEALLOCATE(CV, STAT = IERR)
      DEALLOCATE(XCORR, STAT = IERR)
      DEALLOCATE(YCORR, STAT = IERR)
C
C Format statements
C      
  100 FORMAT (  
     + 'Multilinear regression analysis',I4
     +/
     +/'Current status of x-variables:'
     +/
     +/
     +/'File:',1X,A
     +/'Title:',1X,A
     +/'Use weights supplied:',1X,A
     +/'Include a constant term:',1X,A
     +/'Number of variables supplied =',I4
     +/'Number of variables included =',I4
     +/'Number of variables excluded =',I4
     +/'Number of parameter estimates =',I4
     +/
     +/'Fit data with current settings'
     +/'Change status of constant term'
     +/'Suppress/Restore variables'
     +/'Correlation/plot of y = f(x_i)'
     +/'ANOVA, residuals, F test'
     +/'Compare 2 parameters'
     +/'Compare 2 sets of parameters'
     +/'Evaluate y = f(x)'
     +/'Results'
     +/'Quit ... Exit multilinear regression analysis')
  150 FORMAT (' Results for just fitting the constant term:'
     +/' Constant =',1P,E13.5,', SSQ =',E12.5,', Mallows'' Cp =',E12.5) 
  155 FORMAT (' Results for just fitting the constant term:'
     +/' Constant = ',A,', SSQ = ',A,', Mallows'' Cp = ',A)          
  200 FORMAT (' Number of parameters = ',A,
     +', Rank = ',A,', Number of points = ',A,', NDOF = ',A)
  300 FORMAT (' Residual-SSQ =',1P,E12.5,', Mallows'' Cp =',E12.5,
     +', R-squared =',0P,F7.4)
  305 FORMAT (' Residual-SSQ = ',A,', Mallows'' Cp = ',A,
     +', R-squared =',F7.4)     
  350 FORMAT (' Weighted-SSQ =',1P,E12.5,' (w = 1/s^2 used)')
  355 FORMAT (' Weighted-SSQ = ',A,' (w = 1/s^2 used)')  
  400 FORMAT (' Parameter        Value     Lower95%cl    Upper95%cl',
     +'     Std.Error     p')
  500 FORMAT ('  Constant',     1P,4(1X,E13.5),0P,F9.4,A)
  505 FORMAT ('  Constant',        4(1X,A13),F9.4,A)
  600 FORMAT ('    B(',I1,')  ',1P,4(1X,E13.5),0P,F9.4,A)
  605 FORMAT ('    B(',I1,')  ',   4(1X,A13),F9.4,A)  
  610 FORMAT ('    B(',I2,') ', 1P,4(1X,E13.5),0P,F9.4,A)
  615 FORMAT ('    B(',I2,') ',    4(1X,A13),F9.4,A)  
  620 FORMAT ('    B(',I3,')',  1P,4(1X,E13.5),0P,F9.4,A)  
  625 FORMAT ('    B(',I3,')',     4(1X,A13),F9.4,A)    
  700 FORMAT (
     + 'Display the residuals'
     +/'File the residuals'
     +/'Display/file ANOVA table'
     +/'Plot residuals: against theory'
     +/'Plot residuals: half normal'
     +/'Plot residuals: full normal'
     +/'F tests'
     +/'Quit ... Exit residuals analysis')
  800 FORMAT ('Number      Y-value        Theory      Wtd.Res.',
     +'      Leverage    Studentized')
  900 FORMAT ('Number      Y-value        Theory      Residual',
     +'      Leverage    Studentized')
 1000 FORMAT (I5,1P,5(1X,E13.5))
 1050 FORMAT (I5,5(1X,A13))
 1100 FORMAT (
     +/' ANOVA'
     +/' Source      NDOF          SSQ       Mean-SSQ       F-value',
     +'      p'
     +/' Total     ', I6,1P,1X,E13.5,
     +/' Regression', I6,   3(1X,E13.5),0P,F10.4
     +/' Residual  ', I6,1P,2(1X,E13.5))
 1150 FORMAT (
     +/' ANOVA'
     +/' Source      NDOF          SSQ       Mean-SSQ       F-value',
     +'      p'
     +/' Total     ', I6,  1X,A13,
     +/' Regression', I6,3(1X,A13),F10.4
     +/' Residual  ', I6,2(1X,A13))    
 1200 FORMAT ('Values for x(1) to x(',I3,')')
 1300 FORMAT ('x(',I3,') =',1P,E13.5, ', coefficient =',E13.5,A)
 1350 FORMAT ('x(',I3,') =',1X,A13,', coefficient =',1X,A13,A)
 1400 FORMAT ('y(x) =',1P,E13.5)
 1450 FORMAT ('y(x) =',1X,A)
      END
C
C
