c
c
c subroutine orthog with related procedures as follows:
c =====================================================
c module_orthog
c deriv_orthog
c fit_orthog
c funct_orthog
c mod_orthog
c
c Action: Fit weighted least squares, major axis, reduced major axes
c         lines using module workspaces defined in module_orthog
c         instead of the original common blocks
c Note: the module is necesary to enable the subroutines to use
c       the methods employed by qnfit for function evaluations, gradients,
c       covariances, etc.
c
c-----------------------------------------------------------------------
c
      module module_orthog
      integer lwtype, np, npts1
      double precision fact(2), rtol, rndof
      double precision, allocatable :: error(:), fjacc(:,:), fval(:),
     +                                 resid(:), theory(:), wresid(:),
     +                                 xval(:)
      logical, allocatable :: equal(:)
      end module module_orthog
c
c-----------------------------------------------------------------------
c
      subroutine orthog (ifail, nin, nout, npts, nrmax, ntype,
     +                   p, s, se, x, y,
     +                   abort, covar, goffit, plot, supply, table)
c
c action : orthogonal regression
c author : w.g.bardsley, university of manchester, u.k., 15/09/99
c          06/06/2000 added call to nxyr2p
c          20/02/2006 replaced common blocks by module_orthog  
c          12/04/2007 introduced facmin to adjusted fact(i) and avoid fact(i) = 0
c          01/07/2021 added e_formats and e_numbers, etc.      
c          21/09/2021 replaced d09 and form09 by d10 and formgr
c
c ifail  : (output) 0 for success
c nin    : (input/unchanged) input unit for data (if supply = .false.)
c nout   : (input/unchanged) output unit for results (if table = .true.)
c npts   : (input/output depending on supply) dimension of data
c nrmax  : (input/unchanged) max dimension of data
c ntype  : (input/unchanged) 1 = weighted least squares
c                            2 = weighted reduced (triangular)
c                            3 = weighted major axis regression (orthogonal)
c p      : (output) p(1) = slope, p(2) = constant term
c se     : (output) parameter standard errors (if covar = .true.)
c s, x, y: (input/output depending on supply) data
c abort  : (output) flags failure
c covar  : (input/unchanged) invert hessian
c goffit : (input/unchanged) residuals analysis and goodness of fit
c plot   : (input/unchanged) create a plot
c supply : (input/unchanged) supply data (or read in on nin)
c table  : (input/unchanged) create a table (for display and writing to nout)
c

      use module_orthog
      implicit   none
c
c arguments
c
      integer    ifail, nin, nout, npts, nrmax, ntype
      double precision p(2), s(nrmax), se(2), x(nrmax), y(nrmax)
      logical    abort, covar, goffit, plot, supply, table
c
c locals
c
      integer    npar, nhess
      parameter (npar = 2, nhess = npar + 1)
      integer    n0, n1, n2, n5
      parameter (n0 = 0, n1 = 1, n2 = 2, n5 = 5)
      integer    index(npar), istate(npar), nfree
      integer    i, ierr
      integer    icolor
      double precision resul(20), tstat, xpar(npar)
      double precision xplot(n2), yplot(n2)
      double precision corr(npar,npar), cv(nhess,nhess), diagv(npar),
     +                 g(npar), hessex(nhess,nhess), sigma, w(3*npar)
      double precision prob, r, rsqd, temp
      double precision zero, one, two
      parameter (zero = 0.0d+00, one = 1.0d+00, two = 2.0d+00)
      double precision g01fbf$, g01ebf$, x02amf$, pval(2), pnt01, pnt05
      parameter (pnt01 = 0.01d+00, pnt05 = 0.05d+00) 
      double precision facmin
      parameter (facmin = 1.0e-04)
      character  fname*1024, line*100, stars(2)*2, title*80,
     +           word12(2)*12
      character (len = 1 )  xtitle, ytitle
      parameter (xtitle = 'x', ytitle = 'y')
      character (len = 80) trim80, chop80
      character (len = 50) ptitle
      character (len = 13) d13(4), showrj
      character (len = 10 ) d10(2), formgr
      logical    e_formats, e_numbers
      logical    free(npar), testit, weight
      logical    file1, file2, graph, tshow1, tshow2
      parameter (file1 = .true., file2 = .true., graph = .true.,
     +           tshow1 = .true., tshow2 = .true.)
      logical    gsave
      parameter (gsave = .true.)
      external   e_formats, formgr, showrj
      external   deriv_orthog, funct_orthog
      external   fit_orthog
      external   x02amf$, g02caf$, putifa, putfat, table1, gks004,
     +           qncov1, datfil, g01ebf$, g01fbf$, gksr01, nxyr2p,
     +           chop80, trim80
      intrinsic  sqrt, abs, trim
c
c set defaults then test ntype
c
      ifail = - 1
      do i = 1, 2
         p(i) = zero
         pval(i) = one
         se(i) = zero
         stars(i) = '**'
      enddo
      abort = .true.
      if (ntype.lt.1 .or. ntype.gt.3) then
         call putfat ('ntype out of range in call to ORTHOG')
         return
      endif
      e_numbers = e_formats()
c
c get data if required
c

      if (.not.supply) then
         close (unit = nin)
         call datfil (nin, nrmax, npts,
     +                s, x, y,
     +                fname, title,
     +                abort)
         close (unit = nin)
         if (abort) return
         write (nout,'(a)') ' '
         write (nout,'(a)') 'Filename:'
         write (nout,'(a)') trim80(fname)
         write (nout,'(a)') 'Title: '
         write (nout,'(a)') chop80(title)
      endif
      if (npts.lt.2) then
         call putfat ('Insufficient data')
         return
      endif
c
c initialise
c
      rtol = 1.0d+09*x02amf$()
      tstat = zero
      lwtype = ntype
      np = npts
      npts1 = npts
      rndof = dble(npts) - two
      word12(1) = 'slope    (m)'
      word12(2) = 'constant (c)'
      abort = .true.
      weight = .false.
      testit = .true.
c
c allocate workspace arrays in the module using np >= npts1 >= npts
c
      ierr = 0
      if (allocated(error)) deallocate(error, stat = ierr)
      if (ierr.ne.0) return
      if (allocated(fjacc)) deallocate(fjacc, stat = ierr)
      if (ierr.ne.0) return
      if (allocated(fval)) deallocate(fval, stat = ierr)
      if (ierr.ne.0) return
      if (allocated(resid)) deallocate(resid, stat = ierr)
      if (ierr.ne.0) return
      if (allocated(theory)) deallocate(theory, stat = ierr)
      if (ierr.ne.0) return
      if (allocated(wresid)) deallocate(wresid, stat = ierr)
      if (ierr.ne.0) return
      if (allocated(xval)) deallocate(xval, stat = ierr)
      if (ierr.ne.0) return
      if (allocated(equal)) deallocate(equal, stat = ierr)
      if (ierr.ne.0) return
      allocate(error(np), stat = ierr)
      if (ierr.ne.0) return
      allocate(fjacc(np,npar), stat = ierr)
      if (ierr.ne.0) return
      allocate(fval(np), stat = ierr)
      if (ierr.ne.0) return
      allocate(resid(np), stat = ierr)
      if (ierr.ne.0) return
      allocate(theory(np), stat = ierr)
      if (ierr.ne.0) return
      allocate(wresid(np), stat = ierr)
      if (ierr.ne.0) return
      allocate(xval(np), stat = ierr)
      if (ierr.ne.0) return
      allocate(equal(np), stat = ierr)
      if (ierr.ne.0) return
c
c copy data into the workspaces and test for weighting
c
      do i = 1, npts
         error(i) = s(i)
         xval(i) = x(i)
         fval(i) = y(i)
         if (testit) then
            if (abs(one - s(i)).gt.pnt01) then
               weight = .true.
               testit = .false.
            endif
         endif
      enddo
c
c unweighted least squares regression to get starting estimates
c
      ifail = 1
      call g02caf$(npts, xval, fval, resul, ifail)
      call putifa (ifail, nout, 'G02CAF$/ORTHOG')
      if (ifail.ne.0) then
         deallocate(error, stat = ierr)
         deallocate(fjacc, stat = ierr)
         deallocate(fval, stat = ierr)
         deallocate(resid, stat = ierr)
         deallocate(theory, stat = ierr)
         deallocate(wresid, stat = ierr)
         deallocate(xval, stat = ierr)
         deallocate(equal, stat = ierr)
         return
       endif
c
c use the least squares parameters as starting estimates
c
      fact(1) = resul(6)
      fact(2) = resul(7) 
      do i = 1, 2 
         if (abs(fact(i)).le.facmin) then
            if (fact(i).ge.zero) then
               fact(i) = facmin
            else
               fact(i) = - facmin
            endif      
         endif
      enddo
      xpar(1) = one
      xpar(2) = one
      call fit_orthog (deriv_orthog, funct_orthog,
     +                 ifail,
     +                 xpar,
     +                 abort)
c
c set external parameters = internal_parameters*scaling_factors
c
      p(1) = fact(1)*xpar(1)
      p(2) = fact(2)*xpar(2)
c
c covariance matrix if required
c
      if (covar) then
         do i = 1, npts
            equal(i) = .false.
         enddo
         do i = 1, npar
            istate(i) = 1
         enddo
         call funct_orthog (npar, xpar, sigma)
         call qncov1 (funct_orthog,
     +                index, istate, nout, nfree, nhess, np, npar,
     +                npts1, npar,
     +                corr, cv, diagv, error, fact, fjacc, g, hessex,
     +                sigma, w, xpar, xval, fval, theory,
     +                equal, free)
         tstat = g01fbf$('S', pnt05, rndof, ifail)
         call putifa (ifail, nout, 'G01FBF/ORTHOG')
         do i = 1, 2
            se(i) = sqrt(diagv(i))
            if (se(i).gt.rtol) then
               pval(i) = g01ebf$('S', abs(p(i)/se(i)), rndof, ifail)
               call putifa (ifail, nout, 'G01EBF/ORTHOG')
               if (pval(i).gt.pnt05) then
                  stars(i) = '**'
               elseif (pval(i).gt.pnt01) then
                  stars(i) = ' *'
               else
                  stars(i) = '  '
               endif
            endif
         enddo
      endif
c
c table and output to file if requested
c
      if (table) then
         icolor = 15
         call table1 (icolor, 'OPEN')
         icolor = 4
         line = ' '
         write (nout,'(a)') line
         if (weight) then
           line = 'Results for weighted fitting (w = 1/s^2)'
         else
           line = 'Results for unweighted fitting (all s = 1)'
         endif
         call table1 (icolor, line)
         write (nout,'(a)') line
         if (lwtype.eq.1) then
            line = 'Model = Least squares best fit line'
         elseif (lwtype.eq.2) then
            line = 'Model = Reduced major axis best fit line'
         else
            line = 'Model = Major axis best fit line'
         endif
         call table1 (icolor, line)
         write (nout,'(a)') line
         write (line,100)
         call table1 (icolor, line)
         write (nout,'(a)') line
         icolor = 0
         if (e_numbers) then
            do i = 2, 1, - 1
               write (line,200) word12(i), p(i), se(i),
     +                          p(i) - tstat*se(i),
     +                          p(i) + tstat*se(i), pval(i), stars(i)
               call table1 (icolor, line)
               write (nout,'(a)') line
            enddo
         else
            do i = 2, 1, - 1
               d13(1) = showrj(p(i))
               d13(2) = showrj(se(i))
               temp = p(i) - tstat*se(i)
               d13(3) = showrj(temp)
               temp = p(i) + tstat*se(i)
               d13(4) = showrj(temp)
               write (line,250) word12(i), d13(1), d13(2),
     +                          d13(3), d13(4), pval(i), stars(i)
               call table1 (icolor, line)
               write (nout,'(a)') line
            enddo
           
         endif
         call nxyr2p (npts,
     +                prob, r, rsqd, x, y,
     +                abort)
         if (.not.abort) then
            write (line,300) rsqd, r, prob
            call table1 (icolor, line)
            write (nout,300) rsqd, r, prob
         endif
         call table1 (icolor, 'CLOSE')
      endif
c
c goodness of fit if requested
c
      if (goffit) then
         call gksr01 (nout, npar, npts1,
     +                resid, error, theory, wresid, xval, fval,
     +                file1, file2, graph, tshow1, tshow2)
      endif
c
c plot if requested
c
      if (plot) then
         xplot(1) = zero
         xplot(2) = zero
         yplot(1) = zero
         yplot(2) = zero
         if (e_numbers) then
            if (lwtype.eq.1) then
               write (ptitle,400) p(1), p(2)
            elseif (lwtype.eq.2) then
               write (ptitle,500) p(1), p(2)
            else
               write (ptitle,600) p(1), p(2)
            endif
         else
            d10(1) = formgr(p(1))
            d10(2) = formgr(p(2))
            if (lwtype.eq.1) then
               write (ptitle,450) trim(d10(1)), trim(d10(2))
            elseif (lwtype.eq.2) then
               write (ptitle,550) trim(d10(1)), trim(d10(2))
            else
               write (ptitle,650) trim(d10(1)), trim(d10(2))  
            endif 
         endif   
         call gks004 (n0, n1, n0, n0,
     +                n5, n0, n0, n0,
     +                npts, npts, n2, n2,
     +                xval, xval, xplot, xplot,
     +                fval, theory, yplot, yplot,
     +                ptitle, xtitle, ytitle,
     +                gsave, gsave)
      endif
c
c deallocate workspaces
c
      deallocate(error, stat = ierr)
      deallocate(fjacc, stat = ierr)
      deallocate(fval, stat = ierr)
      deallocate(resid, stat = ierr)
      deallocate(theory, stat = ierr)
      deallocate(wresid, stat = ierr)
      deallocate(xval, stat = ierr)
      deallocate(equal, stat = ierr)
c
c set abort = .false. before successful return
c
      abort = .false.
c
c format statements
c
  100 format ('Parameter         Value       Std. Error',
     +'    Lower95%cl    Upper95%cl    p')
  200 format (A,1p,4(1x,e13.5),0p,f8.4,1x,A)
  250 format (A,4(1x,a13),f8.4,1x,A)
  300 FORMAT ('(R-squared =',F7.4,', R =',F8.4,', p =',F7.4,')')
  400 format ('Least squares m =',1p,1x,e13.5,',c =',1x,e13.5)
  450 format ('Least squares m =',1x,a,',c =',1x,a)
  500 format ('Rd. major axis m =',1p,1x,e13.5,',c =',1x,e13.5)
  550 format ('Rd. major axis m =',1x,a,',c =',1x,a)
  600 format ('Major axis m =',1p,1x,e13.5,',c =',1x,e13.5)
  650 format ('Major axis m =',1x,a,',c =',1x,a)
      end
c
c
      subroutine deriv_orthog (funct,
     +                         n,
     +                         g, w, x)
c
c action : finite difference approximation to derivatives using qngrd1
c author : w.g.bardsley, university of manchester, u.k., 15/09/99
c
      implicit   none
      integer    n
      integer    inform
      double precision g(n), w(3*n), x(n)
      logical    tpoint
      parameter (tpoint = .false.)
      external   funct, qngrd1
      call qngrd1 (funct,
     +             inform, n,
     +             g, w, x,
     +             tpoint)
      end
c
c
      subroutine fit_orthog (deriv_orthog, funct_orthog,
     +                       ifail,
     +                       xpar,
     +                       abort)
c
c action : call setulb for minimisation of lines
c author : w.g.bardsley, university of manchester, u.k., 15/9/99
c          05/12/99 increased dimension lw1
c          note: liw >= 3*n
c                lw1 >= 2*(2*m*n + 4*n + 11*m*m + 8*m)
c                lw2 >= 3*n
c arguments : ifail = 0 for success
c             xpar = estimates
c             abort = .true./.false.
c locals :
c             nbd   = type of bounds (0 = unbounded, 1 = only lower,
c                     2 = both, 3 = only upper)
c             nf    = output unit
c             bl    = lower bounds
c             bu    = upper bounds
c             g     = gradient
c             w1    = double precision workspace for optimiser
c             w2    = double precision workspace for gradient
c             x     = variables
c
c extra parameters : m = no. corrections in limited memory matrix 3=< m =< 20
c                    iprint = frequency and type of output < 0 => no output
c                    factr = tolerance 1.d+7 for moderate accuracy
c                    pgtol = tolerance on projected gradient
c
      implicit         none
c
c arguments
c
      integer          ifail
      double precision xpar(2)
      logical          abort
      integer          m, n, liw, lw1, lw2
c
c locals
c
      parameter       (m = 8, n = 2,
     +                 liw = 3*n,
     +                 lw1 = 2*(2*m*n + 4*n + 11*m*m + 8*m),
     +                 lw2 = 3*n)
      integer          iw(liw), nbd(n)
      integer          i, mtry, iprint, isave(44)
      parameter       (iprint = - 1)
      integer          maxfev, numfev
      double precision zero, one, ftol
      parameter       (zero = 0.0d+00, one = 1.0d+00, ftol = 1.0d-10)
      double precision bl(n), bu(n), f, g(n), w1(lw1), w2(lw2), x(n)
      double precision factr, pgtol, dsave(29)
      character        csave*60, task*60
      logical          lsave(4)
      external         deriv_orthog, funct_orthog
      external         setulb
      intrinsic        abs
c
c Set the default output parameters in case of premature return
c
      do i = 1, 5
         w1(i) = zero
      enddo
c
c Check the input parameters before fitting
c
      abort = .true.
      ifail = 0
      i = n
      if (i.lt.1) then
         ifail = - 1
         return
      endif
      i = 3*n
      if (liw.lt.i) then
         ifail = - 2
         return
      endif
      i = 2*m*n + 4*n + 11*m*m + 8*m
      if (lw1.lt.i) then
         ifail = - 3
         return
      endif
      i = 3*n
      if (lw2.lt.i) then
         ifail = - 4
         return
      endif
      do i = 1, n
         nbd(i) = 0
      enddo
c
c We start the iteration by initializing task.
c
      task = 'START'
c
c Now we set the precision ... low, medium or high
c
      mtry = m
      factr = 1.0d+5
      pgtol = 1.0d-5
c
c ------- the beginning of the loop ----------
c

      ifail = 0
      maxfev = 200*(n + 1)
      numfev = 0
      x(1) = xpar(1)
      x(2) = xpar(2)
  20  continue
c
c This is the call to the L-BFGS-B code.
c
      call setulb (n, mtry, x, bl, bu, nbd, f, g, factr, pgtol, w1, iw,
     +             task, iprint, csave, lsave, isave, dsave)
c
c Check the number of iterations
c
      numfev = numfev + 1
      if (numfev.eq.maxfev) then
         ifail = 2
         goto 40
      endif
      if (task(1:2) .eq. 'FG') then
c
c the minimization routine has returned to request the
c function f and gradient g values at the current x.
c
         call funct_orthog (n, x, f)
         call deriv_orthog (funct_orthog, n, g, w2, x)
c
c go back to the minimization routine.
c
         goto 20
      endif
c
c
      if (task(1:5) .eq. 'NEW_X')  then
c
c the minimization routine has returned with a new iterate,
c

         if (numfev.lt.2 .or. dsave(13).gt.ftol*(one + abs(f))) then
            goto 20
         else
            task =
     +     'CONVERGENCE: projected gradient < 1.d-10*(1 + abs(f))'
         endif
      endif
c           ---------- the end of the loop -------------
c
c If task is neither FG nor NEW_X we terminate execution.
c
   40 continue
      if (ifail.eq.2) then
         ifail = 2
      elseif (task(1:4) .eq. 'CONV' .or.
     +        task .eq. 'ABNORMAL_TERMINATION_IN_LNSRCH') then
         ifail = 0
      else
         ifail = 1
      endif
      abort = .false.
      xpar(1) = x(1)
      xpar(2) = x(2)
      end
c
c
      subroutine funct_orthog (n,
     +                         xc, fc)
c
c action : objective function for orthogonal regression
c author : w.g.bardsley, university of manchester, u.k., 15/09/99
c          20/02/2006 replaced common blocks by module_orthog
c
      use module_orthog
      implicit   none
c
c arguments
c
      integer    n
      double precision fc, xc(n)
c
c locals
c
      integer    i
      double precision zero, one
      parameter (zero = 0.0d+00, one = 1.0d+00)
      double precision scale1
      logical    normal
      external   mod_orthog
      intrinsic  abs, sqrt
C
C Special action if lwtype = 2 or 3
C
      normal = .true.
      if (lwtype.eq.1) then
         scale1 = one
      elseif (lwtype.eq.2) then
         scale1 = abs(fact(1)*xc(1))
         if (scale1.gt.rtol) then
            scale1 = one/scale1
            normal = .false.
         endif
      elseif (lwtype.eq.3) then
         scale1 = fact(1)*xc(1)
         scale1 = one/(one + scale1**2)
         normal = .false.
      endif
C
C Call mod_orthog to define THEORY = calculated theoretical value
C
      call mod_orthog (n, xc)
      if (normal) then
         if (npts1.eq.1) then
C
C Get THEORY(1) for VCOVAR to calculate gradient/Jacobian
C
            fc = (fval(1) - theory(1))/error(1)
         else
C
C Calculate objective function WSSQ/NDOF
C
            fc = zero
            do i = 1, npts1
               fc = fc + ((fval(i) - theory(i))/error(i))**2
            enddo
            fc = fc/rndof
         endif
      else
         if (npts1.eq.1) then
C
C Get THEORY(1) for VCOVAR to calculate gradient/Jacobian using SCALE
C
            fc = sqrt(scale1)*(fval(1) - theory(1))/error(1)
         else
C
C Calculate objective function WSSQ/NDOF using SCALE
C
            fc = zero
            do i = 1, npts1
               fc = fc + ((fval(i) - theory(i))/error(i))**2
            enddo
            fc = scale1*fc/rndof
         endif
      endif
      end
c
c
      subroutine mod_orthog (n,
     +                       xc)
c
c action : model function for orthogonal regression
c author : w.g.bardsley, university of manchester, uk, 15/09/99
c          20/02/2006 replaced common blocks by module_orthog
c
      use module_orthog
      implicit   none
c
c arguments
c
      integer    n
      double precision xc(n)
c
c locals
c
      integer    i
      double precision constant, slope
      slope = fact(1)*xc(1)
      constant = fact(2)*xc(2)
      do i = 1, npts1
         theory(i) = slope*xval(i) + constant
      enddo
      end
c
c
