

c
c-----------------------------------------------------------------
c The SIMFIT Interfaces to lbfgs-B
c
c In addition to several independent calls, e.g. for maximum likelihood
c minimisation (Fitting Weibull for censored data), or orthogonal
c regression, the main interfaces to setulb (= lbfgsb) are as follows:
c
c QNFIT0: completely silent version
c QNFIT1: output only if bad fitting results
c QNFIT2: includes a winio control to monitor progress
c QNFIT3: inludes an interface to list01 to control random fitting
c-----------------------------------------------------------------
c
c =========================================================================
c This version is silent unless errors are encountered or bad fit results.
c Use QNFIT0 for a completely silent version or QNFIT2 for detailed output.
c =========================================================================
c
      subroutine qnfit1 (deriv, funct,
     +                   ifail, iw, liw, lw1, lw2, n, nbd, nf, npts,
     +                   bl, bu, f, g, w1, w2, x,
     +                   type1, type2)
c
c action : call setulb for minimisation
c author : w.g.bardsley, university of manchester, u.k., 9/9/97
c          13/10/1997 Added wssq1, wssq2, pcent, time, numfev to w1 output
c          14/01/1998 Revised but this version does not provide the user
c                     the chance to interrupt
c          09/11/1998 reduced dimension required for lw1 and value of m
c          18/09/1999 added call to waiter
c          05/12/1999 increased dimension required for lw1
c          27/09/2002 replaced patch1 by table1
c          18/02/2005 switched waiter on/off for unfinished optimisation,
c                     added ntotal, and imroved diagnostics
c          19/11/2009 added intents
c          19/11/2023 added e_formats and changed critical values from 2%,10% to 1%, 5%  
c
c advice : this is a replacement for e04jaf in the simfit quasi newton
c          programs
c dimensions :
c          note: liw >= 3*n
c                lw1 >= 2*(2*m*n + 4*n + 11*m*m + 8*m)
c                lw2 >= 3*n
c arguments : deriv = subroutine to calculate g (may need w2)
c             funct = subroutine to calculate f
c             iw    = integer workspace
c             liw   = size of iw
c             lw1   = size of w1
c             lw2   = size of w2
c             n     = size of bl, bu, g, x
c             nbd   = type of bounds (0 = unbounded, 1 = only lower,
c                     2 = both, 3 = only upper)
c             nf    = preconnected output unit for results/messages
c             npts  = no. of data points
c             bl    = lower bounds
c             bu    = upper bounds
c             g     = gradient
c             w1    = double precision workspace for optimiser
c             w2    = double precision workspace for gradient
c             x     = variables, i.e. parameters to be estimated
c             type1 = 'exact' or 'approximate' derivatives
c             type2 = 'low', 'medium', 'high' precision
c
c extra parameters : m = no. corrections in limited memory matrix 3=< m =< 20
c                    iprint = frequency and type of output < 0 => no output
c                    factr = tolerance 1.d+7 for moderate accuracy
c                    pgtol = tolerance on projected gradient
c
      implicit         none
c
c arguments
c
      integer,             intent (in)    :: liw, lw1, lw2, n, nf, npts
      integer,             intent (inout) :: ifail
      integer,             intent (in)    :: nbd(n)
      integer,             intent (inout) :: iw(liw)
      double precision,    intent (in)    :: bl(n), bu(n)
      double precision,    intent (inout) :: f, g(n), w1(lw1), w2(lw2),
     +                                       x(n)
      character (len = *), intent (in)    :: type1, type2
c
c locals
c
      integer          icolor, ix, iy, numtxt
      parameter       (ix = 4, iy = 4, numtxt = 21)
      integer          i, m, mtry, iprint, isave(44)
      parameter       (m = 10, iprint = - 1)
      integer          maxfev, ntotal, numfev
      double precision factr, pgtol, dsave(29)
      double precision clock2, pcent, time, t1, t2, wssq1, wssq2
      double precision zero, one, f1, f5, f100, ftol
      parameter       (zero = 0.0d+00, one = 1.0d+00, f1 = 1.0d+00,
     +                 f5 = 5.0d+00, f100 = 100.0d+00, ftol = 1.0d-10)
      character (len = 12) form12, i12(3)
      character (len = 13) d13(5), showlj
      character        csave*60, line*100, word8*8, resul*32, task*60,
     +                 text(numtxt)*100
      character        blank*1, minfit*32, nofit*32
      parameter       (blank = ' ',
     +                 minfit = 'Only a small reduction in WSSQ',
     +                 nofit = 'No appreciable reduction in WSSQ')
      logical          align1
      logical          action, lsave(4), yes
      logical          e_formats, e_numbers
      external         yesno2, table1, lcase1, triml1, clock2, waiter,
     +                 putfat
      external         deriv, funct, setulb
      external         form12, showlj, e_formats
      intrinsic        abs, dble
      e_numbers = e_formats() 
c
c Set the default output parameters in case of premature return
c
      do i = 1, 5
         w1(i) = zero
      enddo
c
c Check the input parameters before fitting
c
      ifail = 0
      if (n.lt.1) then
         ifail = - 1
         return
      endif
      i = 3*n
      if (liw.lt.i) then
         ifail = - 2
         return
      endif
      i = 2*(2*m*n + 4*n + 11*m*m + 8*m)
      if (lw1.lt.i) then
         write (line,100)
         call putfat (line)
         ifail = - 3
         return
      endif
      i = 3*n
      if (lw2.lt.i) then
         ifail = - 4
         return
      endif
      do i = 1, n
         if (bl(i).gt.bu(i)) then
            ifail = - 5
            return
         endif
      enddo
      do i = 1, n
         if (x(i).lt.bl(i) .or. x(i).gt.bu(i)) then
            ifail = - 6
            return
         endif
      enddo
      do i = 1, n
         if (nbd(i).lt.0 .or. nbd(i).gt.3) then
            ifail = - 7
            return
         endif
      enddo
c
c Call waiter with action = .true.
c
      action = .true.
      call waiter (action)
c
c Calculate wssq1
c
      call funct (n,
     +            x, wssq1)
c
c We start the iteration by initializing task.
c
      task = 'START'
c
c Now we decide on the precision ... low, medium or high
c
      if (type2(1:1).eq.'l' .or. type2(1:1).eq.'L') then
         mtry = m - 4
         factr = 1.0d+10
         pgtol = 1.0d-5
      elseif (type2(1:1).eq.'h' .or. type2(1:1).eq.'H') then
         mtry = m
         factr = 1.0d+1
         pgtol = 1.0d-7
      else
         mtry = m - 2
         factr = 1.0d+5
         pgtol = 1.0d-6
      endif
c
c ------- the beginning of the loop ----------
c

      ifail = 0
      maxfev = 200*(n + 1)
      ntotal = 0
      numfev = 0
      time = zero
      align1 = .true.
      t1 = clock2 (align1)
      align1 = .false.
c
c Label 20: the reverse communication point
c ========
c
  20  continue
c
c This is the call to the L-BFGS-B code.
c
      call setulb (n, mtry, x, bl, bu, nbd, f, g, factr, pgtol, w1, iw,
     +             task, iprint, csave, lsave, isave, dsave)
c
c Check the number of iterations
c
      numfev = numfev + 1
      ntotal = ntotal + 1
      if (numfev.eq.maxfev) then
         action = .false.
         call waiter (action)
         t2 = clock2 (align1)
         time = time + t2 - t1
         yes = .false.
         icolor = 1
         write (line,200) ntotal
         call yesno2 (icolor, ix, iy,
     +                line,
     +                yes)
         if (yes) then
            action = .true.
            call waiter (action)
            time = zero
            t1 = clock2 (align1)
            numfev = 0
            goto 20
         else
            ifail = 2
            goto 40
         endif
      endif
      if (task(1:2) .eq. 'FG') then
c
c the minimization routine has returned to request the
c function f and gradient g values at the current x.
c
         call funct (n,
     +               x, f)
         call deriv (funct,
     +               n,
     +               g, w2, x)
c
c go back to the minimization routine.
c
         goto 20
      endif
c
c
      if (task(1:5) .eq. 'NEW_X')  then
c
c the minimization routine has returned with a new iterate,
c

         if (numfev.lt.2 .or. dsave(13).gt.ftol*(one + abs(f))) then
            goto 20
         else
            task =
     +     'CONVERGENCE: projected gradient < 1.d-10*(1 + abs(f))'
         endif
      endif
c           ---------- the end of the loop -------------
c
c Label 40: If task is neither FG nor NEW_X we terminate execution.
c =========
c
   40 continue
      if (ifail.ne.2) then
         t2 = clock2 (align1)
         time = time + t2 - t1
      endif
      if (ifail.eq.2) then
         ifail = 2!to silence ftn95
      elseif (task(1:4) .eq. 'CONV' .or.
     +        task .eq. 'ABNORMAL_TERMINATION_IN_LNSRCH') then
         ifail = 0
      else
         ifail = 1
      endif
      call funct (n, x, wssq2)
      pcent = f100*(wssq1 - wssq2)/(wssq1 + ftol)
c
c call waiter with action = .false.
c
      if (action) then
         action = .false.
         call waiter (action)
      endif
c
c assess the outcome and advise user if there is any cause to worry
c
      if (pcent.lt.f5 .or. ifail.ne.0) then
         if (pcent.lt.f1) then
            resul = nofit
         elseif (pcent.lt.f5) then
            resul = minfit
         else
            resul = blank
         endif
         write (word8,'(I8)') ntotal
         call triml1 (word8)
         call lcase1 (task)
         if (e_numbers) then
            write (text,300) type1, factr, type2, pgtol, wssq1, wssq2,
     +                       ifail, pcent, resul, time, word8, npts, n,
     +                       task
         else
            d13(1) = showlj(factr)
            d13(2) = showlj(pgtol)  
            d13(3) = showlj(wssq1) 
            d13(4) = showlj(wssq2)
            d13(5) = showlj(time)
            i12(1) = form12(npts)
            i12(2) = form12(n)  
            write (text,350) type1, d13(1), type2, d13(2), d13(3),
     +                       d13(4), ifail, pcent, resul, d13(5),
     +                       word8, i12(1), i12(2), task
         endif  
         icolor = 15
         call table1 (icolor, 'OPEN')
         do i = 1, numtxt
            if (i.eq.1 .or. i.eq.numtxt) then
               icolor = 4
            elseif (i.eq.15 .and. pcent.lt.f5) then
               icolor = 4
            else
               icolor = 0
            endif
            call table1 (icolor, text(i))
         enddo
         call table1 (icolor, 'CLOSE')
         write (nf,'(A)') blank
         if (e_numbers) then
            write (nf,300) type1, factr, type2, pgtol, wssq1, wssq2,
     +                     ifail, pcent, resul, time, word8, npts, n,
     +                     task
         else
            d13(2) = showlj(pgtol)  
            d13(3) = showlj(wssq1) 
            d13(4) = showlj(wssq2)
            d13(5) = showlj(time)
            i12(1) = form12(npts)
            i12(2) = form12(n)  
            write (nf,350) type1, d13(1), type2, d13(2), d13(3),
     +                     d13(4), ifail, pcent, resul, d13(5),
     +                     word8, i12(1), i12(2), task
         endif  
         write (nf,'(A)') blank
      endif
c
c Store results in workspace w1
c
      w1(1) = wssq1
      w1(2) = wssq2
      w1(3) = pcent
      w1(4) = time
      w1(5) = dble(ntotal)
c
c format statements
c      
  100 format (
     +'Dimension LW1 too small in call to QNFIT1')
  200 format (i5,1x,
     +'iterations used ... re-enter with current parameters ?')
  300 format (
     + 'Optimisation Method: Quasi-Newton with ',A,' gradient'
     +/
     +/'Procedure QNFIT1: limited progress could be due to:'
     +/'a) wrong model or inconsistent gradient vector for these data'
     +/'b) starting estimates too close to best fit parameters'
     +/'c) starting estimates too far from best fit parameters'
     +/'d) noisy, ill-conditioned, sparse, or numerically exact data'
     +/'e) badly scaled data (i.e. units too large or too small)'
     +/'f) badly chosen data weighting factors (i.e. s-values)'
     +/'g) insufficient number of iterations (re-enter ?), or'
     +/'h) incorrect tolerance parameters factr and pgtol.'
     +/'factr              =',1P,E11.3,',  Precision: ',A
     +/'pgtol              =',1P,E11.3
     +/'wssq/ndof before   =',1P,E11.3
     +/'wssq/ndof after    =',1P,E11.3,',  IFAIL =',I3
     +/'Percent reduction  =',0P,F11.2,'% ',A
     +/'cpu time (sec)     =',1P,E11.3,',  no. of iterations = ',A
     +/'No. of data points =',I6
     +/'No. of parameters  =',I6
     +/'Convergence message (TASK on exit from lbfgsb/setulb) is:'
     +/A)
  350 format (
     + 'Optimisation Method: Quasi-Newton with ',A,' gradient'
     +/
     +/'Procedure QNFIT1: limited progress could be due to:'
     +/'a) starting estimates too close to best fit parameters'
     +/'b) starting estimates too far from best fit parameters'
     +/'c) noisy, ill-conditioned, sparse, or numerically exact data'
     +/'d) badly scaled data (i.e. units too large or too small)'
     +/'e) badly chosen data weighting factors (i.e. s-values)'
     +/'f) wrong model or inconsistent gradient vector for these data'
     +/'g) insufficient number of iterations (re-enter ?), or'
     +/'h) incorrect tolerance parameters factr and pgtol.'
     +/'factr                 =',1X,A13,' Precision: ',A
     +/'pgtol                 =',1X,A13
     +/'wssq/ndof before      =',1X,A13
     +/'wssq/ndof after       =',1X,A13,' IFAIL =',I3
     +/'Percent reduction     =',0P,F8.3,'% ',A
     +/'cpu time (sec)        =',1X,A13,' number of iterations = ',A
     +/'Number of data points =',1X,A12
     +/'Number of parameters  =',1X,A12
     +/'Convergence message (TASK on exit from lbfgsb/setulb) is: '
     +/A)    
      end
c
c
