C
C
      SUBROUTINE SURV01 (N, NCMAX, NIN, NOUT, NRMAX, 
     +                   A, 
     +                   FNAME, TITLE,
     +                   SUPPLY)
C
C ACTION : Analyse one survival time
C AUTHOR : W.G.Bardsley, University of Manchester, U.K.
C          02/10/2007 derived from SURVIV
C          24/10/2009 extended plot to t = 0, p = 1 and corrected dimension error
C          04/07/2010 added call to SURV03 to calculate step curve
C          14/05/2013 removed P from call to SVDATA
C          28/11/2013 added call to SURV04 
C          10/12/2014 now outputs parameters with more significant figures
C
C Note: the units (NIN, NOUT) and dimensions (NCMAX, NRMAX, NVMAX) are
C       (input/unchanged) and everything else is workspace
C       also NCMAX >= 3 and NVMAX >= NRMAX since used for unravelling
C
C      N = sample size (must be >= 2)
C     IC = censoring code, : 0 = failure, 1 = right censored
C  ICVEC = full censoring code after unravelling
C  IFREQ = frequencies
C    IWK = workspace
C  NCMAX = dimension (must be >= 3)
C    NIN = input unit
C   NOUT = output unit
C  NRMAX = dimension (must be >= N)
C  NVMAX = dimension after unravelling
C      A = array to hold data matrix
C      P = probabilities, i.e. KMS(t)
C   PSIG = std. dev. P
C  PSTEP = workspace
C      T = times (may be replicates)
C   TIME = workspace
C     TP = distinct times
C   TVEC = all times (no replicates after unravelling)
C     WK = workspace
C FNAME = file name
C TITLE = title
C SUPPLY = flag for data input as follows:
C          SUPPLY = .TRUE. then FNAME, TITLE, A and N are supplied
C          SUPPLY = .FALSE. then FNAME, TITLE, A, and N are set interactively 
C
      IMPLICIT   NONE
C
C Arguments
C
      INTEGER,             INTENT (IN)    :: NCMAX, NIN, NOUT, NRMAX
      INTEGER,             INTENT (INOUT) :: N
      DOUBLE PRECISION,    INTENT (INOUT) :: A(NRMAX,NCMAX)
      CHARACTER (LEN = *), INTENT (INOUT) :: FNAME, TITLE
      LOGICAL,             INTENT (IN)    :: SUPPLY
C
C Allocatables
C      
      INTEGER,          ALLOCATABLE :: IC(:), IFREQ(:), IWK(:)
      INTEGER,          ALLOCATABLE :: ICVEC(:) 
      DOUBLE PRECISION, ALLOCATABLE :: P(:), PSIG(:), T(:),
     +                                 TP(:), WK(:)
      DOUBLE PRECISION, ALLOCATABLE :: STEP(:), TIME(:), TVEC(:)
      DOUBLE PRECISION, ALLOCATABLE :: XGRAF(:), YGRAF(:)
      DOUBLE PRECISION, ALLOCATABLE :: PCENS(:), TCENS(:)
C
C Locals
C
      INTEGER    MAXIT, NGRAF, NMIN
      PARAMETER (MAXIT = 250, NGRAF = 120, NMIN = 10000)
      INTEGER    N0, N1, N2
      PARAMETER (N0 = 0, N1 = 1, N2 = 2)
      INTEGER    JCOLOR, NUMDEC, NUMOPT, NUMSTA, NUMTXT
      INTEGER    NUMBLD(30)
      INTEGER    I, ICOUNT, IERR, IFAIL, NMAX, NSUM(3), NVMAX
      INTEGER    NAP1, NCENS, ND, NDOF, NIT, NSTEP, NTEMP, NVEC
      DOUBLE PRECISION BETA, CORR, DEV, GAMMA, SEBETA, SEGAM
      DOUBLE PRECISION ONE, PNT05, PNT975, RTOL, START, TOL, TWO,
     +                 ZERO
      PARAMETER (ONE = 1.0D+00, PNT05 = 0.05D+00,
     +           PNT975 = 0.975D+00, RTOL = 1.0D-250, START = -1.0D+00,
     +           TOL = 0.0001D+00, TWO = 2.0D+00, ZERO = 0.0D+00)
      DOUBLE PRECISION HATA, HATLAM, SEHATA, SEHLAM
      DOUBLE PRECISION XBETA, XSTART, XSTOP
      DOUBLE PRECISION G01FBF$, G01EBF$
      DOUBLE PRECISION ALPHA, ARGVAL, CV, D1, D2, PARAM(5), STDERR(5),
     +                 TE, TH, TSIG(5), TSTAT, TVAL(5)
      DOUBLE PRECISION TEMP
      CHARACTER (LEN = 13) D13(4), SHOWRJ
      CHARACTER  BLANK*1, CENS*1, FREQ*1
      PARAMETER (BLANK = ' ', CENS = 'C', FREQ = 'F')
      CHARACTER  LINE*100, TEXT(30)*100
      CHARACTER  SYMPAR(5)*8, TYPE1(5)*2
      CHARACTER  PTITLE*50, XTITLE*40, YTITLE*40
      CHARACTER  CHOP80*80, TRIM80*80
      CHARACTER (LEN = 12) FORM12, WORD12(3) 
      LOGICAL    YES
      PARAMETER (YES = .TRUE.)
      LOGICAL    E_FORMATS, E_NUMBERS
      LOGICAL    ABORT, AGAIN, DONE
      EXTERNAL   E_FORMATS, SHOWRJ
      EXTERNAL   PUTIFA, LSTBOX, TABLE1, DIVIDE, GKS004, PUTADV,
     +           CHOP80, TRIM80
      EXTERNAL   SVDATA, THWEIB, SURV00, SURV03, SURV04, FORM12
      EXTERNAL   G12AAF$, G07BEF$, G01FBF$, G01EBF$
      INTRINSIC  EXP, ABS, MAX, SQRT, DBLE
      SAVE       ICOUNT
      DATA       ICOUNT / 0 /
      DATA       NUMBLD / 30*0 /
      E_NUMBERS = E_FORMATS()
C
C Check input dimensions
C      
      IF (NIN.LT.1 .OR. NOUT.LT.1 .OR. NRMAX.LT.2 .OR.
     +    NCMAX.LT.3) RETURN
C
C Initialise if required then allocate
C 
      IF (SUPPLY) THEN    
         CALL SURV00 (NCMAX, NRMAX, N, NVMAX,
     +                A)
         IF (NVMAX.LE.0) RETURN
         IF (NVMAX.LT.NMIN) NVMAX = NMIN  
      ELSE
         NVMAX = MAX(10*NRMAX, NMIN)  
         FNAME = BLANK
      ENDIF   
      IERR = 0
      IF (ALLOCATED(IC)) DEALLOCATE(IC, STAT = IERR)
      IF (IERR.NE.0) RETURN 
      IF (ALLOCATED(IFREQ)) DEALLOCATE(IFREQ, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(IWK)) DEALLOCATE(IWK, STAT = IERR)
      IF (IERR.NE.0) RETURN     
      IF (ALLOCATED(ICVEC)) DEALLOCATE(ICVEC, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(P)) DEALLOCATE(P, STAT = IERR)
      IF (IERR.NE.0) RETURN 
      IF (ALLOCATED(PSIG)) DEALLOCATE(PSIG, STAT = IERR)
      IF (IERR.NE.0) RETURN   
      IF (ALLOCATED(T)) DEALLOCATE(T, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(TP)) DEALLOCATE(TP, STAT = IERR)
      IF (IERR.NE.0) RETURN 
      IF (ALLOCATED(WK)) DEALLOCATE(WK, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(STEP)) DEALLOCATE(STEP, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(TIME)) DEALLOCATE(TIME, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(TVEC)) DEALLOCATE(TVEC, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(XGRAF)) DEALLOCATE(XGRAF, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(YGRAF)) DEALLOCATE(YGRAF, STAT = IERR)
      IF (IERR.NE.0) RETURN
C
C Allocate arrays to hold up to NRMAX items
C        
      NMAX = NRMAX  
      ALLOCATE(IC(NMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN 
      ALLOCATE(IFREQ(NMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(IWK(NMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(P(NMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(PSIG(NMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(T(NMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(TP(NMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
C
C Allocate arrays to hold data after unrolling for G07BEF
C    
      ALLOCATE(ICVEC(NVMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(TVEC(NVMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(WK(NVMAX), STAT = IERR)
      IF (IERR.NE.0) RETURN  
C
C Allocate arrays for best-fit plot
C        
      ALLOCATE(XGRAF(NGRAF), STAT = IERR)
      IF (IERR.NE.0) RETURN 
      ALLOCATE(YGRAF(NGRAF), STAT = IERR)
      IF (IERR.NE.0) RETURN 
C
C Read in if .NOT.SUPPLY then analyse data format
C                 
      CALL SVDATA (IC, ICVEC, IFREQ, N, NCMAX, NIN, NRMAX, NVEC,
     +             NVMAX,
     +             A, T, TVEC,
     +             FNAME, TITLE,
     +             ABORT, SUPPLY)
      IF (ABORT) GOTO 20
      NSUM(1) = 0
      NSUM(2) = 0
      DO I = 1, N
         IF (IC(I).EQ.0) THEN
            NSUM(1) = NSUM(1) + IFREQ(I)
         ELSE
            NSUM(2) = NSUM(2) + IFREQ(I)
         ENDIF
      ENDDO
      NSUM(3) = NSUM(1) + NSUM(2)
      WORD12(1) = FORM12(NSUM(1))            
      WORD12(2) = FORM12(NSUM(2))
      WORD12(3) = FORM12(NSUM(3))           
      DONE = .FALSE.  
C
C Call G12AAF
C
      IFAIL = 1
      CALL G12AAF$(N, T, IC, FREQ, IFREQ, ND, TP, P, PSIG, IWK,
     +             IFAIL)
      CALL PUTIFA (IFAIL, NOUT, 'G12AAF/SURVIV')
      IF (IFAIL.NE.N0) GOTO 20
C
C Call G07BEF
C
      IFAIL = N1
      GAMMA = START
      CALL G07BEF$(CENS, NVEC, TVEC, ICVEC, BETA, GAMMA, TOL, MAXIT,
     +             SEBETA, SEGAM, CORR, DEV, NIT, WK, IFAIL)
      CALL PUTIFA (IFAIL, NOUT, 'G07BEF/SURVIV')
      IF (IFAIL.NE.N0) GOTO 20
C
C Transform beta
C
      HATLAM = EXP(BETA)
      SEHLAM = HATLAM*SEBETA
      HATA = HATLAM**(ONE/GAMMA)
      D1 = HATA/GAMMA
      D2 = - HATA*BETA/(GAMMA**2)
      CV = CORR*SEBETA*SEGAM
      ARGVAL = (D1*SEBETA)**2 + (D2*SEGAM)**2 + TWO*D1*D2*CV
      SEHATA = SQRT(MAX(RTOL,ARGVAL))
C
C Calculate parameter details
C
      CALL THWEIB (BETA, CORR, GAMMA, SEBETA, SEGAM, TH, TE)
      SYMPAR(1) = '       B'
      SYMPAR(2) = '    beta'
      SYMPAR(3) = '  lambda'
      SYMPAR(4) = '       A'
      SYMPAR(5) = '  t-half'
      PARAM(1) = GAMMA
      PARAM(2) = BETA
      PARAM(3) = HATLAM
      PARAM(4) = HATA
      PARAM(5) = TH
      STDERR(1) = SEGAM
      STDERR(2) = SEBETA
      STDERR(3) = SEHLAM
      STDERR(4) = SEHATA
      STDERR(5) = TE
      IFAIL = 1
      NDOF = N - 2
      TSTAT = G01FBF$('Lower-tail', PNT975, DBLE(NDOF), IFAIL)
      CALL PUTIFA (IFAIL, NOUT, 'G01FBF/DATOUT')
      DO I = 1, 5
         IF (STDERR(I).LT.RTOL) STDERR(I) = RTOL
         ARGVAL = ABS(PARAM(I)/STDERR(I))
         IFAIL = 1
         ALPHA = ONE - G01EBF$('Lower-tail', ARGVAL, DBLE(NDOF),
     +                         IFAIL)
         CALL PUTIFA (IFAIL, NOUT, 'G01EBF/DATOUT')
         TSIG(I) = TWO*ALPHA
         TVAL(I) = TSTAT*STDERR(I)
         IF (TSIG(I).GT.PNT05) THEN
            TYPE1(I) = ' *'
         ELSE
            TYPE1(I) = BLANK
         ENDIF
      ENDDO
C
C Create best fit curve
C
      XSTART = ZERO
      XSTOP = TP(ND)
      CALL DIVIDE (NGRAF, 
     +             XGRAF, XSTART, XSTOP)
      XBETA = EXP(BETA)
      DO I = N1, NGRAF
         YGRAF(I) = EXP(-XBETA*XGRAF(I)**GAMMA)
      ENDDO
C
C Create step curve
C
      IF (ALLOCATED(STEP)) DEALLOCATE(STEP, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(TIME)) DEALLOCATE(TIME, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(PCENS)) DEALLOCATE(PCENS, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(TCENS)) DEALLOCATE(TCENS, STAT = IERR)
      IF (IERR.NE.0) RETURN 
C
C The step curve starts at t = 0, p = 1 and requires at least 2*ND + 1
C TIME, STEP pairs (plus 1 to extend to a final censored point)  
C        
      NSTEP = N2*ND + N1
      NTEMP = NSTEP + N1 
      ALLOCATE(STEP(NTEMP), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(TIME(NTEMP), STAT = IERR)
      IF (IERR.NE.0) RETURN
      NTEMP = N      
      ALLOCATE(PCENS(NTEMP), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE(TCENS(NTEMP), STAT = IERR)
      IF (IERR.NE.0) RETURN    
      CALL SURV03 (ND,
     +             P, STEP, TIME, TP,
     +             ABORT) 
      IF (ABORT) GOTO 20 
      IF (IC(N).EQ.0) THEN
         NAP1 = NSTEP
      ELSE
         NAP1 = NSTEP + 1
         TIME(NAP1) = T(N)
         STEP(NAP1) = P(ND)
      ENDIF     
      CALL SURV04 (IC, IFREQ, N, NCENS,
     +             P, PCENS, T, TCENS)                    
C
C Write details and parameters to output file
C
      ICOUNT = ICOUNT + N1
      WRITE (NOUT,100) ICOUNT, TRIM80(FNAME), CHOP80(TITLE),
     +                 WORD12(1), WORD12(2), WORD12(3)
      WRITE (NOUT,200)
      IF (E_NUMBERS) THEN
         WRITE (NOUT,300) (SYMPAR(I), PARAM(I), STDERR(I),
     +                     PARAM(I) - TVAL(I),
     +                     PARAM(I) + TVAL(I), TSIG(I),
     +                     TYPE1(I), I = 1, 5)
      ELSE
         DO I = 1, 5 
            D13(1) = SHOWRJ(PARAM(I))
            D13(2) = SHOWRJ(STDERR(I))
            TEMP = PARAM(I) - TVAL(I)
            D13(3) = SHOWRJ(TEMP)
            TEMP = PARAM(I) + TVAL(I)
            D13(4) = SHOWRJ(TEMP)
            WRITE (NOUT,310) SYMPAR(I), D13(1), D13(2), D13(3),
     +                       D13(4), TSIG(I)         
         ENDDO  
      ENDIF   
      WRITE (NOUT,400) CORR
C
C Subsidiary loop for data output from analysis of 1 data set
C
      AGAIN = .TRUE.
      DO WHILE (AGAIN)
         NUMDEC = 1
         NUMOPT = 7
         NUMSTA = 7
         NUMTXT = NUMSTA + NUMOPT - 1
         WRITE (TEXT,500) WORD12(1), WORD12(2), WORD12(3)
         NUMBLD(1) = 4
         CALL LSTBOX (NUMBLD, NUMDEC, NUMOPT, NUMSTA, NUMTXT,
     +                TEXT)
         NUMBLD(1) = 0
         IF (NUMDEC.EQ.1) THEN
C
C Plot
C
            PTITLE = 'Kaplan-Meier S(t)'
            XTITLE = 'Time'
            YTITLE = 'KMS(t)'
            CALL GKS004 (N1, N0, N0, N0,
     +                   N0, N0, N0, N0,
     +                   NSTEP, NGRAF, NGRAF, NGRAF,
     +                   TIME, XGRAF, XGRAF, XGRAF,
     +                   STEP, YGRAF, YGRAF, YGRAF,
     +                   PTITLE, XTITLE, YTITLE,
     +                   YES, YES)
           ELSEIF (NUMDEC.EQ.2) THEN
C
C Plot with Weibull
C
            PTITLE = 'Kaplan-Meier S(t) and Best Fit Weibull Curve'
            XTITLE = 'Time'
            YTITLE = 'KMS(t)'
            CALL GKS004 (N1, N2, N0, N0,
     +                   N0, N0, N0, N0,
     +                   NSTEP, NGRAF, NGRAF, NGRAF,
     +                   TIME, XGRAF, XGRAF, XGRAF,
     +                   STEP, YGRAF, YGRAF, YGRAF,
     +                   PTITLE, XTITLE, YTITLE,
     +                   YES, YES)
         ELSEIF (NUMDEC.EQ.3) THEN
C
C Advanced plot
C         
            PTITLE = 'Extended KMS(t) Curve (+ if Censored)'
            XTITLE = 'Time'
            YTITLE = 'KMS(t)'
            CALL GKS004 (N1, N0, N0, N0,
     +                   N0, N2, N0, N0,
     +                   NAP1, NCENS, NGRAF, NGRAF,
     +                   TIME, TCENS, XGRAF, XGRAF,
     +                   STEP, PCENS, YGRAF, YGRAF,
     +                   PTITLE, XTITLE, YTITLE,
     +                   YES, YES)
         ELSEIF (NUMDEC.EQ.4) THEN
C
C Output data from G12AAF to screen
C
            JCOLOR = 15
            CALL TABLE1 (JCOLOR, 'OPEN')
            WRITE (LINE,600)
            JCOLOR = 4
            CALL TABLE1 (JCOLOR, LINE)
            JCOLOR = 0
            IF (E_NUMBERS) THEN
               DO I = 1, ND
                  WRITE (LINE,700) TP(I), P(I), PSIG(I)
                  CALL TABLE1 (JCOLOR, LINE)
               ENDDO
            ELSE
               DO I = 1, ND
                  D13(1) = SHOWRJ(TP(I))
                  WRITE (LINE,710) D13(1), P(I), PSIG(I)
                  CALL TABLE1 (JCOLOR, LINE)
               ENDDO     
            ENDIF  
            CALL TABLE1 (JCOLOR, 'CLOSE')
         ELSEIF (NUMDEC.EQ.5) THEN
C
C Output data from G12AAF to results file
C
            IF (DONE) THEN
               WRITE (LINE,800)
               CALL PUTADV (LINE) 
            ELSE
               DONE = .TRUE.
               WRITE (NOUT,'(A)') BLANK
               WRITE (NOUT,600)
               IF (E_NUMBERS) THEN
                  DO I = 1, ND
                     WRITE (NOUT,700) TP(I), P(I), PSIG(I)
                  ENDDO
               ELSE
                  DO I = 1, ND
                     D13(1) = SHOWRJ(TP(I))
                     WRITE (NOUT,710) D13(1), P(I), PSIG(I)
                  ENDDO 
               ENDIF  
               WRITE (LINE,900)
               CALL PUTADV (LINE)
            ENDIF   
         ELSEIF (NUMDEC.EQ.6) THEN
C
C Output data from G07BEF to screen
C
            WRITE (TEXT,200)
            JCOLOR = 15
            CALL TABLE1 (JCOLOR, 'OPEN')
            DO I = 1, 7
               IF (I.EQ.1 .OR. I.EQ.7) THEN
                  JCOLOR = 4
               ELSE
                  JCOLOR = 0
               ENDIF
                  CALL TABLE1 (JCOLOR, TEXT(I))
            ENDDO
            IF (E_NUMBERS) THEN
               WRITE (TEXT,300) (SYMPAR(I), PARAM(I), STDERR(I),
     +                           PARAM(I) - TVAL(I),
     +                           PARAM(I) + TVAL(I), TSIG(I),
     +                           TYPE1(I), I = 1, 5)
            ELSE
               DO I = 1, 5 
                  D13(1) = SHOWRJ(PARAM(I))
                  D13(2) = SHOWRJ(STDERR(I))
                  TEMP = PARAM(I) - TVAL(I)
                  D13(3) = SHOWRJ(TEMP)
                  TEMP = PARAM(I) + TVAL(I)
                  D13(4) = SHOWRJ(TEMP)
                  WRITE (TEXT(I),310) SYMPAR(I), D13(1), D13(2), D13(3),
     +                                D13(4), TSIG(I)
               ENDDO 
            ENDIF  
            JCOLOR = 0
            DO I = 1, 5
               CALL TABLE1 (JCOLOR, TEXT(I))
            ENDDO
            WRITE (TEXT(1),400) CORR
            CALL TABLE1 (JCOLOR, TEXT(1))
            CALL TABLE1 (JCOLOR, 'CLOSE')
         ELSE
            AGAIN = .FALSE.
         ENDIF
      ENDDO  
C
C LABEL 20: deallocate
C      
   20 CONTINUE 
      DEALLOCATE(IC, STAT = IERR)
      DEALLOCATE(IFREQ, STAT = IERR)
      DEALLOCATE(IWK, STAT = IERR)
      DEALLOCATE(ICVEC, STAT = IERR)
      DEALLOCATE(P, STAT = IERR)
      DEALLOCATE(PSIG, STAT = IERR)
      DEALLOCATE(T, STAT = IERR)
      DEALLOCATE(TP, STAT = IERR)
      DEALLOCATE(WK, STAT = IERR)
      DEALLOCATE(STEP, STAT = IERR)
      DEALLOCATE(TIME, STAT = IERR)
      DEALLOCATE(TVEC, STAT = IERR)
      DEALLOCATE(XGRAF, STAT = IERR)
      DEALLOCATE(YGRAF, STAT = IERR)
      DEALLOCATE(PCENS, STAT = IERR)
      DEALLOCATE(TCENS, STAT = IERR)
C
C Format statements
C
  100 FORMAT (
     +/1X,'Single Survival Analysis no.',I4
     +/1X,'================================'
     +/1X,'File name'
     +/A
     +/1X,'Data title'
     +/A
     +/'Number failed =',1x,a
     +/'Number censored =',1x,a
     +/'Sample size =',1x,a
     +/)
  200 FORMAT (
     + ' Alternative MLE Weibull parameterizations',
     +/
     +/' S(t) = exp[-{exp(beta)}t^B]'
     +/'      = exp[-{lambda}t^B]'
     +/'      = exp[-{A*t}^B]'
     +/
     +/'Parameter     Value        Std. err.    Lower95%cl',
     +'    Upper95%cl    p')
  300 FORMAT (A8,1P,1X,E13.5,1X,E13.5,1X,E13.5,1X,E13.5,0P,F8.4,A2)
  310 FORMAT (A8,1X,A13,1X,A13,1X,A13,1X,A13,F8.4,A2)
  400 FORMAT (' Correlation coefficient(beta,B) =',F8.4)
  500 FORMAT (
     + 'Analysis of one set of survival data'
     +/  
     +/'Number failed =',1x,a
     +/'Number censored =',1x,a
     +/'Sample size =',1x,a
     +/
     +/'Plot a KMS(t) curve'
     +/'Plot a KMS(t) curve with best-fit Weibull'
     +/'Plot a KMS(t) curve with censored points'
     +/'Display a KMS(t) table'
     +/'Write table to file'
     +/'Display parameters'
     +/'Quit ... Exit survival analysis for 1 data set')
  600 FORMAT ('         Time   KMS(t)  Std.Err.')
  700 FORMAT (1P,E13.5,0P,2F9.4)
  710 FORMAT (A13,2F9.4)
  800 FORMAT ('Already done')
  900 FORMAT ('The table has been written to the results file')
      END
C
C
