C
C
      SUBROUTINE SVDVAL (IRANK, LWORK, NCMAX, NCOL, NDMAX, NIN, NF,
     +                   NRMAX, NROW, NSVD,
     +                   A, D, WORK,
     +                   TITLE,
     +                   ABORT, DISPLY, FILE, SUPPLY)
C
C ACTION : SVD using BLAS/LAPACK
C AUTHOR : W.G.Bardsley, University of Manchester, U.K., 30/7/97
C          27/07/2001 Altered output so that results are filed only if
C                     DISPLAY and FILE are both .TRUE. and also wrote the
C                     singular vectors to U and VT if successful
C          01/04/2003 added control over output
C          03/12/2003 revised and added FILE1, FILE2, FILE3, FILE4
C          09/01/2006 moved B, C, U, VT from argument list to allocatables
C          05/02/2006 moved TITLE to argument list
C          21/08/2006 added proportions and percentages to output
C          23/08/2006 added calls to F06QFF and used new scheme for dimensions
C                     so that U and VT always have the minimum dimensions
C          06/10/2006 made sure TITLE1 is defined before calling VECOUT and MATOUT
C          27/11/2006 introduced NCADD and NRADD to allow use of /check_mate
C          24/02/2008 introduced tabulating and plotting fractions for sigma^2 
C          18/03/2010 FIXED nradd = 0 but allowed NCADD to vary (despite /RAGGED_ARRAYS) since
C                     DGEBD2 calls DLARF(G) which actually does need an extra column
C          11/05/2010 introduced NKLCFG to switch on/off the test file advice 
C          30/04/2011 introduced call to TFILEQ
C          17/09/2015 set NCADD = 2 so it can be called with SUPPLy = .TRUE. and NCMAX > NCOL
C          01/11/2021 added E_NUMBERS and E_FORMATS, etc.
C   
C ADVICE :  LWORK should be >= 64*(NCOL + NROW) for optimum results
C           If A is supplied it is returned unchanged while if A is
C           read in it is returned unchanged because A is copied into B
C           If DISPLY = FILE = .FALSE. and SUPPLY = .TRUE. the routine is silent
C           On successful exit for a M by N matrix WORK contains the first three
C           components of U and VT as follows:
C                               WORK(1) to WORK(M) =  U(1,1) to  U(M,1)
C                         WORK(M + 1) to WORK(2*M) =  U(1,2) to  U(M,2)
C                       WORK(2*M + 1) to WORK(3*M) =  U(1,3) to  U(M,3)
C                       WORK(3*M + 1) to WORK(4*M) =  U(1,4) to  U(M,4)
C                   WORK(4*M + 1) to WORK(4*M + N) = VT(1,1) to VT(1,N)
C             WORK(4*M + N + 1) to WORK(4*M + 2*N) = VT(2,1) to VT(2,N)
C           WORK(4*M + 2*N + 1) to WORK(4*M + 3*N) = VT(3,1) to VT(3,N)
C           WORK(4*M + 3*N + 1) to WORK(4*M + 4*N) = VT(4,1) to VT(4,N)
C
C           IRANK: (output) rank
C           LWORK: (input/unchanged) dimension of WORK >= 64*(NCMAX + NRMAX)
C           NCMAX: (input/unchanged) max. column dimension
C            NCOL: (input/output) as follows: input if SUPPLY= .TRUE. o/w output
C           NDMAX: (input/unchanged) max. no. singular values MAX(NCMAX, NRMAX) to
C                                    be safe but actually MIN(NCMAX,NRMAX) will do
C             NIN: (input/unchanged) unconnected data input unit
C              NF: (input/unchanged) preconnected unit for results
C           NRMAX: (input/unchanged) max. row dimension
C            NROW: (input/output) as follows: input if SUPPLY = .TRUE. o/w output
C            NSVD: (output) no. of singular values
C               A: (input/output) as follows: input if SUPPLY = .TRUE. o/w output
C               D: (output) NSD singular values
C            WORK: workspace
C           TITLE: (input/output) as follows: input if SUPPLY o/w output
C           ABORT: (output) error indicator
C          DISPLY: (input/unchanged) display results table if .TRUE.
C            FILE: (input/unchanged) write to results file if .TRUE.
C                                    and also DSPLY = .TRUE.
C          SUPPLY: (input/unchanged) supply matrix in A
C
      IMPLICIT   NONE
C
C Arguments
C
      INTEGER,             INTENT (IN)    :: LWORK, NCMAX, NDMAX, NIN,
     +                                       NF, NRMAX 
      INTEGER,             INTENT (OUT)   :: IRANK, NSVD  
      INTEGER,             INTENT (INOUT) :: NCOL, NROW
      DOUBLE PRECISION,    INTENT (INOUT) :: A(NRMAX,NCMAX), D(NDMAX),
     +                                       WORK(LWORK)
      CHARACTER (LEN = *), INTENT (INOUT) :: TITLE
      LOGICAL,             INTENT (IN)    :: DISPLY, FILE, SUPPLY 
      LOGICAL,             INTENT (OUT)   :: ABORT
C
C Local allocatable arrays
C
      DOUBLE PRECISION, ALLOCATABLE :: B(:,:), C(:,:), U(:,:), VT(:,:)
      DOUBLE PRECISION, ALLOCATABLE :: E(:), TAUP(:), TAUQ(:), Y(:)
C
C Locals
C
      INTEGER    LDB, LDVT, LDU, LDC, NCOLU, NCOLVT, NROW1
      INTEGER    I, IERR, INFO, J, K, M, N
      INTEGER    KVAL9, NKLCFG
      INTEGER    ISEND
      INTEGER    ICOLOR, IX, IY, NUMDEC, NUMOPT
      PARAMETER (ICOLOR = 7, IX = 4, IY = 4, NUMOPT = 14)
      INTEGER    NUMPOS(NUMOPT), NBIG
      INTEGER    NTYPE, N0, N1, N2, N4, N5, N15, N21
      PARAMETER (NTYPE = 3, N0 = 0, N1 = 1, N2 = 2, N4 = 4, N5 = 5,
     +           N15 = 15, N21 = 21)
      INTEGER    NCADD, NRADD
      DOUBLE PRECISION SSQ, SUMSVD, TEMP, TEMP1, TEMP1_SSQ, TEMP2,
     +                 TEMP2_SSQ, TOL
      DOUBLE PRECISION X02AJF$
      DOUBLE PRECISION ZERO, F100
      PARAMETER (ZERO = 0.0D+00, F100 = 100.0D+00)
      CHARACTER (LEN = 13) D13(2), SHOWRJ
      CHARACTER  FNAME*1024
      CHARACTER  FNAME1*1024, TEXT1(1)*80, TITLE1*80, WORD8*8
      CHARACTER  LINE*100, TEXT(30)*100
      CHARACTER  PTITLE*50, XTITLE*40, YTITLE*40
      CHARACTER  BLANK*1
      PARAMETER (BLANK = ' ')
      LOGICAL    E_NUMBERS, E_FORMATS
      LOGICAL    ABORT1, FILE1, FILE2, FILE3, FILE4, REPEET
      LOGICAL    FIXCOL, FIXROW, LABEL
      PARAMETER (FIXCOL = .FALSE., FIXROW = .FALSE., LABEL = .TRUE.)
      LOGICAL    HEADER, QTEXT, QTITLE
      PARAMETER (HEADER = .TRUE., QTEXT = .TRUE., QTITLE = .TRUE.)
      EXTERNAL   E_FORMATS, SHOWRJ
      EXTERNAL   MATTIN, TABLE1, DSPLAY, LBOX02, MATOUT, VECOUT,
     +           TRIML1, GKS001, REVPRO
      EXTERNAL   NKLCFG, TFILEQ
      EXTERNAL   F08KEF$, F08KFF$, F08MEF$, F06QFF$, X02AJF$
      INTRINSIC  MIN, MAX, DBLE
      DATA       NUMPOS / NUMOPT*1 /
C
C Initialise
C
      NCADD = 2
      NRADD = 0
      IRANK = 0
      NSVD = 0
      ABORT = .TRUE.
      LDB = NRMAX + NRADD
      LDC = 1
C
C Allocate the workspaces
C
      IERR = 0
      IF (ALLOCATED(B)) DEALLOCATE(B, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(C)) DEALLOCATE(C, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(E)) DEALLOCATE(E, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(TAUP)) DEALLOCATE(TAUP, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(TAUQ)) DEALLOCATE(TAUQ, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(U)) DEALLOCATE(U, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(VT)) DEALLOCATE(VT, STAT = IERR)
      IF (IERR.NE.0) RETURN
      IF (ALLOCATED(Y)) DEALLOCATE(Y, STAT = IERR)
      IF (IERR.NE.0) RETURN  
      ALLOCATE (B(LDB,NCMAX + NCADD), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE (C(LDC,LDC), STAT = IERR)
      IF (IERR.NE.0) RETURN
        
      NBIG = MAX(NCMAX,NRMAX)
      
      ALLOCATE (E(NBIG), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE (TAUP(NBIG), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE (TAUQ(NBIG), STAT = IERR)
      IF (IERR.NE.0) RETURN
      ALLOCATE (Y(NBIG), STAT = IERR)
      IF (IERR.NE.0) RETURN
C
C Read in the matrix if .NOT.SUPPLY
C
      FNAME = BLANK
      IF (.NOT.SUPPLY) THEN
         KVAL9 = NKLCFG(N21)
         IF (KVAL9.EQ.N1) THEN
            WRITE (LINE,100)
            CALL TFILEQ (LINE)
         ENDIF   
         I = N0
         CLOSE (UNIT = NIN)
         CALL MATTIN (I, NCMAX, NCOL, NIN, NRMAX, NROW,
     +                A, D,
     +                FNAME, TITLE,
     +                ABORT, FIXCOL, FIXROW, LABEL)
         CLOSE (UNIT = NIN)
         IF (ABORT) THEN
            DEALLOCATE(B, STAT = IERR)
            DEALLOCATE(C, STAT = IERR)
            DEALLOCATE(E, STAT = IERR)
            DEALLOCATE(TAUP, STAT = IERR)
            DEALLOCATE(TAUQ, STAT = IERR)
            DEALLOCATE(U, STAT = IERR)
            DEALLOCATE(VT, STAT = IERR)
            DEALLOCATE(Y, STAT = IERR)
            RETURN
         ENDIF
      ENDIF
      IF (NCOL.LT.N2 .OR. NROW.LT.N2) THEN
         DEALLOCATE(B, STAT = IERR)
         DEALLOCATE(C, STAT = IERR)
         DEALLOCATE(E, STAT = IERR)
         DEALLOCATE(TAUP, STAT = IERR)
         DEALLOCATE(TAUQ, STAT = IERR)
         DEALLOCATE(U, STAT = IERR)
         DEALLOCATE(VT, STAT = IERR)
         ABORT = .TRUE.
         RETURN
      ENDIF
C
C Define M and N then copy A into B since the matrix is overwritten
C
      
      M = NROW
      N = NCOL
      DO J = N1, NCOL
         DO I = N1, NROW
            B(I,J) = A(I,J)
         ENDDO
      ENDDO
C
C Reduce A (i.e. B now) to bidiagonal form
C
      CALL F08KEF$(M, N, B, LDB, D, E, TAUQ, TAUP, WORK, LWORK, INFO)
      IF (INFO.NE.N0) THEN
         DEALLOCATE(B, STAT = IERR)
         DEALLOCATE(C, STAT = IERR)
         DEALLOCATE(E, STAT = IERR)
         DEALLOCATE(TAUP, STAT = IERR)
         DEALLOCATE(TAUQ, STAT = IERR)
         DEALLOCATE(U, STAT = IERR)
         DEALLOCATE(VT, STAT = IERR)
         DEALLOCATE(Y, STAT = IERR)
         ABORT = .TRUE.
         RETURN
      ENDIF
C
C The SVD
C

      IF (M.GE.N) THEN
          NSVD = N 
          LDVT = N + NRADD
          NCOLVT = N         
          NROW1 = N
          IF (ALLOCATED(VT)) DEALLOCATE(VT, STAT = IERR)
          IF (IERR.NE.0) RETURN
          ALLOCATE(VT(LDVT,NCOLVT + NCADD), STAT = IERR)
          IF (IERR.NE.0) RETURN
          CALL F06QFF$('Upper', N, N, B, LDB, VT, LDVT)
          LDU = M + NRADD
          NCOLU = N
          IF (ALLOCATED(U)) DEALLOCATE(U, STAT = IERR)
          IF (IERR.NE.0) RETURN
          ALLOCATE(U(LDU,NCOLU + NCADD), STAT = IERR)
          IF (IERR.NE.0) RETURN
          CALL F06QFF$('Lower', M, N, B, LDB, U, LDU)
          CALL F08KFF$('P', N, N, M, VT, LDVT, TAUP, WORK, LWORK,
     +                 INFO)
          CALL F08KFF$('Q', M, N, N, U, LDU, TAUQ, WORK, LWORK,
     +                 INFO)
          CALL F08MEF$('Upper', N, N, M, N0, D, E, VT, LDVT, U, LDU,
     +                 C, LDC, WORK, INFO)
      ELSE
          NSVD = M
          LDVT = N + NRADD
          NCOLVT = N 
          NROW1 = M
          IF (ALLOCATED(VT)) DEALLOCATE(VT, STAT = IERR)
          IF (IERR.NE.0) RETURN
          ALLOCATE(VT(LDVT,NCOLVT + NCADD), STAT = IERR)
          IF (IERR.NE.0) RETURN
          CALL F06QFF$('Upper', M, N, B, LDB, VT, LDVT)
          LDU = M + NRADD
          NCOLU = M 
          IF (ALLOCATED(U)) DEALLOCATE(U, STAT = IERR)
          IF (IERR.NE.0) RETURN
          ALLOCATE(U(LDU,NCOLU + NCADD), STAT = IERR)
          IF (IERR.NE.0) RETURN
          CALL F06QFF$('Lower', M, M, B, LDB, U, LDU)
          CALL F08KFF$('P', M, N, M, VT, LDVT, TAUP, WORK, LWORK,
     +                 INFO)
          CALL F08KFF$('Q', M, M, N, U, LDU, TAUQ, WORK, LWORK,
     +                 INFO)
          CALL F08MEF$('Lower', M, N, M, N0, D, E, VT, LDVT, U, LDU,
     +                 C, LDC, WORK, INFO)
      ENDIF
      IF (INFO.EQ.N0) THEN
C
C SVD has succeeded so calculate the rank
C
         E_NUMBERS = E_FORMATS()  
         ABORT = .FALSE.
         TOL = X02AJF$()*D(1)
         IRANK = N1
         DO I = N2, NSVD
            IF (D(I).GE.TOL) THEN
               IRANK = IRANK + N1
            ELSE
               D(I) = ZERO
            ENDIF
         ENDDO
C
C Load the first four components U and VT into WORK
C
         IF (IRANK.GE.1) THEN
            DO I = N1, M
               WORK(I) = U(I,1)
            ENDDO
            K = 4*M
            DO J = N1, N
               WORK(K + J) = VT(1,J)
            ENDDO
         ENDIF
         IF (IRANK.GE.2) THEN
            K = M
            DO I = N1, M
               WORK(K + I) = U(I,2)
            ENDDO
            K = 4*M + N
            DO J = N1, N
               WORK(K + J) = VT(2,J)
            ENDDO
         ENDIF
         IF (IRANK.GE.3) THEN
            K = 2*M
            DO I = N1, M
               WORK(K + I) = U(I,3)
            ENDDO
            K = 4*M + 2*N
            DO J = N1, N
               WORK(K + J) = VT(3,J)
            ENDDO
         ENDIF
         IF (IRANK.GE.4) THEN
            K = 3*M
            DO I = N1, M
               WORK(K + I) = U(I,4)
            ENDDO
            K = 4*M + 3*N
            DO J = N1, N
               WORK(K + J) = VT(4,J)
            ENDDO
         ENDIF
C
C**********************************************
C Note: WORK must NOT be used again from now on
C**********************************************
C
         
C
C Calculate SUMSVD and SSQ
C
         SUMSVD = ZERO
         SSQ = ZERO
         DO I = N1, IRANK
            SUMSVD = SUMSVD + D(I)
            SSQ = SSQ + D(I)*D(I)
         ENDDO
C
C Define WORD8
C         
         WRITE (WORD8,'(I8)') IRANK
         CALL TRIML1 (WORD8)
C
C Output a table if DISPLY = .TRUE.
C
         IF (DISPLY) THEN
            FILE1 = FILE
            FILE2 = FILE
            FILE3 = FILE
            FILE4 = FILE
            IF (FILE) THEN
               WRITE (NF,'(A)') BLANK
               WRITE (NF,200) WORD8
               WRITE (NF,'(A)') TITLE
            ENDIF
            IF (MIN(NCOL,NROW).GT.20) THEN
                FILE1 = .FALSE.
                IF (MIN(NCOL,NROW).GT.30) FILE2 = .FALSE.
                FILE3 = .FALSE.
                FILE4 = .FALSE.
            ENDIF
            WRITE (TEXT,300)
            NUMDEC = N1
            REPEET = .TRUE.
            DO WHILE (REPEET)
C
C Loop to choose options
C
               ISEND = N1
               CALL LBOX02 (ICOLOR, IX, IY, NUMDEC, NUMOPT, NUMPOS,
     +                      TEXT)
               IF (NUMDEC.EQ.1) THEN
                  WRITE (LINE,400)
                  CALL DSPLAY (NCMAX, NCOL, NF, NRMAX, NROW, NTYPE,
     +                         A,
     +                         LINE,
     +                         FILE1)
                  IF (FILE1) WRITE (NF,'(A)') BLANK
                  FILE1 = .FALSE.
               ELSEIF (NUMDEC.EQ.2) THEN
                  CALL TABLE1 (N15, 'OPEN')
               
                  WRITE (LINE,500) WORD8
                  CALL TABLE1 (N4, LINE)
                  TEMP2 = ZERO
                  TEMP2_SSQ = ZERO
                  DO J = N1, IRANK
                     TEMP1 = D(J)/SUMSVD
                     TEMP2 = TEMP2 + TEMP1
                     TEMP1_SSQ = D(J)*D(J)/SSQ
                     TEMP2_SSQ = TEMP2_SSQ + TEMP1_SSQ
                     IF (E_NUMBERS) THEN
                        WRITE (LINE,600) J, D(J), TEMP1, TEMP2,
     +                                   D(J)*D(J), TEMP1_SSQ, TEMP2_SSQ                     
                     ELSE
                        D13(1) = SHOWRJ(D(J))
                        TEMP = D(J)*D(J)
                        D13(2) = SHOWRJ(TEMP) 
                        WRITE (LINE,650) J, D13(1), TEMP1, TEMP2,
     +                                   D13(2), TEMP1_SSQ, TEMP2_SSQ   
                     ENDIF  
                     CALL TABLE1 (N0, LINE)
                  ENDDO
                  CALL TABLE1 (N0, 'CLOSE')
C
C Write to file if FILE = .TRUE.
C
                  IF (FILE2) THEN
                     WRITE (NF,500) WORD8
                     TEMP2 = ZERO
                     TEMP2_SSQ = ZERO
                     DO J = N1, NSVD
                        TEMP1 = D(J)/SUMSVD
                        TEMP2 = TEMP2 + TEMP1
                        TEMP1_SSQ = D(J)*D(J)/SSQ
                        TEMP2_SSQ = TEMP2_SSQ + TEMP1_SSQ
                        IF (E_NUMBERS) THEN
                           WRITE (NF,600) J, D(J), TEMP1, TEMP2,
     +                                    D(J)*D(J), TEMP1_SSQ,
     +                                    TEMP2_SSQ   
                        ELSE
                           D13(1) = SHOWRJ(D(J))
                           TEMP = D(J)*D(J) 
                           D13(2) = SHOWRJ(TEMP) 
                           WRITE (NF,650) J, D13(1), TEMP1, TEMP2,
     +                                    D13(2), TEMP1_SSQ, TEMP2_SSQ   
                        ENDIF
                     ENDDO
                     FILE2 = .FALSE.
                  ENDIF

C Output the singular vectors
C
               ELSEIF (NUMDEC.LE.4) THEN
                  IF (NUMDEC.EQ.3) THEN
                     WRITE (LINE,700)
                     CALL DSPLAY (NCOLVT, NCOLVT, NF, LDVT, NROW1,
     +                            NTYPE,
     +                            VT,
     +                            LINE,
     +                            FILE3)
                     FILE3 = .FALSE.
                  ELSE
                     WRITE (LINE,800)
                     CALL DSPLAY (NCOLU, NCOLU, NF, LDU, M, NTYPE,
     +                            U,
     +                            LINE,
     +                            FILE4)
                     FILE4 = .FALSE.
                  ENDIF
                  NUMDEC = NUMOPT
               ELSEIF (NUMDEC.EQ.5) THEN
C
C Write svd to file
C
                  ISEND = N1
                  TITLE1 = 'Singular values'
                  CLOSE (UNIT = NIN)
                  CALL VECOUT (ISEND, NSVD, NIN, NSVD,
     +                         D,
     +                         FNAME1, TITLE1,
     +                         ABORT1, QTEXT, QTITLE)
                  CLOSE (UNIT = NIN)
                  NUMDEC = NUMOPT
C
C Output the singular vectors
C
               ELSEIF (NUMDEC.LE.7) THEN
                  IF (NUMDEC.EQ.6) THEN
                     ISEND = N1
                     TEXT1(1) = BLANK
                     TITLE1 = 'SVD Matrix V^T'
                     CLOSE (UNIT = NIN)
                     CALL MATOUT (ISEND, NCOLVT, NIN, LDVT, NROW1, N1,
     +                            VT,
     +                            FNAME1, TEXT1, TITLE1,
     +                            ABORT1, HEADER, QTEXT, QTITLE)
                     CLOSE (UNIT = NIN)
                  ELSE
                     ISEND = N1  
                     TEXT1(1) = BLANK
                     TITLE1 = 'SVD Matrix U'
                     CLOSE (UNIT = NIN)
                     CALL MATOUT (ISEND, NCOLU, NIN, LDU, M, N1,
     +                            U,
     +                            FNAME1, TEXT1, TITLE1,
     +                            ABORT1, HEADER, QTEXT, QTITLE)
                     CLOSE (UNIT = NIN)
                  ENDIF
                  NUMDEC = NUMOPT
               ELSEIF (NUMDEC.LT.13) THEN
C
C Plot
C
                  DO I = N1, IRANK
                     E(I) = DBLE(I)
                  ENDDO
                  XTITLE = 'Index i'
                  
                  IF (NUMDEC.EQ.8) THEN
                     PTITLE = 'Singular Value Decomposition'
                     YTITLE = 'sigma(i)'
                     DO I = 1, IRANK
                        Y(I) = D(I) 
                     ENDDO   
                  ELSEIF (NUMDEC.EQ.9) THEN
                     PTITLE = 'Cumulative sigma(i)'
                     YTITLE = 'Proportion'
                     TEMP1 = ZERO 
                     DO I = 1, IRANK
                        TEMP1 = TEMP1 + D(I)
                        Y(I) = TEMP1/SUMSVD
                     ENDDO  
                  ELSEIF (NUMDEC.EQ.10) THEN
                     PTITLE = 'Sigma(i)^2 Values'
                     YTITLE = 'sigma(i)^2'
                     DO I = 1, IRANK
                        Y(I) = D(I)*D(I)
                     ENDDO  
                  ELSEIF (NUMDEC.EQ.11) THEN
                     PTITLE = 'Cumulative sigma(i)^2'
                     YTITLE = 'Proportion'
                     TEMP1 = ZERO 
                     DO I = 1, IRANK
                        TEMP1 = TEMP1 + D(I)*D(I)
                        Y(I) = TEMP1/SSQ
                     ENDDO            
                  ELSEIF (NUMDEC.EQ.12) THEN
                     PTITLE = '100*cumulative sigma(i)^2'
                     YTITLE = 'Percentage Variance'
                     TEMP1 = ZERO 
                     DO I = 1, IRANK
                        TEMP1 = TEMP1 + D(I)*D(I)
                        Y(I) = F100*TEMP1/SSQ
                     ENDDO               
                  ENDIF 
                  CALL GKS001 (N1, N5, IRANK,
     +                         E, Y,
     +                         PTITLE, XTITLE, YTITLE)
     +
                  NUMDEC = NUMOPT
               ELSEIF (NUMDEC.EQ.NUMOPT - 1) THEN
                  CALL REVPRO (NF)
                  NUMDEC = NUMOPT   
               ELSE
                  REPEET = .FALSE.
               ENDIF
            ENDDO
         ENDIF
      ELSE
         ABORT = .TRUE. 
      ENDIF
C
C
      DEALLOCATE(B, STAT = IERR)
      DEALLOCATE(C, STAT = IERR)
      DEALLOCATE(E, STAT = IERR)
      DEALLOCATE(TAUP, STAT = IERR)
      DEALLOCATE(TAUQ, STAT = IERR)
      DEALLOCATE(U, STAT = IERR)
      DEALLOCATE(VT, STAT = IERR)
      DEALLOCATE(Y, STAT = IERR)

C
C Format statements
C
  100 FORMAT ('Now input a n by m matrix formatted like matrix.tf2')
  200 FORMAT ('Title of current data with rank =',1X,A)
  300 FORMAT (
     + 'Display matrix U*S*(V-transpose)'
     +/'Display singular values'
     +/'Display V-transpose'
     +/'Display U'
     +/'File: Save singular values As ...'
     +/'File: Save V-transpose As ...'
     +/'File: Save U As ...'
     +/'Plot sigma'
     +/'Plot cumulative sigma'
     +/'Plot sigma squared'
     +/'Plot cumulative sigma squared'
     +/'Plot percentage variance'
     +/'Results'
     +/'Quit ... Exit these SVD options')
  400 FORMAT ('Current matrix:')
  500 FORMAT ('Index       Sigma(i) Fraction Cumulative',
     +        '    Sigma(i)^2 Fraction Cumulative: rank =',1X,A)
  600 FORMAT (I5,1P,E15.7,0P,F9.4,F11.4,1P,1X,E13.5,0P,F9.4,F11.4)
  650 FORMAT (I5,2X,A13,F9.4,F11.4,1X,A13,F9.4,F11.4)
  700 FORMAT ('Right singular vectors by row (V-transpose)')
  800 FORMAT ('Left singular vectors by column (U)')
      END
C
C
