C
C Code for Trinomial Analysis
C ===========================
C TRINOM
C TRINXY
C FCNXYZ
C
C
      SUBROUTINE TRINOM (NCMAX, NIN, NOBS, NOUT, NRMAX, NROW,
     +                   A, B, PRED,
     +                   FNAME, TITLE,
     +                   NEWDAT, SUPPLY)
C
C ACTION : Analyse a trinomial distribution given x, y, and N = x + y + z
C AUTHOR : W.G.Bardsley, University of Manchester, U.K., 21/11/96
C          19/09/2005 revised to allow up to 12 plots
C          24/10/2005 revised to add 0.5 when empty cells encountered
C          22/09/2007 added INTENTS and FNAME, TITLE, NEWDAT, SUPPLY to argument list 
C          04/12/2020 increased NGRAF from 200 to 300 and NPTS starts at 150 instead of 100
C
C  NCMAX: (input/unchanged) dimension
c    NIN: (input/unchanged) unconnected unit for data input
C   NOBS: workspace
C   NOUT: (input/unchanged) unit pre-connected unit for results
C  NRMAX: (input/unchanged) dimension
C   NROW: (input/output) row dimension 
C      A: workspace
C      B: workspace
C   PRED: workspace
C  FNAME: (input/output) filename
C  TITLE: (input/output) filename
C NEWDAT: (input/output)
C SUPPLY: (input/unchanged) 
C
C ADVICE : Adjust NGRAF for a smoother graph and use ITYPE to
C          fix the method: 1 = long hand, 2 use C05AZF
C          Since G01AFF is called it is essential to overdimension
C          the arrays NOBS and PRED (for borders/marginals)
C
      IMPLICIT   NONE
C
C Arguments
C
      INTEGER,             INTENT (IN)    :: NCMAX, NIN, NOUT, NRMAX
      INTEGER,             INTENT (INOUT) :: NROW
      INTEGER,             INTENT (OUT)   :: NOBS(NRMAX + 1,NCMAX + 1)
      DOUBLE PRECISION,    INTENT (INOUT) :: A(NRMAX,NCMAX), B(NRMAX)
      DOUBLE PRECISION,    INTENT (OUT)   :: PRED(NRMAX + 1,NCMAX + 1)
      CHARACTER (LEN = *), INTENT (INOUT) :: FNAME, TITLE
      LOGICAL,             INTENT (OUT)   :: NEWDAT 
      LOGICAL,             INTENT (IN)    :: SUPPLY
C
C Locals
C
      INTEGER    I, IFAIL, ISEND, ITYPE, J, K, L(12), M, N(12), NCOL, 
     +           NDATA, NPLOTS, NPTS, NR(12), NRSAV(12)
      PARAMETER (ISEND = 0, ITYPE = 2, NCOL = 3, M = 0)
      INTEGER    NGRAF
      PARAMETER (NGRAF = 300)
      INTEGER    NUMDEC, NUMOPT, NUMTXT
      PARAMETER (NUMOPT = 8, NUMTXT = 21)
      INTEGER    NUMBLD(30)
      INTEGER    ICOUNT, INOB, IPRED, MM, M1, NDF, NPOS, NUM, NN, N1
      DOUBLE PRECISION X1(NGRAF), X2(NGRAF), X3(NGRAF), X4(NGRAF),
     +                 X5(NGRAF), X6(NGRAF), X7(NGRAF), X8(NGRAF),
     +                 X9(NGRAF), X10(NGRAF), X11(NGRAF), X12(NGRAF)
      DOUBLE PRECISION Y1(NGRAF), Y2(NGRAF), Y3(NGRAF), Y4(NGRAF),
     +                 Y5(NGRAF), Y6(NGRAF), Y7(NGRAF), Y8(NGRAF),
     +                 Y9(NGRAF), Y10(NGRAF), Y11(NGRAF), Y12(NGRAF)
      DOUBLE PRECISION CHISQD, PVAL, PSIG, X, XHAT, Y, YHAT, Z
      DOUBLE PRECISION P(21)
      DOUBLE PRECISION ZERO, EPSI, HALF, TWO, THREE, F100
      PARAMETER (ZERO = 0.0D+00, EPSI = 1.0D-03, HALF = 0.5D+00,
     +           TWO = 2.0D+00, THREE = 3.0D+00,
     +           F100 = 100.0D+00)
      DOUBLE PRECISION G01ECF$, G01FCF$
      CHARACTER (LEN = 13) D13, SHOWLJ
      CHARACTER (LEN = 12) I12, FORM12
      CHARACTER  SYMBOL*30, TEXT(30)*100
      CHARACTER  INFO(NGRAF)*10, LINE*100
      CHARACTER  PTITLE*33 , XTITLE*6, YTITLE*6
      PARAMETER (XTITLE = 'px-hat',
     +           YTITLE = 'py-hat')
      LOGICAL    E_NUMBERS, E_FORMATS
      LOGICAL    PLOT(NGRAF)
      LOGICAL    ABORT, FIXCOL, FIXROW, LABEL
      PARAMETER (FIXCOL = .TRUE., FIXROW = .FALSE., LABEL = .TRUE.)
      LOGICAL    AXES, GSAVE, READY, REPEET
      PARAMETER (AXES = .TRUE., GSAVE = .TRUE.)
      EXTERNAL   E_FORMATS, FORM12, SHOWLJ
      EXTERNAL   MATTIN, GKS004, PUTIFA, PUTADV, LISTBX, PUTFAT,
     +           GETDM1, PATCH2, GKS012, GETJM1, CHKBOX, PUTWAR, REVPRO
      EXTERNAL   TRINXY
      EXTERNAL   G01AFF$, G01ECF$, G01FCF$
      INTRINSIC  NINT, DBLE, MIN, ABS
      SAVE       ICOUNT, L, NPTS, NRSAV, PVAL
      DATA       ICOUNT, NPTS  / 0, 150 /
      DATA       L / 12*0 /
      DATA       NRSAV / 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12 /
      DATA       PVAL / 0.95D+00 /
      DATA       NUMBLD / 30*0 /
C
C Initialise
C 
      E_NUMBERS = E_FORMATS()
      NEWDAT = .FALSE.
      IF (SUPPLY) THEN
         READY = .TRUE.
         NDATA = NROW
         READY = .TRUE.
C
C*********start of checking data supplied
C               
               DO I = 1, NROW
C
C First deal with columns 1 and 2, i.e. X and Y
C
                  IF (READY) THEN
                     DO J = 1, 2
                        IF (A(I,J).LT.HALF) THEN
                           IF (A(I,J).LT.ZERO) THEN
C
C Error 1: X < 0 or Y < 0 ... abort
C
                              WRITE (LINE,200) I, J
                              CALL PUTFAT (LINE)
                              RETURN
                           ELSE
C
C Error 2: X = 0 or Y = 0 ... repair
C
                              DO K = 1, 2
                                 A(I,K) = A(I,K) + HALF
                              ENDDO
                              A(I,3) = A(I,3) + THREE*HALF
                              WRITE (LINE,300) I
                              CALL PUTWAR (LINE)
                           ENDIF
                        ENDIF
                     ENDDO
C
C Now deal with column 3, i.e. N but checking for Z = 0
C
                     IF (A(I,3).LE.ZERO .OR.
     +                   A(I,3).LT.A(I,1) + A(I,2) - EPSI) THEN
C
C Error 3: N < 0 or N < X + Y ... abort
C
                        WRITE (LINE,200) I, J
                        CALL PUTFAT (LINE)
                        RETURN
                     ENDIF
                     IF (READY) THEN
                        IF (ABS(A(I,3) - A(I,1) - A(I,2)).LT.EPSI) THEN
C
C Error 4: Z = 0 ... repair
C
                           DO J = 1, 2
                              A(I,J) = A(I,J) + HALF
                           ENDDO
                           A(I,3) = A(I,3) + THREE*HALF
                           WRITE (LINE,300) I
                           CALL PUTWAR (LINE)
                        ENDIF
                        IF (A(I,3).LT.HALF .OR.
     +                      A(I,3) - A(I,1) - A(I,2) + EPSI.LT.HALF)
     +                      THEN
C
C Error 5: N still inconsistent ... abort
C
                           CALL PUTWAR (LINE)
                           WRITE (LINE,200) I, J
                           CALL PUTFAT (LINE)
                           RETURN
                        ENDIF
                     ENDIF
                  ENDIF
               ENDDO
            IF (READY .AND. NROW.GT.1) THEN
C
C Chi-square test: First try NUM = 1
C
               MM = NROW + 1
               NN = NCOL + 1
               DO I = 1, NCOL
                  DO J = 1, NROW
                     NOBS(J,I) = NINT(A(J,I))
                  ENDDO
               ENDDO
               INOB = NRMAX + 1
               IPRED = NRMAX + 1
               NUM = 1
               IFAIL = 1
               CALL G01AFF$(INOB, IPRED, MM, NN, NOBS, NUM, PRED,
     +                      CHISQD, P, NPOS, NDF, M1, N1, IFAIL)
               IF (IFAIL.EQ.1) THEN
C
C Shrinking failed so try NUM = 0
C
                  CALL PUTADV (
     +'Singular data ... Automatic shrinking abandoned')
                  MM = NROW + 1
                  NN = NCOL + 1
                  DO I = 1, NCOL
                     DO J = 1, NROW
                        NOBS(J,I) = NINT(A(J,I))
                     ENDDO
                  ENDDO
                  INOB = NRMAX
                  IPRED = NRMAX
                  NUM = 0
                  IFAIL = 1
                  CALL G01AFF$(INOB, IPRED, MM, NN, NOBS, NUM, PRED,
     +                         CHISQD, P, NPOS, NDF, M1, N1, IFAIL)
               ENDIF
               IF (IFAIL.NE.0) THEN
                  CALL PUTIFA (IFAIL, NOUT, 'G01AFF/TRINOM')
                  READY = .FALSE.
               ELSE
C
C Record the results from the chi-square test
C
                  IFAIL = 0
                  PSIG = G01ECF$('Upper', CHISQD, DBLE(NDF), IFAIL)
                  IF (IFAIL.NE.0) CALL PUTIFA (IFAIL, NOUT,
     +                                         'G01ECF/TRINOM')
                  ICOUNT = ICOUNT + 1
                  IF (E_NUMBERS) THEN
                     WRITE (NOUT,400) ICOUNT, TITLE, NDF, CHISQD, PSIG
                     WRITE (LINE,500) NDF, CHISQD, PSIG
                  ELSE
                     I12 = FORM12(NDF)
                     D13 = SHOWLJ(CHISQD) 
                     WRITE (NOUT,450) ICOUNT, TITLE, I12, D13, PSIG
                     WRITE (LINE,550) TRIM(I12), TRIM(D13), PSIG
                  ENDIF  
                  CALL PUTADV (LINE)
               ENDIF
            ENDIF
C
C*****end of checking data supplied
C            
      ELSE  
         READY = .FALSE.
         NDATA = 0
      ENDIF   
      NPLOTS = 0
      NUMDEC = NUMOPT - 1
      REPEET = .TRUE.
C
C Main loop
C
      DO WHILE (REPEET)
         IF (READY) THEN
            SYMBOL = '[ready]'
            DO I = 1, 12
               IF (I.LE.NDATA) THEN
                  IF (NRSAV(I).GT.NDATA) NRSAV(I) = I
                  NR(I) = NRSAV(I)
               ELSE
                  NR(I) = 0
               ENDIF
            ENDDO
            NPLOTS = 0
            DO I = 1, 12
               IF (NR(I).GT.0) NPLOTS = NPLOTS + 1
            ENDDO
         ELSE
            SYMBOL = '[***data required***]'
            NDATA = 0
            NPLOTS = 0
         ENDIF
         IF (NUMDEC.LE.0 .OR. NUMDEC.GT.NUMOPT) NUMDEC = 1
         WRITE (TEXT,100) SYMBOL, F100*PVAL, NPTS, NPLOTS, NDATA
         CALL LISTBX (NUMDEC, NUMOPT,
     +                TEXT)
         IF (.NOT.READY) THEN
            IF (NUMDEC.EQ.2 .OR. NUMDEC.EQ.5) THEN
               CALL PUTFAT ('First read in some data')
               NUMDEC = 0
            ENDIF
         ENDIF
         IF (NUMDEC.EQ.1) THEN
C
C NUMDEC = 1: Read in and check the data
C ===========
C
            IF (SUPPLY) THEN
               NEWDAT = .TRUE.
               RETURN
            ENDIF   
            READY = .FALSE.
            CLOSE (UNIT = NIN)
            I = ISEND
            CALL MATTIN (I, NCMAX, NCOL, NIN, NRMAX, NROW,
     +                   A, B,
     +                   FNAME, TITLE,
     +                   ABORT, FIXCOL, FIXROW, LABEL)
            CLOSE (UNIT = NIN)
            IF (NROW.LT.1) ABORT = .TRUE.
            IF (ABORT) THEN
               READY = .FALSE.
               NDATA = 0
            ELSE
               READY = .TRUE.
               NDATA = NROW
               DO I = 1, NROW
C
C First deal with columns 1 and 2, i.e. X and Y
C
                  IF (READY) THEN
                     DO J = 1, 2
                        IF (A(I,J).LT.HALF) THEN
                           IF (A(I,J).LT.ZERO) THEN
C
C Error 1: X < 0 or Y < 0 ... abort
C
                              WRITE (LINE,200) I, J
                              CALL PUTFAT (LINE)
                              READY = .FALSE.
                           ELSE
C
C Error 2: X = 0 or Y = 0 ... repair
C
                              DO K = 1, 2
                                 A(I,K) = A(I,K) + HALF
                              ENDDO
                              A(I,3) = A(I,3) + THREE*HALF
                              WRITE (LINE,300) I
                              CALL PUTWAR (LINE)
                           ENDIF
                        ENDIF
                     ENDDO
C
C Now deal with column 3, i.e. N but checking for Z = 0
C
                     IF (A(I,3).LE.ZERO .OR.
     +                   A(I,3).LT.A(I,1) + A(I,2) - EPSI) THEN
C
C Error 3: N < 0 or N < X + Y ... abort
C
                        WRITE (LINE,200) I, J
                        CALL PUTFAT (LINE)
                        READY = .FALSE.
                     ENDIF
                     IF (READY) THEN
                        IF (ABS(A(I,3) - A(I,1) - A(I,2)).LT.EPSI) THEN
C
C Error 4: Z = 0 ... repair
C
                           DO J = 1, 2
                              A(I,J) = A(I,J) + HALF
                           ENDDO
                           A(I,3) = A(I,3) + THREE*HALF
                           WRITE (LINE,300) I
                           CALL PUTWAR (LINE)
                        ENDIF
                        IF (A(I,3).LT.HALF .OR.
     +                      A(I,3) - A(I,1) - A(I,2) + EPSI.LT.HALF)
     +                      THEN
C
C Error 5: N still inconsistent ... abort
C
                           CALL PUTWAR (LINE)
                           WRITE (LINE,200) I, J
                           CALL PUTFAT (LINE)
                           READY = .FALSE.
                        ENDIF
                     ENDIF
                  ENDIF
               ENDDO
            ENDIF
            IF (READY .AND. NROW.GT.1) THEN
C
C Chi-square test: First try NUM = 1
C
               MM = NROW + 1
               NN = NCOL + 1
               DO I = 1, NCOL
                  DO J = 1, NROW
                     NOBS(J,I) = NINT(A(J,I))
                  ENDDO
               ENDDO
               INOB = NRMAX + 1
               IPRED = NRMAX + 1
               NUM = 1
               IFAIL = 1
               CALL G01AFF$(INOB, IPRED, MM, NN, NOBS, NUM, PRED,
     +                      CHISQD, P, NPOS, NDF, M1, N1, IFAIL)
               IF (IFAIL.EQ.1) THEN
C
C Shrinking failed so try NUM = 0
C
                  CALL PUTADV (
     +'Singular data ... Automatic shrinking abandoned')
                  MM = NROW + 1
                  NN = NCOL + 1
                  DO I = 1, NCOL
                     DO J = 1, NROW
                        NOBS(J,I) = NINT(A(J,I))
                     ENDDO
                  ENDDO
                  INOB = NRMAX
                  IPRED = NRMAX
                  NUM = 0
                  IFAIL = 1
                  CALL G01AFF$(INOB, IPRED, MM, NN, NOBS, NUM, PRED,
     +                         CHISQD, P, NPOS, NDF, M1, N1, IFAIL)
               ENDIF
               IF (IFAIL.NE.0) THEN
                  CALL PUTIFA (IFAIL, NOUT, 'G01AFF/TRINOM')
                  READY = .FALSE.
               ELSE
C
C Record the results from the chi-square test
C
                  IFAIL = 0
                  PSIG = G01ECF$('Upper', CHISQD, DBLE(NDF), IFAIL)
                  IF (IFAIL.NE.0) CALL PUTIFA (IFAIL, NOUT,
     +                                         'G01ECF/TRINOM')
                  ICOUNT = ICOUNT + 1
                  IF (E_NUMBERS) THEN
                     WRITE (NOUT,400) ICOUNT, TITLE, NDF, CHISQD, PSIG
                     WRITE (LINE,500) NDF, CHISQD, PSIG
                  ELSE
                     I12 = FORM12(NDF)
                     D13 = SHOWLJ(CHISQD) 
                     WRITE (NOUT,450) ICOUNT, TITLE, I12, D13, PSIG
                     WRITE (LINE,550) TRIM(I12), TRIM(D13), PSIG
                  ENDIF 
                  CALL PUTADV (LINE)
               ENDIF
            ENDIF
            IF (READY) THEN
               NUMDEC = 2
            ELSE
               NUMDEC = 1
            ENDIF
         ELSEIF (NUMDEC.EQ.2) THEN
C
C NUMDEC = 2: Plot
C ===========
C Set the CHISQD value to significance level for plotting
C
            IFAIL = 1
            CHISQD = G01FCF$(PVAL, TWO, IFAIL)
            WRITE (PTITLE,600) NINT(F100*PVAL)
C
C Initialise the graph coordinates
C
            DO I = 1, 12
               L(I) = 0
               N(I) = 0
               IF (I.LE.NDATA) THEN
                  IF (NRSAV(I).GT.NDATA) NRSAV(I) = I
                  NR(I) = NRSAV(I)
               ELSE
                  NR(I) = 0
               ENDIF
            ENDDO
C
C Calculate X, XHAT, Y, YHAT and graph coordinates
C
            K = NPTS
            NPLOTS = 0
            IF (NR(1).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(1),1)
               Y = A(NR(1),2)
               Z = A(NR(1),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X1, XHAT, Y1, YHAT, Z)
            ENDIF
            IF (NR(2).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(2),1)
               Y = A(NR(2),2)
               Z = A(NR(2),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X2, XHAT, Y2, YHAT, Z)
            ENDIF
            IF (NR(3).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(3),1)
               Y = A(NR(3),2)
               Z = A(NR(3),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X3, XHAT, Y3, YHAT, Z)
            ENDIF
            IF (NR(4).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(4),1)
               Y = A(NR(4),2)
               Z = A(NR(4),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X4, XHAT, Y4, YHAT, Z)
            ENDIF
            IF (NR(5).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(5),1)
               Y = A(NR(5),2)
               Z = A(NR(5),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X5, XHAT, Y5, YHAT, Z)
            ENDIF
            IF (NR(6).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(6),1)
               Y = A(NR(6),2)
               Z = A(NR(6),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X6, XHAT, Y6, YHAT, Z)
            ENDIF
            IF (NR(7).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(7),1)
               Y = A(NR(7),2)
               Z = A(NR(7),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X7, XHAT, Y7, YHAT, Z)
            ENDIF
            IF (NR(8).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(8),1)
               Y = A(NR(8),2)
               Z = A(NR(8),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X8, XHAT, Y8, YHAT, Z)
            ENDIF
            IF (NR(9).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(9),1)
               Y = A(NR(9),2)
               Z = A(NR(9),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X9, XHAT, Y9, YHAT, Z)
            ENDIF
            IF (NR(10).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(10),1)
               Y = A(NR(10),2)
               Z = A(NR(10),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X10, XHAT, Y10, YHAT, Z)
            ENDIF
            IF (NR(11).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(11),1)
               Y = A(NR(11),2)
               Z = A(NR(11),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X11, XHAT, Y11, YHAT, Z)
            ENDIF
            IF (NR(12).GT.0) THEN
               NPLOTS = NPLOTS + 1
               L(NPLOTS) = 1
               N(NPLOTS) = K
               X = A(NR(12),1)
               Y = A(NR(12),2)
               Z = A(NR(12),3)
               XHAT = X/Z
               YHAT = Y/Z
               CALL TRINXY (ITYPE, K,
     +                      CHISQD, X12, XHAT, Y12, YHAT, Z)
            ENDIF
C
C Plot the graphs
C
            IF (NDATA.NE.NPLOTS) THEN
               WRITE (LINE,700) NDATA, NPLOTS
               CALL PUTADV (LINE)
            ENDIF
            IF (NPLOTS.LT.1) THEN
               CALL PUTADV ('No data selected for plotting')
            ELSEIF (NPLOTS.LE.4) THEN
               CALL GKS004 (L(1), L(2), L(3), L(4),
     +                      M, M, M, M,
     +                      N(1), N(2), N(3), N(4),
     +                      X1, X2, X3, X4, Y1, Y2, Y3, Y4,
     +                      PTITLE, XTITLE, YTITLE,
     +                      AXES, GSAVE)
            ELSE
               CALL GKS012 (L(1), L(2), L(3), L(4),
     +                      L(5), L(6), L(7), L(8),
     +                      L(9), L(10), L(11), L(12),
     +                      M, M, M, M, M, M, M, M, M, M, M, M,
     +                      N(1), N(2), N(3), N(4),
     +                      N(5), N(6), N(7), N(8),
     +                      N(9), N(10), N(11), N(12),
     +                      X1, X2, X3, X4,
     +                      X5, X6, X7, X8,
     +                      X9, X10, X11, X12,
     +                      Y1, Y2, Y3, Y4,
     +                      Y5, Y6, Y7, Y8,
     +                      Y9, Y10, Y11, Y12,
     +                      PTITLE, XTITLE, YTITLE,
     +                      AXES, GSAVE)
            ENDIF
            NUMDEC = 1
         ELSEIF (NUMDEC.EQ.3) THEN
C
C NUMDEC = 3: set significance level PVAL
c ===========
C
            X = 1.0D+00
            Y = F100*PVAL
            Z = 99.0D+00
            CALL GETDM1 (X, Y, Z,
     +'Chi-square percentage significance level required')
            PVAL = Y/F100
            NUMDEC = 2
         ELSEIF (NUMDEC.EQ.4) THEN
C
C NUMDEC = 4: set no. plotting points NPTS
C ===========
C
            I = 20
            CALL GETJM1 (I, NPTS, NGRAF,
     +'Number of points to plot for each contour')
            NUMDEC = 2
         ELSEIF (NUMDEC.EQ.5) THEN
C
C NUMDEC = 5: select data for plotting
C ===========
C
            K = MIN(NDATA, NGRAF)
            DO I = 1, K
               PLOT(I) = .FALSE.
               WRITE (INFO(I),900) I
            ENDDO
            DO I = 1, 12
               IF (I.LE.NDATA) THEN
                  IF (NRSAV(I).GT.NDATA) NRSAV(I) = I
                  IF (NRSAV(I).GT.0) THEN
                     PLOT(NRSAV(I)) = .TRUE.
                  ENDIF
               ENDIF
            ENDDO
            WRITE (PTITLE,800)
            CALL CHKBOX (K,
     +                   INFO, PTITLE,
     +                   PLOT)
            DO I = 1, 12
               NRSAV(I) = 0
            ENDDO
            J = 0
            DO I = 1, K
               IF (J.LT.12) THEN
                  IF (PLOT(I)) THEN
                     J = J + 1
                     NRSAV(J) = I
                  ENDIF
               ENDIF
            ENDDO
         ELSEIF (NUMDEC.EQ.NUMOPT - 2) THEN
            CALL REVPRO (NOUT)   
         ELSEIF (NUMDEC.EQ.NUMOPT - 1) THEN
C
C NUMDEC = NUMOPT - 1: help
C ====================
C
            NUMBLD(1) = 1
            WRITE (TEXT,1000)
            CALL PATCH2 (NUMBLD, NUMTXT,
     +                   TEXT)
            NUMDEC = 1
         ELSEIF (NUMDEC.EQ.NUMOPT) THEN
C
C NUMDEC = NUMOPT: cancel
C ================
C
            REPEET = .FALSE.
         ENDIF
      ENDDO
  100 FORMAT (
     + 'Input: new x,y,N data',2X,A
     +/'Plot: parameter confidence contours'
     +/'Change: significance level (',F4.1,'%)'
     +/'Change: number of contour points (',I3,')'
     +/'Change: items plotted (',I3,' out of',I4,')'
     +/'Results'
     +/'Help'
     +/'Quit ... Exit trinomial analysis')
  200 FORMAT ('Value at A(',I1,',',I1,') is unacceptable')
  300 FORMAT ('0.5 added to all cells in row',I3)
  400 FORMAT (
     +/' Trinomial Analysis number',I3
     +/' ----------------------------'
     +/' Title of Data:'
     +/A
     +/' Number of degrees of freedom (N) =',I7
     +/' Chi-square test value (C)        =',1P,E10.3
     +/' p = P(chi-sqd.>= C)              =',0P,F7.4)
  450 FORMAT (
     +/' Trinomial Analysis number',I3
     +/' ----------------------------'
     +/' Title of Data:'
     +/A
     +/' Number of degrees of freedom =',1X,A
     +/' Chi-square test value (C)    =',1X,A
     +/' p = P(chi-sqd. >= C)         =',F7.4)   
  500 FORMAT ('Deg. Free =',I4,', Chi-sq. =',1P,E11.3,', p =',0P,F7.4)
  550 FORMAT ('Deg. Free =',1X,A,', Chi-sq. =',1X,A,', p =',F7.4)
  600 FORMAT (I2,'% Trinomial Confidence Contours')
  700 FORMAT ('Number of items =',I4,', Number to be plotted =',I3)
  800 FORMAT ('Select up to 12 items')
  900 FORMAT ('Data',I4)
 1000 FORMAT (
     + 'Trinomial confidence contours'
     +/
     +/'This procedure is used when N data have been obtained for the'
     +/'occurence of objects in one of three categories, say'
     +/'x > 0 in category X'
     +/'y > 0 in category Y, and'
     +/'z > 0 in category Z, so that N = x + y + z'
     +/'The three trinomial probability parameter estimates are'
     +/'px = x/N, py = y/N, and pz = z/N, but only two are independent'
     +/'since px + py + pz = 1.'
     +/
     +/'To use this procedure you must prepare a data file with columns'
     +/'consisting of x, y, N (not x, y, z , see test file trinom.tf1)'
     +/'and the chi-square test will be done followed by a plot of the'
     +/'px and py estimates with confidence regions. The significance'
     +/'level is 95 percent by default, but this can be altered.'
     +/
     +/'The plot provides a convenient visual method to detect which of'
     +/'the data sets are inconsistent, if the chi-square test suggests'
     +/'it. If contours are pairwise disjoint, then it can be assumed'
     +/'that corresponding data triples differ significantly.')
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE TRINXY (ITYPE, NGRAF,
     +                   CHISQD, XGRAF, XHAT, YGRAF, YHAT, Z)
C
C
C ACTION : Calculate x,y for plotting trinomial confidence limits
C AUTHOR : W.G.Bardsley, University of Manchester, U.K.,21/11/96
C ADVICE : Adjust NJUMPS or TOLX for greater accuracy
C          The root is located by travelling along a radiating line
C          ITYPE = 1: Method 1 ... by travelling out along the line
C          ITYPE = 2: Method 2 ... by root finding
C
      IMPLICIT   NONE
C
C Arguments
C
      INTEGER,          INTENT (IN)  :: ITYPE, NGRAF
      DOUBLE PRECISION, INTENT (IN)  :: CHISQD, XHAT, YHAT, Z
      DOUBLE PRECISION, INTENT (OUT) :: XGRAF(NGRAF), YGRAF(NGRAF)
C
C Locals
C
      INTEGER    I, IFAIL, IND, J
      INTEGER    IR, NJUMPS
      PARAMETER (IR = 0, NJUMPS = 500)
      DOUBLE PRECISION A, B, DELTA, FNOW, FPREV, T, TDIFF, TJUMP, THETA,
     +                 THETA1, THETA2, THETA3, X, XNOW, XPREV, XSTART,
     +                 XSTOP, Y, YNOW, YPREV, YSTART, YSTOP
      DOUBLE PRECISION ONE, PI, TWO, TWOPI, ZERO
      PARAMETER (ONE = 1.0D+00, PI = 3.1415927D+00, TWO = 2.0D+00,
     +           TWOPI = TWO*PI, ZERO = 0.0D+00)
      DOUBLE PRECISION C(17), FT, F1, F2, TOLX, T1, T2
      PARAMETER (TOLX = 1.0D-05)
      DOUBLE PRECISION FCNXYZ
      LOGICAL   NOTYET
      EXTERNAL  FCNXYZ
      EXTERNAL  C05AZF$
      INTRINSIC ATAN, TAN, DBLE
C
C Calculate the line segment parameters
C
      XSTART = XHAT
      YSTART = YHAT
      DELTA = TWOPI/DBLE(NGRAF - 1)
      THETA = ZERO
      THETA1 = PI - ATAN((ONE - YSTART)/(XSTART))
      THETA2 = PI + ATAN(YSTART/XSTART)
      THETA3 = TWOPI - ATAN(YSTART/(ONE - XSTART))
C
C Loop for each x,y cordinate pair along the line y = Ax + B
C
      DO I = 1, NGRAF
C
C Set angle THETA and A, B as in y = Ax + B
C
         A = TAN(THETA)
         B = YHAT - A*XHAT
C
C Find intersection of y = Ax + B with limits of range
C
         IF (THETA.GE.THETA3) THEN
            XSTOP = (ONE - B)/(ONE + A)
            YSTOP = A*XSTOP + B
         ELSEIF (THETA.GE.THETA2) THEN
            XSTOP = - B/A
            YSTOP = ZERO
         ELSEIF (THETA.GE.THETA1) THEN
            XSTOP = ZERO
            YSTOP = B
         ELSE
            XSTOP = (ONE - B)/(ONE + A)
            YSTOP = A*XSTOP + B
         ENDIF
C
C Default plotting coordinates in case root finding fails
C
         XGRAF(I) = XSTART
         YGRAF(I) = YSTART
C
C Method 1: Use T as line segment parameter then search along the line
C ========

         IF (ITYPE.EQ.1) THEN
            TJUMP = ONE/DBLE(NJUMPS - 1)
            T = ZERO
            X = XSTART
            Y = YSTART
            FPREV = FCNXYZ(CHISQD, X, XHAT, Y, YHAT, Z)
            XPREV = X
            YPREV = Y
            NOTYET = .TRUE.
C
C Search for the root by comparing adjacent function values for sign change
C
            DO J = 2, NJUMPS - 1
               IF (NOTYET) THEN
                  T = T + TJUMP
                  TDIFF = ONE - T
                  XNOW = TDIFF*XSTART + T*XSTOP
                  YNOW = TDIFF*YSTART + T*YSTOP
                  FNOW = FCNXYZ(CHISQD, XNOW, XHAT, YNOW, YHAT, Z)
                  IF (FNOW*FPREV.LE.ZERO) THEN
                     NOTYET = .FALSE.
                     XGRAF(I) = (XNOW + XPREV)/TWO
                     YGRAF(I) = (YNOW + YPREV)/TWO
                  ELSE
                     FPREV = FNOW
                     XPREV = XNOW
                     YPREV = YNOW
                  ENDIF
               ENDIF
            ENDDO
         ELSEIF (ITYPE.EQ.2) THEN
C
C Method 2: Find the root using C05AZF with T as line parameter
C ========
            T1 = ZERO
            TDIFF = ONE - T1
            X = TDIFF*XSTART + T1*XSTOP
            Y = TDIFF*YSTART + T1*YSTOP
            F1 = FCNXYZ(CHISQD, X, XHAT, Y, YHAT, Z)
C
C First attempt to find a range with T2 = 0.8
C
            T2 = 0.8D+00
            TDIFF = ONE - T2
            X = TDIFF*XSTART + T2*XSTOP
            Y = TDIFF*YSTART + T2*YSTOP
            F2 = FCNXYZ(CHISQD, X, XHAT, Y, YHAT, Z)
            IF (F1*F2.GT.ZERO) THEN
C
C Failure: Now attempt with T2 = 0.9
C
               T2 = 0.9D+00
               TDIFF = ONE - T2
               X = TDIFF*XSTART + T2*XSTOP
               Y = TDIFF*YSTART + T2*YSTOP
               F2 = FCNXYZ(CHISQD, X, XHAT, Y, YHAT, Z)
C
C Failure: Now attempt with T2 = 0.95
C
               IF (F1*F2.GT.ZERO) THEN
                  T2 = 0.95D+00
                  TDIFF = ONE - T2
                  X = TDIFF*XSTART + T2*XSTOP
                  Y = TDIFF*YSTART + T2*YSTOP
                  F2 = FCNXYZ(CHISQD, X, XHAT, Y, YHAT, Z)
C
C Failure: Now attempt with T2 = 0.99
C
                  IF (F1*F2.GT.ZERO) THEN
                     T2 = 0.99D+00
                     TDIFF = ONE - T2
                     X = TDIFF*XSTART + T2*XSTOP
                     Y = TDIFF*YSTART + T2*YSTOP
                     F2 = FCNXYZ(CHISQD, X, XHAT, Y, YHAT, Z)
                  ENDIF
               ENDIF
            ENDIF
            IF (F1*F2.LE.ZERO) THEN
C
C A range has been found that brackets the root so use C05AZF
C
               IND = 1
               IFAIL = 1
   20          CONTINUE
               CALL C05AZF$(T1, T2, FT, TOLX, IR, C, IND, IFAIL)
               IF (IND.EQ.2 .OR. IND.EQ.3 .OR. IND.EQ.4) THEN
                  TDIFF = ONE - T1
                  X = TDIFF*XSTART + T1*XSTOP
                  Y = TDIFF*YSTART + T1*YSTOP
                  FT = FCNXYZ(CHISQD, X, XHAT, Y, YHAT, Z)
                  GOTO 20
               ENDIF
               T = T1
               TDIFF = ONE - T
               XGRAF(I) = TDIFF*XSTART + T*XSTOP
               YGRAF(I) = TDIFF*YSTART + T*YSTOP
            ENDIF
         ENDIF
C
C Increment THETA to update y = Ax + B
C
         THETA = THETA + DELTA
      ENDDO
      END
C
C-------------------------------------------------------------
C
      DOUBLE PRECISION FUNCTION FCNXYZ(CHISQD, X, XHAT, Y, YHAT, Z)
C
C The quadratic form equation
C
      IMPLICIT NONE
C
C Arguments
C
      DOUBLE PRECISION, INTENT (IN) :: CHISQD, X, XHAT, Y, YHAT, Z
C
C Locals
C
      DOUBLE PRECISION EPSI, F1, F2, F3, F4, ONE, TWO, XDIFF, YDIFF
      PARAMETER (EPSI = 1.0D-010, ONE = 1.0D+00, TWO = 2.0D+00)
      INTRINSIC MAX
      XDIFF = XHAT - X
      YDIFF = YHAT - Y
      F1 = Z/MAX(ONE - X - Y, EPSI)
      F2 = XDIFF**2*(ONE - Y)/MAX(X, EPSI)
      F3 = YDIFF**2*(ONE - X)/MAX(Y, EPSI)
      F4 = TWO*XDIFF*YDIFF
      FCNXYZ = F1*(F2 + F3 + F4) - CHISQD
      END
C
C

