C
C DEQSOL6.FOR ... The differential equation routines
C ===========
C
C Note: this version uses KMAX_A, KMAX_F, KMAX_J and KMAX_Y to dimension
C       the call to DEQMOD
C
C In order to silence ftn95 several dummy assignments have been made.
C These can all be commented out in the final version.
C
C DEQ001: DEQF01 ... 1 eqn.
C         DEQJ01
C DEQ002: DEQF02 ... 2 eqn.
C         DEQJ02
C DEQ003: DEQF03 ... 3 eqn.
C         DEQJ03
C DEQ004: DEQF04 ... 4 eqn.
C         DEQJ04
C DEQ005: DEQF05 ... 5 eqn.
C         DEQJ05
C DEQCOM:        ... compare with analytic solution
C DEQINI:        ... initialise parameters
C DEQMOD:        ... select model
C DEQUSE: USEDEQ ... user-supplied
C         USEJAC
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQF01 (NEQ, T, Y, F, P, IP)
C
C ACTION : The differential equations for NEQ = 1
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: IP(*), NEQ
      DOUBLE PRECISION, INTENT (IN)  :: P(*), T, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: F(NEQ)
C
C Locals
C      
      INTEGER    NMOD
      DOUBLE PRECISION ZERO, YMIN, RTOL
      PARAMETER (ZERO = 0.0D+00, YMIN = 1.0D-50, RTOL = 1.0D-200)
      DOUBLE PRECISION TEMP1, TEMP2
      INTRINSIC ABS, EXP

      TEMP1 = T!to silence ftn95

      NMOD = IP(2)
      IF (NMOD.EQ.1) THEN
         TEMP1 = P(1) + Y(1)
         IF (ABS(TEMP1).LE.RTOL) THEN
            F(1) = ZERO
         ELSE
            F(1) = - P(2)*Y(1)/TEMP1
         ENDIF
      ELSEIF (NMOD.EQ.2) THEN
         TEMP1 = P(3) - Y(1)
         TEMP2 = P(1) + TEMP1
         IF (ABS(TEMP2).LE.RTOL) THEN
            F(1) = ZERO
         ELSE
            F(1) = P(2)*TEMP1/TEMP2
         ENDIF
      ELSEIF (NMOD.EQ.3) THEN
         TEMP1 = P(1) + Y(1)
         IF (ABS(TEMP1).LE.RTOL) THEN
            F(1) = ZERO
         ELSE
            F(1) = - P(2)*Y(1)/TEMP1 - P(3)*Y(1) - P(4)
         ENDIF
      ELSEIF (NMOD.EQ.4) THEN
         TEMP1 = P(5) - Y(1)
         TEMP2 = P(1) + TEMP1
         IF (ABS(TEMP2).LE.RTOL) THEN
            F(1) = ZERO
         ELSE
            F(1) = P(2)*TEMP1/TEMP2 + P(3)*TEMP1 + P(4)
         ENDIF
      ELSEIF (NMOD.EQ.5) THEN
         TEMP1 = P(3)*(Y(1) - P(4))
         TEMP2 = Y(1)**2 + P(1)*Y(1) + P(2)
         IF (ABS(TEMP2).LE.RTOL) THEN
            F(1) = ZERO
         ELSE
            F(1) = TEMP1/TEMP2
         ENDIF
      ELSEIF (NMOD.EQ.6) THEN
         IF (Y(1).GT.YMIN) THEN
            F(1) = P(1)*Y(1)**P(2) - P(3)*Y(1)**P(4)
         ELSE
            F(1) = YMIN
         ENDIF
      ELSEIF (NMOD.EQ.7) THEN
         IF (Y(1).GT.YMIN .AND. T.GE.ZERO) THEN
            F(1) = EXP(-P(5)*T)*P(1)*Y(1)**P(2) - P(3)*Y(1)**P(4)
         ELSE
            F(1) = YMIN
         ENDIF
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQJ01 (NEQ, X, Y, ML, MU, PW, NROWPW, P, IP)
C
C ACTION : The Jacobian matrix for NEQ = 1
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: IP(*), ML, MU, NEQ, NROWPW
      DOUBLE PRECISION, INTENT (IN)  :: P(*), X, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: PW(NROWPW,NEQ)
C
C Locals
C      
      INTEGER    NMOD
      DOUBLE PRECISION ONE, TWO, ZERO, RTOL, YMIN
      PARAMETER (ONE = 1.0D+00, TWO = 2.0D+00, ZERO = 0.0D+00,
     +           RTOL = 1.0D-200, YMIN = 1.0D-50)
      DOUBLE PRECISION TEMP1, TEMP2
      INTRINSIC ABS, EXP

      TEMP1 = X!to silence ftn95
      NMOD = ML!to silence ftn95
      NMOD = MU!to silence ftn95

      NMOD = IP(2)
      IF (NMOD.EQ.1) THEN
         TEMP1 = (P(1) + Y(1))**2
         IF (ABS(TEMP1).LE.RTOL) THEN
            PW(1,1) = ZERO
         ELSE
            PW(1,1) = - P(1)*P(2)/TEMP1
         ENDIF
      ELSEIF (NMOD.EQ.2) THEN
         TEMP1 = P(3) - Y(1)
         TEMP2 = (P(1) + TEMP1)**2
         IF (ABS(TEMP2).LE.RTOL) THEN
            PW(1,1) = ZERO
         ELSE
            PW(1,1) = - P(1)*P(2)/TEMP2
         ENDIF
      ELSEIF (NMOD.EQ.3) THEN
         TEMP1 = (P(1) + Y(1))**2
         IF (ABS(TEMP1).LE.RTOL) THEN
            PW(1,1) = ZERO
         ELSE
            PW(1,1) = - P(1)*P(2)/TEMP1 - P(3)
         ENDIF
      ELSEIF (NMOD.EQ.4) THEN
         TEMP1 = P(5) - Y(1)
         TEMP2 = (P(1) + TEMP1)**2
         IF (ABS(TEMP2).LE.RTOL) THEN
            PW(1,1) = ZERO
         ELSE
            PW(1,1) = - P(1)*P(2)/TEMP2 - P(3)
         ENDIF
      ELSEIF (NMOD.EQ.5) THEN
         TEMP1 = (Y(1)**2 + P(1)*Y(1) + P(2))**2
         IF (ABS(TEMP1).LE.RTOL) THEN
            PW(1,1) = ZERO
         ELSE
            PW(1,1) = P(3)*(Y(1)*(TWO*P(4) - Y(1)) + P(1)*P(4))/TEMP1
         ENDIF
      ELSEIF (NMOD.EQ.6) THEN
         IF (Y(1).GT.YMIN) THEN
            PW(1,1) = P(1)*P(2)*Y(1)**(P(2) - ONE)
     +                - P(3)*P(4)*Y(1)**(P(4) - ONE)
         ELSE
            PW(1,1) = YMIN
         ENDIF
       ELSEIF (NMOD.EQ.7) THEN
         IF (Y(1).GT.YMIN .AND. X.GE.ZERO) THEN
            PW(1,1) = EXP(-P(5)*X)*P(1)*P(2)*Y(1)**(P(2) - ONE)
     +                - P(3)*P(4)*Y(1)**(P(4) - ONE)
         ELSE
            PW(1,1) = YMIN
         ENDIF         
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQF02 (NEQ, T, Y, F, P, IP)
C
C ACTION : The differential equations for N = 2
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: IP(*), NEQ
      DOUBLE PRECISION, INTENT (IN)  :: P(*), T, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: F(NEQ)
C
C Locals
C      
      INTEGER    NMOD
      DOUBLE PRECISION TEMP1
      DOUBLE PRECISION ONE
      PARAMETER (ONE = 1.0D+00)
      INTRINSIC SIN

      TEMP1 = T!to silence ftn95

      NMOD = IP(2)
      IF (NMOD.EQ.1) THEN
         F(1) = Y(2)
         F(2) = - P(1)*Y(2) - P(2)*Y(1)
      ELSEIF (NMOD.EQ.2) THEN
         TEMP1 = Y(1)*Y(2)
         F(1) = P(1)*Y(1) - P(2)*TEMP1
         F(2) = - P(3)*Y(2) + P(4)*TEMP1
      ELSEIF (NMOD.EQ.3) THEN
         F(1) = Y(1)*(P(1) - P(2)*Y(1) - P(3)*Y(2))
         F(2) = Y(2)*(P(4) - P(5)*Y(2) - P(6)*Y(1))
      ELSEIF (NMOD.EQ.4) THEN
         F(1) = Y(2)
         F(2) = - P(1)*SIN(Y(1)) 
      ELSEIF (NMOD.EQ.5) THEN
         F(1) = Y(2)
         F(2) = - P(1)*SIN(Y(1)) - P(2)*Y(2)    
      ELSEIF (NMOD.EQ.6) THEN
         F(1) = Y(2)
         F(2) = - Y(1) - P(1)*(Y(1)**2 - ONE)*Y(2)     
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQJ02 (NEQ, X, Y, ML, MU, PW, NROWPW, P, IP)
C
C ACTION : The Jacobian matrix for N = 2
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: IP(*), ML, MU, NEQ, NROWPW
      DOUBLE PRECISION, INTENT (IN)  :: P(*), X, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: PW(NROWPW,NEQ)
C
C Locals
C      
      INTEGER    NMOD
      DOUBLE PRECISION TWO, ONE, ZERO
      PARAMETER (TWO = 2.0D+00, ONE = 1.0D+00, ZERO = 0.0D+00)
      INTRINSIC  COS
      PW(1,1) = X!to silence ftn95
      NMOD = ML!to silence ftn95
      NMOD = MU!to silence ftn95
      NMOD = IP(2)
      IF (NMOD.EQ.1) THEN
         PW(1,1) = ZERO
         PW(1,2) = ONE
         PW(2,1) = - P(2)
         PW(2,2) = - P(1)
      ELSEIF (NMOD.EQ.2) THEN
         PW(1,1) = P(1) - P(2)*Y(2)
         PW(1,2) = - P(2)*Y(1)
         PW(2,1) =   P(4)*Y(2)
         PW(2,2) = - P(3) + P(4)*Y(1)
      ELSEIF (NMOD.EQ.3) THEN
         PW(1,1) = P(1) - TWO*P(2)*Y(1) - P(3)*Y(2)
         PW(1,2) = - P(3)*Y(1)
         PW(2,1) = - P(6)*Y(2)
         PW(2,2) = P(4) - TWO*P(5)*Y(2) - P(6)*Y(1)
      ELSEIF (NMOD.EQ.4) THEN
         PW(1,1) = ZERO
         PW(1,2) = ONE   
         PW(2,1) = - P(1)*COS(Y(1))   
         PW(2,2) = ZERO
      ELSEIF (NMOD.EQ.5) THEN
         PW(1,1) = ZERO
         PW(1,2) = ONE   
         PW(2,1) = - P(1)*COS(Y(1))   
         PW(2,2) = - P(2)        
      ELSEIF (NMOD.EQ.6) THEN
         PW(1,1) = ZERO
         PW(1,2) = ONE
         PW(2,1) = - ONE - TWO*P(1)*Y(1)*Y(2)   
         PW(2,2) = - P(1)*(Y(1)**2 - ONE)
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQF03 (NEQ, T, Y, F, P, IP)
C
C ACTION : The differential equations for N = 3
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: IP(*), NEQ
      DOUBLE PRECISION, INTENT (IN)  :: P(*), T, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: F(NEQ)
C
C Locals
C      
      INTEGER    NMOD
      DOUBLE PRECISION ONE, ZERO, RTOL
      PARAMETER (ONE = 1.0D+00, ZERO = 0.0D+00, RTOL = 1.0D-200)
      DOUBLE PRECISION TEMP1, TEMP2, TEMP3, TEMP4
      INTRINSIC ABS

      TEMP1 = T!to silence ftn95

      NMOD = IP(2)
      IF (NMOD.EQ.1) THEN
         TEMP1 = P(1)*Y(1)
         TEMP2 = P(2)*Y(2)
         TEMP3 = P(3)*Y(2)
         TEMP4 = P(4)*Y(3)
         F(1) = - TEMP1 + TEMP2
         F(2) =   TEMP1 - TEMP2 - TEMP3 + TEMP4
         F(3) =   TEMP3 - TEMP4
      ELSEIF (NMOD.EQ.2) THEN
         IF (Y(3).GT.ZERO) THEN
            TEMP1 = ONE  + P(7)*Y(3)**P(8)
         ELSE
            TEMP1 = ONE
         ENDIF
         IF (ABS(TEMP1).LE.RTOL) THEN
            F(1) = ZERO
         ELSE
            F(1) = P(6)/TEMP1 - P(1)*Y(1)
         ENDIF
         F(2) = P(2)*Y(1) - P(3)*Y(2)
         F(3) = P(4)*Y(2) - P(5)*Y(3)
      ELSEIF (NMOD.EQ.3) THEN
         IF (Y(3).GT.ZERO) THEN
            TEMP1 = P(7)*Y(3)**P(8)
         ELSE
            TEMP1 = ZERO
         ENDIF
         TEMP2 = P(6)*(ONE + TEMP1)
         TEMP3 = P(9) + TEMP1
         IF (ABS(TEMP3).LE.RTOL) THEN
            F(1) = ZERO
         ELSE
            F(1) = TEMP2/TEMP3 - P(1)*Y(1)
         ENDIF
         F(2) = P(2)*Y(1) - P(3)*Y(2)
         F(3) = P(4)*Y(2) - P(5)*Y(3)
      ELSEIF (NMOD.EQ.4) THEN
         TEMP1 = P(1)*Y(1)*Y(2)
         TEMP2 = P(2)*Y(2)
         F(1) = - TEMP1
         F(2) = TEMP1 - TEMP2
         F(3) = TEMP2
      ELSEIF (NMOD.EQ.5) THEN
         TEMP1 = P(1)*Y(1)*Y(2)
         TEMP2 = P(2)*Y(2)
         F(1) = - TEMP1 + P(3)
         F(2) = TEMP1 - TEMP2
         F(3) = TEMP2  - P(3) 
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQJ03 (NEQ, X, Y, ML, MU, PW, NROWPW, P, IP)
C
C ACTION : The Jacobian matrix for N = 3
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: IP(*), ML, MU, NEQ, NROWPW
      DOUBLE PRECISION, INTENT (IN)  :: P(*), X, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: PW(NROWPW,NEQ)
C
C Locals
C      
      INTEGER    NMOD
      DOUBLE PRECISION ONE, ZERO, RTOL
      PARAMETER (ONE = 1.0D+00, ZERO = 0.0D+00, RTOL = 1.0D-200)
      DOUBLE PRECISION TEMP1, TEMP2
      INTRINSIC ABS

      TEMP1 = X!to silence ftn95
      NMOD = ML!to silence ftn95
      NMOD = MU!to silence ftn95

      NMOD = IP(2)
      IF (NMOD.EQ.1) THEN
         PW(1,1) = - P(1)
         PW(1,2) =   P(2)
         PW(1,3) =   ZERO
         PW(2,1) =   P(1)
         PW(2,2) = - P(2) - P(3)
         PW(2,3) =   P(4)
         PW(3,1) =   ZERO
         PW(3,2) =   P(3)
         PW(3,3) = - P(4)
      ELSEIF (NMOD.EQ.2) THEN
         IF (Y(3).GT.ZERO) THEN
            TEMP1 = (ONE + P(7)*Y(3)**P(8))**2
            IF (TEMP1.LE.RTOL) THEN
               PW(1,3) = ZERO
            ELSE
               PW(1,3) = - P(6)*P(7)*P(8)*(Y(3)**(P(8) - ONE))/TEMP1
            ENDIF
         ELSE
            PW(1,3) = ZERO
         ENDIF
         PW(1,1) = - P(1)
         PW(1,2) =   ZERO
         PW(2,1) =   P(2)
         PW(2,2) = - P(3)
         PW(2,3) =   ZERO
         PW(3,1) =   ZERO
         PW(3,2) =   P(4)
         PW(3,3) = - P(5)
      ELSEIF (NMOD.EQ.3) THEN
         IF (Y(3).GT.ZERO) THEN
            TEMP1 = (P(9) + P(7)*Y(3)**P(8))**2
            IF (ABS(TEMP1).LE.RTOL) THEN
               PW(1,3) = ZERO
            ELSE
               PW(1,3) = P(6)*P(7)*P(8)*(Y(3)**(P(8)-ONE))*(P(9)-ONE)
     +                   /TEMP1
            ENDIF
         ELSE
            PW(1,3) = ZERO
         ENDIF
         PW(1,1) = - P(1)
         PW(1,2) =   ZERO
         PW(2,1) =   P(2)
         PW(2,2) = - P(3)
         PW(2,3) =   ZERO
         PW(3,1) =   ZERO
         PW(3,2) =   P(4)
         PW(3,3) = - P(5)
      ELSEIF (NMOD.EQ.4 .OR. NMOD.EQ.5) THEN
         TEMP1 = P(1)*Y(1)
         TEMP2 = P(1)*Y(2)
         PW(1,1) = - TEMP2
         PW(1,2) = - TEMP1
         PW(1,3) = ZERO
         PW(2,1) = TEMP2
         PW(2,2) = TEMP1  - P(2)
         PW(2,3) = ZERO
         PW(3,1) = ZERO
         PW(3,2) = P(2)
         PW(3,3) = ZERO
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQF04 (NEQ, T, Y, F, P, IP)
C
C ACTION : The differential equations for N = 4
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: IP(*), NEQ
      DOUBLE PRECISION, INTENT (IN)  :: P(*), T, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: F(NEQ)
C
C Locals
C      
      INTEGER    NMOD
      DOUBLE PRECISION TEMP1, TEMP2, TEMP3, TEMP4, TEMP5, TEMP6

      TEMP1 = T!to silence ftn95

      NMOD = IP(2)
      IF (NMOD.EQ.1) THEN
         TEMP1 = P(1)*Y(1)
         TEMP2 = P(2)*Y(2)
         TEMP3 = P(3)*Y(2)
         TEMP4 = P(4)*Y(3)
         TEMP5 = P(5)*Y(3)
         TEMP6 = P(6)*Y(4)
         F(1) = - TEMP1 + TEMP2
         F(2) =   TEMP1 - TEMP2 - TEMP3 + TEMP4
         F(3) =                   TEMP3 - TEMP4 - TEMP5 + TEMP6
         F(4) =                                   TEMP5 - TEMP6
      ELSEIF (NMOD.EQ.2) THEN
         TEMP1 = P(1)*Y(1)*Y(3)
         TEMP2 = P(4)*Y(2)*Y(3)
         TEMP3 = (P(2) + P(3))*Y(4)
         F(1) = - TEMP1 + P(2)*Y(4)
         F(2) =   P(3)*Y(4) - TEMP2
         F(3) = - TEMP1 - TEMP2 + TEMP3
         F(4) = - F(3)
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQJ04 (NEQ, X, Y, ML, MU, PW, NROWPW, P, IP)
C
C ACTION : The Jacobian matrix for N = 4
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: IP(*), ML, MU, NEQ, NROWPW
      DOUBLE PRECISION, INTENT (IN)  :: P(*), X, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: PW(NROWPW,NEQ)
C
C Locals
C      
      INTEGER    NMOD
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO = 0.0D+00)

      PW(1,1) = X!to silence ftn95
      NMOD = ML!to silence ftn95
      NMOD = MU!to silence ftn95

      NMOD = IP(2)
      IF (NMOD.EQ.1) THEN
         PW(1,1) = - P(1)
         PW(1,2) =   P(2)
         PW(1,3) =   ZERO
         PW(1,4) =   ZERO
         PW(2,1) =   P(1)
         PW(2,2) = - P(2) - P(3)
         PW(2,3) =   P(4)
         PW(2,4) =   ZERO
         PW(3,1) =   ZERO
         PW(3,2) =   P(3)
         PW(3,3) = - P(4) - P(5)
         PW(3,4) =   P(6)
         PW(4,1) =   ZERO
         PW(4,2) =   ZERO
         PW(4,3) =   P(5)
         PW(4,4) = - P(6)
      ELSEIF (NMOD.EQ.2) THEN
         PW(1,1) = - P(1)*Y(3)
         PW(1,2) =   ZERO
         PW(1,3) = - P(1)*Y(1)
         PW(1,4) =   P(2)
         PW(2,1) =   ZERO
         PW(2,2) = - P(4)*Y(3)
         PW(2,3) = - P(4)*Y(2)
         PW(2,4) =   P(3)
         PW(3,1) =   PW(1,1)
         PW(3,2) = - P(4)*Y(3)
         PW(3,3) =   PW(1,3) + PW(2,3)
         PW(3,4) =   P(2) + P(3)
         PW(4,1) = - PW(3,1)
         PW(4,2) = - PW(3,2)
         PW(4,3) = - PW(3,3)
         PW(4,4) = - PW(3,4)
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQF05 (NEQ, T, Y, F, P, IP)
C
C ACTION : The differential equations for N = 5
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: IP(*), NEQ
      DOUBLE PRECISION, INTENT (IN)  :: P(*), T, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: F(NEQ)
C
C Locals
C      
      INTEGER    NMOD
      DOUBLE PRECISION TEMP1, TEMP2, TEMP3, TEMP4, TEMP5, TEMP6, TEMP7,
     +                 TEMP8

      TEMP1 = T!to silence ftn95

      NMOD = IP(2)
      IF (NMOD.EQ.1) THEN
         TEMP1 = P(1)*Y(1)
         TEMP2 = P(2)*Y(2)
         TEMP3 = P(3)*Y(2)
         TEMP4 = P(4)*Y(3)
         TEMP5 = P(5)*Y(3)
         TEMP6 = P(6)*Y(4)
         TEMP7 = P(7)*Y(4)
         TEMP8 = P(8)*Y(5)
         F(1) = - TEMP1 + TEMP2
         F(2) =   TEMP1 - TEMP2 - TEMP3 + TEMP4
         F(3) =                   TEMP3 - TEMP4 - TEMP5 + TEMP6
         F(4) =                                   TEMP5 - TEMP6
     +                                          - TEMP7 + TEMP8
         F(5) =                                   TEMP7 - TEMP8
      ELSEIF (NMOD.EQ.2) THEN
         TEMP1 = P(1)*Y(1)*Y(3)
         TEMP2 = P(6)*Y(2)*Y(3)
         TEMP3 = (P(2) + P(3))*Y(4)
         TEMP4 = (P(4) + P(5))*Y(5)
         F(1) = - TEMP1 + P(2)*Y(4)
         F(2) =   P(5)*Y(5) - TEMP2
         F(3) = - TEMP1 - TEMP2 + P(2)*Y(4) + P(5)*Y(5)
         F(4) =   TEMP1 - TEMP3 + P(4)*Y(5)
         F(5) =   TEMP2 - TEMP4 + P(3)*Y(4)
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQJ05 (NEQ, X, Y, ML, MU, PW, NROWPW, P, IP)
C
C ACTION : The Jacobian matrix for N = 5
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: IP(*), ML, MU, NEQ, NROWPW
      DOUBLE PRECISION, INTENT (IN)  :: P(*), X, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: PW(NROWPW,NEQ)
C
C Locals
C      
      INTEGER    NMOD
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO = 0.0D+00)

      PW(1,1) = X!to silence ftn95
      NMOD = ML!to silence ftn95
      NMOD = MU!to silence ftn95

      NMOD = IP(2)
      IF (NMOD.EQ.1) THEN
         PW(1,1) = - P(1)
         PW(1,2) =   P(2)
         PW(1,3) =   ZERO
         PW(1,4) =   ZERO
         PW(1,5) =   ZERO
         PW(2,1) =   P(1)
         PW(2,2) = - P(2) - P(3)
         PW(2,3) =   P(4)
         PW(2,4) =   ZERO
         PW(2,5) =   ZERO
         PW(3,1) =   ZERO
         PW(3,2) =   P(3)
         PW(3,3) = - P(4) - P(5)
         PW(3,4) =   P(6)
         PW(3,5) =   ZERO
         PW(4,1) =   ZERO
         PW(4,2) =   ZERO
         PW(4,3) =   P(5)
         PW(4,4) = - P(6) - P(7)
         PW(4,5) =   P(8)
         PW(5,1) =   ZERO
         PW(5,2) =   ZERO
         PW(5,3) =   ZERO
         PW(5,4) =   P(7)
         PW(5,5) = - P(8)
      ELSEIF (NMOD.EQ.2) THEN
         PW(1,1) = - P(1)*Y(3)
         PW(1,2) =   ZERO
         PW(1,3) =   ZERO
         PW(1,4) =   P(2)
         PW(1,5) =   ZERO
         PW(2,1) =   ZERO
         PW(2,2) = - P(6)*Y(3)
         PW(2,3) = - P(6)*Y(2)
         PW(2,4) =   ZERO
         PW(2,5) =   P(5)
         PW(3,1) =   PW(1,1)
         PW(3,2) =   PW(2,2)
         PW(3,3) =   P(1)*Y(1)
         PW(3,4) =   P(2)
         PW(3,5) =   P(5)
         PW(4,1) = - PW(1,1)
         PW(4,2) =   ZERO
         PW(4,3) =   ZERO
         PW(4,4) = - P(2) - P(3)
         PW(4,5) =   P(4)
         PW(5,1) =   ZERO
         PW(5,2) =   ZERO
         PW(5,3) = - PW(2,3)
         PW(5,4) =   P(3)
         PW(5,5) = - P(4) - P(5)
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQCOM (IWANT, M, MODEL, NMOD, NPTS, NTMAX, NUMY,
     +                   NYMAX, P, TX, XSTART, YCOM, Y0, ABORT)
C
C ACTION: Compare with values for an alternative (exact/approx) solution
C AUTHOR: w.g.bardsley, university of manchester, u.k.
C
C These examples written out in longhand style for illustration
C
      IMPLICIT   NONE
      INTEGER    M, MODEL, NMOD, NPTS, NTMAX, NUMY, NYMAX
      INTEGER    IWANT(12)
      INTEGER    I, IDISC
      DOUBLE PRECISION P(M), TX(NTMAX), XSTART, YCOM(NTMAX,NYMAX),
     +                 Y0(NYMAX)
      DOUBLE PRECISION DISC, DUMMY, TEMP, TEMP1, TEMP2, W, Z
      DOUBLE PRECISION A, B, B1, B2, C, D, DET
      DOUBLE PRECISION ARG, CO, EX, EX1, EX2, SI
      DOUBLE PRECISION A11, A12, A21, A22, E1, E2
      DOUBLE PRECISION ZERO, ONE, TWO, FOUR
      PARAMETER (ZERO = 0.0D+00, ONE = 1.0D+00, TWO = 2.0D+00,
     +           FOUR = 4.0D+00)
      LOGICAL    ABORT
      INTRINSIC  EXP, COS, SIN, SQRT
      ABORT = .TRUE.
      IF (NUMY.LT.1) RETURN
      I = IWANT(1)!to silence ftn95
      IF (MODEL.EQ.2 .AND. NMOD.EQ.1) THEN
C
C OK for MODEL = 2, NMOD = 1
C
         ABORT = .FALSE.
         DISC = P(1)**2 - FOUR*P(2)
         IF (DISC.LT.ZERO) THEN
            IDISC = 1
         ELSEIF (DISC.GT.ZERO) THEN
            IDISC = 3
         ELSE
            IDISC = 2
         ENDIF
         IF (IDISC.EQ.1) THEN
            W = SQRT(- DISC)/TWO
            Z = - P(1)/TWO
            DUMMY = W*XSTART
            TEMP = EXP(XSTART*Z)
            A11 = COS(DUMMY)*TEMP
            A12 = SIN(DUMMY)*TEMP
            A21 = A11*Z - A12*W
            A22 = A11*W + A12*Z
         ELSEIF (IDISC.EQ.2) THEN
            Z = - P(1)/TWO
            TEMP1 = XSTART*Z
            TEMP2 = EXP(TEMP1)
            A11 = TEMP2
            A12 = TEMP2*XSTART
            A21 = TEMP2*Z
            A22 = (ONE + TEMP1)*TEMP2
         ELSE
            A = - P(1)/TWO
            B = SQRT(DISC)/TWO
            E1 = A + B
            E2 = A - B
            A11 = EXP(E1*XSTART)
            A12 = EXP(E2*XSTART)
            A21 = A11*E1
            A22 = A12*E2
         ENDIF
         B1 = Y0(1)
         B2 = Y0(2)
         DET = A11*A22 - A12*A21
         A = (A22*B1 - A12*B2)/DET
         B = (A11*B2 - A21*B1)/DET
         IF (IDISC.EQ.1) THEN
            C = A*Z + B*W
            D = B*Z - A*W
            DO I = 1, NPTS
               ARG = TX(I)*W
               CO = COS(ARG)
               SI = SIN(ARG)
               EX = EXP(TX(I)*Z)
               YCOM(I,1) = (A*CO + B*SI)*EX
               YCOM(I,2) = (C*CO + D*SI)*EX
            ENDDO
         ELSEIF (IDISC.EQ.2) THEN
            DO I = 1, NPTS
               EX = EXP(Z*TX(I))
               YCOM(I,1) = (A + B*TX(I))*EX
               YCOM(I,2) = YCOM(I,1)*Z + B*EX
            ENDDO
         ELSE
            C = A*E1
            D = B*E2
            DO I = 1, NPTS
               EX1 = EXP(E1*TX(I))
               EX2 = EXP(E2*TX(I))
               YCOM(I,1) = A*EX1 + B*EX2
               YCOM(I,2) = C*EX1 + D*EX2
            ENDDO
         ENDIF
      ENDIF
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQINI (IRELAB, M, MODEL, MPED, N, NMOD, NPMAX, NPTS,
     +                   NYMAX,
     +                   P, TOL, XEND, XSTART, Y0)
C
C ACTION: Initialise values 
C AUTHOR: w.g.bardsley, university of manchester, u.k.
C
C IRELAB = Error test: mixed(0), decimal(1), digit(2)
C M      = No. of parameters
C MODEL  = Model number (type)
C MPED   = Jacobian: internally(0), by PEDERV(1)
C N      = No. of differential equations
C NMOD   = No. of model
C NPMAX  = Dimension of P
C NPTS   = No. of time points
C NYMAX  = Dimension of Y0
C P(I)   = Fixed parameters
C TOL    = Tolerance
C XEND   = Final time point
C XSTART = Starting time point
C Y0(I)  = Initial values for Y(I)
C USER   = User supplied model Y/N
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: NPMAX, NYMAX
      INTEGER,          INTENT (IN)  :: MODEL, N, NMOD
      INTEGER,          INTENT (OUT) :: IRELAB, M, MPED, NPTS
      DOUBLE PRECISION, INTENT (OUT) :: P(NPMAX), TOL, XEND, XSTART,
     +                                  Y0(NYMAX)
C
C Locals
C      
      INTEGER    N0, N1, N121
      PARAMETER (N0 = 0, N1 = 1, N121 = 121)
      INTEGER    I, J
      DOUBLE PRECISION EPSI
      PARAMETER (EPSI = 1.0D-04)
      DOUBLE PRECISION ONE, TEN, TWO, THREE, ZERO
      PARAMETER (ONE = 1.0D+00, TEN = 10.0D+00, TWO = 2.0D+00,
     +           THREE = 3.0D+00, ZERO = 0.0D+00)
      DOUBLE PRECISION PNT0001, PNT01, PNT1, PNT25, PNT5, PNT666
      PARAMETER (PNT0001 = 0.0001D+00, PNT01 = 0.01D+00, PNT1 = 0.1D+00,
     +           PNT25 = 0.25D+00, PNT5 = 0.5D+00,
     +           PNT666 = 0.6666667D+00)
C
C Basic defaults
C
      IRELAB = N0
      MPED = N1
      TOL = EPSI
      NPTS = N121
      XEND = TEN
      XSTART = ZERO
      DO I = N1, NPMAX
         P(I) = ONE
      ENDDO
      DO I = N1, NYMAX
         Y0(I) = ONE
      ENDDO
C
C MODEL = 1 .... 1 differential equation
C
      IF (MODEL.EQ.1) THEN
         IF (NMOD.EQ.1) THEN
            M = N + 2
         ELSEIF (NMOD.EQ.2) THEN
            M = N + 3
            Y0(1) = ZERO
         ELSEIF (NMOD.EQ.3) THEN
            M = N + 4
         ELSEIF (NMOD.EQ.4) THEN
            M = N + 5
            Y0(1) = ZERO
         ELSEIF (NMOD.EQ.5) THEN
            M = N + 4
            P(3) = - ONE
            P(4) = PNT5
            Y0(1) = ZERO
         ELSEIF (NMOD.EQ.6) THEN
            M = N + 4
            P(2) = PNT666
            Y0(1) = PNT01
         ELSEIF (NMOD.EQ.7) THEN
            M = N + 5
            P(2) = PNT666   
            Y0(1) = PNT0001 
         ENDIF
C
C MODEL = 2 ... 2 differential equations
C
      ELSEIF (MODEL.EQ.2) THEN
         IF (NMOD.EQ.1) THEN
            M = N + 2
         ELSEIF (NMOD.EQ.2) THEN
            M = N + 4
            P(1) = 0.7D+00
            Y0(2) = PNT5
         ELSEIF (NMOD.EQ.3) THEN
            M = N + 6
            P(3) = PNT25
            P(6) = PNT25
            Y0(1) = ONE + PNT5
            Y0(2) = PNT1
         ELSEIF (NMOD.EQ.4) THEN
            M = N + 1
            P(1) = ONE
            Y0(1) = PNT25
            Y0(2) = ZERO
            XEND = 6.3D+00
         ELSEIF (NMOD.EQ.5) THEN
            M = N + 2
            P(1) = ONE
            P(2) = PNT1
            Y0(1) = PNT25
            Y0(2) = ZERO
            XEND = THREE*TEN            
         ELSEIF (NMOD.EQ.6) THEN
            M = N + 1
            P(1) = ONE
            Y0(1) = TWO
            Y0(2) = ZERO    
         ENDIF
C
C MODEL = 3 ... 3 differential equations
C
      ELSEIF (MODEL.EQ.3) THEN
         IF (NMOD.EQ.1) THEN
            M = N + 4
            P(1) = PNT5
            P(2) = PNT25
            P(4) = PNT25
            Y0(2) = ZERO
            Y0(3) = ZERO
         ELSEIF (NMOD.EQ.2) THEN
            M = N + 8
            Y0(2) = PNT01
            Y0(3) = PNT01
         ELSEIF (NMOD.EQ.3) THEN
            M = N + 9
            P(9) = TWO
            Y0(2) = PNT01
            Y0(3) = PNT01
         ELSEIF (NMOD.EQ.4) THEN
            M = N + 2
            P(1) = 0.0025D+00
            P(2) = PNT5
            Y0(1) = 900.0D+00
            Y0(2) =  10.0D+00
            Y0(3) =  90.0D+00
         ELSEIF (NMOD.EQ.5) THEN
            M = N + 3
            P(1) = 0.001D+00
            P(2) = PNT5
            P(3) = 20.0d+00
            Y0(1) = 900.0D+00
            Y0(2) =  10.0D+00
            Y0(3) =  90.0D+00   
         ENDIF
C
C MODEL = 4 ... 4 differential equations
C
      ELSEIF (MODEL.EQ.4) THEN
         IF (NMOD.EQ.1) THEN
            XEND = TEN
            M = N + 6
            P(1) = PNT5
            P(2) = PNT25
            P(4) = PNT25
            P(6) = PNT25
            Y0(2) = ZERO
            Y0(3) = ZERO
            Y0(4) = ZERO
         ELSEIF (NMOD.EQ.2) THEN
            XEND = ONE
            M = N + 4
            P(1) = TEN
            P(4) = TEN
            Y0(2) = ZERO
            Y0(3) = PNT1
            Y0(4) = ZERO
         ENDIF
C
C MODEL = 5 ... 5 differential equations
C
      ELSEIF (MODEL.EQ.5) THEN
         IF (NMOD.EQ.1) THEN
            XEND = TEN
            M = N + 8
            P(1) = PNT5
            P(2) = PNT25
            P(4) = PNT25
            P(6) = PNT25
            P(8) = PNT25
            Y0(2) = ZERO
            Y0(3) = ZERO
            Y0(4) = ZERO
            Y0(5) = ZERO
         ELSEIF (NMOD.EQ.2) THEN
            XEND = ONE
            M = N + 6
            P(1) = TEN
            P(6) = TEN
            Y0(2) = ZERO
            Y0(3) = PNT1
            Y0(4) = ZERO
            Y0(5) = ZERO
         ENDIF
      ENDIF
      J = M - N
      DO I = 1, N
         J = J + 1
         P(J) = Y0(I)
      ENDDO
      END
C
C----------------------------------------------------------------------
C
      SUBROUTINE DEQMOD (MODEL, N, NMOD, NYMAX,
     +                   MODEL_FILE, Z,
     +                   USER)
C
C ACTION: Choose a model
C AUTHOR: w.g.bardsley, university of manchester, u.k.
C
C      MODEL = no. differential equations
C      N = MODEL
C      NMOD = no. in series of order MODEL
C      NMOD = no. parameters (M) if USER = .TRUE.
C      N2 for dimensions since only used to read and parse the user file
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,             INTENT (IN)    :: NYMAX
      INTEGER,             INTENT (INOUT) :: MODEL 
      INTEGER,             INTENT (OUT)   :: N, NMOD
      CHARACTER (LEN = *), INTENT (INOUT) :: MODEL_FILE, Z(24)
      LOGICAL,             INTENT (OUT)   :: USER
C
C Locals
C      
      INTEGER    ISEND, NVAR, N2
      PARAMETER (ISEND = 1, NVAR = 1, N2 = 2)
      INTEGER    KMAX_A, KMAX_F, KMAX_J, KMAX_Y
      PARAMETER (KMAX_A = N2, KMAX_F = N2, KMAX_J = N2*N2,
     +           KMAX_Y = N2)
      INTEGER    ICOLOR, IX, IY, NUMDEC, NUMOPT
      PARAMETER (ICOLOR = 3, IX = 4, IY = 4)
      INTEGER    NUMBLD(30), NUMPOS(20), NUMSTA, NUMTXT
      DOUBLE PRECISION F(N2), P(N2), TEMP, X, Y, YDE(N2), YJA(N2*N2)
      CHARACTER  TEXT(30)*100
      LOGICAL    ABORT, DEQN
      PARAMETER (DEQN = .TRUE.)
      EXTERNAL   PUTADV, LBOX02, LSTBOX
      EXTERNAL   DEUSER
      EXTERNAL   DEQSOL_REQUIRED
      DATA       NUMBLD / 30*0 /
      DATA       NUMPOS / 20*1 /
      NUMDEC = NYMAX!to silence ftn95
   20 CONTINUE
      WRITE (TEXT,100)
      NUMOPT = 7
      NUMSTA = 17
      NUMTXT = NUMSTA + NUMOPT - 1
      NUMDEC = NUMOPT
      NUMBLD(1) = 4
      CALL LSTBOX (NUMBLD, NUMDEC, NUMOPT, NUMSTA, NUMTXT,
     +             TEXT)
      CALL DEQSOL_REQUIRED (NUMDEC)
      NUMBLD(1) = 0
      MODEL = NUMDEC
      N = MODEL
      USER = .FALSE.
C
C MODEL = 1 ... 1 differential equation
C
      IF (MODEL.EQ.1) THEN
         WRITE (TEXT,200)
         NUMDEC = 1
         NUMOPT = 7
         CALL LBOX02 (ICOLOR, IX, IY, NUMDEC, NUMOPT, NUMPOS,
     +                TEXT)
         NMOD = NUMDEC
         IF (NMOD.EQ.1) THEN
            Z(2) = ' Irreversible MM substrate-depletion'
            Z(4) = ' dy/dx = -p(2)y/[p(1) + y]'
            Z(6) = ' y = substrate, p(1) = Km, p(2) = Vmax'
         ELSEIF (NMOD.EQ.2) THEN
            Z(2) = ' Irreversible MM product-accumulation'
            Z(4) = ' dy/dx = p(2)(p(3) - y)/[p(1) + (p(3) - y)]'
            Z(6) = ' y = product,p(1) = Km,p(2) = Vmax,p(3) = S(0)'
         ELSEIF (NMOD.EQ.3) THEN
            Z(2) = ' Generalised substrate-depletion'
            Z(4) = ' dy/dx = -p(2)y/[p(1) + y] - p(3)y - p(4)'
            Z(6) = ' y = substrate, p(1) = Km, p(2) = Vmax'
         ELSEIF (NMOD.EQ.4) THEN
            Z(2) = ' Generalised product-accumulation'
            Z(4) = ' dy/dx = p(2)(p(5) - y)/[p(1) + (p(5) - y)] +'
            Z(5) = '         p(3)(p(5) - y) + p(4)'
            Z(7) = ' y = product,p(1) = Km,p(2) = Vmax,p(5) = S(0)'
         ELSEIF (NMOD.EQ.5) THEN
            Z(2) = ' Membrane transport (variable volume, etc.)'
            Z(4) = ' dy/dx = p(3)(y - p(4))/[y^2 + p(1)y + p(2)]'
            Z(6) = ' y = solute, p(4) = y(infinity)'
         ELSEIF (NMOD.EQ.6) THEN
            Z(2) = ' Von Bertalannfy growth model'
            Z(4) = ' dy/dx = p(1)y^p(2) - p(3)y^p(4)'
         ELSEIF (NMOD.EQ.7) THEN
            Z(2) = ' Von Bertalannfy growth/decay model'
            Z(4) = ' dy/dx = exp[-p(5)*x]p(1)y^p(2) - p(3)y^p(4)'   
         ENDIF
C
C MODEL = 2 ... 2 differential equations
C
      ELSEIF (MODEL.EQ.2) THEN
         WRITE (TEXT,300)
         NUMDEC = 1
         NUMOPT = 6
         CALL LBOX02 (ICOLOR, IX, IY, NUMDEC, NUMOPT, NUMPOS,
     +                TEXT)
         NMOD = NUMDEC
         IF (NMOD.EQ.1) THEN
            Z(2) = ' d^2y/dt^2 + p(1)dy/dt + p(2)y = 0'
            Z(4) = ' This is equivalent to the autonomous system'
            Z(6) = ' dy(1)/dx = y(2)'
            Z(7) = ' dy(2)/dx = - p(1)*y(2) - p(2)*y(1)'
            Z(8) = ' The original ODE can of course be integrated'
         ELSEIF (NMOD.EQ.2) THEN
            Z(2) = ' Lotka-Volterra Predator-Prey equations'
            Z(4) = ' dy(1)/dx = p(1)y(1) - p(2)y(1)y(2)'
            Z(6) = ' dy(2)/dx = - p(3)y(2) + p(4)y(1)y(2)'
         ELSEIF (NMOD.EQ.3) THEN
            Z(2) = ' Competing species ecological model'
            Z(4) = ' dy(1)/dx = y(1)[p(1) - p(2)y(1) - p(3)y(2)]'
            Z(6) = ' dy(2)/dx = y(2)[p(4) - p(5)y(2) - p(6)y(1)]'
         ELSEIF (NMOD.EQ.4) THEN
            Z(1) = 'Nonlinear undamped pendulum'   
            Z(2) = 'd^2(theta)/dt^2 + (g/l)sin(theta) = 0' 
            Z(4) = 'This is equivalent to the autonomous system'
            Z(6) = 'dy(1)/dx = y(2)'
            Z(7) = 'dy(2)/dx = - p(1)sin(y(1))'
            Z(8) = '    p(1) > 0'  
         ELSEIF (NMOD.EQ.5) THEN
            Z(1) = 'Nonlinear damped pendulum'   
            Z(2) = 'd^2(x)/dt^2 + (c/ml)d(x)/dt + (g/l)sin(x) = 0' 
            Z(4) = 'This is equivalent to the autonomous system'
            Z(6) = 'dy(1)/dx = y(2)'
            Z(7) = 'dy(2)/dx = - p(1)sin(y(1)) - p(2)y(2)'
            Z(8) = '    p(1) > 0 and p(2) > 0'              
         ELSEIF (NMOD.EQ.6) THEN
            Z(1) = 'Van der Pol oscillator'   
            Z(2) = 'd^2(x)/dt^2 + (mu)(x^2 - 1)dx/dt + x = 0' 
            Z(4) = 'This is equivalent to the autonomous system'
            Z(6) = 'dy(1)/dx = y(2)'
            Z(7) = 'dy(2)/dx = - y(1) - p(1)[y(1)^2 - 1]y(2)'
            z(8) = '    p(1) > 0'
         ENDIF
C
C MODEL = 3 ... 3 differential equations
C
      ELSEIF (MODEL.EQ.3) THEN
         WRITE (TEXT,400)
         NUMDEC = 1
         NUMOPT = 5
         CALL LBOX02 (ICOLOR, IX, IY, NUMDEC, NUMOPT, NUMPOS,
     +                TEXT)
         NMOD = NUMDEC
         IF (NMOD.EQ.1) THEN
            Z(2) = ' Consecutive reversible reactions'
            Z(4) = ' dy(1)/dx = - p(1)y(1) + p(2)y(2)'
            Z(6) =
     +      ' dy(2)/dx =   p(1)y(1) - [p(2)+p(3)]y(2) + p(4)y(3)'
            Z(8) = ' dy(3)/dx =   p(3)y(2) - p(4)y(3)'
         ELSEIF (NMOD.EQ.2) THEN
            Z(2) = ' Negative feedback (e.g. on mRNA)'
            Z(4) = ' '
            Z(5) = ' '
            Z(6) =
     +      ' dy(1)/dx = p(6)/[1 +  p(7)y(3)^p(8)] - p(1)y(1)'
            Z(8) = ' dy(2)/dx = p(2)y(1) - p(3)y(2)'
            Z(10) = ' dy(3)/dx = p(4)y(2) - p(5)y(3)'
         ELSEIF (NMOD.EQ.3) THEN
            Z(2) = ' Positive feedback (e.g. on mRNA)'
            Z(4) = ' '
            Z(5) =
     +      ' dy(1)/dx ='
            Z(6) =
     +' p(6)[1 + p(7)y(3)^p(8)]/[p(9) +  p(7)y(3)^p(8)] - p(1)y(1)'
            Z(8) = ' dy(2)/dx = p(2)y(1) - p(3)y(2)'
            Z(10) = ' dy(3)/dx = p(4)y(2) - p(5)y(3)'
         ELSEIF (NMOD.EQ.4) THEN!epidemic equations 
            Z(2) = ' Epidemic: y(1) = S, y(2) = I, y(3) = R'
            Z(4) = ' dy(1)/dx = -p(1)y(1)y(2)'
            Z(6) = ' dy(2)/dx =  p(1)y(1)y(2) - p(2)y(2)'
            Z(8) = ' dy(3)/dx =  p(2)y(2)'
         ELSEIF (NMOD.EQ.5) THEN!recurrent epidemic equations 
            Z(2) = ' Recurrent epidemic: y(1) = S, y(2) = I, y(3) = R'
            Z(4) = ' dy(1)/dx = -p(1)y(1)y(2) + p(3)'
            Z(6) = ' dy(2)/dx =  p(1)y(1)y(2) - p(2)y(2)'
            Z(8) = ' dy(3)/dx =  p(2)y(2) - p(3)'   
         ENDIF
C
C MODEL = 4 ... 4 differential equations
C
      ELSEIF (MODEL.EQ.4) THEN
         WRITE (TEXT,500)
         NUMDEC = 1
         NUMOPT = 2
         CALL LBOX02 (ICOLOR, IX, IY, NUMDEC, NUMOPT, NUMPOS,
     +                TEXT)
         NMOD = NUMDEC
         IF (NMOD.EQ.1) THEN
            Z(2) = ' Consecutive reversible reactions'
            Z(4) = ' dy(1)/dx = -p(1)y(1) + p(2)y(2)'
            Z(6) =
     +      ' dy(2)/dx = p(1)y(1) - [p(2)+p(3)]y(2) + p(4)y(3)'
            Z(8) =
     +      ' dy(3)/dx = p(3)y(2) - [p(4)+p(5)]y(3) + p(6)y(4)'
            Z(10) = ' dy(4)/dx = p(5)y(3) - p(6)y(4)'
         ELSEIF (NMOD.EQ.2) THEN
            Z(2) = ' Michaelis-Menten kinetics'
            Z(4) = ' dy(1)/dx = - p(1)y(1)y(3) + p(2)y(4)'
            Z(6) = ' dy(2)/dx =   p(3)y(4) - p(4)y(2)y(3)'
            Z(8) =
     +  ' dy(3)/dx = - p(1)y(1)y(3) - p(4)y(2)y(3) + [p(2) + p(3)]y(4)'
            Z(10) =
     +  ' dy(4)/dx =   p(1)y(1)y(3) + p(4)y(2)y(3) - [p(2) + p(3)]y(4)'
            Z(12) = ' y(1) = [S], y(2) = [P], y(3) = [E], y(4) = [ES]'
            Z(14) =
     +  ' p(1) = k(+1), p(2) = k(-1), p(3) = k(+2), p(4) = k(-2)'
         ENDIF
C
C MODEL = 5 ... 5 differential equations
C
      ELSEIF (MODEL.EQ.5) THEN
         WRITE (TEXT,600)
         NMOD = 1
         NUMDEC = 1
         NUMOPT = 2
         CALL LBOX02 (ICOLOR, IX, IY, NUMDEC, NUMOPT, NUMPOS, 
     +                TEXT)
         NMOD = NUMDEC
         IF (NMOD.EQ.1) THEN
            Z(2) = ' Consecutive reversible reactions'
            Z(4) = ' dy(1)/dx = -p(1)y(1) + p(2)y(2)'
            Z(6) =
     +      ' dy(2)/dx = p(1)y(1) - [p(2)+p(3)]y(2) + p(4)y(3)'
            Z(8) =
     +      ' dy(3)/dx = p(3)y(2) - [p(4)+p(5)]y(3) + p(6)y(4)'
            Z(10) =
     +      ' dy(4)/dx = p(5)y(3) - [p(6)+p(7)]y(4) + p(8)y(5)'
            Z(12) =
     +      ' dy(5)/dx = p(7)y(4) - p(8)y(5)'

         ELSEIF (NMOD.EQ.2) THEN
            Z(2) = ' Briggs-Haldane kinetics'
            Z(4) = ' dy(1)/dx = -p(1)y(1)y(3) + p(2)y(4)'
            Z(6) = ' dy(2)/dx =  p(5)y(5) - p(6)y(2)y(3)'
            Z(8) =
     +  ' dy(3)/dx = -p(1)y(1)y(3) + p(2)y(4) + p(5)y(5) -p(6)y(2)y(3)'
            Z(10) =
     +  ' dy(4)/dx =  p(1)y(1)y(3) - [p(2) + p(3)]y(4) + p(4)y(5)'
            Z(12) =
     +  ' dy(5)/dx =  p(6)y(2)y(3) + p(3)y(4) - [p(4) + p(5)]y(5)'
            Z(14) = ' y(1)=[S],y(2)=[P],y(3)=[E],y(4)=[ES],y(5)=[EP]'
            Z(16) =
     +  ' p(1) = k(+1), p(2) = k(-1), p(3) = k(+2), p(4) = k(-2)'
            Z(18) =
     +  ' p(5) = k(+3), p(6) = k(-3)'
         ENDIF
      ELSEIF (MODEL.EQ.6) THEN
         CALL PUTADV (
     +  'Now read in an ascii text model file (see deqmod?.tf?)')
         N = 0
         CALL DEUSER (ISEND,
     +                KMAX_A, KMAX_F, KMAX_J, KMAX_Y,
     +                N, NMOD, NVAR, N2,
     +                P, F, X, Y, YDE, YJA, TEMP,
     +                MODEL_FILE, Z, 
     +                ABORT, DEQN)
         IF (ABORT) THEN
            MODEL = 0
            N = 0
            GOTO 20
         ELSE
            MODEL = N
            USER = .TRUE.
         ENDIF
      ELSE
         MODEL = 0
         N = 0
         USER = .FALSE.
      ENDIF
C
C Format statements
C      
  100 FORMAT (
     + 'DEQSOL equation installation options'
     +/
     +/'The library has a limited number of models as it is anticipated'
     +/'that users will want to supply their own models. A set of test'
     +/'models deqmod?.tf? is provided to illustrate how to do this and'
     +/'the model-defining syntax is explained in the reference manual.'
     +/
     +/'Note that user-defined models should have two sections, i.e.'
     +/'begin{limits} ... end{limits} to define parameter limits, and'
     +/'begin{range} ... end{range} to define the integration range,'  
     +/'as demonstrated in the deqmod?.tf? demonstration files.'
     +/
     +/'With small sets of equations you can add code for a Jacobian,'
     +/'and demonstration files deqmod3.tf1 and deqmod3.tf2 show how to'
     +/'supply a model with an explicit Jacobian, or with no Jacobian.'
     +/
     +/'1 library-defined differential equation'
     +/'2 library-defined differential equations (all autonomous)'
     +/'3 library-defined differential equations'
     +/'4 library-defined differential equations'
     +/'5 library-defined differential equations'
     +/'n user-defined differential equations'
     +/'Quit ... Exit program DEQSOL')
  200 FORMAT (
     + 'Irreversible MM substrate-depletion'
     +/'Irreversible MM product-accumulation'
     +/'Generalised substrate-depletion curve'
     +/'Generalised product-accumulation-curve'
     +/'Membrane transport (variable volume, etc.)'
     +/'Von Bertalannfy growth model'
     +/'Von Bertalannfy growth/decay model')
  300 FORMAT (
     + 'dy(1)/dx = y(2), dy(2)/dx = - p(1)y(2) - p(2)y(1)'
     +/'Lotka-Volterra Predator-Prey equations'
     +/'Competing species ecological model'
     +/'The undamped nonlinear pendulum'
     +/'The damped nonlinear pendulum'
     +/'The Van der Pol oscillator')
  400 FORMAT (
     + 'Consecutive reactions (A,B,C)'
     +/'Negative feedback (e.g. on mRNA)'
     +/'Positive feedback (e.g. on mRNA)'
     +/'Epidemic: Susceptible/Infected/Resistant'
     +/'Epidemic: Recurrent Susceptible/Infected/Resistant')
  500 FORMAT (
     + 'Consecutive reactions (A,B,C,D)'
     +/'Michaelis-Menten enzyme kinetics')
  600 FORMAT (
     + 'Consecutive reactions (A,B,C,D,E)'
     +/'Briggs-Haldane enzyme kinetics')
      END

c
c
      subroutine deqsol_required (isend)
c
c action: set mask for the main deqsol menu demo files
c author: w.g.bardsley, university of manchester, u.k., 01/07/2020
c      
      implicit none
c
c argument
c      
      integer, intent (in) :: isend
c
c locals
c      
      integer i
      integer nmask
      parameter (nmask = 7)
      integer mask(nmask)
      logical required(nmask)
      logical store
      parameter (store = .true.)
      external x_putadv, query_files_required
c
c check isend
c      
      if (isend.lt.1 .or. isend.gt.7) then
         call x_putadv (
     +'ISEND out of range in call to DEQSOL_REQUIRED')
         return
      endif   
      if (isend.eq.nmask) then
         do i = 1, nmask
            required(i) = .true.
         enddo
      else     
         required(nmask) = .true. 
         do i = 1, nmask - 1
            if (i.eq.isend) then
               required(i) = .true.
            else
               required(i) = .false.
            endif   
         enddo    
      endif
      do i = 1, nmask
         if (required(i)) then
            mask(i) = 1
         else
            mask(i) = 0
         endif  
      enddo
      call query_files_required (mask, nmask,
     +                           store)
      end
c
c          
      
C
C----------------------------------------------------------------------
C
      SUBROUTINE USEDEQ (NEQ, T, Y, F, P, IP)
C
C ACTION : The differential equations for user supplied model
C ADVICE : DVODE version
C
      IMPLICIT   NONE
C
C Arguments
C   
      INTEGER,          INTENT (IN)  :: IP(*), NEQ
      DOUBLE PRECISION, INTENT (IN)  :: P(*), T, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: F(NEQ)
C
C Locals
C      
C      INTEGER    I
      INTEGER    JSEND, NPAR, NVAR, NYMAX
      PARAMETER (JSEND = 2, NVAR = 1, NYMAX = 100)
      INTEGER    KMAX_A, KMAX_F, KMAX_J, KMAX_Y
      DOUBLE PRECISION PW(NYMAX*NYMAX)
      DOUBLE PRECISION TEMP1, TEMP2
      CHARACTER (LEN = 80)   :: MODNAM(24)
      CHARACTER (LEN = 1024) :: MODEL_FILE 
      LOGICAL    ABORT, DEQN
      PARAMETER (DEQN = .TRUE.)
      EXTERNAL   DEUSER
      NPAR = IP(4)
      KMAX_A = NPAR
      KMAX_F = NEQ
      KMAX_J = NEQ*NEQ
      KMAX_Y = NEQ
      CALL DEUSER (JSEND,
     +             KMAX_A, KMAX_F, KMAX_J, KMAX_Y,
     +             NEQ, NPAR, NVAR, NYMAX,
     +             P, F, T, TEMP1, Y, PW, TEMP2,
     +             MODEL_FILE, MODNAM,
     +             ABORT, DEQN)
C      WRITE (*,'(A,1PE12.4)') 'T = ', T
C      WRITE (*,'(A,1P,10E12.4)') 'F =', (F(I), I = 1, NEQ)
      END
C
C-----------------------------------------------------------------------
C
      SUBROUTINE USEJAC (NEQ, X, Y, ML, MU, PW, NROWPW, P, IP)
C
C ACTION : The Jacobian matrix for user supplied models
C ADVICE : DVODE version
C          12/01/1998 Dimensioned F(NYMAX) since estimation of Jacobian
C                     could involve calculation of F
C          04/09/2009 minor editing and introduced YJA
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: NEQ, NROWPW
      INTEGER,          INTENT (IN)  :: IP(*), ML, MU
      DOUBLE PRECISION, INTENT (IN)  :: P(*), X, Y(NEQ)
      DOUBLE PRECISION, INTENT (OUT) :: PW(NROWPW,NEQ)
C
C Locals
C      
      INTEGER    JSEND, NPAR, NVAR, NYMAX
      PARAMETER (JSEND = 3, NVAR = 1, NYMAX = 100)
      INTEGER    I, ICOUNT, J, KMAX_A, KMAX_F, KMAX_J, KMAX_Y
      DOUBLE PRECISION F(NYMAX)
      DOUBLE PRECISION YJA(NYMAX*NYMAX)
      DOUBLE PRECISION TEMP1, TEMP2
      CHARACTER (LEN = 80)   :: MODNAM(24)
      CHARACTER (LEN = 1024) :: MODEL_FILE
      LOGICAL    ABORT, DEQN
      PARAMETER (DEQN = .TRUE.)
      EXTERNAL   DEUSER
      NPAR = ML!to silence ftn95
      NPAR = MU!to silence ftn95
      NPAR = IP(4)
      KMAX_A = NPAR
      KMAX_F = NYMAX
      KMAX_J = NROWPW*NEQ
      KMAX_Y = NEQ
      CALL DEUSER (JSEND,
     +             KMAX_A, KMAX_F, KMAX_J, KMAX_Y,
     +             NEQ, NPAR, NVAR, NYMAX,
     +             P, F, X, TEMP1, Y, YJA, TEMP2,
     +             MODEL_FILE, MODNAM,
     +             ABORT, DEQN)
C
C Copy Jacobian from YJA into PW to avoid problems with /f_stdcall + /checkmate
C     
      ICOUNT = 0
      DO J = 1, NEQ
         DO I = 1, NEQ
            ICOUNT = ICOUNT + 1
            PW(I,J) = YJA(ICOUNT)
         ENDDO  
      ENDDO  
C      WRITE (*,'(A,1PE12.4)') 'T = ', X
C      DO I = 1, NEQ
C         WRITE (*,'(1P,10E12.4)') (PW(I,J), J = 1, NEQ)
C      ENDDO   
      END
C
C
