C
C EOQSOL3.FOR
C ===========
C
C VERSION : EOQSOL ... WITH 8 MODELS
C AUTHOR  : W. G. BARDSLEY, UNIVERSITY OF MANCHESTER, U.K. 2/12/91
C           Date of this version 10/11/97
C
C MODELS  : SUBROUTINE TO SET UP MODNUM, NPHI AND NTHETA
C TCHECK  : SUBROUTINE TO CHECK THETA
C LIBRARY : FUNCTIONS G1, G2 AND DG/DX FOR PROGRAM EOQSOL
C
C
      SUBROUTINE MODELS (MODNUM, NPHI, NTHETA,
     +                   MODEL)
C
C     ... CHOOSE CURRENT MODEL
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,             INTENT (OUT) :: MODNUM, NPHI, NTHETA
      CHARACTER (LEN = *), INTENT (OUT) :: MODEL
C
C Locals
C      
      INTEGER    NUMOPT
      PARAMETER (NUMOPT = 8)
      CHARACTER  TEXT(30)*100
      EXTERNAL   LISTBX
      WRITE (TEXT,100)
      MODNUM = 1
      CALL LISTBX (MODNUM, NUMOPT,  
     +             TEXT)
      IF (MODNUM.EQ.1) THEN
         MODEL = 'g2 = order 2 saturation function, ... g1 = order 1'
         NPHI = 1
         NTHETA = 1
      ELSEIF (MODNUM.EQ.2) THEN
         MODEL = 'g2 = order 3 saturation function, ... g1 = order 2'
         NPHI = 2
         NTHETA = 2
      ELSEIF (MODNUM.EQ.3) THEN
         MODEL = 'g2 = 2 High/Low affinity sites, ... g1 = 1 site'
         NPHI = 1
         NTHETA = 2
      ELSEIF (MODNUM.EQ.4) THEN
         MODEL = 'g2 = 3 High/Low affinity sites, ... g1 = 2 sites'
         NPHI = 2
         NTHETA = 4
      ELSEIF (MODNUM.EQ.5) THEN
         MODEL = 'g2 = 2 parallel exponentials, ... g1 = 1 exponential'
         NPHI = 1
         NTHETA = 2
      ELSEIF (MODNUM.EQ.6) THEN
         MODEL = 'g2 = 3 parallel exponentials, ... g1 = 2 exponentials'
         NPHI = 2
         NTHETA = 4
      ELSEIF (MODNUM.EQ.7) THEN
         MODEL = 'g2 = 2 sequential exponentials, .. g1 = 1 exponential'
         NPHI = 1
         NTHETA = 1
      ELSEIF (MODNUM.EQ.8) THEN
         MODEL = 'g2 = Hill equation, ... g1 = 1:1 function'
         NPHI = 1
         NTHETA = 1
      ENDIF
C
C Format statement
C      
  100 FORMAT (
     + 'g2 = order 2 saturation function`g1 = order 1'
     +/'g2 = order 3 saturation function`g1 = order 2'
     +/'g2 = 2 High/Low affinity sites  `g1 = 1 site'
     +/'g2 = 3 High/Low affinity sites  `g1 = 2 sites'
     +/'g2 = 2 parallel exponentials    `g1 = 1 exponential'
     +/'g2 = 3 parallel exponentials    `g1 = 2 exponentials'
     +/'g2 = 2 sequential exponentials  `g1 = 1 exponential'
     +/'g2 = Hill equation              `g1 = 1:1 rational')
      END
C
C--------------------------------------------------------------
C
      SUBROUTINE TCHECK (MODNUM, NTHETA, 
     +                   EPSI, THETA,
     +                   ABORT)
C
C CHECK THETA VALUES SUPPLIED
C
      IMPLICIT  NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: MODNUM, NTHETA
      DOUBLE PRECISION, INTENT (IN)  :: EPSI, THETA(NTHETA)
      LOGICAL,          INTENT (OUT) :: ABORT
C
C Locals
C      
      INTEGER   I
      DOUBLE PRECISION HUGE1, ONE, TSUM
      CHARACTER (LEN = 13) D13, SHOWLJ
      CHARACTER LINE*100
      LOGICAL   E_NUMBERS, E_FORMATS
      EXTERNAL  E_FORMATS, SHOWLJ
      EXTERNAL PUTADV
      E_NUMBERS = E_FORMATS()
      ONE = 1.0D+00 - EPSI
      HUGE1 = 1.0D+00/EPSI
      ABORT = .FALSE.
      DO I = 1, NTHETA
         IF (THETA(I).LT.EPSI) THEN
            ABORT = .TRUE.
            IF (E_NUMBERS) THEN
               WRITE (LINE,100) I, EPSI
            ELSE
               D13 = SHOWLJ(EPSI)  
               WRITE (LINE,150) I, TRIM(D13)
            ENDIF    
            CALL PUTADV (LINE)
         ELSEIF (THETA(I).GT.HUGE1) THEN
            ABORT = .TRUE.
            IF (E_NUMBERS) THEN 
               WRITE (LINE,200) I, HUGE1
            ELSE
               D13 = SHOWLJ(HUGE1) 
               WRITE (LINE,250) I, TRIM(D13)
            ENDIF     
            CALL PUTADV (LINE)
         ENDIF
         IF (MODNUM.EQ.3 .AND. I.EQ.1 .OR.
     +       MODNUM.EQ.4 .AND. I.LT.3 .OR.
     +       MODNUM.EQ.5 .AND. I.EQ.1 .OR.
     +       MODNUM.EQ.6 .AND. I.LT.3) THEN
            IF (THETA(I).GT.ONE) THEN
               ABORT = .TRUE.
               IF (E_NUMBERS) THEN
                  WRITE (LINE,200) I, ONE
               ELSE
                  D13 = SHOWLJ(ONE) 
                  WRITE (LINE,250) I, TRIM(D13)
               ENDIF       
               CALL PUTADV (LINE)
            ENDIF
         ENDIF
      ENDDO
      IF (MODNUM.EQ.4 .OR. MODNUM.EQ.6) THEN
         TSUM = THETA(1) + THETA(2)
         IF (TSUM.GT.ONE) THEN
            ABORT = .TRUE.
            IF (E_NUMBERS) THEN
               WRITE (LINE,300) ONE
            ELSE
               D13 = SHOWLJ(ONE)  
               WRITE (LINE,350) TRIM(D13)
            ENDIF     
            CALL PUTADV (LINE)
         ENDIF
      ENDIF
C
C Format statements
C      
  100 FORMAT ('theta(',I2,') <',1P,E13.5,' ... Try again')
  150 FORMAT ('theta(',I2,') <',1X,A,' ... Try again')
  200 FORMAT ('theta(',I2,') >',1P,E13.5,' ... Try again')
  250 FORMAT ('theta(',I2,') >',1X,A,' ... Try again')
  300 FORMAT ('theta(1) + theta(2) >',1P,E13.5,1X,
     +        '... Try again')
  350 FORMAT ('theta(1) + theta(2) >',1X,A,1X,
     +        '... Try again')   
      END
C
C-------------------------------------------------------------------
C
C LIBRARY OF MODELS G1, G2 AND DG2/DX
C
      DOUBLE PRECISION FUNCTION G1(X)

      USE MODULE_EOQSOL, ONLY : MODNUM
C
C     ... DEFICIENT MODELS
C
      IMPLICIT NONE
C
C Argument
C      
      DOUBLE PRECISION, INTENT (IN) :: X
C
C Locals
C      
      DOUBLE PRECISION G11, G12, G13, G14, G15, G16, G17, G18
      EXTERNAL G11, G12, G13, G14, G15, G16, G17, G18
      IF (MODNUM.EQ.1) THEN
         G1 = G11(X)
      ELSEIF (MODNUM.EQ.2) THEN
         G1 = G12(X)
      ELSEIF (MODNUM.EQ.3) THEN
         G1 = G13(X)
      ELSEIF (MODNUM.EQ.4) THEN
         G1 = G14(X)
      ELSEIF (MODNUM.EQ.5) THEN
         G1 = G15(X)
      ELSEIF (MODNUM.EQ.6) THEN
         G1 = G16(X)
      ELSEIF (MODNUM.EQ.7) THEN
         G1 = G17(X)
      ELSEIF (MODNUM.EQ.8) THEN
         G1 = G18(X)
      ENDIF
      END
C
C--------------------------------------------------------
C
      DOUBLE PRECISION FUNCTION G2(X)

      USE MODULE_EOQSOL, ONLY : MODNUM
C
C     ... CORRECT MODELS
C
      IMPLICIT NONE
C
C Argument
C      
      DOUBLE PRECISION, INTENT (IN) :: X
C
C Locals
C      
      DOUBLE PRECISION G21, G22, G23, G24, G25, G26, G27, G28
      EXTERNAL G21, G22, G23, G24, G25, G26, G27, G28
      IF (MODNUM.EQ.1) THEN
         G2 = G21(X)
      ELSEIF (MODNUM.EQ.2) THEN
         G2 = G22(X)
      ELSEIF (MODNUM.EQ.3) THEN
         G2 = G23(X)
      ELSEIF (MODNUM.EQ.4) THEN
         G2 = G24(X)
      ELSEIF (MODNUM.EQ.5) THEN
         G2 = G25(X)
      ELSEIF (MODNUM.EQ.6) THEN
         G2 = G26(X)
      ELSEIF (MODNUM.EQ.7) THEN
         G2 = G27(X)
      ELSEIF (MODNUM.EQ.8) THEN
         G2 = G28(X)
      ENDIF
      END
C
C------------------------------------------------------
C
      DOUBLE PRECISION FUNCTION DG2DX(X)

      USE MODULE_EOQSOL, ONLY : MODNUM
C
C     ... DERIVATIVES OF CORRECT MODEL
C
      IMPLICIT NONE
C
C Argument
C      
      DOUBLE PRECISION, INTENT (IN) :: X
C
C Locals
C      
      DOUBLE PRECISION DG2DX1, DG2DX2, DG2DX3, DG2DX4
      DOUBLE PRECISION DG2DX5, DG2DX6, DG2DX7, DG2DX8
      EXTERNAL DG2DX1, DG2DX2, DG2DX3, DG2DX4
      EXTERNAL DG2DX5, DG2DX6, DG2DX7, DG2DX8
      IF (MODNUM.EQ.1) THEN
         DG2DX = DG2DX1(X)
      ELSEIF (MODNUM.EQ.2) THEN
         DG2DX = DG2DX2(X)
      ELSEIF (MODNUM.EQ.3) THEN
         DG2DX = DG2DX3(X)
      ELSEIF (MODNUM.EQ.4) THEN
         DG2DX = DG2DX4(X)
      ELSEIF (MODNUM.EQ.5) THEN
         DG2DX = DG2DX5(X)
      ELSEIF (MODNUM.EQ.6) THEN
         DG2DX = DG2DX6(X)
      ELSEIF (MODNUM.EQ.7) THEN
         DG2DX = DG2DX7(X)
      ELSEIF (MODNUM.EQ.8) THEN
         DG2DX = DG2DX8(X)
      ENDIF
      END
C
C----------------------------------------------------------------------
C The following functions are all double precsion with x as intent (in)
C
      FUNCTION G11(X)

      USE MODULE_EOQSOL, ONLY : PHI
      
C     ...   PHI(1)*X/(PHI(1)*X + 1.0), NPHI = 1
      IMPLICIT   NONE
      DOUBLE PRECISION X, G11
      DOUBLE PRECISION ONE
      PARAMETER (ONE = 1.0D+00)
      DOUBLE PRECISION TEMP
      TEMP = PHI(1)*X
      G11 = TEMP/(ONE + TEMP)
      END
C
C
      FUNCTION G21(X)

      USE MODULE_EOQSOL, ONLY : THETA
      
C     ...   (X + THETA(1)*X**2)/(1.0 + 2.0*X + THETA(1)*X**2)
C     ...   THETA(1) = K2/K1, NTHETA = 1
      IMPLICIT   NONE
      DOUBLE PRECISION X, G21
      DOUBLE PRECISION ONE, TWO
      PARAMETER (ONE = 1.0D+00, TWO = 2.0D+00)
      DOUBLE PRECISION TEMP
      TEMP = THETA(1)*X
      G21 = X*(TEMP + ONE)/(X*(TEMP + TWO) + ONE)
      END
C
C
      FUNCTION DG2DX1(X)

      USE MODULE_EOQSOL, ONLY : THETA
      
C     ...   (X + THETA(1)*X**2)/(1.0 + 2.0*X + THETA(1)*X**2)
      IMPLICIT   NONE
      DOUBLE PRECISION X, DG2DX1
      DOUBLE PRECISION ONE, TWO
      PARAMETER (ONE = 1.0D+00, TWO = 2.0D+00)
      DOUBLE PRECISION BOT, TEMP, TOP
      TEMP = THETA(1)*X
      TOP = ONE + TWO*TEMP + X*TEMP
      BOT = ONE + X*(TWO + TEMP)
      DG2DX1 = TOP/(BOT**2)
      END
C
C
      FUNCTION G12(X)

      USE MODULE_EOQSOL, ONLY : PHI
      
C     ...   P(X) = PHI(2)*PHI(1)*X**2 + 2.0*PHI(1)*X + 1.0, NPHI = 2
      IMPLICIT   NONE
      DOUBLE PRECISION X, G12
      DOUBLE PRECISION ONE, TWO
      PARAMETER (ONE = 1.0D+00, TWO = 2.0D+00)
      DOUBLE PRECISION BOT, P1, P2, TOP
      P1 = PHI(1)*X
      P2 = PHI(2)*X*P1
      TOP = P2 + P1
      BOT = P2 + TWO*P1 + ONE
      G12 = TOP/BOT
      END
C
C
      FUNCTION G22(X)

      USE MODULE_EOQSOL, ONLY : THETA
      
C     ...   3:3 SATURATION FUNCTION WITH STATISTICAL FACTORS, NTHETA = 3
C     ...   THETA(1) = K2/K1, THETA(2) = K3/K1
      IMPLICIT   NONE
      DOUBLE PRECISION X, G22
      DOUBLE PRECISION BOT, TOP, T1, T2
      T1 = THETA(1)*X
      T2 = THETA(2)*X*T1
      TOP = X*(T2 + 2.0D+00*T1 + 1.0D+00)
      BOT = X*(T2 + 3.0D+00*T1 + 3.0D+00) + 1.0D+00
      G22 = TOP/BOT
      END
C
C
      FUNCTION DG2DX2(X)

      USE MODULE_EOQSOL, ONLY : THETA
       
C     ...   3:3 SATURATION FUNCTION WITH STATISTICAL FACTORS
      IMPLICIT   NONE
      DOUBLE PRECISION X, DG2DX2
      DOUBLE PRECISION BOT, TOP, T1, T2
      T1 = THETA(1)*X
      T2 = THETA(2)*X*T1
      TOP = 1.0D+00 + 4.0D+00*T1 + 3.0D+00*T2 +
     +      X*(3.0D+00*T1 + 4.0D+00*T2 + X*T1*T2)
      BOT = 1.0D+00 + X*(3.0D+00 + 3.0D+00*T1 + T2)
      DG2DX2 = TOP/(BOT**2)
      END
C
C
      FUNCTION G13(X)

      USE MODULE_EOQSOL, ONLY : PHI
      
C     ...   PHI(1)*X/(PHI(1)*X + 1.0), NPHI = 1
      IMPLICIT   NONE
      DOUBLE PRECISION X, G13
      DOUBLE PRECISION TEMP
      TEMP = PHI(1)*X
      G13 = TEMP/(1.0D+00 + TEMP)
      END
C
C
      FUNCTION G23(X)

      USE MODULE_EOQSOL, ONLY : THETA
      
C     ...   THETA(1)*X/(1 + X) + (1 - THETA(1))*THETA(2)*X/(1 + THETA(2)*X)
C     ...   THETA(2) = K2/K1, NTHETA = 2
      IMPLICIT   NONE
      DOUBLE PRECISION X, G23
      DOUBLE PRECISION TEMP
      TEMP = THETA(2)*X
      G23 = THETA(1)*X/(1.0D+00 + X) +
     +      (1.0D+00 - THETA(1))*TEMP/(1.0D+00 + TEMP)
      END
C
C
      FUNCTION DG2DX3(X)

      USE MODULE_EOQSOL, ONLY : THETA
      
C     ...   THETA(1)*X/(1 + X) + (1 - THETA(1))*THETA(2)*X/(1 + THETA(2)*X)
      IMPLICIT   NONE
      DOUBLE PRECISION X, DG2DX3
      DOUBLE PRECISION TEMP
      TEMP = THETA(2)*X
      DG2DX3 = THETA(1)/((1.0D+00 + X)**2)
     +       + (1.0D+00 - THETA(1))*THETA(2)/((1.0D+00 + TEMP)**2)
      END
C
C
      FUNCTION G14(X)

      USE MODULE_EOQSOL, ONLY : PHI
      
C     ...   PHI(1)*X/(1 + X) + (1 - PHI(1))*PHI(2)*X/(1 + PHI(2)*X)
C     ...   PHI(2) = K2/K1, NPHI = 2
      IMPLICIT   NONE
      DOUBLE PRECISION X, G14
      DOUBLE PRECISION TEMP
      TEMP = PHI(2)*X
      G14 = PHI(1)*X/(1.0D+00 + X) +
     +      (1.0D+00 - PHI(1))*TEMP/(1.0 + TEMP)
      END
C
C
      FUNCTION G24(X)

      USE MODULE_EOQSOL, ONLY : THETA
      
C     ...   THETA(1)*X/(1 + X) + THETA(2)*THETA(3)*X/(1.0 + THETA(3)*X)
C           + (1 - THETA(1) - THETA(2))*THETA(4)*X/(1 + THETA(4)*X)
C     ...   THETA(3) = K2/K1, THETA(4) = K3/K1, NTHETA = 4
      IMPLICIT   NONE
      DOUBLE PRECISION X, G24
      DOUBLE PRECISION T1, T2
      T1 = THETA(3)*X
      T2 = THETA(4)*X
      G24 = THETA(1)*X/(1.0D+00 + X) + THETA(2)*T1/(1.0D+00 + T1) +
     +      (1.0D+00 - THETA(1) - THETA(2))*T2/(1.0D+00 + T2)
      END
C
C
      FUNCTION DG2DX4(X)

      USE MODULE_EOQSOL, ONLY : THETA
      
C     ...   THETA(1)*X/(1 + X) + THETA(2)*THETA(3)*X/(1 + THETA(3)*X)
C           + (1.0 - THETA(1) - THETA(2))*THETA(4)*X/(1.0 + THETA(4)*X)
      IMPLICIT   NONE
      DOUBLE PRECISION X, DG2DX4
      DOUBLE PRECISION T0, T1, T2
      T0 = (1.0D+00 + X)**2
      T1 = (1.0D+00 + THETA(3)*X)**2
      T2 = (1.0D+00 + THETA(4)*X)**2
      DG2DX4 = THETA(1)/T0 + THETA(2)*THETA(3)/T1
     +       + (1.0D+00 - THETA(1) - THETA(2))*THETA(4)/T2
      END
C
C
      FUNCTION G15(X)

      USE MODULE_EOQSOL, ONLY : ENEG, EPOS, PHI
      
C     ...   EXP( - PHI(1)*X), NPHI = 1
      IMPLICIT   NONE
      DOUBLE PRECISION X, G15
      DOUBLE PRECISION TEMP
      EXTERNAL  MIDDLE
      INTRINSIC EXP
      TEMP =  - PHI(1)*X
      CALL MIDDLE (ENEG, TEMP, EPOS)
      G15 = EXP(TEMP)
      END
C
C
      FUNCTION G25(X)

      USE MODULE_EOQSOL, ONLY : ENEG, EPOS, THETA
      
C     ...   THETA(1)*EXP(-X) + (1.0 - THETA(1))*EXP(-THETA(2)*X)
C     ...   THETA(2) = K2/K1, NTHETA = 2
      IMPLICIT   NONE
      DOUBLE PRECISION X, G25
      DOUBLE PRECISION T1, T2
      EXTERNAL  MIDDLE
      INTRINSIC EXP
      T1 = - X
      CALL MIDDLE (ENEG, T1, EPOS)
      T2 = - THETA(2)*X
      CALL MIDDLE (ENEG, T2, EPOS)
      G25 = THETA(1)*EXP(T1) + (1.0D+00 - THETA(1))*EXP(T2)
      END
C
C
      FUNCTION DG2DX5(X)

      USE MODULE_EOQSOL, ONLY : ENEG, EPOS, THETA
      
C     ...   THETA(1)*EXP(-X) + (1.0 - THETA(1))*EXP(-THETA(2)*X)
      IMPLICIT   NONE
      DOUBLE PRECISION X, DG2DX5
      DOUBLE PRECISION T1, T2
      EXTERNAL  MIDDLE
      INTRINSIC EXP
      T1 = - X
      CALL MIDDLE (ENEG, T1, EPOS)
      T2 = - THETA(2)*X
      CALL MIDDLE (ENEG, T2, EPOS)
      DG2DX5 = - THETA(1)*EXP(T1) -
     +           (1.0D+00 - THETA(1))*THETA(2)*EXP(T2)
      END
C
C
      FUNCTION G16(X)

      USE MODULE_EOQSOL, ONLY : ENEG, EPOS, PHI
      
C     ...   PHI(1)*EXP( - X) + (1.0 - PHI(1))*EXP( - PHI(2)*X), NPHI = 2
      IMPLICIT   NONE
      DOUBLE PRECISION X, G16
      DOUBLE PRECISION T1, T2
      EXTERNAL  MIDDLE
      INTRINSIC EXP
      T1 = - X
      CALL MIDDLE (ENEG, T1, EPOS)
      T2 =  - PHI(2)*X
      CALL MIDDLE (ENEG, T2, EPOS)
      G16 = PHI(1)*EXP(T1) + (1.0D+00 - PHI(1))*EXP(T2)
      END
C
C
      FUNCTION G26(X)

      USE MODULE_EOQSOL, ONLY : ENEG, EPOS, THETA
      
C     ...   THETA(1)*EXP(-X) + THETA(2)*EXP(-THETA(3)*X)
C           + (1.0 - THETA(1) - THETA(2))*EXP(- THETA(4)*X)
C     ...   THETA(3) = K2/K1, THETA(4) = K3/K1, NTHETA = 4
      IMPLICIT   NONE
      DOUBLE PRECISION X, G26
      DOUBLE PRECISION T1, T2, T3
      EXTERNAL  MIDDLE
      INTRINSIC EXP
      T1 = - X
      CALL MIDDLE (ENEG, T1, EPOS)
      T2 = - THETA(3)*X
      CALL MIDDLE (ENEG, T2, EPOS)
      T3 = - THETA(4)*X
      CALL MIDDLE (ENEG, T3, EPOS)
      G26 = THETA(1)*EXP(T1) + THETA(2)*EXP(T2) +
     +      (1.0D+00 - THETA(1) - THETA(2))*EXP(T3)
      END
C
C
      FUNCTION DG2DX6(X)

      USE MODULE_EOQSOL, ONLY : ENEG, EPOS, THETA
      
C     ...   THETA(1)*EXP( - X) + THETA(2)*EXP( - THETA(3)*X)
C           + (1.0 - THETA(1) - THETA(2))*EXP( - THETA(4)*X)
      IMPLICIT   NONE
      DOUBLE PRECISION X, DG2DX6
      DOUBLE PRECISION T1, T2, T3 
      EXTERNAL  MIDDLE
      INTRINSIC EXP
      T1 = - X
      CALL MIDDLE (ENEG, T1, EPOS)
      T2 = - THETA(3)*X
      CALL MIDDLE (ENEG, T2, EPOS)
      T3 = - THETA(4)*X
      CALL MIDDLE (ENEG, T3, EPOS)
      DG2DX6 = - THETA(1)*EXP(T1) - THETA(2)*THETA(3)*EXP(T2)
     +         - (1.0D+00 - THETA(1) - THETA(2))*THETA(4)*EXP(T3)
      END
C
C
      FUNCTION G17(X)

      USE MODULE_EOQSOL, ONLY : ENEG, EPOS, PHI
      
C     ...   1.0 - EXP( - PHI(1)*X), NPHI = 1
      IMPLICIT   NONE
      DOUBLE PRECISION X, G17
      DOUBLE PRECISION T1
      EXTERNAL  MIDDLE
      INTRINSIC EXP
      T1 = - PHI(1)*X
      CALL MIDDLE (ENEG, T1, EPOS)
      G17 = 1.0D+00 - EXP(T1)
      END
C
C
      FUNCTION G27(X)

      USE MODULE_EOQSOL, ONLY : ENEG, EPOS, EPSI, THETA
      
C     ...   1.0-(THETA(1)*EXP(-X)-EXP(-THETA(1)*X))/(THETA(1)-1.0)
C     ...   1.0-(X+1)*EXP(-X) IF THETA(1) = 1.0
C     ...   THETA(1) = K2/K1, NTHETA = 1
      IMPLICIT   NONE
      DOUBLE PRECISION X, G27
      DOUBLE PRECISION DN, T1, T2
      EXTERNAL  MIDDLE
      INTRINSIC ABS, EXP
      T1 = - X
      CALL MIDDLE (ENEG, T1, EPOS)
      T2 = - THETA(1)*X
      CALL MIDDLE (ENEG, T2, EPOS)
      IF (ABS(THETA(1) - 1.0D+00).LE.EPSI) THEN
         G27 = 1.0D+00 - (X + 1.0D+00)*EXP(T1)
      ELSE
         DN = THETA(1) - 1.0D+00
         G27 = 1.0D+00 - (THETA(1)*EXP(T1) - EXP(T2))/DN
      ENDIF
      END
C
C
      FUNCTION DG2DX7(X)

      USE MODULE_EOQSOL, ONLY : ENEG, EPOS, EPSI, THETA
C     ...   1.0-(THETA(1)*EXP(-X)-EXP(-THETA(1)*X))/(THETA(1)-1.0)
C     ...   1.0-(X+1)*EXP(-X) IF THETA(1) = 1.0
      IMPLICIT   NONE
      DOUBLE PRECISION X, DG2DX7
      DOUBLE PRECISION DN, T1, T2
      EXTERNAL  MIDDLE
      INTRINSIC ABS, EXP
      T1 = - X
      CALL MIDDLE (ENEG, T1, EPOS)
      T2 = - THETA(1)*X
      CALL MIDDLE (ENEG, T2, EPOS)
      IF (ABS(THETA(1) - 1.0D+00).LE.EPSI) THEN
         DG2DX7 = X*EXP(T1)
      ELSE
         DN = THETA(1) - 1.0D+00
         DG2DX7 = THETA(1)*(EXP(T1) - EXP(T2))/DN
      ENDIF
      END
C
C
      FUNCTION G18(X)

      USE MODULE_EOQSOL, ONLY : PHI
      
C     ...   PHI(1)*X/(PHI(1)*X + 1.0), NPHI = 1
      IMPLICIT   NONE
      DOUBLE PRECISION X, G18
      DOUBLE PRECISION TEMP
      TEMP = PHI(1)*X
      G18 = TEMP/(1.0D+00 + TEMP)
      END
C
C
      FUNCTION G28(X)

      USE MODULE_EOQSOL, ONLY : RTOL, THETA
      
C     ...   X**THETA(1)/(1.0 + X**THETA(1))
C     ...   THETA(1) = N, HILL COEFFICIENT, NTHETA = 1
      IMPLICIT   NONE
      DOUBLE PRECISION X, G28
      DOUBLE PRECISION TEMP
      IF (X.GT.RTOL) THEN
         TEMP = X**THETA(1)
         G28 = TEMP/(TEMP + 1.0D+00)
      ELSE
         G28 = 0.0D+00
      ENDIF
      END
C
C
      FUNCTION DG2DX8(X)

      USE MODULE_EOQSOL, ONLY : RTOL, THETA
      
C     ...   X**THETA(1)/(1.0 + X**THETA(1))
      IMPLICIT   NONE
      DOUBLE PRECISION X, DG2DX8
      DOUBLE PRECISION T1, T2
      IF (X.GT.RTOL) THEN
         T1 = X**THETA(1)
         T2 = THETA(1)*X**(THETA(1) - 1.0D+00)
         DG2DX8 = T2/((1.0D+00 + T1)**2)
      ELSE
         DG2DX8 = 0.0D+00
      ENDIF
      END
C
C
