C
C FTN95 version
C =============
C
      PROGRAM MAIN
C
C VERSION : Defined in SIMVER/DLLCHK
C FORTRAN : 95, Double precision
C INPUT   : TITLE, NPTS, X, Y, S = Standard error of Y and text
C OUTPUT  : File ready for curve-fitting programs
C AUTHOR  : W. G. Bardsley, University of Manchester, U.K.
C REVISED : 08/12/1989 ADVISE/OFILES/GRAF2A/HPSORT
C           05/11/1990 NROW and NCOL/GRF001
C           24/04/1991 SCREEN and GRFGK1
C           04/05/1991 CHECK and WEIGHT added
C           22/05/1991 GKS001
C           10/07/1992 Simplified and extensively re-organised
C           12/02/1993 GET???, PUT??? subroutines and compressed
C           21/12/1993 First FTN77 version
C           25/03/1994 DBOS version 
C           20/02/1995 Salamanca version
C           24/11/1995 Changed SMIN from 1.0e-35 to 1.0e-20
C           16/02/1997 Win32 version
C           07/08/1998 added dllchk
C           03/10/1998 FTN95 version
C           08/10/1998 added W_EDITTX and LABEL = FNAME in DATAIN
C           14/08/1999 replaced w_edittx by edittx
C           13/09/1999 added call to WINDOW
C           07/02/2000 added DILUTE for dilution mode
C           14/02/2000 added SIMVER
C           17/03/2001 simplified interface as follows:
C                      ORDERX = .TRUE., X must be in increasing order
C                      DILUTE = .TRUE., X must be in increasing order for dilution
C                      WEIGHT = .TRUE., S must be supplied
C                      EXPERT = .TRUE., S is to be calculated
C                      XTEXT = .TRUE., add extra text data to end of file
C                      XYONLY = .TRUE., only X and Y sent to editor and table
C           18/12/2002 replaced TRIM80 by TRIM40 and WORD40 in DATAIN and
C                      introduced TRIM60 and WORD60 into SUB003
C           24/03/2003 added CALCS to force calculation of S if required
C           27/07/2005 used SIMVER to retrieve the version identifier and
C                      pass it as a version identifier to ADVISE and
C                      increased DVER to length = 30
C           23/03/2008 edited for version 6
C           01/12/2009 changed TABLE1 to TABLE5
C           01/10/2010 restored TABLE1 and increased number of signifuicant digits to 8
C           17/03/2018 added YTEXT and calls to EDITPL and MFTIDY to add a parameter limits section 
C
      IMPLICIT   NONE
      INTEGER    ISEND, NOUT
      PARAMETER (ISEND = 1, NOUT = 4)
      INTEGER    NMAX
      PARAMETER (NMAX = 250)
      INTEGER    ICOUNT, ITYPE, JSEND, NOPT, NPTS
      DOUBLE PRECISION ORDER(NMAX), S(NMAX), X(NMAX), Y(NMAX)
      DOUBLE PRECISION STORE(NMAX), TEMP(NMAX,3)
      DOUBLE PRECISION XVER, YVER
      CHARACTER  FNAME*1024, TITLE*80
      CHARACTER  CIPHER(2,NMAX)*6
      CHARACTER  BLANK*1, PNAME*6
      PARAMETER (BLANK = ' ', PNAME = 'MAKFIL')
      CHARACTER  DVER*30, PVER*15
      PARAMETER (PVER = 'w_makfil.exe')
      LOGICAL    CALL_EDITOR
      PARAMETER (CALL_EDITOR = .TRUE.)
      LOGICAL    ABORT, ACTION, AGAIN, CHECK, DILUTE, EXPERT, OK,
     +           ORDERX, READY, REPEET, SHOW, WEIGHT, XTEXT, XYONLY,
     +           YTEXT
      LOGICAL    CALCS
      EXTERNAL   ADVISE, OFILES, DATAIN, CHKDAT, STOPGO
      EXTERNAL   DETAIL, SUB001, SUB002, SUB003, SUB004, SUB005, SUB006,
     +           SUB007, SUB008, SUB009
      EXTERNAL   DLLCHK, WINDOW, SIMVER, PUTADV

C
C======================================================================
C Open an inactive background window and then check the DLLs
C The following values must be edited at each release:
C XVER = version number
C YVER = release number
C DVER = version name
C These must be consistent with the same values in the SIMFIT DLLs
C
      JSEND = 1
      ACTION = .TRUE.
      TITLE = 'Simfit: program '// PNAME
      CALL WINDOW (JSEND,
     +             TITLE,
     +             ACTION)
      CALL SIMVER (XVER, YVER,
     +             DVER)
      ABORT = .FALSE.
      SHOW = .FALSE.
      CALL DLLCHK (XVER, YVER,
     +             DVER, PVER,
     +             ABORT, SHOW)
C
C Checking completed so now proceed to the main program
C======================================================================
C

C
C Initialise then advise user and open output file or abort
C
      FNAME = BLANK
      DILUTE = .FALSE.
      EXPERT = .FALSE.
      ORDERX = .TRUE.
      WEIGHT = .FALSE.
      XTEXT = .TRUE.
      YTEXT = .TRUE.
      XYONLY = .TRUE.
      ICOUNT = 0
      ITYPE = 1
      CALL ADVISE (ITYPE, NMAX,
     +             DVER,
     +             ABORT, DILUTE, EXPERT, ORDERX, WEIGHT,
     +             XTEXT, XYONLY, YTEXT)
      IF (ABORT) THEN
         REPEET = .FALSE.
      ELSE
         REPEET = .TRUE.
      ENDIF
C
C The main cycle point
C ====================
C
      DO WHILE (REPEET)
         CALL PUTADV (
     +'Now specify a file in which to save your curve fitting data')
         CLOSE (UNIT = NOUT)
         CALL OFILES (ISEND, NOUT,
     +                FNAME,
     +                ABORT)
         IF (ABORT) THEN
            CLOSE (UNIT = NOUT)
            REPEET = .FALSE.
C
C Enter primary data points
C
         ELSE
            ICOUNT = ICOUNT + 1
            IF (ICOUNT.GT.1) THEN
               ITYPE = 3
               CALL ADVISE (ITYPE, NMAX,
     +                      DVER,
     +                      ABORT, DILUTE, EXPERT, ORDERX, WEIGHT,
     +                      XTEXT, XYONLY, YTEXT)
            ENDIF
            CALL DATAIN (NMAX, NPTS,
     +                   S, STORE, X, Y,
     +                   FNAME, TITLE,
     +                   DILUTE, ORDERX, WEIGHT)
C
C Use CALCS to force calculation of S if calculating S has been selected
C
            IF (EXPERT .AND. .NOT.WEIGHT .AND. XYONLY) THEN
               CALCS = .TRUE.
            ELSE
               CALCS = .FALSE.
            ENDIF
C
C The lesser cycle point
C ======================
C
            AGAIN = .TRUE.
            CHECK = .TRUE.
            DO WHILE (AGAIN)
               IF (CHECK) CALL CHKDAT (NPTS,
     +                                 S, X, Y,
     +                                 CIPHER,
     +                                 OK, READY)
               CALL DETAIL (NOPT,
     +                      CALCS, EXPERT, OK, READY, WEIGHT)
               IF (NOPT.EQ.1) THEN
C
C Table
C
                  CALL SUB001 (NPTS,
     +                         S, X, Y,
     +                         CIPHER,
     +                         EXPERT)
                  AGAIN = .TRUE.
                  CHECK = .FALSE.
               ELSEIF (NOPT.EQ.2) THEN
C
C Plot
C
                  CALL SUB002 (NPTS,
     +                         X, Y)
                  AGAIN = .TRUE.
                  CHECK = .FALSE.
               ELSEIF (NOPT.EQ.3) THEN
C
C Edit
C
                  CALL SUB003 (NMAX, NPTS,
     +                         S, TEMP, X, Y, 
     +                         FNAME,
     +                         CALL_EDITOR, XYONLY)
                  AGAIN = .TRUE.
                  CHECK = .TRUE.
               ELSEIF (NOPT.EQ.4) THEN
C
C Put into order
C
                  CALL SUB004 (NPTS,
     +                         ORDER, S, X, Y,
     +                         READY)
                  AGAIN = .TRUE.
                  CHECK = .TRUE.
               ELSEIF (NOPT.EQ.5) THEN
C
C Write file
C
                  CALL SUB005 (NOUT, NPTS, 
     +                         S, X, Y,
     +                         FNAME, TITLE,
     +                         READY, XTEXT, YTEXT) 
                  IF (.NOT.READY) THEN
                     AGAIN = .TRUE.
                     CHECK = .TRUE.
                  ELSE
                     AGAIN = .FALSE.
                  ENDIF
               ELSEIF (NOPT.EQ.6) THEN
C
C Calculate s from replicates
C
                  CALL SUB006 (NPTS,
     +                         ORDER, S, X, Y,
     +                         READY)
                  AGAIN = .TRUE.
                  CHECK = .TRUE.
               ELSEIF (NOPT.EQ.7) THEN
C
C Set s = k
C
                  CALL SUB007 (NPTS,
     +                         S)
                  AGAIN = .TRUE.
                  CHECK = .TRUE.
               ELSEIF (NOPT.EQ.8) THEN
C
C Set s = %|y|
C
                  CALL SUB008 (NPTS,
     +                         S, Y)
                  AGAIN = .TRUE.
                  CHECK = .TRUE.
               ELSEIF (NOPT.EQ.9) THEN
C
C Restore s = s typed in
C
                  CALL SUB009 (NPTS,
     +                         S, STORE)
                  AGAIN = .TRUE.
                  CHECK = .TRUE.
               ELSE
                  ITYPE = 2
                  CALL ADVISE (ITYPE, NMAX,
     +                         DVER,
     +                         ABORT, DILUTE, EXPERT, ORDERX, WEIGHT,
     +                         XTEXT, XYONLY, YTEXT)
                  AGAIN = .TRUE.
               ENDIF
            ENDDO
C
C Is another run required ?
C
            CALL STOPGO (BLANK, FNAME, PNAME,
     +                   ABORT)
            IF (ABORT) THEN
               REPEET = .FALSE.
            ELSE
               REPEET = .TRUE.
            ENDIF
         ENDIF
      ENDDO

C
C======================================================================
C The program is finished so we can close down the background window
C
      JSEND = 1
      ACTION = .FALSE.
      CALL WINDOW (JSEND,
     +             TITLE, 
     +             ACTION)
C
C======================================================================
C

      END
C
C------------------------------------------------------------------------
C
      SUBROUTINE ADVISE (ITYPE, NMAX,
     +                   DVER,
     +                   ABORT, DILUTE, EXPERT, ORDER,
     +                   WEIGHT, XTEXT, XYONLY, YTEXT)
C
C Advise user: ITYPE = 1: first time call
C              ITYPE = 2: display help
C              ITYPE = 3: choose mode
C
      IMPLICIT   NONE
C
C Arguments
C
      INTEGER,             INTENT (IN)    :: ITYPE, NMAX
      CHARACTER (LEN = *), INTENT (IN)    :: DVER
      LOGICAL,             INTENT (OUT)   :: ABORT
      LOGICAL,             INTENT (INOUT) :: DILUTE, EXPERT, ORDER,
     +                                       WEIGHT, XTEXT, XYONLY,
     +                                       YTEXT  
C
C Locals
C
      INTEGER    ISEND
      INTEGER    ICOLOR, NUMHDR, NUMOPT
      PARAMETER (ICOLOR = 7, NUMHDR = 14, NUMOPT = 3)
      INTEGER    NUMBLD(NUMHDR), NUMPOS(NUMOPT)
      INTEGER    KCOLOR, KX, KY, KSHADE, KUMDEC, KUMOPT,
     +           KSTART, KTEXT
      PARAMETER (KCOLOR = 7, KX = 4, KY = 4, KSHADE = 1, KUMDEC = 1,
     +           KUMOPT = 9, KSTART = 3, KTEXT = 11)
      INTEGER    KUMBLD(KTEXT), KUMPOS(KUMOPT)
      CHARACTER  HEADER(NUMHDR)*100, OPTION(NUMOPT)*50
      CHARACTER  TEXT(30)*100
      LOGICAL    ACCEPT, REPEET
      LOGICAL    BORDER, FLASH, HIGH
      PARAMETER (BORDER = .FALSE., FLASH = .FALSE., HIGH = .TRUE.)
      EXTERNAL   TITLES, PUTADV, RBOX01, HELP_MAKFIL
      SAVE       ACCEPT
      DATA       ACCEPT / .FALSE. /
      DATA       NUMBLD / NUMHDR*0 /
      DATA       NUMPOS / 1, 1, 1 /
      DATA       OPTION /
     +'Help           ',
     +'Run the program',
     +'Quit  ...  Exit'/
      ABORT = .FALSE.
      REPEET = .TRUE.
      DO WHILE (REPEET)
         IF (ITYPE.EQ.1) THEN
            REPEET = .TRUE.
            WRITE (HEADER,100) DVER, NMAX
            ISEND = 2
            CALL TITLES (ICOLOR, NUMBLD, ISEND, NUMHDR, NUMOPT, NUMPOS,
     +                   HEADER, OPTION)
         ELSEIF (ITYPE.EQ.2) THEN
            REPEET = .FALSE.
            ISEND = 1
         ELSEIF (ITYPE.EQ.3) THEN
            ISEND = 2
         ENDIF
         IF (ISEND.EQ.1) THEN
            CALL HELP_MAKFIL ('makfil')
         ELSEIF (ISEND.EQ.2) THEN
            IF (ACCEPT) RETURN
            KUMBLD(1) = 1
            KUMBLD(2) = 0
            KUMBLD(3) = 100
            KUMBLD(4) = 100
            KUMBLD(5) = 100
            KUMBLD(6) = 200
            KUMBLD(7) = 200
            KUMBLD(8) = 200
            KUMBLD(9) = 0
            KUMBLD(10) = 0
            KUMBLD(11) = 0
            IF (DILUTE) THEN
               ORDER = .TRUE.
               KUMPOS(1) = 0
               KUMPOS(2) = 1
               KUMPOS(3) = 0
            ELSE
               KUMPOS(2) = 0
               IF (ORDER) THEN
                  KUMPOS(1) = 1
                  KUMPOS(3) = 0
               ELSE
                  KUMPOS(1) = 0
                  KUMPOS(3) = 1
               ENDIF
            ENDIF
            IF (WEIGHT) THEN
               EXPERT = .TRUE.
               KUMPOS(4) = 0
               KUMPOS(5) = 0
               KUMPOS(6) = 1
            ELSE
               KUMPOS(6) = 0
               IF (EXPERT) THEN
                  KUMPOS(4) = 0
                  KUMPOS(5) = 1
               ELSE
                  KUMPOS(4) = 1
                  KUMPOS(5) = 0
               ENDIF
            ENDIF
            IF (XTEXT) THEN
               KUMPOS(7) = 1
            ELSE
               KUMPOS(7) = 0
            ENDIF
            IF (YTEXT) THEN
               KUMPOS(8) = 1
            ELSE
               KUMPOS(8) = 0
            ENDIF
            KUMPOS(9) = 0
            WRITE (TEXT,200)
            CALL RBOX01 (KCOLOR, KX, KY, KSHADE, KUMBLD, KUMDEC,
     +                   KUMOPT, KUMPOS, KSTART, KTEXT,
     +                   TEXT,
     +                   BORDER, FLASH, HIGH)
            IF (KUMPOS(1).EQ.1) THEN
               ORDER = .TRUE.
               DILUTE = .FALSE.
            ELSEIF (KUMPOS(2).EQ.1) THEN
               ORDER = .TRUE.
               DILUTE = .TRUE.
            ELSEIF (KUMPOS(3).EQ.1) THEN
               ORDER = .FALSE.
               DILUTE = .FALSE.
            ENDIF
            IF (KUMPOS(4).EQ.1) THEN
               WEIGHT = .FALSE.
               EXPERT = .FALSE.
               XYONLY = .TRUE.
            ELSEIF (KUMPOS(5).EQ.1) THEN
               WEIGHT = .FALSE.
               EXPERT = .TRUE.
               XYONLY = .TRUE.
            ELSEIF (KUMPOS(6).EQ.1) THEN
               WEIGHT = .TRUE.
               EXPERT = .TRUE.
               XYONLY = .FALSE.
            ENDIF
            IF (KUMPOS(7).EQ.1) THEN
               XTEXT = .TRUE.
            ELSE
               XTEXT = .FALSE.
            ENDIF
             IF (KUMPOS(8).EQ.1) THEN
               YTEXT = .TRUE.
            ELSE
               YTEXT = .FALSE.
            ENDIF
            IF (KUMPOS(9).EQ.1) ACCEPT = .TRUE.
            IF (DILUTE) CALL PUTADV (
     +'Input x (>=1) as x = 1, 2, 4, etc. for x = 1:1, 1:2, 1:4')
            IF (WEIGHT) CALL PUTADV (
     +'Input s (> 0) for weights w = 1/s^2 (i.e. s = std. dev. y)')
            ABORT = .FALSE.
            REPEET = .FALSE.
         ELSEIF (ISEND.EQ.3) THEN
            ABORT = .TRUE.
            REPEET = .FALSE.
         ENDIF
      ENDDO
C
C Format statements
C      
  100 FORMAT (
     + 'Package `SIMFIT'
     +/'        `      '
     +/'Program `MAKFIL'
     +/'        `      '
     +/'Action  `Make a file with (x,y) or (x,y,s) data for curve'
     +/'        `fitting, graph plotting or calibration. Weights'
     +/'        `can be input or calculated from data values.'
     +/'        `      '
     +/'Version `',A
     +/'        `Maximum number of rows',I5
     +/'        `      '
     +/'Graphics`Windows types plus EPS, PDF, PNG, and SVG.'
     +/'        `      '
     +/'Author  `W.G.Bardsley, University of Manchester, U.K.')
  200 FORMAT (
     + 'Data input options'
     +/'...'
     +/'x supplied in increasing order (normal mode)'
     +/'x supplied in increasing order (dilution assay)'
     +/'x supplied in arbitrary order'
     +/'s not typed in (input x,y: set to s = 1)'
     +/'s not typed in (input x,y: s calculated)'
     +/'s typed in (input x,y,s: i.e. weights known)'
     +/'Option to add descriptive text to end of file'
     +/'Option to add parameter starting estimates and limits' 
     +/'Do not ask again during this run (switch of)')
      END
C
C--------------------------------------------------------------------
C
      SUBROUTINE DATAIN (NMAX, NPTS,
     +                   S, STORE, X, Y,
     +                   FNAME, TITLE,
     +                   DILUTE, ORDER, WEIGHT)
C
C Set XYONLY, read in TITLE, X, Y, S = 1 or S
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,             INTENT (IN)  :: NMAX     
      INTEGER,             INTENT (OUT) :: NPTS
      DOUBLE PRECISION,    INTENT (OUT) :: S(NMAX), STORE(NMAX),
     +                                     X(NMAX), Y(NMAX)
      CHARACTER (LEN = *), INTENT (IN)  :: FNAME
      CHARACTER (LEN = *), INTENT (OUT) :: TITLE
      LOGICAL,             INTENT (IN)  :: DILUTE, ORDER, WEIGHT
C
C Locals
C      
      INTEGER    N1, N8, N12, NLARGE
      PARAMETER (N1 = 1, N8 = 8, N12 = 12, NLARGE = 50)
      INTEGER    JCOLOR
      PARAMETER (JCOLOR = 8)
      INTEGER    NUMBLD(N12), NUMHDR
      INTEGER    I, J, K, L, LEN200
      DOUBLE PRECISION RTOL, STEMP, XMAX, XMIN, XTEMP, YTEMP
      DOUBLE PRECISION ONE
      PARAMETER (ONE = 1.0D+00)
      DOUBLE PRECISION X02AMF$
      CHARACTER  LINE*1024, HEADER(N12)*100, TRIM40*40, WORD8*8,
     +           WORD40*40
      LOGICAL    REPEET, YES
      EXTERNAL   GETIL1, GETSXY, ANSWER, GETSTR, LEN200, PUTFAT, PUTWAR,
     +           TRIM40, YMDHMS
      EXTERNAL   X02AMF$
      DATA       NUMBLD / N12*0 /
      CALL YMDHMS (WORD8)
      TITLE = 'New Data: '//WORD8
      LINE = 'An informative title for your data'
      CALL GETSTR (LINE, TITLE)
      REPEET = .TRUE.
      DO WHILE (REPEET)
         WRITE (LINE,100) NMAX
         CALL GETIL1 (N1, NPTS, NMAX,
     +                LINE)
         IF (NPTS.GT.NLARGE) THEN
            WRITE (HEADER,200) NLARGE
            LINE = 'Type full data set requested (usually no) ?'
            NUMHDR = N8
            YES = .FALSE.
            NUMBLD(1) = 1
            CALL ANSWER (JCOLOR, NUMBLD, NUMHDR,
     +                   HEADER, LINE, 
     +                   YES)
            NUMBLD(1) = 0
            IF (YES) THEN
               REPEET = .FALSE.
            ELSE
               REPEET = .TRUE.
            ENDIF
         ELSE
            REPEET = .FALSE.
         ENDIF
      ENDDO
C
C Read in the primary data points
C
      WORD40 = TRIM40(FNAME)
      L = LEN200(WORD40)
      IF (ORDER) THEN
         LINE =  WORD40(N1:L)//' (x in increasing order)'
      ELSE
         LINE =  WORD40(N1:L)//' (x in any order)'
      ENDIF
      CALL GETSXY (NPTS,
     +             S, X, Y,
     +             LINE, 
     +             ORDER, WEIGHT)
C
C Store original S values
C
      DO I = 1, NPTS
         STORE(I) = S(I)
      ENDDO
      IF (DILUTE) THEN
C
C If in dilution mode invert the data set
C
         RTOL = 1.0D+09*X02AMF$()
         XMAX = ONE/(1.0D+100*RTOL)
         XMIN = ONE - RTOL
         DO I = 1, NPTS
            IF (X(I).GT.XMAX) THEN
               WRITE (LINE,300) I
               CALL PUTFAT (LINE)
               WRITE (LINE,400)
               CALL PUTWAR (LINE)
               RETURN
            ENDIF
            IF (X(I).LT.XMIN) THEN
               WRITE (LINE,500) I
               CALL PUTFAT (LINE)
               WRITE (LINE,400)
               CALL PUTWAR (LINE)
               RETURN
            ENDIF
         ENDDO
         DO I = 1, NPTS
            STEMP = S(NPTS)
            XTEMP = X(NPTS)
            YTEMP = Y(NPTS)
            DO J = NPTS, I + 1, - 1
               K = J - 1
               S(J) = S(K)
               X(J) = X(K)
               Y(J) = Y(K)
            ENDDO
            S(I) = STEMP
            X(I) = ONE/XTEMP
            Y(I) = YTEMP
         ENDDO
      ENDIF
C
C Format statements
C      
  100 FORMAT (
     +'No. of x,y data pairs you want to type in now: max. =',I4)
  200 FORMAT (
     +'Warning     `You have requested to type in >',I3,1X,
     +'data points'
     +/
     +/'Advice     `To prepare large data files it is better to make'
     +/'           `smaller files, then join them together using the'
     +/'           `program EDITFL. This avoids the errors that tend'
     +/'           `to occur when typing in too many lines of data.'
     +/'           `You can of course type in large data sets if you'
     +/'           `insist.')
  300 FORMAT (
     +'Data cannot be rearranged. X too large at line number',I4)
  400 FORMAT (
     +'Data file will be unsuitable for use in dilution mode')
  500 FORMAT (
     +'Data cannot be rearranged. X < 1 at line number',I4)
      END
C
C------------------------------------------------------------------------
C
      SUBROUTINE DETAIL (NDEC,
     +                   CALCS, EXPERT, OK, READY, WEIGHT)
C
C Decide option required
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER, INTENT (OUT)   :: NDEC
      LOGICAL, INTENT (IN)    :: EXPERT, OK, READY, WEIGHT
      LOGICAL, INTENT (INOUT) :: CALCS
C
C Locals
C      
      INTEGER    N1, N4, N5, N6, N7, N8, N10, N14
      PARAMETER (N1 = 1, N4 = 4, N5 = 5, N6 = 6, N7 = 7, N8 = 8, 
     +           N10 = 10, N14 = 14)
      INTEGER    ICOLOR, IX, IY, LSHADE, NSTART, NTEXT
      PARAMETER (ICOLOR = 3, IX = 4, IY = 4, LSHADE = 1)
      INTEGER    JCOLOR
      PARAMETER (JCOLOR = 2)
      INTEGER    NUMBLD(N14), NUMOPT, NUMPOS(N7)
      CHARACTER  SYMBOL(3)*4
      CHARACTER  ARROW*4, BLANK*4
      CHARACTER  OPTS(30)*100
      PARAMETER (ARROW = ' ** ', BLANK = '    ')
      LOGICAL    REPEET
      LOGICAL    BORDER, FLASH, HIGH
      PARAMETER (BORDER = .FALSE., FLASH = .FALSE., HIGH = .TRUE.)
      EXTERNAL   LBOX01
      DATA       NUMPOS / N7*1 /
      DATA       NUMBLD / N14*0 /
      REPEET = .TRUE.
      DO WHILE (REPEET)
         REPEET = .FALSE.
         IF (CALCS) THEN
            NDEC = N5
            SYMBOL(1) = BLANK
            SYMBOL(2) = BLANK
            SYMBOL(3) = BLANK
         ELSEIF (READY) THEN
            IF (OK) THEN
               IF (EXPERT .OR. WEIGHT) THEN
                  NDEC = N6
               ELSE
                  NDEC = N5
               ENDIF
               SYMBOL(1) = BLANK
               SYMBOL(2) = BLANK
               SYMBOL(3) = ARROW
            ELSE
               NDEC = 1
               SYMBOL(1) = ARROW
               SYMBOL(2) = BLANK
               SYMBOL(3) = BLANK
            ENDIF
         ELSE
            NDEC = N4
            SYMBOL(1) = BLANK
            SYMBOL(2) = ARROW
            SYMBOL(3) = BLANK
         ENDIF
         IF (EXPERT) THEN
            WRITE (OPTS,100) SYMBOL(1), SYMBOL(2), SYMBOL(3)
            NUMOPT = N7
            NSTART = N8
            NTEXT = NSTART + NUMOPT - N1
            NUMBLD(1) = N4
            CALL LBOX01 (ICOLOR, IX, IY, LSHADE, NUMBLD, NDEC, NUMOPT,
     +                   NUMPOS, NSTART, NTEXT,
     +                   OPTS,
     +                   BORDER, FLASH, HIGH)
            IF (CALCS .AND. NDEC.EQ.N6) NDEC = N5
            IF (NDEC.EQ.N5) THEN
               IF (WEIGHT) THEN
                  WRITE (OPTS,200)
                  NUMOPT = N5
               ELSE
                  WRITE (OPTS,300)
                  NUMOPT = N4
               ENDIF
               NDEC = NUMOPT
               NSTART = N1
               CALL LBOX01 (JCOLOR, IX, IY, LSHADE, NUMBLD, NDEC,
     +                      NUMOPT, NUMPOS, NSTART, NUMOPT,
     +                      OPTS,
     +                      BORDER, FLASH, HIGH)
               IF (NDEC.EQ.NUMOPT) THEN
                  REPEET = .TRUE.
               ELSE
                  CALCS = .FALSE.
               ENDIF
               NDEC = NDEC + N5
            ELSEIF (NDEC.EQ.N6) THEN
               NDEC = N5
            ELSEIF (NDEC.EQ.NUMOPT) THEN
               NDEC = N10
            ENDIF
         ELSE
            WRITE (OPTS,400) SYMBOL(1), SYMBOL(2), SYMBOL(3)
            NUMOPT = N6
            NSTART = N8
            NTEXT = NSTART + NUMOPT - N1
            NUMBLD(1) = N4
            CALL LBOX01 (ICOLOR, IX, IY, LSHADE, NUMBLD, NDEC, NUMOPT,
     +                   NUMPOS, NSTART, NTEXT,
     +                   OPTS,
     +                   BORDER, FLASH, HIGH)
            IF (NDEC.EQ.NUMOPT) NDEC = N10
         ENDIF
      ENDDO
C
C Format statements
C      
  100 FORMAT (
     + 'Options to check data before writing the output file'
     +/ 
     +/'You have now typed in all of your (x,y) data values so'
     +/'you can inspect the current data as a table or graph to'
     +/'identify any mistakes and make final corrections before'
     +/'creating the new data file.'
     +/
     +/'Display a table of current (x,y,s) data',A
     +/'Plot a graph using current (x,y) data'
     +/'Edit the current (x,y,s) data'
     +/'Put data into order of increasing x-values',A
     +/'Calculate new s values from replicates'
     +/'Save to file',A
     +/'Help')
  200 FORMAT (
     + 's = sample standard deviation'
     +/'s = constant (const. variance)'
     +/'s = cv%|y|/100 (const. rel. err.)'
     +/'s = s before editing/calculating'
     +/'Cancel')
  300 FORMAT (
     + 's = sample standard deviation'
     +/'s = constant (const. variance)'
     +/'s = cv%|y|/100 (const. rel. err.)'
     +/'Cancel')
  400 FORMAT (
     + 'Options to check data before writing the output file'
     +/ 
     +/'You have now typed in all of your (x,y) data values so'
     +/'you can inspect the current data as a table or graph to'
     +/'identify any mistakes and make final corrections before'
     +/'creating the new data file.'
     +/
     +/'Display a table of current (x,y) data',A
     +/'Plot a graph using current (x,y) data'
     +/'Edit the current (x,y) data'
     +/'Put data into order of increasing x-values',A
     +/'Save to file',A
     +/'Help')
      END
C
C-------------------------------------------------------------------
C
      SUBROUTINE SUB001 (NPTS,
     +                   S, X, Y,
     +                   CIPHER, 
     +                   EXPERT)
C
C Table
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,             INTENT (IN) :: NPTS
      DOUBLE PRECISION,    INTENT (IN) :: S(NPTS), X(NPTS), Y(NPTS)
      CHARACTER (LEN = *), INTENT (IN) :: CIPHER(2,NPTS)
      LOGICAL    EXPERT
C
C Locals
C      
      INTEGER    COLOUR
      PARAMETER (COLOUR = 15)
      INTEGER    ICOLOR, JCOLOR
      PARAMETER (ICOLOR = 0, JCOLOR = 4)
      INTEGER    I, J
      CHARACTER (LEN = 100) LINE
      CHARACTER (LEN = 15 ) SHOW15, WORD15X, WORD15Y, WORD15S 
      EXTERNAL   TABLE1, SHOW15
      CALL TABLE1 (COLOUR, 'OPEN')
      IF (.NOT.EXPERT) THEN
         WRITE (LINE,100)
         CALL TABLE1 (JCOLOR, LINE)
         DO I = 1, NPTS
            WORD15X = SHOW15(X(I))
            WORD15Y = SHOW15(Y(I))
            WRITE (LINE,200) I, WORD15X, WORD15Y,
     +                      (CIPHER(J,I),J = 1, 2)
            CALL TABLE1 (ICOLOR, LINE)
         ENDDO
      ELSE
         WRITE (LINE,300)
         CALL TABLE1 (JCOLOR, LINE)
         DO I = 1, NPTS
            WORD15X = SHOW15(X(I))
            WORD15Y = SHOW15(Y(I))
            WORD15S = SHOW15(S(I))
            WRITE (LINE,400) I, WORD15X, WORD15Y, WORD15S,
     +                      (CIPHER(J,I),J = 1, 2)
            CALL TABLE1 (ICOLOR, LINE)
         ENDDO
      ENDIF
      CALL TABLE1 (COLOUR, 'CLOSE')
C
C Format statements
C      
  100 FORMAT (1X,'Line number',15X,'x',18X,'y')
  200 FORMAT (1X,I6,6X,2(A15,4X),A6,2X,A6)
  300 FORMAT (1X,'Line number',15X,'x',18X,'y',18X,'s')
  400 FORMAT (1X,I6,6X,3(A15,4X),A6,2X,A6)
      END
C
C----------------------------------------------------------------
C
      SUBROUTINE SUB002 (NPTS, 
     +                   X, Y)
C
C Graph
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: NPTS
      DOUBLE PRECISION, INTENT (IN) :: X(NPTS), Y(NPTS)
C
C Locals
C      
      INTEGER    L, M
      PARAMETER (L = 1, M = 1)
      CHARACTER  PTITLE*16, XTITLE*1, YTITLE*1
      PARAMETER (PTITLE = 'Current x,y data',
     +           XTITLE = 'X',
     +           YTITLE = 'Y')
      EXTERNAL   GKS001
      IF (NPTS.LT.2) RETURN
      CALL GKS001 (L, M, NPTS, 
     +             X, Y, 
     +             PTITLE, XTITLE, YTITLE)
      END
C
C------------------------------------------------------------
C
      SUBROUTINE SUB003 (NMAX, NPTS,
     +                   S, TEMP, X, Y,
     +                   FNAME,
     +                   CALL_EDITOR, XYONLY)
C
C Change a line ... can now use W_EDITOR if CALL_EDITOR = .TRUE.
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,             INTENT (IN)    ::  NMAX, NPTS
      DOUBLE PRECISION,    INTENT (INOUT) :: S(NMAX), TEMP(NMAX,3),
     +                                       X(NMAX), Y(NMAX)
      CHARACTER (LEN = *), INTENT (IN)    :: FNAME
      LOGICAL,             INTENT (IN)    :: CALL_EDITOR, XYONLY
C
C Locals
C      
      INTEGER    I, J, NALTER, NCOLS
      INTEGER    ISEND, ITYPE, N1, N2
      PARAMETER (ISEND = 2, ITYPE = 1, N1 = 1, N2 = 2)
      DOUBLE PRECISION STEMP(1), XTEMP, YTEMP
      DOUBLE PRECISION ONE, XSMALL
      PARAMETER (ONE = 1.0D+00, XSMALL = 1.0D-150)
      CHARACTER  LINE*100, WORD60*60, TRIM60*60
      LOGICAL    CURVE, FIXCOL, FIXROW, LABEL, ORDER, REPEET, WEIGHT
      PARAMETER (CURVE = .TRUE., FIXCOL = .TRUE., FIXROW = .FALSE.,
     +           LABEL = .TRUE., ORDER = .TRUE.)
      EXTERNAL   GETJM1, GETD02, GETD03, TESTER, PUTFAT, EDITOR, TRIM60
      SAVE       NALTER
      DATA       NALTER / 1 /
      IF (NALTER.GT.NPTS) NALTER = 1
      IF (CALL_EDITOR) THEN
C
C Copy the current data into TEMP then call the editor
C
         DO I = 1, NPTS
            TEMP(I,1) = X(I)
            TEMP(I,2) = Y(I)
            IF (XYONLY) THEN
               TEMP(I,3) = ONE
            ELSE  
               TEMP(I,3) = S(I)
            ENDIF   
         ENDDO
         IF (XYONLY) THEN
            NCOLS = 2
            WEIGHT = .FALSE.
         ELSE
            NCOLS = 3
            WEIGHT = .TRUE.
         ENDIF
         WORD60 = TRIM60(FNAME)
         CALL EDITOR (ISEND, ITYPE, NCOLS, NMAX, NPTS, 
     +                TEMP,
     +                WORD60, CURVE, FIXCOL, FIXROW, LABEL, ORDER,
     +                WEIGHT)
         DO I = 1, NPTS
            X(I) = TEMP(I,1)
            Y(I) = TEMP(I,2)
            IF (.NOT.XYONLY) S(I) = TEMP(I,3)
         ENDDO
      ELSE
C
C Select the line
C
         WRITE (LINE,100)
         CALL GETJM1 (N1, NALTER, NPTS, 
     +                LINE)
         I = NALTER
         STEMP(1) = S(I)
         XTEMP = X(I)
         YTEMP = Y(I)
C
C Read in new values
C
         REPEET = .TRUE.
         DO WHILE (REPEET)
            REPEET = .FALSE.
            IF (XYONLY) THEN
               WRITE (LINE,200) X(I), Y(I)
               CALL GETD02 (XTEMP, YTEMP, LINE)
            ELSE
               WRITE (LINE,300) X(I), Y(I), S(I)
               CALL GETD03 (XTEMP, YTEMP, STEMP(1), LINE)
               J = 1
               CALL TESTER (STEMP, I, J, XSMALL)
            ENDIF
C
C Check new values
C
            IF (I.EQ.N1) THEN
               IF (XTEMP.GT.X(N2)) THEN
                  WRITE (LINE,400) X(N2)
                  CALL PUTFAT (LINE)
                  REPEET = .TRUE.
               ENDIF
            ELSEIF (I.EQ.NPTS) THEN
               IF (XTEMP.LT.X(NPTS - N1)) THEN
                  WRITE (LINE,500) X(NPTS - N1)
                  CALL PUTFAT (LINE)
                  REPEET = .TRUE.
               ENDIF
            ELSE
               IF (XTEMP.GT.X(I + N1)) THEN
                  WRITE (LINE,400) X(I + N1)
                  CALL PUTFAT (LINE)
                  REPEET = .TRUE.
               ELSEIF(XTEMP.LT.X(I - N1)) THEN
                  WRITE (LINE,500) X(I - N1)
                  CALL PUTFAT (LINE)
                  REPEET = .TRUE.
               ENDIF
            ENDIF
C
C Assign final corrections
C
            IF (.NOT.REPEET) THEN
               S(I) = STEMP(1)
               X(I) = XTEMP
               Y(I) = YTEMP
            ENDIF   
         ENDDO
      ENDIF
C
C Format statements
C      
  100 FORMAT ('Number of the line of data to be corrected')
  200 FORMAT ('The new x,y values  (...current values =',
     +1P,E15.7,',',E15.7,')')
  300 FORMAT ('The new x,y,s values  (...current values =',
     +1P,E15.7,',',E15.7,',',E15.7,')')
  400 FORMAT ('New x-value must be =<',1P,E15.7,' ... Try again')
  500 FORMAT ('New x-value must be >=',1P,E15.7,' ... Try again')
      END
C
C-------------------------------------------------------------------
C
      SUBROUTINE SUB004 (NPTS, 
     +                   ORDER, S, X, Y,
     +                   READY)
C
C Order data
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)    :: NPTS
      DOUBLE PRECISION, INTENT (INOUT) :: ORDER(NPTS), S(NPTS), X(NPTS),
     +                                    Y(NPTS)
      LOGICAL,          INTENT (IN)    :: READY
C
C Locals
C      
      INTEGER    I
      INTEGER    ICOLOR, NUMHDR
      PARAMETER (ICOLOR = 3, NUMHDR = 7)
      INTEGER    NUMBLD(NUMHDR)
      CHARACTER  HEADER(NUMHDR)*80
      CHARACTER  LINE*40
      PARAMETER (LINE = 'Proceed to rearrange data (usually no) ?')
      LOGICAL    YES
      EXTERNAL   HPSORT, ORDERS, ANSWER
      INTRINSIC  DBLE
      DATA       NUMBLD / 2*1, 5*0 /
      IF (NPTS.LT.2) RETURN
      IF (READY) THEN
         RETURN
      ELSE
         WRITE (HEADER,100)
         YES = .FALSE.
         CALL ANSWER (ICOLOR, NUMBLD, NUMHDR,
     +                HEADER, LINE,
     +                YES)
         IF (.NOT.YES) RETURN
      ENDIF
      DO I = 1, NPTS
         ORDER(I) = DBLE(I)
      ENDDO
      CALL HPSORT (NPTS,
     +             X, Y, S, ORDER)
      CALL ORDERS (NPTS,
     +             X, Y, S, ORDER)
C
C Format statement
C     
  100 FORMAT (
     + 'Warning `The data could be out of order due to a mistake'
     +/'        `while editing.'
     +/'        `For instance you might have inserted a new line'
     +/'        `in a wrong position or input an incorrect power'
     +/'        `of ten for an x-value. You must check the table'
     +/'        `and graph for this before rearranging the whole'
     +/'        `data set.')

      END
C
C-------------------------------------------------------------------
C
      SUBROUTINE SUB005 (NOUT, NPTS,
     +                   S, X, Y,
     +                   FNAME, TITLE, 
     +                   READY, XTEXT, YTEXT)
C
C Write output file
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,             INTENT (IN) :: NOUT, NPTS
      DOUBLE PRECISION,    INTENT (IN) :: S(NPTS), X(NPTS), Y(NPTS)
      CHARACTER (LEN = *), INTENT (IN) :: FNAME, TITLE
      LOGICAL,             INTENT (IN) :: READY, XTEXT, YTEXT
C
C Local allocatable array
C      
      DOUBLE PRECISION, ALLOCATABLE :: A(:,:) 
C
C Locals
C      
      INTEGER    ISEND, NHIGH, NWIDE, N3
      PARAMETER (ISEND = 3, NHIGH = 1, NWIDE = 80, N3 = 3)
      INTEGER    JSEND, NMAX, NPAR
      PARAMETER (JSEND = 4, NMAX = 20)
      INTEGER    I, IERR, IOS, NRMAX
      DOUBLE PRECISION PLOW(NMAX), PMID(NMAX), PHIGH(NMAX)
      CHARACTER  TEXT(NHIGH)*(NWIDE), WORD32*32
      LOGICAL    ABORT
      LOGICAL    FALSE, TRUE
      PARAMETER (FALSE = .FALSE., TRUE = .TRUE.)
      LOGICAL    FILEIT, SUPPLY
      PARAMETER (FILEIT = .TRUE., SUPPLY = .FALSE.)
      EXTERNAL   PUTFAT, MATOUT, YMDHMS, EDITPL, MFTIDY
      SAVE       TEXT
      DATA       TEXT / NHIGH*' ' /
      IF (.NOT.READY) THEN
         CALL PUTFAT ('Put data into order before writing file')
      ELSE
         IERR = 0
         IF (ALLOCATED(A)) DEALLOCATE(A, STAT = IERR)
         IF (IERR.NE.0) RETURN
         NRMAX = NPTS  
         ALLOCATE (A(NRMAX,N3), STAT = IERR)
         IF (IERR.NE.0) RETURN    
         DO I = 1, NPTS
            A(I,1) = X(I)
            A(I,2) = Y(I)
            A(I,3) = S(I)
         ENDDO     
         CALL YMDHMS (WORD32)
         TEXT(1) = WORD32
         CLOSE (UNIT = NOUT)
         OPEN (UNIT = NOUT, FILE = FNAME, IOSTAT = IOS)
         CALL MATOUT (ISEND, N3, NOUT, NRMAX, NPTS, NHIGH,
     +                A,
     +                FNAME, TEXT, TITLE,
     +                ABORT, TRUE, XTEXT, FALSE)         
         DEALLOCATE (A, STAT = IERR)
         NPAR = 0
         IF (YTEXT) THEN
            CALL EDITPL (JSEND, NPAR, NMAX, NOUT,
     +                   PLOW, PMID, PHIGH,
     +                   FILEIT, SUPPLY)  
            
         ENDIF           
         CLOSE (UNIT = NOUT)
         IF (YTEXT) CALL MFTIDY (FNAME)
      ENDIF  
      END
C
C------------------------------------------------------------------
C
      SUBROUTINE SUB006 (NPTS,
     +                   ORDER, S, X, Y,
     +                   READY)
C
C Order data then replace S by sample estimates
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN   ) :: NPTS
      DOUBLE PRECISION, INTENT (INOUT) :: ORDER(NPTS), S(NPTS), X(NPTS),
     +                                    Y(NPTS)
      LOGICAL,          INTENT (IN)    :: READY
C
C Locals
C      
      INTEGER    I
      INTEGER    ICOLOR, NUMHDR
      PARAMETER (ICOLOR = 3, NUMHDR = 7)
      INTEGER    NUMBLD(NUMHDR)
      CHARACTER  HEADER(NUMHDR)*100
      CHARACTER  LINE*22
      PARAMETER (LINE = 'Proceed to rearrange ?')
      LOGICAL    YES
      EXTERNAL   HPSORT, ORDERS, SIGMAS, ANSWER
      DATA       NUMBLD / 2*1, 5*0 /
      IF (NPTS.LT.2) RETURN
      IF (.NOT.READY) THEN
         WRITE (HEADER,100)
         YES = .FALSE.
         CALL ANSWER (ICOLOR, NUMBLD, NUMHDR,
     +                HEADER, LINE,
     +                YES)
         IF (.NOT.YES) RETURN
         DO I = 1, NPTS
            ORDER(I) = I
         ENDDO
         CALL HPSORT (NPTS,
     +                X, Y, S, ORDER)
         CALL ORDERS (NPTS,
     +                X, Y, S, ORDER)
      ENDIF
      CALL SIGMAS (NPTS,
     +             X, Y, S)
C
C Format statement
C     
  100 FORMAT (
     + 'Warning `The data could be out of order due to a mistake'
     +/'        `while editing.'
     +/'        `For instance you might have inserted a new line'
     +/'        `in a wrong position or input an incorrect power'
     +/'        `of ten for an x-value. You must check the table'
     +/'        `and graph for this before rearranging the whole'
     +/'        `data set.')
      END
C
C-----------------------------------------------------------------
C
      SUBROUTINE SUB007 (NPTS,
     +                   S)
C
C Set S = a constant
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: NPTS
      DOUBLE PRECISION, INTENT (OUT) :: S(NPTS)
C
C Locals
C      
      INTEGER    I
      INTEGER    ICOLOR, IX, IY, LSHADE, NUMTXT
      PARAMETER (ICOLOR = 0, IX = 10, IY = 4, LSHADE = 2, NUMTXT = 17)
      INTEGER    NUMBLD(NUMTXT)
      DOUBLE PRECISION SMIN
      PARAMETER (SMIN = 1.0D-150)
      DOUBLE PRECISION CONST
      CHARACTER  TEXT(NUMTXT)*80
      LOGICAL    BORDER
      PARAMETER (BORDER = .FALSE.)
      EXTERNAL   GETDGE, PATCH1
      DATA       NUMBLD / 6*1, 11*0 /
      WRITE (TEXT,100)
      CALL PATCH1 (ICOLOR, IX, IY, LSHADE, NUMBLD, NUMTXT, TEXT,
     +             BORDER)
      CONST = 1.0D+00
      CALL GETDGE (CONST, SMIN, 'The constant value required')
      DO I = 1, NPTS
         S(I) = CONST
      ENDDO
C
C Format statement
C      
  100 FORMAT (
     + '   `This option is for when you think that constant variance'
     +/'   `is appropriate, i.e. standard deviation of y is constant'
     +/'   `(sigma) independent of the x- or y-values.'
     +/
     +/'   `You input your value for sigma and the program will then'
     +/'   `set all s-values = your sigma.'
     +/
     +/'   `There are three possibilities for sigma.'
     +/
     +/'a) `Sigma = your independent estimate for the standard'
     +/'   `deviation of y (obtained by sampling ?)'
     +/
     +/'b) `Sigma = an inspired guess for a standard deviation'
     +/'   `e.g. 7.5% of average absolute y-value'
     +/
     +/'c) `Sigma = 1 since you are not sure what to do or you'
     +/'   `wish to use unweighted regression analysis')
      END
C
C
      SUBROUTINE SUB008 (NPTS,
     +                   S, Y)
C
C Set S = CV% Y
C
      IMPLICIT   NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN) :: NPTS
      DOUBLE PRECISION, INTENT (IN)  :: Y(NPTS)
      DOUBLE PRECISION, INTENT (OUT) :: S(NPTS)
C
C Locals
C      
      INTEGER    I, ICOUNT
      INTEGER    ICOLOR, IX, IY, LSHADE, NUMTXT
      PARAMETER (ICOLOR = 1, IX = 10, IY = 4, LSHADE = 2, NUMTXT = 17)
      INTEGER    NUMBLD(NUMTXT)
      DOUBLE PRECISION FACTOR, PCMAX, PCMIN
      PARAMETER (FACTOR = 100.0D+00, PCMAX = 200.0D+00,
     +           PCMIN = 0.001D+00)
      DOUBLE PRECISION BOTLIM, TOPLIM, SMIN
      PARAMETER (BOTLIM = 1.0D+00, TOPLIM = 25.0D+00, SMIN = 1.0D-150)
      DOUBLE PRECISION FRACN, PCENT, STEMP
      CHARACTER  LINE*80, TEXT(NUMTXT)*80
      LOGICAL    BORDER
      PARAMETER (BORDER = .FALSE.)
      EXTERNAL   GETDM1, PUTCAU, PUTADV, PATCH1
      INTRINSIC  ABS
      DATA       NUMBLD / 3*1, 14*0 /
      WRITE (TEXT,100)
      CALL PATCH1 (ICOLOR, IX, IY, LSHADE, NUMBLD, NUMTXT, TEXT,
     +             BORDER)
      PCENT = 7.5D+00
      CALL GETDM1 (PCMIN, PCENT, PCMAX, 'Percent required for cv%')
      IF (PCENT.LT.BOTLIM) THEN
         CALL PUTCAU ('Value unrealistically small')
      ELSEIF (PCENT.GT.TOPLIM) THEN
         CALL PUTCAU ('Value unrealistically large')
      ENDIF
      FRACN = PCENT/FACTOR
      ICOUNT = 0
      DO I = 1, NPTS
         STEMP = FRACN*ABS(Y(I))
         IF (STEMP.LT.SMIN) THEN
            STEMP = BOTLIM
            ICOUNT = ICOUNT + 1
         ENDIF
         S(I) = STEMP
      ENDDO
      IF (ICOUNT.GT.0) THEN
         WRITE (LINE,200) ICOUNT
         CALL PUTADV (LINE)
      ENDIF
C
C Format statements
C      
  100 FORMAT (
     + '   `This option is for when you think that constant relative'
     +/'   `error is appropriate, i.e. you think standard deviation'
     +/'   `of y is a constant proportion of absolute y-value.'/
     +/'   `You must now input a value for cv% which is an estimate'
     +/'   `of the coefficient of variation of true response y, i.e.'
     +/'   `you think standard deviation of y = (cv%/100)true|y|.'/
     +/'   `The program will then set all s = (cv%/100)actual|y|.'/
     +/'   `There are two possibilities for cv%.'/
     +/'a) `cv% = your independent estimate of the coefficient of'
     +/'   `variation of y (obtained by sampling ?)'/
     +/'b) `cv% = an inspired guess for the coefficient of'
     +/'   `variation, e.g. 5, 7.5, 10 etc.')
  200 FORMAT ('y too small ... No. times default s-value used =',I5)
      END
C
C---------------------------------------------------------------------
C
      SUBROUTINE SUB009 (NPTS,
     +                   S, STORE)
C
C Re-set s = default values or values typed in
C
      IMPLICIT NONE
C
C Arguments
C      
      INTEGER,          INTENT (IN)  :: NPTS
      DOUBLE PRECISION, INTENT (IN)  :: STORE(NPTS)
      DOUBLE PRECISION, INTENT (OUT) :: S(NPTS)
C
C Locals
C      
      INTEGER  I
      DO I = 1, NPTS
         S(I) = STORE(I)
      ENDDO
      END
C
C
