C
C FTN95 version using MODULE_QNFIT
C ================================
C
C Extra source code is required as follows:
C ----------------------------------------
C QNFIT.FOR  : MAIN
C QNFIT01.FOR: DATAIN, DECIDE
C QNFIT02.FOR: DERIV2, FUNCT1, FUNCT2, FUNCT3, GOFFIT, OUTDAT
C QNFIT03.FOR: PARAIN, PSCALE, QNCFIG, QNEDIT
C QNFIT04.FOR: QMODEL, QMODEX (i.e. similar to MAKDAT6.FOR), D02 routines:
C              D02FCN, D02JAC, D02FCN_USE, D02JAC_USE, D02SOL
C QNFIT05.FOR: RANDOM, SETSUP, TESTPS, VCOVAR, WEIGHT
C QNFIT06.FOR: ZMAREA, ZMCALI, ZMDERI, ZMEVAL, ZMFUNC
C QNFIT07.FOR: ZMCUTS, ZMSURF, ZMWSSQ
C QNFIT08.FOR: ZMPLOT, ZMDCON, ZMSTOR, ZMTYPE
C QNFIT09.FOR: MULT1A, MULT1B, MULT1C
C QNFIT10.FOR: ADVISE, QNDAT5, QNLGLS, QNPLOT, QNPCHK
C====================================================================
C
C====================================================================
C QNFIT: a quasi Newton nonlinear fitting program
C====================================================================
C
C Details of this version
C -----------------------
C This version of QNFIT uses QNFIT2/LBFGS_B and does not use the QNUSER
C routine from the original QNFIT7.FOR. It uses it from the MODELS DLL.
C The old QNFIT8.FOR is now the new QNFIT07.FOR
C In this version QNFIT04.FOR is similar to MAKDAT6.FOR 
C Precision in QNFIT2/LBFGS-B is controlled by ATYPE and OTYPE
C
C The module is mainly used to communicate between the following:
C ---------------------------------------------------------------
C FUNCT1
C QMODEL
C QMODEX
C
C The default ODE solver is DVODE which requires these parameters:
C ---------------------------------------------------------------
C IRELAB: 0 = mixed, 1 = absolute, 2 = relative (similar to D02EBF)
C METH: 1 = Adams, 2 = BDF
C MITER: 0 = no Jacobian, 1 = Analytic Jacobian, 2 = Estimated Jacobian
C
C Now include the extra source code
C =================================
C
C     INCLUDE 'qnfit01.for', NOLIST
C     INCLUDE 'qnfit02.for', NOLIST
C     INCLUDE 'qnfit03.for', NOLIST
C     INCLUDE 'qnfit04.for', NOLIST
C     INCLUDE 'qnfit05.for', NOLIST
C     INCLUDE 'qnfit06.for', NOLIST
C     INCLUDE 'qnfit07.for', NOLIST
C     INCLUDE 'qnfit08.for', NOLIST
C     INCLUDE 'qnfit09.for', NOLIST
C     INCLUDE 'qnfit10.for', NOLIST
C     INCLUDE 'dllchk.for', NOLIST
C

C
C-----------------------------------------------------------------------
C-----------------------------------------------------------------------
C Start of module for QNFIT
C This module holds all the arrays and parameters necessary to run QNFIT
C The module is chiefly used to communicate data to and from the models
C Note that XVAL and EQUAL must be dimensioned (NPTS + 1) for DVODE
C-----------------------------------------------------------------------
C-----------------------------------------------------------------------
C
      MODULE MODULE_QNFIT
C
C Dimensions: maximum array size parameters for normal mode:
C NP: data
C NX: parameters

C Dimensions: maximum array size parameters for multifunction mode
C NPMAX: parameters 
C NTMAX: data 
C NYMAX: number of equations 
      
      IMPLICIT NONE
C
C ----------------------------------------------
C Part 1: Fixed parameters and fixed size arrays
C ----------------------------------------------
C
      INTEGER    N2, N3, N4, N8, N10, N11, N16, N24
      PARAMETER (N2 = 2, N3 = 3, N4 = 4, N8 = 8, N10 = 10, N11 = 11,
     +           N16 = 16, N24 = 24)
      INTEGER    NITER, NPLOT
      PARAMETER (NITER = 99, NPLOT = 600)
      INTEGER    LIP
      PARAMETER (LIP = 10)
      INTEGER    IP(LIP)
      DOUBLE PRECISION XPLOT(NPLOT),
     +                 Y2(NPLOT),  Y4(NPLOT),  Y6(NPLOT),
     +                 Y8(NPLOT), Y10(NPLOT), Y12(NPLOT)
      CHARACTER (LEN = 80) NAMMOD(N24)
C
C ---------------
C Part 2: Scalars
C ---------------
C 
      INTEGER    LWTYPE, MODEL, NCALLS, NMOD, NPAR, NPTS, NVAR, NZEROS
      INTEGER    IRELAB, METH, MITER
      DOUBLE PRECISION DTOL, D02TOL, ENEG, EPOS, EPSI, RTOL, XTOL, ZTOL
      DOUBLE PRECISION ASYMP, RNDOF, TIME 
      CHARACTER (LEN = 1) RELABS
      LOGICAL    CONST, DEQN 
      LOGICAL    USE_D02CJF, USE_D02EJF, USE_JACOBIAN      
C
C --------------------------------------------------------------
C Part 3: Parameters and arrays for communicating in normal mode
C --------------------------------------------------------------
C
      INTEGER    NP
      PARAMETER (NP = 10000)
      INTEGER    NX, NHESS
      PARAMETER (NX = 100, NHESS = NX + 1)
C
C Integers dimensioned NX
C 
      INTEGER    INDEX(NX), ISTART(NX), ISTATE(NX), NBD(NX)
C
C Doubles dimensioned NP
C      
      DOUBLE PRECISION ERRSAV(NP), S(NP), T(NP), U(NP), V(NP), W(NP)
      DOUBLE PRECISION XSAV(NP + 1), TSAV(NP)
      DOUBLE PRECISION Y1(NP), Y3(NP),  Y5(NP),
     +                 Y7(NP), Y9(NP), Y11(NP)
      DOUBLE PRECISION ERROR(NP), FVAL(NP), XVAL(NP + 1), YVAL(NP),
     +                 ZVAL(NP), THEORY(NP)
C
C Doubles dimensioned NX
C     
      DOUBLE PRECISION A(NX), B(NX), FACT(NX)
      DOUBLE PRECISION BL(NX), BL1(NX), BL2(NX), BL3(NX), BU(NX),
     +                 BU1(NX), BU2(NX), BU3(NX), DIAGV(NX), EIGVAL(NX),
     +                 ERR(NX), G(NX), P(NX), PAR(NX), PX(NX), X(NX),
     +                 XSTORE(NX), X1(NX)
C
C Doubles dimensioned NHESS
C     
      DOUBLE PRECISION CORR(NHESS,NHESS), CV(NHESS,NHESS), 
     +                 HESSEX(NHESS,NHESS), HESSIN(NHESS,NHESS)     
C
C Doubles dimension NP,NX
C     
      DOUBLE PRECISION FJACC(NP,NX)
C
C Characters dimensioned NP
C      
      CHARACTER  LABELS(NP)*20
C
C Characters dimensioned NX
C      
      CHARACTER (LEN = 9) RECORD(NX,NX)
      CHARACTER (LEN = 5) SYMBOL(NX)
C
C Logicals dimensioned NP
C      
      LOGICAL    EQUAL(NP + 1), EQSAV(NP + 1) 
C
C Logicals dimensioned NX
C
      LOGICAL    FREE(NX)  
C
C -----------------
C Part 4: Workspace
C -----------------
C  
      INTEGER    LIW, LW1, LW2
      PARAMETER (LIW = N3*NX,
     +           LW1 = N2*(N2*N10*NX + N4*NX + N11*N10*N10 + N8*N10),
     +           LW2 = NX*(NX +7) + 1)
      INTEGER    IW(LIW) 
      DOUBLE PRECISION W1(LW1), W2(LW2)    
C
C -----------------------------------------------------
C Part 5: Data for communicating in multi-function mode
C -----------------------------------------------------
C
      INTEGER    NPMAX, NTMAX, NYMAX, IMW
      PARAMETER (NPMAX = 100, NTMAX = 2000, NYMAX = 10,
     +           IMW = NTMAX*NYMAX)
      INTEGER    KMAX_A, KMAX_F, KMAX_J, KMAX_Y
      PARAMETER (KMAX_A = NPMAX, KMAX_F = NYMAX, KMAX_J = NYMAX**2,
     +           KMAX_Y = NYMAX)
      INTEGER    IADDUP, INDEXM(NTMAX,NYMAX), NPTBIG, NUMEQN,
     +           NUMPAR, NUMPOS, NUMPNT(NYMAX), NUMVAR
      DOUBLE PRECISION AMULT(NPMAX), FMULT(NYMAX), SMULT(NTMAX,NYMAX),
     +                 STEMP(NTMAX), WMULT(IMW),
     +                 XMULT(NTMAX,NYMAX), XTEMP(NTMAX),
     +                 YMJACC(KMAX_J),
     +                 YMDE(NYMAX), YMULT(NTMAX,NYMAX), YTEMP(NTMAX)
      LOGICAL    MULTI1, M1DATA, M1MOD


      END MODULE MODULE_QNFIT  
C
C------------------------------------------------------------------------
C------------------------------------------------------------------------
C End of module for QNFIT
C------------------------------------------------------------------------
C------------------------------------------------------------------------
C   
   
C
C
      PROGRAM QNFIT
C
C PACKAGE : SIMFIT
C PROGRAM : QNFIT
C VERSION : details from SIMVER/DLLCHK
C FORTRAN : Double precision
C ACTION  : Curve fitting by the quasi-Newton routine QNFIT2/LBFGS-B
C           Model or family of models called from subroutine QMODEL.
C           NAG E04 and D02 routines can be used
C OUTPUT  : Sum of weighted squared residuals, parameters, standard
C           errors, eigenvalues of Hessian and matrix of correlations.
C           Best-fit parameters and curve fitting details are written
C           to a file along with residuals if required.
C           Also does AUC, WSSQ plot, z = f(x,y) plot (slices), derivative,
C           evaluation and calibration.
C ADVICE  : Use FACTORS to keep internal parameters and condition
C           no. of the Hessian of order unity at the solution point.
C           The covariance matrix is obtained by inverting the Hessian.
C           Model is set to - 1 until a consistent model is initialised
C ARRAYS  : Set parameters to dimension the arrays etc. as follows:-
C           NIN = unit for input
C           NOUT = unit for output
C           NFILE = unit for residuals
C           NP = maximum no. of data points
C           NX = maximum no. of parameters
C AUTHOR  : W. G. Bardsley, University of Manchester, U.K., 08/06/1987
C REVISED : 26/08/1988 Hessian from Jacobian, program flow, graphics
C           11/10/1988 Replace E04JBE by E04JAF and E04HBE by ZMGRAD
C           21/09/1989 Add run test and improve subroutine GOFFIT.
C           05/10/1989 Replace ZMGRAD by QNGRAD and use parameters NF,
C                      NP, NX, NHESS, NUMF and NUMX for array dimensions.
C           24/01/1990 INVERT, OFILES, RANDOM, ELIMINATE FACTIN
C           19/12/1991 DEQN, DATTIN, EXPERT, STATS, DATTCHK, MODEL, NVAR,
C                      RESGKS
C           06/02/1992 GETNUM, GETCHR
C           13/02/1992 CHECKT
C           10/03/1992 IWARNU, VEC5IN, GKSR02
C           15/05/1992 Changed PARAIN menu, defined ZTOL, RTOL in SETSUP
C           22/05/1992 GKST02 and ASYMP
C           22/10/1992 DECIDE: Code when MODEL<1 and JSEND=NFREE for NDOF
C           05/02/1993 GET???, PUT??? and compressed
C           19/04/1993 Added NFIX
C           19/06/1993 RESFIL
C           23/06/1993 Removed array TEMP,added /IWK/ and /RWK/ for models
C           25/06/1993 Added t values to parameter output table
C           16/11/1993 Added QNDATA to fit subsets of data using S,T,U,V,W
C           28/11/1993 Added call to DIVIDE
C           09/01/1994 Version for use with models in dynamic link library
C                      DBOS version ... 15/4/94
C           07/10/1994 Added functions of three variables
C           20/12/1994 Added differential equations
C           13/02/1995 Added XYLIMS, calibration, areas etc.
C           20/02/1995 Version for Salamanca
C           11/07/1995 Added NZMOD,QNUSER so consistent with deqsol/makdat
C           13/02/1996 WSSQ(p(i),p(j)) ZMWSSQ and surface slices ZMCUTS
C           22/05/1996 Added option to calculate s^2 = A + B*y^C
C           07/10/1997 WIN32 version ... reorganised and added QNCFIG
C           15/02/1998 Added FNAME1 to call to GOFFIT for ZMCUTS and opened
C                      w_simfit.err for error piping to unit 6 from dvode
C           23/03/1998 Added QNEDIT to allow in-line data editing
C           07/08/1998 added dllchk
C           06/10/1998 removed TEXT
C           20/11/1998 added DEFGKS$
C           22/11/1098 extrapolation, deconvolution and error bars in
C                      ZMPLOT/ZMDCON
C           14/12/1998 replaced TUTORS by TUTOR1
C           03/02/1999 added LWTYPE for reduced major and major axis regression
C           03/03/1999 added multifunction mode: n functions of 1 variable
C           24/06/1999 corrected errors associated with reading in a
C                      parameter file in PARAIN and redefining YMULT so
C                      leading to incorect graphics in multifunction mode
C           24/08/1999 added code to call PARLIM
C           13/09/1999 added call to WINDOW
C           05/12/1999 increased dimension LW1
C           25/01/2000 changed DERIV1 to DERIV2
C           14/02/2000 added SIMVER
C           05/04/2001 revised
C           16/05/2001 added PCVTST
C           15/01/2002 added BL2 and BU2 and moved ADVISE into QNFITB.INS
C           20/03/2002 opened/closed file = 'iterate.dat' on unit = 8 (now called iterate.txt')
C           14/02/2003 altered test for MODE = 2/3 before call to PCVTST
C           10/05/2004 added LABELS in call to DATAIN
C           01/08/2005 increased DVER to *30 and added to call to ADVISE
C           24/01/2007 added FULL_PATH and SIM256
C           26/07/2007 added WORD32 and call to YMDHMS
C           16/11/2009 replaced common blocks by module_qnfit 
C           17/12/2009 added CHKPAR and QNPCHK to monitor parameters 
C           08/03/2017 added NITER to write iteration data to iterate.dat then introduced 
C                      subroutine QNITER to view the iteration files
C           14/08/2018 replaced E04JYF by E04UFF as default NAG routine 
C           11/05/2022 added E_NUMBERS and E_FORMATS 
C

C
C LINK MODULE_QNFIT to initialise all the MODULE variables in the MAIN program
C
      USE MODULE_QNFIT

      IMPLICIT   NONE
C
C Local variables and parameters
C      
      INTEGER    NFILE, NIN, NOUT
      PARAMETER (NFILE = 10, NIN = 3, NOUT = 4)
      INTEGER    NCHECK, NGRAF, NZMOD
      PARAMETER (NCHECK = 10, NGRAF = 320, NZMOD = 24)
      INTEGER    NSMALL, N0
      PARAMETER (NSMALL = 49, N0 = 0)
      INTEGER    I, IFAIL, ISEND, J, JSEND, NDEC, NDOF, NFIX, NFREE,
     +           NPAR1, NPTSAV, NZSAV
      INTEGER    MODE, MTYPE
      INTEGER    MARK
      DOUBLE PRECISION AWT, BWT, CWT, EPSX, RNMOD, SIGMA, WSSQ, XMAX,
     +                 XMIN, YMAX, YMIN
      DOUBLE PRECISION XVER, YVER
      DOUBLE PRECISION ZERO, ONE, TEN
      PARAMETER (ZERO = 0.0D+00, ONE = 1.0D+00, TEN = 100.0D+00)  
      CHARACTER (LEN = 1024) FULL_PATH, SIM256
      CHARACTER (LEN = 1024) ITDAT(10)
      CHARACTER  FSMALL(NSMALL)*1024, TSMALL(NSMALL)*80
      CHARACTER  FNAME1*1024, FNAME2*1024, MODNAM(NZMOD)*80, TITLE*80
      CHARACTER  OTYPE*9, WORD32*32
      CHARACTER  DVER*30, PVER*15
      PARAMETER (PVER = 'w_qnfit.exe')
      CHARACTER  ATYPE*11, BLANK*1, PNAME*5
      PARAMETER (ATYPE = 'approximate', BLANK = ' ', PNAME = 'QNFIT')
      LOGICAL    ABORT, ACTION, EXPERT, FIRST, OLDMOD, SHOW, STATS,
     +           SUPPLY
      LOGICAL    CHKPAR(NCHECK)
      LOGICAL    FILEIT, OK, SHOWIT, USE_E04JYF, USE_E04KZF, USE_E04UFF
      LOGICAL    QNLGLS 
      LOGICAL    SPLASH
      PARAMETER (SPLASH = .FALSE.)
C
C External procvedures
C
      EXTERNAL  G05CCF$
      EXTERNAL  DEFGKS
      EXTERNAL  QNFIT2, DERIV2, CHECKW, QNLGLS, QNPCHK
      EXTERNAL  MULT1B, MULT1C
      EXTERNAL  ADVISE, SETSUP, DATAIN, DECIDE, PARAIN, RANDOM, PSCALE,
     +          FUNCT1, OUTDAT, STOPGO, ZMAREA, ZMCALI, ZMDERI, ZMEVAL,
     +          ZMWSSQ, WEIGHT, XYLIMS, ZMPLOT, ZMSTOR, ZMTYPE
      EXTERNAL  REVPRO, VIEWER, PUTADV, PCVTST, SIM256, YMDHMS
      EXTERNAL  DLLCHK, WINDOW, SIMVER
      EXTERNAL  FUNCT2, E04QN1, DLLNAG, QNCFIG, E04MN1, E04SQ1, FUNCT3
      EXTERNAL  QNITER
C
C------------------------------------------------------
C The logical array CHKPAR is used during program
C development to follow the path of limits, scaling
C factors and parameters between internal and external
C coordinates. These checks should be switched off
C when program development is completed.
C 
C Initialise CHKPAR as follows:
C
C 1: before call to PARAIN
C 2:  after call to PARAIN
C 3: before call to RANDOM
C 4:  after call to RANDOM
C 5: before call to PSCALE
C 6:  after call to PSCALE
C 7: before call to fitting routine
C 8:  after call to fitting routine
C 9: before calculations 
C
      DO I = 1, NCHECK
         CHKPAR(I) = .FALSE.
      ENDDO   
C------------------------------------------------------

C
C======================================================================
C Open an inactive background window and then check the DLLs
C The following values must be edited at each release:
C XVER = version number
C YVER = release number
C DVER = release date
C These must be consistent with the same values in the SIMFIT DLLs
C
      
      ISEND = 1
      ACTION = .TRUE.
      TITLE = 'Simfit: program '// PNAME
      CALL WINDOW (ISEND,
     +             TITLE,
     +             ACTION)
      CALL SIMVER (XVER, YVER, 
     +             DVER)
      ABORT = .FALSE.
      SHOW = .FALSE.
      CALL DLLCHK (XVER, YVER,
     +             DVER, PVER, 
     +             ABORT, SHOW)
C
C Checking completed so now proceed to the main program
C======================================================================
C

C
C Advise the user
C
      FIRST = .TRUE.
      CALL ADVISE (NP,
     +             DVER,
     +             ABORT, FIRST)
      IF (ABORT) GOTO 70
C
C Initialise random number generator then the other control parameters
C
      CALL G05CCF$
      CALL DEFGKS
      ACTION = QNLGLS(N0)
      
      CALL SETSUP (IRELAB, METH, MITER,
     +             DTOL, D02TOL, ENEG, EPOS, EPSI, RTOL, XTOL, ZTOL,
     +             OTYPE, RELABS,
     +             SPLASH)
 
C
C Check for NAG
C    
      CALL DLLNAG (MARK,
     +             OK)
      IF (MARK.GE.20 .AND. OK) THEN
         USE_D02CJF = .FALSE.
         USE_D02EJF = .TRUE.
         USE_E04JYF = .FALSE.
         USE_E04KZF = .FALSE.
         USE_E04UFF = .TRUE.
         USE_JACOBIAN = .TRUE.
         CALL QNCFIG (IRELAB, METH, MITER,
     +                DTOL, D02TOL, 
     +                OTYPE, RELABS,
     +                USE_D02CJF, USE_D02EJF,
     +                USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +                USE_JACOBIAN)
      ELSE
         USE_D02CJF = .FALSE.
         USE_D02EJF = .FALSE.
         USE_E04JYF = .FALSE.
         USE_E04KZF = .FALSE.  
         USE_E04UFF = .FALSE.
         USE_JACOBIAN = .TRUE. 
      ENDIF
C
C Initialise scalars
C      
      LWTYPE = 1
      MODEL = 0
      NFIX = 0
      NMOD = 0
      NZEROS = 0
      ASYMP = - 1.0D+00
      AWT = 1.0D-10
      BWT = 0.01D+00
      CWT = 2.0D+00
      TIME = ZERO
      FNAME1 = BLANK
      FNAME2 = BLANK
      CONST = .FALSE.
      M1DATA = .FALSE.
      M1MOD = .FALSE.
      MULTI1 = .FALSE.
      SUPPLY = .FALSE.
C
C Initialise the work-space arrays
C
      DO I = 1, LIP
         IP(I) = 1
      ENDDO
      DO I = 1, NX
         BL(I) = -TEN
         BL1(I) = -TEN
         BL2(I) = -TEN
         BL3(I) = -TEN
         BU(I) = TEN
         BU1(I) = TEN
         BU2(I) = TEN
         BU3(I) = TEN
         FACT(I) = ONE
         X(I) = ONE
         X1(I) = ONE
         PX(I) = ONE
         DO J = 1, NX
            RECORD(J,I) = BLANK
         ENDDO
      ENDDO
      DO I = 1, LIW
         IW(I) = 0
      ENDDO
      DO I = 1, LW1
         W1(I) = ZERO
      ENDDO
      DO I = 1, LW2
         W2(I) = ZERO
      ENDDO         
      DO I = 1, NZMOD
         MODNAM(I) = BLANK
         NAMMOD(I) = BLANK
      ENDDO
      DO I = 1, NP
         ERROR(I) = ONE
         FVAL(I) = ZERO
         THEORY(I) = ZERO
         XVAL(I) = ZERO
         YVAL(I) = ZERO
         ZVAL(I) = ZERO
         EQUAL(I) = .FALSE.
      ENDDO
      XVAL(NP + 1) = ZERO
      EQUAL(NP + 1) = .FALSE.
      DO I = 1, NSMALL
         FSMALL(I) = BLANK
         TSMALL(I) = BLANK
      ENDDO
      DO I = 1, 10
        ITDAT(I) = BLANK
      ENDDO  
   
C
C Open error output to unit 6 from DVODE and LBFGSB
C                  
      CALL YMDHMS (WORD32)                       
      FULL_PATH = SIM256('w_qnfit.txt')
      OPEN (UNIT = 6, FILE = FULL_PATH)
      WRITE (6,'(A)')
     +'SIMFIT: Program QNFIT ... messages from DVODE/LBFGSB'
      WRITE (6,'(A)') WORD32
      ITDAT(1) = FULL_PATH
      
C
C Note: the iterations are now written to iterate.txt not iterate.dat 
C Open 'iterate.txt' on unit 8 to record iteration data from LBFGSB
C                                     
      FULL_PATH = SIM256('iterate.txt')
      OPEN (UNIT = 8, FILE = FULL_PATH)
      WRITE (8,'(A)')
     +'SIMFIT: program QNFIT ... iteration data from LBFGSB'
      WRITE (8,'(A)') WORD32
      ITDAT(2) = FULL_PATH
      
C
C LABEL 10: Read in new data file or make a selection from the old data set
C =========
C
   10 CONTINUE
C
C Set EPSX = RTOL to test for replicates in DATAIN
C
      EPSX = RTOL
      CALL DATAIN (IRELAB, METH, MITER, MODEL, MTYPE, NIN, NOUT, NP,
     +             NPAR1, NPTS, NVAR, NX, NZEROS,
     +             BL1, BU1, DTOL, D02TOL, EPSX, ERROR, FVAL, S, T, U,
     +             V, W, XVAL, X1, YVAL, ZVAL,
     +             FNAME1, FNAME2, LABELS, OTYPE, RELABS, TITLE,
     +             ABORT, DEQN, EQUAL, EXPERT, OLDMOD, SUPPLY,
     +             USE_D02CJF, USE_D02EJF, 
     +             USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +             USE_JACOBIAN)
      IF (ABORT) GOTO 60
      IF (MTYPE.LE.4) THEN
         M1DATA = .FALSE.
         M1MOD = .FALSE.
         MULTI1 = .FALSE.
       ENDIF
C
C Find the data range
C
      IF (NVAR.EQ.1 .OR. DEQN) THEN
         CALL XYLIMS (NPTS,
     +                XVAL, XMAX, XMIN, FVAL, YMAX, YMIN)
      ENDIF
      IF (OLDMOD) GOTO 30
C
C LABEL 20: Choose a new model for fitting
C =========
C
   20 CONTINUE
      IF (M1DATA) THEN
C
C If multi data is present try to read in an ASCII text model
C NOTE: the vectors ERROR, FVAL, XVAL, of length NPTS contain all the
C       data stored consecutively, i.e. NPTS = NPTBIG
C
         CALL MULT1B (ABORT)
         IF (ABORT) THEN
            GOTO 10
         ELSE
            MODEL = 17
            NMOD = 1
            NPAR = NUMPAR
            NVAR = NUMVAR
            DEQN = .FALSE.
            DO I = 1, NZMOD
               MODNAM(I) = NAMMOD(I)
            ENDDO
         ENDIF
      ELSE
C
C Otherwise a normal model
C
         ISEND = 1
         CALL DECIDE (IRELAB, ISEND, JSEND, METH, MITER, MODEL, NDOF,
     +                NFIX, NMOD, NPAR, NPTS, NVAR, NX,
     +                B, DTOL, D02TOL, RNDOF, RNMOD,
     +                MODNAM, OTYPE, RELABS,
     +                CONST, DEQN, STATS,
     +                USE_D02CJF, USE_D02EJF,
     +                USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +                USE_JACOBIAN)
C
C See if a special line type is required: set LWTYPE
C
         CALL ZMTYPE (LWTYPE, MODEL, NMOD, NPAR, NVAR,
     +                DEQN)
      ENDIF
C
C LABEL 30: New parameter start-values/limits, then random search and scaling
C =========
C
   30 CONTINUE
      IF (MODEL.LT.1) GOTO 10
      CALL QNPCHK (NPAR,
     +             BL, BU, FACT, PX, X,
     +             'Check 1: Before call to PARAIN',
     +             CHKPAR(1))       
      CALL PARAIN (ISTART, ISTATE, NFREE, NPAR, NPAR1, NPMAX, NSMALL,
     +             NVAR, NX,
     +             AMULT, BL, BL1, BL2, BU, BU1, BU2, EPSI, FACT, RTOL,
     +             PX, X, X1, FSMALL, MODNAM, TSMALL,
     +             ABORT, DEQN, EXPERT, M1DATA)
      CALL QNPCHK (NPAR,
     +             BL, BU, FACT, PX, X,
     +             'Check 2: After call to PARAIN',
     +             CHKPAR(2))
      IF (ABORT) GOTO 10
      ISEND = 2
      JSEND = NFREE
      CALL DECIDE (IRELAB, ISEND, JSEND, METH, MITER, MODEL, NDOF,
     +             NFIX, NMOD, NPAR, NPTS, NVAR, NX,
     +             B, DTOL, D02TOL, RNDOF, RNMOD,
     +             MODNAM, OTYPE, RELABS,
     +             CONST, DEQN, STATS, 
     +             USE_D02CJF, USE_D02EJF,
     +             USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +             USE_JACOBIAN)
      IF (QNLGLS(N16)) THEN
C
C Random search
C        
         CALL QNPCHK (NPAR,
     +                BL, BU, FACT, PX, X,
     +               'Check 3: Before call to RANDOM',
     +                CHKPAR(3))
         CALL RANDOM (ISTATE, NOUT, NPAR,
     +                BL, BU, PX, RNDOF, X,
     +                ABORT)
         CALL QNPCHK (NPAR,
     +                BL, BU, FACT, PX, X,
     +               'Check 4: After call to RANDOM',
     +                CHKPAR(4))
         IF (ABORT) GOTO 10
      ENDIF 
C
C At this point the parameters, scaling factors, and limits are in
C external coordinates so the call to PSCALE is necessary to at this
C point to map the values into internal coordinates i.e. X(i) = 1, etc.
C before proceeding to curve fitting
C      
      CALL QNPCHK (NPAR,
     +             BL, BU, FACT, PX, X,
     +            'Check 5: Before call to PSCALE',
     +             CHKPAR(5))    
      CALL PSCALE (ISTATE, NPAR,
     +             BL, BU, EPSI, FACT, PX, X)
      CALL QNPCHK (NPAR,
     +             BL, BU, FACT, PX, X,
     +            'Check 6: After call to PSCALE',
     +             CHKPAR(6))    
      ISEND = 3
      CALL DECIDE (IRELAB, ISEND, JSEND, METH, MITER, MODEL, NDOF,
     +             NFIX, NMOD, NPAR, NPTS, NVAR, NX,
     +             B, DTOL, D02TOL, RNDOF, RNMOD,
     +             MODNAM, OTYPE, RELABS,
     +             CONST, DEQN, STATS,
     +             USE_D02CJF, USE_D02EJF,
     +             USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +             USE_JACOBIAN)
      IF (JSEND.LT.1) GOTO 10
C
C LABEL 40: Curve fitting, goodness of fit then output the results
C =========
C
   40 CONTINUE
      CALL FUNCT1 (NPAR,
     +             X, SIGMA)
      WSSQ = RNDOF*SIGMA
      CALL CHECKW (NDOF,
     +             WSSQ)
      ISEND = 1
      CALL OUTDAT (IFAIL, INDEX, ISEND, ISTATE, MODEL, NDOF, NFILE,
     +             NFIX, NFREE, NGRAF, NHESS, NMOD, NOUT, NP, NPAR,
     +             NPTS, NUMPOS, NVAR, NX, NZEROS,
     +             BL, BU, CORR, CV, DIAGV, EIGVAL, EPSI, ERR, ERROR,
     +             FACT, FJACC, FVAL, G, HESSEX, HESSIN, P, PAR, PX,
     +             RNDOF, RTOL, S, SIGMA, T, THEORY, TIME, U, V, W,
     +             WSSQ, W2, X, XVAL, YVAL, ZVAL,
     +             FNAME1, FNAME2, MODNAM, RECORD, SYMBOL, TITLE,
     +             CONST, DEQN, EQSAV, EQUAL, FREE, MULTI1, STATS,
     +             USE_D02CJF, USE_D02EJF,
     +             USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +             USE_JACOBIAN)

C
C Initialise PAR and CV
C
      DO I = 1, NPAR
         PAR(I) = ZERO
         DO J = 1, NPAR
            IF (I.EQ.J) THEN
               CV(J,I) = - ONE
            ELSE
               CV(J,I) = ZERO
            ENDIF
         ENDDO
      ENDDO
C
C Set NBD to define types of limits required by QNFIT2/LBFGS
C
      DO I = 1, NPAR
         NBD(I) = 2
         BL3(I) = BL(I)
         BU3(I) = BU(I)
      ENDDO
      
C      
C===================================================================
C Start of code to call the optimisers
C All limits, scaling factors, etc. must now be in internal
C coordinates, e.g. X(i) = 1 after the call to PSCALE
C====================================================================
C
      FULL_PATH = SIM256('iterate.dat') 
      ITDAT(3) = FULL_PATH
      ACTION = .TRUE.
      I = 4
      CALL QNITER (I,
     +             ITDAT,
     +             ACTION)  
      CLOSE (UNIT = NITER)
      CALL YMDHMS (WORD32)  
      OPEN (UNIT = NITER, FILE = FULL_PATH)
      WRITE (NITER,'(A)')
     +'Parameters 1 to MIN(N,19) and function value from call to FUNCT1'
      WRITE (NITER,'(A)') WORD32
      CALL QNPCHK (NPAR,
     +             BL, BU, FACT, PX, X,
     +             'Check 7: Before call to fitting routine',
     +             CHKPAR(7))
      IF (USE_E04JYF) THEN
         FILEIT = .TRUE.
         SHOWIT = .TRUE.
         CALL E04QN1 (FUNCT2,
     +                IFAIL, NBD, IW, LIW, LW2, NPAR, NOUT,
     +                BL3, BU3, SIGMA, W1, W2, X,
     +                FILEIT, SHOWIT) 
         TIME = W2(1)
      ELSEIF (USE_E04KZF) THEN
         FILEIT = .TRUE.
         SHOWIT = .TRUE.
         CALL E04MN1 (FUNCT3,
     +                IFAIL, NBD, IW, LIW, LW2, NPAR, NOUT,
     +                BL3, BU3, SIGMA, G, W1, W2, X,  
     +                FILEIT, SHOWIT) 
         TIME = W2(1)   
      ELSEIF (USE_E04UFF) THEN
         FILEIT = .TRUE.
         SHOWIT = .TRUE.
         CALL E04SQ1 (FUNCT2, 
     +                IFAIL, NBD, IW, LIW, LW2, NPAR, NOUT,
     +                BL3, BU3, SIGMA, G, W1, W2, X,  
     +                FILEIT, SHOWIT) 
         TIME = W2(1)            
      ELSE
         CALL QNFIT2 (DERIV2, FUNCT1,
     +                IFAIL, IW, LIW, LW1, LW2, NPAR,
     +                NBD, NOUT, NPTS,
     +                BL3, BU3, SIGMA, G, W1, W2, X,
     +                ATYPE, OTYPE)
         TIME = W1(4)
      ENDIF
      CLOSE (UNIT = 99)
      CALL QNPCHK (NPAR,
     +             BL, BU, FACT, PX, X,
     +             'Check 8: After call to fitting routine',
     +             CHKPAR(8))
C      
C===================================================================
C End of code to call the optimisers
C Now X and FACT are at the values from fitting so we have to
C use P = FACT*X to evaluate the models until the next time
C PSCALE is called.
C====================================================================
C
      
      WSSQ = RNDOF*SIGMA
      CALL CHECKW (NDOF,
     +             WSSQ)
      ISEND = 2
      CALL OUTDAT (IFAIL, INDEX, ISEND, ISTATE, MODEL, NDOF, NFILE,
     +             NFIX, NFREE, NGRAF, NHESS, NMOD, NOUT, NP, NPAR,
     +             NPTS, NUMPOS, NVAR, NX, NZEROS,
     +             BL, BU, CORR, CV, DIAGV, EIGVAL, EPSI, ERR, ERROR,
     +             FACT, FJACC, FVAL, G, HESSEX, HESSIN, P, PAR, PX,
     +             RNDOF, RTOL, S, SIGMA, T, THEORY, TIME, U, V, W,
     +             WSSQ, W2, X, XVAL, YVAL, ZVAL,
     +             FNAME1, FNAME2, MODNAM, RECORD, SYMBOL, TITLE,
     +             CONST, DEQN, EQSAV, EQUAL, FREE, MULTI1, STATS,
     +             USE_D02CJF, USE_D02EJF,
     +             USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +             USE_JACOBIAN)
      ISEND = 4
      CALL DECIDE (IRELAB, ISEND, NDEC, METH, MITER, MODEL, NDOF,
     +             NFIX, NMOD, NPAR, NPTS, NVAR, NX,
     +             B, DTOL, D02TOL, RNDOF, RNMOD,
     +             MODNAM, OTYPE, RELABS,
     +             CONST, DEQN, STATS,
     +             USE_D02CJF, USE_D02EJF,
     +             USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +             USE_JACOBIAN)
      CALL OUTDAT (IFAIL, INDEX, NDEC, ISTATE, MODEL, NDOF, NFILE,
     +             NFIX, NFREE, NGRAF, NHESS, NMOD, NOUT, NP, NPAR,
     +             NPTS, NUMPOS, NVAR, NX, NZEROS,
     +             BL, BU, CORR, CV, DIAGV, EIGVAL, EPSI, ERR, ERROR,
     +             FACT, FJACC, FVAL, G, HESSEX, HESSIN, P, PAR, PX,
     +             RNDOF, RTOL, S, SIGMA, T, THEORY, TIME, U, V, W,
     +             WSSQ, W2, X, XVAL, YVAL, ZVAL,
     +             FNAME1, FNAME2, MODNAM, RECORD, SYMBOL, TITLE,
     +             CONST, DEQN, EQSAV, EQUAL, FREE, MULTI1, STATS,
     +             USE_D02CJF, USE_D02EJF,
     +             USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +             USE_JACOBIAN)
C
C LABEL 50: Choose next course of action
C =========
C
   50 CONTINUE
      ISEND = 3
C
C If SUPPLY = .TRUE. the next call restores ERROR and sets SUPPLY = .FALSE.
C
      CALL WEIGHT (ISEND, NOUT, NPTS,
     +             AWT, BWT, CWT, ERROR, ERRSAV, THEORY,
     +             SUPPLY)
      ISEND = 5
      CALL DECIDE (IRELAB, ISEND, NDEC, METH, MITER, MODEL, NDOF,
     +             NFIX, NMOD, NPAR, NPTS, NVAR, NX,
     +             B, DTOL, D02TOL, RNDOF, RNMOD,
     +             MODNAM, OTYPE, RELABS,
     +             CONST, DEQN, STATS,
     +             USE_D02CJF, USE_D02EJF,
     +             USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +             USE_JACOBIAN)
      IF (NDEC.GE.5 .AND. NDEC.LE.10) CALL QNPCHK (NPAR,
     +                                             BL, BU, FACT, PX, X,
     +                                   'Check 9: before calculation',
     +                                             CHKPAR(9))
      IF (NDEC.EQ.1) THEN
         GOTO 10
      ELSEIF (NDEC.EQ.2) THEN
         GOTO 20
      ELSEIF (NDEC.EQ.3) THEN
         GOTO 30
      ELSEIF (NDEC.EQ.4) THEN
         ISEND = 2
         JSEND = NFREE
         CALL DECIDE (IRELAB, ISEND, JSEND, METH, MITER, MODEL, NDOF,
     +                NFIX, NMOD, NPAR, NPTS, NVAR, NX,
     +                B, DTOL, D02TOL, RNDOF, RNMOD,
     +                MODNAM, OTYPE, RELABS,
     +                CONST, DEQN, STATS,
     +                USE_D02CJF, USE_D02EJF,
     +                USE_E04JYF, USE_E04KZF, USE_E04UFF,
     +                USE_JACOBIAN)
C
C The next call adjusts ERROR if required and sets the value for SUPPLY
C
         ISEND = 1
         CALL WEIGHT (ISEND, NOUT, NPTS,
     +                AWT, BWT, CWT, ERROR, ERRSAV, THEORY,
     +                SUPPLY)
         GOTO 40
      ELSEIF (NDEC.EQ.5) THEN
C
C Plot WSSQ/NDOF as a function of p(i), p(j)
C
         CALL ZMWSSQ (NPAR,
     +                FACT, X)
         GOTO 50
      ELSEIF (NDEC.EQ.6) THEN
C
C Area under the best fit curve
C
         IF (MULTI1) THEN
            CALL PUTADV ('Not available in multi-function mode')
         ELSEIF (NVAR.GT.1) THEN
            CALL PUTADV ('Not available in multi-variable mode')
         ELSE
            CALL ZMAREA (NPAR, NOUT, NP, NPTS, NX, NZEROS,
     +                   EPSI, X, THEORY, V, W, XMAX, XMIN, XVAL, YVAL,
     +                   DEQN, EQSAV, EQUAL)
         ENDIF
         GOTO 50
      ELSEIF (NDEC.GE.7 .AND. NDEC.LE.10) THEN
C
C Save EQUAL, NPTS, NZEROS and first points.
C Set NPTS = 1 for QMODEL (1 value) in prediction/calibration, etc.
C
         NPTSAV = NPTS
         DO I = 1, NPTSAV
            EQSAV(I) = EQUAL(I)
            EQUAL(I) = .FALSE.
            TSAV(I) = THEORY(I)
            XSAV(I) = XVAL(I)
         ENDDO
         NZSAV = NZEROS
         IF (NDEC.EQ.7) THEN
C
C Use best fit parameters for derivatives
C
            IF (MULTI1) THEN
               CALL PUTADV ('Not available in multi-function mode')
            ELSEIF (NVAR.GT.1) THEN
               CALL PUTADV ('Not available in multi-variable mode')
            ELSE
               NPTS = 1
               CALL ZMDERI (NGRAF, NIN, NOUT, NP, NPTS, NPAR, NZEROS,
     +                      EPSI, RTOL, THEORY, V, W, X, XMAX, XMIN,
     +                      XVAL, YMAX, YMIN,
     +                      DEQN, EQUAL)
            ENDIF
         ELSEIF (NDEC.EQ.8) THEN
C
C Use best fit curve for calibration
C
            IF (MULTI1) THEN
               CALL PUTADV ('Not available in multi-function mode')
            ELSEIF (NVAR.GT.1) THEN
               CALL PUTADV ('Not available in multi-variable mode')
            ELSE
               NPTS = 1
               CALL ZMCALI (NIN, NOUT, NPAR, NPTS, NP, NZEROS,
     +                      EPSI, THEORY, W, X, XMAX, XMIN, XVAL,
     +                      DEQN, EQUAL)
            ENDIF
         ELSEIF (NDEC.EQ.9) THEN
C
C Use best fit parameters for evaluation
C
            IF (MULTI1) THEN
               CALL PUTADV ('Not available in multi-function mode')
            ELSEIF (NVAR.GT.1) THEN
               CALL PUTADV ('Not available in multi-variable mode')
            ELSE
               NPTS = 1
               CALL ZMEVAL (NIN, NOUT, NP, NPTS, NPAR, NZEROS,
     +                      EPSI, THEORY, W, X, XMAX, XMIN, XVAL,
     +                      YMAX, YMIN,
     +                      DEQN, EQUAL)
            ENDIF
         ELSEIF (NDEC.EQ.10) THEN
C
C Extra plotting: extrapolation, deconvolution and error bars
C
            IF (NVAR.GT.1) THEN
               CALL PUTADV ('Not available in multi-variable mode')
            ELSEIF (MULTI1) THEN
               DO I = 1, NUMPAR
                  AMULT(I) = FACT(I)*X(I)
               ENDDO
               CALL MULT1C (NP, NPLOT,
     +                      XMAX, XMIN, XPLOT,
     +                      T, U, V, W, YVAL, ZVAL,
     +                      Y1, Y2, Y3, Y4, Y5, Y6, Y7, Y8, Y9, Y10,
     +                      Y11, Y12)
            ELSE
               CALL ZMPLOT (MODEL, NMOD, NP, NPAR, NPLOT, NPTS, NPTSAV,
     +                      ASYMP, FACT, FVAL, S, T, THEORY, U, V, X,
     +                      XMAX, XMIN, XSAV, XSTORE, XVAL, Y1, Y2, Y3,
     +                      Y4, Y5, Y6, Y7, Y8, Y9, Y10, Y11, Y12,
     +                      EQUAL)
            ENDIF
         ENDIF
C
C Restore EQUAL, NPTS, NZEROS and the first few points
C
         NPTS = NPTSAV
         DO I = 1, NPTS
            EQUAL(I) = EQSAV(I)
            THEORY(I) = TSAV(I)
            XVAL(I) = XSAV(I)
         ENDDO
         NZEROS = NZSAV
         GOTO 50
      ELSEIF (NDEC.EQ.11) THEN
C
C F test
C
         CALL ZMSTOR (NOUT, NPAR, NPTS,
     +                WSSQ)
         GOTO 50
      ELSEIF (NDEC.EQ.12) THEN
C
C Store/test parameters/covariances
C
         MODE = 2
         DO I = 1, NPAR
            IF (MODE.EQ.2) THEN
               IF (CV(I,I).LT.ZERO) MODE = 3
            ENDIF
         ENDDO
         CALL PCVTST (MODE, NOUT, NPAR, NPTS, NHESS, 
     +                CV, PAR)
         GOTO 50
      ELSEIF (NDEC.EQ.13) THEN
C
C Review progress so far
C
         CALL REVPRO (NOUT)
         GOTO 50
      ELSEIF (NDEC.EQ.14) THEN
C
C View current data
C
         IF (MULTI1) THEN
            CALL PUTADV ('Not available in multi-function mode')
         ELSE
            ISEND = 1
            CALL VIEWER (ISEND, 
     +                   FNAME1, BLANK, BLANK)
         ENDIF
         GOTO 50
      ELSEIF (NDEC.EQ.15) THEN
         ACTION = .FALSE. 
         I = 4
         CALL QNITER(I,
     +               ITDAT,
     +               ACTION)
         GOTO 50            
      ELSEIF (NDEC.EQ.16) THEN
C
C Exit
C
         FNAME1 = BLANK
      ENDIF
C
C LABEL 60: Another go ?
C =========
C
C About this branch point
C =======================
C Note that this next section will only result in a call to
C STOPGO if FNAME1 is not equal to BLANK. This depends on the code in
C DATAIN and the code after the return from DECIDE with the NDEC value
C
   60 CONTINUE
      IF (FNAME1.EQ.BLANK) THEN
C
C No data has yet been analysed so assume termination required
C
         GOTO 70
      ELSE
C
C At least one file has been accessed so warn before terminating
C
         CALL STOPGO (FNAME1, FNAME2, PNAME, 
     +                ABORT)
         IF (ABORT) THEN
            GOTO 70
         ELSE
            GOTO 10
         ENDIF
      ENDIF
C
C LABEL 70: Program termination
C =========
C
   70 CONTINUE
      CLOSE (UNIT = NOUT)
      CLOSE (UNIT = 6)
      CLOSE (UNIT = 8)

C
C======================================================================
C The program is finished so we can close down the background window
C
      ISEND = 1
      ACTION = .FALSE.
      CALL WINDOW (ISEND,
     +             TITLE,
     +             ACTION)
C
C======================================================================
C

      END
C
C
