c
c Program: MAIN (i.e. source code for RUN6.EXE = SIMFIT.EXE)
c Author : W.G.Bardsley, University of Manchester, U.K.
c Action : Program manager to run the Simfit package 
c Version: 8.0.2
c Date   : 29/10/2022
c
c ========================================================
c File RUN6.FOR: The SIMFIT program manager  (version 8)
c ========================================================
c This program can only be compiled using the Salford FTN95
c compiler with the /f_stdcall option. All subroutines and
c functions wth an @ character, e.g. rgb@(red, green, blue)
c are defined in salflibc.dll. Other code is included from
c run6_1.ins, run6_2.ins, and run6_3.ins.
c
c 1) Important information for programmers and translators
c    =====================================================
c This version is specially edited so that programmers and translators
c can develop foreign language versions more easily.  You can make a
c free source form (fortran 95) version to ease translation of text
c strings if it helps problems with fixed source form, but text strings
c much longer than those in the fixed source version should be avoided.
c Program for2f95.for is provided in the source code distribution which
c can be used to map simfit fixed source *.for into free format *.f95
c
c There is an important point to remember:-
c compile dllchk.for BEFORE compiling this program so that object code
c with the correct version identifiers is included in the .exe file.
c
c 1) Other files required in addition to RUN6.FOR
c    ============================================
c RUN6_1.FOR, RUN6_2.FOR, RUN6_3.FOR, RUN6_4.FOR, RUN6.HTM,
c ICO_RUN6.ICO and the link script RUN6.LNK
c DLLCHK.FOR
c
c 2) Sequence of commands required to create the executable x64_simfit.exe
c    =====================================================================
c    This is best done using the batch files in \simfit_x64\work
c    ftn95 /64 dllchk.for           ... makes dllchk.obj
c    ftn95 /64 run6.for             ... makes run6.obj
c    ftn95 /64 run6_1.for           ... makes run6_1.obj
c    ftn95 /64 run6_2.for           ... makes run6_2.obj
c    ftn95 /64 run6_3.for           ... makes run6_3.obj
c    ftn95 /64 run6_4.for           ... makes run6_4.obj
c    slink x64_run6.lnk             ... makes x64_run6.exe
c    copy run6.exe x64_simfit.exe   ... makes x64_simfit.exe
c
c Note that ix_shift and iy_shift are used to control the centralising
c of run6_logo
c
c Uncomment if dllchk, run6_1.for, run6_2.for, and run6_3.for are to be
c linked directly
c*****include 'dllchk.for'
c*****include 'run6_1.for'
c*****include 'run6_2.for'
c*****include 'run6_3.for'
c*****include 'run6_4.for'
c
c
      program main
c
c program: run6
c package: simfit
c action : program manager to run SIMFIT
c author : w.g.bardsley, university of manchester, uk
c          19/12/1998 ftn95 version developed from the ftn77 version
c          22/03/1999 introduced run6_logo
c          25/10/1999 added website to logo and deleted call to mssage
c                     with k = 0 and k = 2 when nval(4) = 1 so that only
c                     advice to first time users is shown as a start up
c          15/11/1999 deleted obsolete calls and added code to warn
c                     about obsolete programs. Note that there is now only
c                     one manual = w_manual.ps.
c                     check_run6_dll was added to communicate version details
c                     without the need for common blocks
c          01/03/2000 Extensive revision for version 5.3
c          10/04/2000 Now calls dllchk using check_run6_dll and no longer
c                     calls w_help.exe to check DLLs
c          10/09/2000 enlarged calibration and viewing options
c          15/12/2000 introduced numrgb$ to make sure w_ps.cfg is initialised
c          18/07/2001 increased format for version from 3.1 and 4.2 to 4.2
c                     and 5.3. The code for checking dlls (54 lines) has been
c                     suppressed as it causes problems if users double click
c                     on the start icon as the first desk top shortcut is
c                     executing, thus causing a double instance and false
c                     warningfs about deleting the dlls.
c          03/09/2001 call to psspec$ from check_run6_dll to check PS specials
c          05/02/2002 added extra call to usermod from Menu 1 and cleaned up
c                     by defining Menu 13 as a sub-menu of Menu 12
c          19/02/2002 increased dimension of nval to nval(12) and added
c                     the clipboard viewer but deleted exit_help and moved
c                     configuration checking to config_wizard
c          21/05/2002 suppressed menu items to view manual, etc. using gsview
c          31/01/2003 added menu 14 and code for results files
c          20/08/2003 added call to w_chkcfg
c          09/11/2003 deleted icon_help, original front page, and added extra
c                     argument to the call to run6_logo
c          14/11/2003 extensive editing
c          12/12/2003 added call to call_xfonts
c          01/06/2004 changed format for version number
c          03/12/2004 release 5.06 ... added check for multiple instances
c          01/03/2005 changed text string from date to identifier
c          07/03/2005 added iver, nag, dllnag, mark, formats 100 and 200
c          03/06/2005 deleted i_edit_results, i_save_results, added
c                     i_view_recent call back for [recent] button
c          27/07/2005 new code to define version and increased dver to
c                     length = 30 this must be done everywhere in Simfit now
c          19/08/2005 submenus and callbacks for simstat_?
c          18/08/2006 edited all main menus and added faq call back
c          29/08/2006 added logical function already_running
c          20/10/2006 improved the operation of already_running
c          04/12/2006 introduced call to simdir and confined w_in.tmp and
c                     w_out.tmp to the simfit folder so fname*10 to fname*256
c                     and other changes for local versus simfit-folder files
c          08/01/2007 introduced call to linux3 and allowed for l2 = 0 from simdir
c          25/07/2007 revised for version 6
c          02/12/2007 added call to i_help_simdem
c          13/01/2009 added call to images
c          24/01/2010 deleted call to images 
c          03/04/2010 added i_run_simfit_programs
c          01/01/2011 added call to i_call_resdef
c          12/09/2013 added call use_url('http://www.simfit.org.uk') and speedup_pdf
c          03/04/2014 added call to driver_closer
c          30/08/2015 added call to switch_on_off
c          31/05/2016 edited for x64_simfit.exe (instead of w_simfit.exe)
c          20/06/2017 checked linux_os before calling driver_closer
c          23/10/2017 renamed url https://simfit.org.uk
c          26/09/2020 added call to i_display_tutorials
c          10/10/2022 added https://simfit.uk to logo and added conpdf and sigpdf to the menus
c
      implicit   none
c
c included subprograms
c      
      include   <windows.ins> 
c
c integer*2 variable
c       
      integer*2  error_code 
c
c integer variables
c      
      integer    nval(12) 
      integer    k, red, green, blue 
c
c integer parameters
c      
      integer    n0, n1, n125, n255
      parameter (n0 = 0, n1 = 1, n125 = 125, n255 = 255)
      integer    mode
      parameter (mode = 0)
c
c integer functions
c      
      integer    menu1_help, menu2_help, menu3_help, menu4_help,
     +           menu5_help, menu6_help, menu7_help, menu8_help,
     +           menu9_help, menu10_help
      integer    adderr, average, binomial, calcurve, chisqd, compare,
     +           csafit, deqsol, editfl, editmt, editps, eoqsol, exfit,
     +           ftest, gcfit, help, hlfit, inrate, linfit, makcsa,
     +           makdat, makfil, maklib, makmat, maksim, mmfit, normal,
     +           polnom, qnfit, rannum, rffit, rstest, sffit, simplot,
     +           simstat, spline, ttest, usermod
      integer    simstat_1, simstat_2, simstat_3, simstat_4, simstat_5,
     +           simstat_6, simstat_7, simstat_8, simstat_9, simstat_10
      integer    modules, release, technical
      integer    i_browse_res, i_browse_usr
      integer    acrobat, calculator, clipboard, configure, editor,
     +           explorer, faq, files, folders, gsview, information
      integer    ms_office_pdf, promote_pdf, pscodes_pdf
      integer    w_examples_pdf, w_manual_pdf
      integer    speedup_pdf, tutorials_pdf, configure_pdf, install_pdf  
      integer    call_xfonts
      integer    view_cfg, view_in, view_out,
     +           view_readme, view_text, view_tf, view_tfl, view_images
      integer    view_deq, view_plf, view_user
      integer    exit_delete, exit_save
      integer    item_jpg, item_pcx, item_bmp, item_tif, item_png,
     +           item_pdf, item_svg, item_xps, svg_item
      integer    i_help_results, i_view_recent, i_view_results
      integer    i_edit_results, i_print_results, i_save_results,
     +           i_table_results
      integer    i_check_second_instance_run6
      integer    i_help_simdem
      integer    pfiles
      integer    simfit_url
      integer    i_call_resdef, i_run_simfit_programs
      integer    switch_on_off
      integer    i_display_tutorials
      integer    conpdf, sigpdf, sumpdf
      integer    open_sv_simfit
c
c double precision variables
c      
      double precision correction, size1 
      double precision xver, yver  
c
c double precision parameters
c      
      double precision pcent, scale1
      parameter (pcent = 1.0d+02, scale1 = 1.0d+00)
c
c character variables
c      
      character  cval(12)*1024
      character  dver*30, pver*15, version*80
      character  reply*100  
c
c character parameters setting w_simfit_exe = x64_simfit_exe to avoid errors
c      
      character   w_simfit_exe*14, x64_simfit_exe*14, star*1 
      parameter (w_simfit_exe = 'x64_simfit.exe', 
     +           x64_simfit_exe = 'x64_simfit.exe',  
     +           star = '*' )   
c
c logical variables
c     
      logical    linux_os, store  
c
c logical functions
c      
      logical    linux3
c
c externals
c      
      external   menu1_help, menu2_help, menu3_help, menu4_help,
     +           menu5_help, menu6_help, menu7_help, menu8_help,
     +           menu9_help, menu10_help
      external   adderr, average, binomial, calcurve, chisqd, compare,
     +           csafit, deqsol, editfl, editmt, editps, eoqsol, exfit,
     +           ftest, gcfit, help, hlfit, inrate, linfit, makcsa,
     +           makdat, makfil, maklib, makmat, maksim, mmfit, normal,
     +           polnom, qnfit, rannum, rffit, rstest, sffit, simplot,
     +           simstat, spline, ttest, usermod
      external   simstat_1, simstat_2, simstat_3, simstat_4, simstat_5,
     +           simstat_6, simstat_7, simstat_8, simstat_9, simstat_10
      external   modules, release, technical
      external   acrobat, calculator, clipboard, configure, editor,
     +           explorer, faq, files, folders, gsview, information
      external   view_cfg, view_in, view_out,
     +           view_readme, view_text, view_tf, view_tfl, view_images
      external   i_browse_res, i_browse_usr
      external   view_deq, view_plf, view_user
      external   ms_office_pdf, promote_pdf, pscodes_pdf
      external   w_examples_pdf, w_manual_pdf
      external   speedup_pdf, tutorials_pdf, configure_pdf, install_pdf 
      external   call_xfonts
      external   exit_delete, exit_save
      external   run6_logo
      external   item_jpg, item_pcx, item_bmp, item_tif, item_png,
     +           item_pdf, item_svg, item_xps, svg_item
      external   w_config, mssage, w_syspar
      external   check_run6_dll, config_wizard
      external   i_help_results, i_view_recent, i_view_results
      external   i_edit_results, i_print_results, i_save_results,
     +           i_table_results
      external   linux3, create_inout_tmp_files
      external   exit@, i_check_second_instance_run6, putadv, simver
      external   i_help_simdem
      external   pfiles
      external   simfit_url
      external   i_call_resdef, i_run_simfit_programs
      external   switch_on_off
      external   i_display_tutorials
      external   conpdf, sigpdf, sumpdf
      external   open_sv_simfit
      c_external driver_closer 'DriverCloser' : integer
c
c intrinsics
c      
      intrinsic  dble, leng  
c 
c---------------------------------------------------------------------------------           
c The first check for multiple instances ... depends on formats %nc, %rm, and the
c call back i_check_second_instance_run6 associated with the main window.
c This will call exit@ only if the run6 window has already been created
c but will not do so if the second instance was started during the preliminary
c checks for missing programs, etc. It will work if the first main window has been
c minimised, so preventing a further round of checking for missing files, etc.
c----------------------------------------------------------------------------------
c
c
c Close down with noisy exit if run6 is already running
c

      k = send_text_message@(x64_simfit_exe, 'already there ?', reply)
      if (k.eq.1) then
         call exit@(error_code)
      endif
c      
c call linux3 to get the operating system but mainly to make sure
c that l_simfit.cfg exists before calling w_config and check before
c calling driver_closer
c                          
      linux_os = linux3(star)      
      if (.not.linux_os) k = driver_closer()
c     
c-----------------------------------------------------------------------------
c

c
c***********************************************************************
c begin checking salflibc.dll and the SIMFIT dlls 
c
      call simver (xver, yver,
     +             dver)     
      if (linux_os) then
         version = 'Linux V'//dver
      else 
         version = 'Version '//dver  
      endif     
      pver = w_simfit_exe
      store = .true.
      call check_run6_dll (xver, yver,
     +                     dver, pver,
     +                     store)
     
c
c end checking salflibc.dll and the SIMFIT dlls
c***********************************************************************
c
     
      call create_inout_tmp_files
c
c start up run6 by first calling w_config and defining local directories
c

      call w_config (mode, nval,
     +               cval)
      if (linux_os) nval(1) = 0

c
c nval(3) ...logo if required
c =======
c

      if (nval(3).eq.n1) then
c
c Blue background to display logo on front page
c
         red = n0
         green = n0
         blue = n255
      else
c
c Grey background for less intrusive front page
c
         red = n125
         green = n125
         blue = n125
      endif
      k = use_rgb_colours@ (n1, n0)

c
c nval(4) ...  start-up message if required
c =======
c

      if (nval(4).eq.n1) call mssage (n1)


c
c call the configuration wizard for remaining configuration
c

      k = n0
      call config_wizard (k, nval,
     +                    cval)

c 
c---------------------------------------------------------------------------------          
c The second check for multiple instances ... depends on formats %nc, %rm, and the
c call back i_check_second_instance_run6 associated with the main window
c This will call exit@ if the further instances were initiated during the
c phase of checking for missing programs, as by this stage the first main run6
c window will already have been created. So it prevents more than one copy
c of the run6 window being created (I think...)
c

      k = send_text_message@(x64_simfit_exe, 'already there ?', reply)
      if (k.eq.1) then
         call putadv (reply)
         call exit@(error_code)
      endif                                 
                          
c-----------------------------------------------------------------------------------
c      

c
c***********************************************************************
c set up the window and create the main menu
c***********************************************************************
c
                            
      call use_windows95_font@()
      call w_syspar (k, 'f')
      correction = dble(k)/pcent
      size1 = scale1*correction
      k = winio@('%`sf%tc[black]%ts&', size1)
      k = winio@('%mi[icon_x64_simfit]&')
      k = winio@('%nc[x64_simfit.exe]&')
      k = winio@('%rm&', i_check_second_instance_run6)
      if (nval(1).eq.n1) then
         k = winio@('%ww[maximise,no_border,independent]&')
      else
         k = winio@('%ww[no_border,independent]&')
      endif
      k = winio@('%bg&', rgb@(red, green, blue))
      k = winio@('%ca[x64_simfit.exe]&')
      k = winio@('%ff&')

c
c***********************************************************************
c now leave the backgound window open and proceed to the main program
c***********************************************************************
c

c
c=======================================================================
c MENU 1
c=======================================================================
c

      k = winio@(
     +'%mn[&File[&Help about: creating Simfit data filesPrograms,|]]&',
     +menu1_help)
      k = winio@(
     +'%mn[[Make a curve-fitting/plotting filemakfil]]&',
     +makfil)
      k = winio@(
     +'%mn[[Make a vector/matrix filemakmat,|]]&',
     +makmat)
      k = winio@(
     +'%mn[[Make/plot graph plotting filessimplot]]&',
     +simplot)
      k = winio@(
     +'%mn[[Make a library archive filemaklib]]&',
     +maklib)
      k = winio@(
     +'%mn[[Make a mathematical model fileusermod]]&',
     +usermod)
      k = winio@(
     +'%mn[[Make a file from Spreadsheet/clipboardmaksim]]&',
     +maksim)
      k = winio@(
     +'%mn[[View Simfit file formatssimfit,|]]&',
     +view_text)
      k = winio@(
     +'%mn[[Close SimfitExit]]&',
     +'exit')

c
c=======================================================================
c MENU 2
c=======================================================================
c

      k = winio@(
     +'%mn[&Edit[&Help about: editing Simfit data filesPrograms,|]]&',
     +menu2_help)
      k = winio@(
     +'%mn[[Curve fitting fileseditfl]]&',
     +editfl)
      k = winio@(
     +'%mn[[Vector/matrix fileseditmt,|]]&',
     +editmt)
      k = winio@(
     +'%mn[[Project archive listingssimfit,|]]&',
     +pfiles)
      k = winio@(
     +'%mn[[Simfit eps filesEditPS]]&',
     +editps)
      k = winio@(
     +'%mn[[eps into pdfGhostscript]]&',
     +item_pdf)
      k = winio@(
     +'%mn[[eps into pngGhostscript]]&',
     +item_png)
      k = winio@(
     +'%mn[[eps into jpegGhostscript]]&',
     +item_jpg)     
      k = winio@(
     +'%mn[[eps into xpsGhostscript]]&',
     +item_xps)       
      k = winio@(
     +'%mn[[eps into pcxGhostscript]]&',
     +item_pcx)
      k = winio@(
     +'%mn[[eps into bmpGhostscript]]&',
     +item_bmp)
      k = winio@(
     +'%mn[[eps into tiffGhostscript]]&',
     +item_tif)
      k = winio@(
     +'%mn[[eps into svgGhostscript]]&',
     +item_svg)
      k = winio@(
     +'%mn[[Simfit svg filesEditSVG]]&',
     +svg_item)




c
c=======================================================================
c MENU 3
c=======================================================================
c

      k = winio@(
     +'%mn[&View[&Help about: viewing filesTypes and Programs,|]]&',
     +menu3_help)
      k = winio@(
     +'%mn[[Readme filesw_readme.*]]&',
     +view_readme)
      k = winio@(
     +'%mn[[Test files*.tf?]]&',
     +view_tf)
      k = winio@(
     +'%mn[[Library files*.tfl]]&',
     +view_tfl)
      k = winio@(
     +'%mn[[User defined modelsuser*.tf? and *.mod]]&',
     +view_user)
      k = winio@(
     +'%mn[[Differential equationsdeq*.tf?]]&',
     +view_deq)
      k = winio@(
     +'%mn[[Parameter limits files*.plf]]&',
     +view_plf)
      k = winio@(
     +'%mn[[Configuration files*.cfg]]&',
     +view_cfg)
      k = winio@(
     +'%mn[[Simfit file formatsAll types,|]]&',
     +view_text)
      k = winio@(
     +'%mn[[Files opened this sessionw_in.tmp]]&',
     +view_in)
      k = winio@(
     +'%mn[[Files saved this sessionw_out.tmp,|]]&',
     +view_out)
      k = winio@(
     +'%mn[[Simfit plotting styles]]&',
     +view_images)
      k = winio@(
     +'%mn[[Simfit font substitution maps,|]]&',
     +call_xfonts)        
      k = winio@(
     +'%mn[[EPS filesEPS viewer,|]]&',
     +gsview)
      k = winio@(
     +'%mn[[PDF filesPDF viewer]]&',
     +acrobat)
      k = winio@(
     +'%mn[[Simfit manualPDF viewer]]&',
     +w_manual_pdf)
      k = winio@(
     +'%mn[[Summary/CollagesPDF viewer]]&',
     +promote_pdf)
      k = winio@(
     +'%mn[[MS OfficePDF viewer]]&',
     +ms_office_pdf)
      k = winio@(
     +'%mn[[PS codesPDF viewer,|]]&',
     +pscodes_pdf)
      k = winio@(
     +'%mn[[Explore clipboardclipit,|]]&',
     +clipboard)
      k = winio@(
     +'%mn[[Explore user folder...Simfit\usr]]&',
     +i_browse_usr)


c
c=======================================================================
c MENU 4
c=======================================================================
c

      k = winio@(
     +'%mn[Fi&t[&Help about: curve fittingPrograms,|]]&',
     +menu4_help)
      k = winio@(
     +'%mn[[Linear: multilinear and GLM modelslinfit]]&',
     +linfit)
      k = winio@(
     +'%mn[[Line; quadratic; cubic;...; polynomialpolnom]]&',
     +polnom)
      k = winio@(
     +'%mn[[Splines: comprehensive optionsspline,|]]&',
     +spline)
      k = winio@(
     +'%mn[[Sum of 1 to n &exponential functionsexfit]]&',
     +exfit)
      k = winio@(
     +'%mn[[Sum of 1 to n &High/Low affinity siteshlfit]]&',
     +hlfit)
      k = winio@(
     +'%mn[[Sum of 1 to n Michaelis-Menten modelsmmfit]]&',
     +mmfit)
      k = winio@(
     +'%mn[[Sum of 1 to n Cooperative binding sitessffit]]&',
     +sffit)
      k = winio@(
     +'%mn[[Order n:n positive rational functionrffit]]&',
     +rffit)
      k = winio@(
     +'%mn[[A sequence of growth/survival curvesgcfit]]&',
     +gcfit)
      k = winio@(
     +'%mn[[Initial-rates/lag-times/asymptotesinrate,|]]&',
     +inrate)
      k = winio@(
     +'%mn[[Advanced: QN/library/user-defined modelqnfit]]&',
     +qnfit)
      k = winio@(
     +'%mn[[Advanced: n differential equationsdeqsol]]&',
     +deqsol)
      k = winio@(
     +'%mn[[Advanced: flow cytometry histogramscsafit]]&',
     +csafit)


c
c=======================================================================
c MENU 5
c=======================================================================
c

      k = winio@(
     +'%mn[&Calibrate[&Help about: calibrationPrograms,|]]&',
     +menu5_help)
      k = winio@(
     +'%mn[[Linear standard curve: y = mx + clinfit]]&',
     +linfit)
      k = winio@(
     +'%mn[[Quadratic or Cubic standard curvepolnom]]&',
     +polnom)
      k = winio@(
     +'%mn[[Fixed knot spline standard curvecalcurve]]&',
     +calcurve)
      k = winio@(
     +'%mn[[Arbitrary spline standard curvespline]]&',
     +spline)
      k = winio@(
     +'%mn[[Transformed polynomial standard curvesimstat,|]]&',
     +simstat)
      k = winio@(
     +'%mn[[EC50 dose-response curve: Michaelis-Mentenmmfit]]&',
     +mmfit)
      k = winio@(
     +'%mn[[EC50 dose-response-curve: cooperative bindingsffit]]&',
     +sffit)
      k = winio@(
     +'%mn[[EC50 dose-response curve: Hill equationinrate]]&',
     +inrate)
      k = winio@(
     +'%mn[[EC50 dose-response curve: growth modelsgcfit,|]]&',
     +gcfit)
      k = winio@(
     +'%mn[[IC50 inhibition curve: Michaelis-Mentenmmfit]]&',
     +mmfit)
      k = winio@(
     +'%mn[[IC50 inhibition curve: survival modelsgcfit,|]]&',
     +gcfit)
      k = winio@(
     +'%mn[[LD50 normal error distribution: survival modelsgcfit]]&',
     +gcfit)
      k = winio@(
     +'%mn[[LD50 binomial error distribution: GLM/probitlinfit,|]]&',
     +linfit)
      k = winio@(
     +'%mn[[t-half: exponential modelsexfit]]&',
     +exfit)
      k = winio@(
     +'%mn[[t-half: growth/survival-models; survival-timesgcfit,|]]&',
     +gcfit)
      k = winio@(
     +'%mn[[Advanced method: you choose modelqnfit]]&',
     +qnfit)


c
c=======================================================================
c MENU 6
c=======================================================================
c

      k = winio@(
     +'%mn[&Plot[&Help about: plottingPrograms]]&',
     +menu6_help)
      k = winio@(
     +'%mn[[Examples of plotting stylesimages,|]]&',
     +view_images)
      k = winio@(
     +'%mn[[Interactively or from filessimplot,|]]&',
     +simplot)
      k = winio@(
     +'%mn[[Select from librarymakdat]]&',
     +makdat)
      k = winio@(
     +'%mn[[User defined equationusermod,|]]&',
     +usermod)
      k = winio@(
     +'%mn[[PostScript overlays and collagesEditPS,|]]&',
     +editps)
      k = winio@(
     +'%mn[[eps into pdfGhostscript]]&',
     +item_pdf)
      k = winio@(
     +'%mn[[eps into pngGhostscript]]&',
     +item_png)
      k = winio@(
     +'%mn[[eps into jpegGhostscript]]&',
     +item_jpg)
      k = winio@(
     +'%mn[[eps into xpsGhostscript]]&',
     +item_xps)     
      k = winio@(
     +'%mn[[eps into pcxGhostscript]]&',
     +item_pcx)
      k = winio@(
     +'%mn[[eps into bmpGhostscript]]&',
     +item_bmp)
      k = winio@(
     +'%mn[[eps into tiffGhostscript]]&',
     +item_tif)
      k = winio@(
     +'%mn[[eps into svgGhostscript,|]]&',
     +item_svg)
      k = winio@(
     +'%mn[[SVG overlays and collagesEditSVG,|]]&',
     +svg_item)
      k = winio@(
     +'%mn[[Restore plotting defaults]]&',
     +i_call_resdef)
      k = winio@(
     +'%mn[[View Simfit character maps]]&',
     +call_xfonts) 
     
     



c
c=======================================================================
c MENU 7
c=======================================================================
c

      k = winio@(
     +'%mn[&Statistics[&Help about: statisticsPrograms,|]]&',
     +menu7_help)
      k = winio@(
     +'%mn[[Comprehensive statisticssimstat (all options)]]&',
     +simstat)
      k = winio@(
     +'%mn[[Data explorationsimstat (option 1)]]&',
     +simstat_1)
      k = winio@(
     +'%mn[[Standard statistical testssimstat (option 2)]]&',
     +simstat_2)
      k = winio@(
     +'%mn[[Analysis of variancesimstat (option 3)]]&',
     +simstat_3)
      k = winio@(
     +'%mn[[Analysis of proportionssimstat (option 4)]]&',
     +simstat_4)
      k = winio@(
     +'%mn[[Multivariate statisticssimstat (option 5)]]&',
     +simstat_5)
      k = winio@(
     +'%mn[[Regression and calibrationsimstat (option 6)]]&',
     +simstat_6)
      k = winio@(
     +'%mn[[Generalized linear modelssimstat (option 7)]]&',
     +simstat_7)
      k = winio@(
     +'%mn[[Data smoothing/Time series/Survival analysissimstat'//
     +' (option 8)]]&',
     +simstat_8)     
c      k = winio@(
c     +'%mn[[Smoothing/Time series/Survivalsimstat (option 8)]]&',
c     +simstat_8)
c     k = winio@(
c     +'%mn[[Time series and survival analysissimstat (option 8)]]&',
c     +simstat_8)     
c      k = winio@(
c     +'%mn[[Smoothing, Time series and Survivalsimstat (option 8)]]&',
c     +simstat_8)
     
      k = winio@(
     +'%mn[[Statistical calculationssimstat (option 9)]]&',
     +simstat_9)
      k = winio@(
     +'%mn[[Numerical analysissimstat (option 10),|]]&',
     +simstat_10)
      k = winio@(
     +'%mn[[Nonparametric testsrstest]]&',
     +rstest)
      k = winio@(
     +'%mn[[Normal distributionnormal]]&',
     +normal)
      k = winio@(
     +'%mn[[Binomial/Trinomial/Poisson distributionsbinomial]]&',
     +binomial)
      k = winio@(
     +'%mn[[Chi-square distributionchisqd]]&',
     +chisqd)
      k = winio@(
     +'%mn[[F distributionftest]]&',
     +ftest)
      k = winio@(
     +'%mn[[t distributionttest]]&',
     +ttest)
      k = winio@(
     +'%mn[[Random numbers/walks/matricesrannum]]&',
     +rannum)
      k = winio@(
     +'%mn[[Multilinear regressionlinfit]]&',
     +linfit)
      k = winio@(
     +'%mn[[Growth and survival curvesgcfit]]&',
     +gcfit)
      k = winio@(
     +'%mn[[Spline smoothingspline]]&',
     +spline)

c
c=======================================================================
c MENU 8
c=======================================================================
c

      k = winio@(
     +'%mn[&Area/Slope[&Help about: areas and slopesPrograms,|]]&',
     +menu8_help)
      k = winio@(
     +'%mn[[Initial-rates/lag-times/asymptotesinrate]]&',
     +inrate)
      k = winio@(
     +'%mn[[Compare-curves/Splines/Smooth-datacompare]]&',
     +compare)
      k = winio@(
     +'%mn[[Derivatives/Areas/Arc-length/curvaturespline]]&',
     +spline)
      k = winio@(
     +'%mn[[Average function value over a rangeaverage]]&',
     +average)
      k = winio@(
     +'%mn[[Estimation of maximum growth rategcfit,|]]&',
     +gcfit)
      k = winio@(
     +'%mn[[AUC (exponential models) exfit]]&',
     +exfit)
      k = winio@(
     +'%mn[[AUC (trapezoidal method) average]]&',
     +average)
      k = winio@(
     +'%mn[[AUC (you choose model)qnfit]]&',
     +qnfit)
      k = winio@(
     +'%mn[[AUC (you supply model)usermod]]&',
     +usermod)

c
c=======================================================================
c MENU 9
c=======================================================================
c

      k = winio@(
     +'%mn[Si&mulate[&Help about: simulationPrograms,|]]&',
     +menu9_help)
      k = winio@(
     +'%mn[[Make data from library of modelsmakdat]]&',
     +makdat)
      k = winio@(
     +'%mn[[Integrate differential equationsdeqsol]]&',
     +deqsol)
      k = winio@(
     +'%mn[[Add error to simulate experimentsadderr]]&',
     +adderr)
      k = winio@(
     +'%mn[[Generate random numbers and walksrannum]]&',
     +rannum)
      k = winio@(
     +'%mn[[Optimal design for data spacingeoqsol]]&',
     +eoqsol)
      k = winio@(
     +'%mn[[Simulate flow cytometry datamakcsa]]&',
     +makcsa)
      k = winio@(
     +'%mn[[User-model: test/zeros/integrateusermod]]&',
     +usermod)

c
c=======================================================================
c MENU 10
c=======================================================================
c

      k = winio@(
     +'%mn[Mo&dules[&Help about: modulesProcedures,|]]&',
     +menu10_help)
      k = winio@(
     +'%mn[[EditorEdit ASCII text files]]&',
     +editor)
      k = winio@(
     +'%mn[[ExplorerExplore computer]]&',
     +explorer)
      k = winio@(
     +'%mn[[PS viewerView/Print PS files]]&',
     +gsview)
      k = winio@(
     +'%mn[[PDF viewerView/print pdf files]]&',
     +acrobat)
      k = winio@(
     +'%mn[[CalculatorDo calculations]]&',
     +calculator)
      k = winio@(
     +'%mn[[ModulesUser defined programs]]&',
     +modules)

c
c=======================================================================
c MENU 11
c=======================================================================
c

      k = winio@(
     +'%mn[&Help[Help about: all subjects,|]]&',
     +help)
    
      k = winio@(
     +'%mn[[Important information]]&',
     +information)
      k = winio@(
     +'%mn[[Program files]]&',
     +files)
      k = winio@(
     +'%mn[[Current folders]]&',
     +folders)
      k = winio@(
     +'%mn[[Readme files]]&',
     +view_readme)
      k = winio@(
     +'%mn[[Technical details]]&',
     +technical)
      k = winio@(
     +'%mn[[Accents and symbols]]&',
     +call_xfonts)
      k = winio@(
     +'%mn[[Manual (pdf)]]&',
     +w_manual_pdf)
      k = winio@(
     +'%mn[[Examples (pdf)]]&',
     +w_examples_pdf)
      k = winio@(
     +'%mn[[Summary (pdf)]]&',
     +promote_pdf)
      k = winio@(
     +'%mn[[MS Office (pdf)]]&',
     +ms_office_pdf)
      k = winio@(
     +'%mn[[PS Codes (pdf)]]&',
     +pscodes_pdf)
      k = winio@(
     +'%mn[[Tutorials (pdf)]]&',
     +tutorials_pdf)
      k = winio@(
     +'%mn[[Configure (pdf)]]&',
     +configure_pdf)
      k = winio@(
     +'%mn[[Install (pdf),|]]&',
     +install_pdf)
     
      k = winio@(
     +'%mn[[The Simdem package,|]]&',
     +i_help_simdem) 

      k = winio@(
     +'%mn[[The Simfit website]]&',
     +simfit_url)
      
      k = winio@(
     +'%mn[[About this Version]]&',
     +release)
     

c
c=======================================================================
c MENU 12 !!!  The next menus must not be edited  !!!
c=======================================================================
c
       k = winio@('%mn[A/&Z]&', i_run_simfit_programs)
c      k = winio@('%mn[A/&Z[adderr]]&', adderr)
c      k = winio@('%mn[[average]]&', average)
c      k = winio@('%mn[[binomial]]&', binomial)
c      k = winio@('%mn[[calcurve]]&', calcurve)
c      k = winio@('%mn[[chisqd]]&', chisqd)
c      k = winio@('%mn[[compare]]&', compare)
c      k = winio@('%mn[[csafit]]&', csafit)
c      k = winio@('%mn[[deqsol]]&', deqsol)
c      k = winio@('%mn[[editfl]]&', editfl)
c      k = winio@('%mn[[editmt]]&', editmt)
c      k = winio@('%mn[[editps]]&', editps)
c      k = winio@('%mn[[eoqsol]]&', eoqsol)
c      k = winio@('%mn[[exfit]]&', exfit)
c      k = winio@('%mn[[ftest]]&', ftest)
c      k = winio@('%mn[[gcfit]]&', gcfit)
c      k = winio@('%mn[[help]]&', help)
c      k = winio@('%mn[[hlfit]]&', hlfit)
c      k = winio@('%mn[[inrate]]&', inrate)
c      k = winio@('%mn[[linfit]]&', linfit)

c
c=======================================================================
c MENU 13  !!!  The next menus must not be edited   !!!
c=======================================================================
c

c      k = winio@('%mn[[M/&Z[makcsa]]]&', makcsa)
c      k = winio@('%mn[[[makdat]]]&', makdat)
c      k = winio@('%mn[[[makfil]]]&', makfil)
c      k = winio@('%mn[[[maklib]]]&', maklib)
c      k = winio@('%mn[[[makmat]]]&', makmat)
c      k = winio@('%mn[[[maksim]]]&', maksim)
c      k = winio@('%mn[[[mmfit]]]&', mmfit)
c      k = winio@('%mn[[[normal]]]&', normal)
c      k = winio@('%mn[[[polnom]]]&', polnom)
c      k = winio@('%mn[[[qnfit]]]&', qnfit)
c      k = winio@('%mn[[[rannum]]]&', rannum)
c      k = winio@('%mn[[[rffit]]]&', rffit)
c      k = winio@('%mn[[[rstest]]]&', rstest)
c      k = winio@('%mn[[[sffit]]]&', sffit)
c      k = winio@('%mn[[[simplot]]]&', simplot)
c      k = winio@('%mn[[[simstat]]]&', simstat)
c      k = winio@('%mn[[[spline]]]&', spline)
c      k = winio@('%mn[[[ttest]]]&', ttest)
c      k = winio@('%mn[[[usermod]]]&', usermod)

c
c=======================================================================
c MENU 14
c=======================================================================
c

      k = winio@(
     +'%mn[&Results[Help about: results,|]]&',
     +i_help_results)
      k = winio@(
     +'%mn[[View results log files]]&',
     +i_view_results)
      k = winio@(
     +'%mn[[Print results log files]]&',
     +i_print_results)
      k = winio@(
     +'%mn[[Save results log files]]&',
     +i_save_results)
      k = winio@(
     +'%mn[[Edit results log files]]&',
     +i_edit_results)
      k = winio@(
     +'%mn[[Extract tables]]&',
     +i_table_results)
      k = winio@(
     +'%mn[[All results files]]&',
     +i_browse_res)
c
c=======================================================================
c MENU 15
c=======================================================================
c
      k = winio@(
     +'%mn[&Speedup[Help about: speedup,|]]&',
     +speedup_pdf)
      k = winio@(
     +'%mn[[Speedup options]]&',
     +switch_on_off)
     
      k = winio@(
     +'%mn[Significant_digits]&',
     +sigpdf)   
     
      k = winio@(
     +'%mn[Contact]&',
     +conpdf)  
c
c Show the button for sv_simfit
c =============================
c
      call select_font@('Microsoft Sans Serif Regular')
      k = winio@('%ts&',1.15d+00)
      k = winio@('%rp&', 4, 1)
      k = winio@('%tc[white]&')
      k = winio@('sv_simfit is a simplified version%nl&')
      k = winio@('%rp&', 4, 0)
      k = winio@('of simfit for inexperienced users.%nl&')
      k = winio@('%rp&', 4, 1)
      k = winio@('%^18bt[ Open sv_simfit ]&', open_sv_simfit)
      k = winio@('%tc[white]&')
      k = winio@('%sf&')
c
c Show logo on front page ... colour will be blue or grey depending on
c =======================     the configuration option chosen for nval(3)
c
      k = winio@('%cn&')
      k = nval(1)
      call run6_logo (k,
     +                version)
c
c The status bar
c ==============
c
      size1 = scale1*correction
      k = winio@('%ob[status, thin_panelled]&')
      k = winio@('%`sf%ts&', size1)
      k = winio@('%`10^bt[Summary]&', sumpdf)
      k = winio@('%10^bt[Tutorials]&', i_display_tutorials)
      k = winio@('%10^bt[Examples]&', w_examples_pdf)
      k = winio@('%10^bt[Manual]&', w_manual_pdf)
      k = winio@('%10^bt[Configure]&', configure)
      k = winio@('%10^bt[FAQ]&', faq)
      k = winio@('%10^bt[Recent]&', i_view_recent)
      k = winio@('%10^bt[Editor]&', editor)
      k = winio@('%10^bt[Explorer]&', explorer)
      k = winio@('%10^bt[Calculator]&', calculator)
      k = winio@('%cb')
c
c
c**********************************************************************
c Final code since exit may be via the window close cross
c**********************************************************************
c

      if (nval(2).eq.1) then
         k = exit_delete()
      else
         call putadv (
     +'Simfit is configured to save temporary files like f$123456.tmp')         
         k = exit_save()
      endif  
      end

c 
c----------------------------------------------------------------------
c
      recursive integer function i_check_second_instance_run6()
c
c call back function used to check for second instances
c it flags attempts to create second instances of the main run6 window
c but it does rely on Clearwin+ having already created the main window
c
      implicit  none
      include  <windows.ins>
      character reply*100
c      reply = clearwin_string@('MESSAGE_TEXT')
      reply = 'x64_simfit.exe is already executing'
      call reply_to_text_message@(reply)
      i_check_second_instance_run6 = 0
      end
c
c----------------------------------------------------------------------
c
      recursive integer function i_help_simdem()
      implicit none
      external help_simdem
      call help_simdem('simdem')
      i_help_simdem = 1
      end
c
c-----------------------------------------------------------------------
c
      recursive integer function switch_on_off()
      implicit none
      integer  n0
      parameter (n0 = 0)
      external switch
      call switch (n0)
      switch_on_off = 1
      end
c
c---------------------------------------------------------------------------
c
      recursive integer function query_sv_simfit()
      implicit none
      external run_sv_simfit
      call run_sv_simfit
      query_sv_simfit = 1
      end
c
c-------------------------------------------------------------------------------
c
      recursive integer function open_sv_simfit()
      implicit none
      integer ifail
      character (len = 1024) fname, sim256
      character (len = 100 ) line
      external run_program, sim256, x_putadv 
      fname = sim256 ('sv_simfit.exe')
      call run_program (ifail,
     +                  fname, ' ')     
      if (ifail.ne.0) then
        write (line,100) ifail
        call x_putadv (line)
      endif 
      open_sv_simfit = 1 
  100 format ('ERROR: from call to sv_simfit IFAIL =',i3) 
      end   
c
c-------------------------------------------------------------------------------
c

      
      subroutine run_sv_simfit
      
c derived from simdem19: answer, display text/yesno
c ================================================= 
c For details read simdem.txt or simdem.html 
c
c subroutine
c ----------
c answer ... display text and a summary question
c
c Meaning of the arguments
c ========================
c icolor: intent (in) 1 = blue, 4 = red, 9 = white, o/w grey
c numbld: intent (in) 1, 2 = Normal; 3,4 = Italic; 4,5 = Bold; 6,7 = Bold Italic
c                     odd number = normal, even number = highlighted
c numhdr: intent (in) number of header lines
c header: intent (in) header text
c option: intent (in) that is the question
c yesno : intent (inout) logical selected (input value sets the default)
c
c Note that a grave accent can be used for tabbing
c
c

      implicit   none
      integer    i, icolor, ifail, numhdr
      parameter (icolor = 7)
      integer    numbld(20)
      character (len = 80  ) header(20), line, option
      character (len = 1024) fname, sim256 
      logical    yesno
      external   sv_answer, run_program, x_putadv
      external   sim256
      do i = 1, 20
         numbld(i) = 0
         header(i) = ' '
      enddo
c
c Typical answer dialogue
c
      write (header,100)
      option = 'Open sv_simfit'
      numhdr = 12
      numbld(1) = 1
      yesno = .true.
      call sv_answer (icolor, numbld, numhdr, header, option, yesno)
      if (yesno) then
         fname = sim256 ('sv_simfit.exe')
         call run_program (ifail,
     +                     fname, ' ')     
         if (ifail.ne.0) then
           write (line,200) ifail
           call x_putadv (line)
         endif  
      endif
      
  100 format (
     + 'Using sv_simfit, the simplified version of Simfit.'
     +/
     +/'The Simfit package provides most of the functionality required'
     +/'for standard data analysis. However it also supports a number'
     +/'of advanced features, e.g. simulating and fitting user-supplied'
     +/'systems of nonlinear differential equations by constrained non-'
     +/'linear optimisation that would only be required by experienced'
     +/'analysts.'
     +/ 
     +/'For that reason a simplified version of Simfit is provided that'
     +/'first-time and inexperienced users can practise with before'
     +/'attempting to use the main Simfit version.')
  200 format ('ERROR: from call to sv_simfit IFAIL =',i3)    
      end
c
c
      subroutine sv_answer (icolor, numbld, numhdr,
     +                      header_in, option_in,
     +                      yes)
c
c action : put out a yes/no question onto a window
c author : w.g.bardsley, university of manchester, u.k., 26/12/96
c          10/02/1997 added tabbing
c          04/04/1997 revised to call w_syspar
c          09/09/1998 removed topmost and made into a normal dialogue window
c          14/11/1998 re-introduced Roman/Courier for button
c          14/04/1999 restored ms sans serif at size courier and supressed
c                     caption cross to avoid ambiguous exits
c          03/09/1999 re-introduced %ww[no_sysmenu]
c          04/12/1999 restored topmost
c          08/01/2001 suppressed %ww to create a dialogue type window
c          12/02/2002 muliplied tab by correction factor and `sf for XP 
c          18/12/2002 added %sy[toolwindow, no_sysmenu]
c          17/11/2006 suppressed toolwindow and added intents 
c          31/01/2007 revised for w_clearwin.dll (x_len200)
c          29/05/2007 added allocatable header, roman, and calls to w_dbleup 
c          03/04/2009 added extra lines to free-up the icon  
c          25/04/2011 added numbld(i) = 4 for bold font and 2 for italic
c          02/08/2017 added Stop and call back function i_stop_this_prpgram
c          
c          icolor: (input/unchanged) colour style
c          numbld: (input/unchanged) text style
c          numhdr: (input/unchanged) no. of header lines
c       header_in: (input/unchanged) header text
c       option_in: (input/unchanged) the 1-line option
c             yes: (input/output) sets the default button on input
c                                 then returns the option selected
c  
c          set font style parameters as follows:
c          fixed ... gives Courier o/w roman
c          high  ... highlights the final question
c
c          icolor background   text     highlight-text (in question line)
c          ====== ==========   ====     ==============
c          0      black        grey     yellow
c          1      blue         grey     yellow
c          2      green        grey     blue
c          3      cyan         grey     blue
c          4      red          grey     yellow
c          9      white        black    red
c          o/w    grey         black    blue
c
c          text size is set by the parameter size_courier or size_roman 
c          this version uses: x_len200 = leng@ = len_trim
c          to be compatible with ftn77 and ftn90
c
      implicit   none
      include   <windows.ins>   
c
c arguments
c      
      integer,             intent (in)    :: numhdr
      integer,             intent (in)    :: icolor, numbld(numhdr)
      character (len = *), intent (in)    :: header_in(numhdr),
     +                                       option_in
      logical,             intent (inout) :: yes
c
c local allocatable array
c                        
      character (len = 129), allocatable :: header(:)
c
c locals
c      
      integer    isend
      parameter (isend = 1)
      integer    i, ierr, j, k, l, x_len200
      integer    n0, n1, n4, n7
      parameter (n0 = 0, n1 = 1, n4 = 4, n7 = 7)
      integer    i_simfit_answer_no, i_simfit_answer_yes
      integer    nmax
      parameter (nmax = 20)
      double precision size_roman, size_courier
      double precision size_roman_1, size_courier_1
      parameter (size_roman_1 = 1.15d+00, size_courier_1 = 1.0d+00)
      double precision correction, tab
      double precision factor, percent
      parameter (factor = 1.0d+00, percent = 100.0d+00)
      character  line*129, option*129, w_dbleup*129
      character  c*1, blank*1, space*3
      parameter (c = 'f', blank = ' ', space = '   ')
      logical    fixed, high
      logical    yes_or_no   
      logical    roman, roman_1
      parameter (roman_1 = .true.)
      external   x_len200
      external   i_simfit_answer_no, i_simfit_answer_yes
      external   w_syspar, w_dbleup
      external   add_stop_option
      intrinsic  dble, index
      common / simfit_answer / yes_or_no  
c
c check numhdr then allocate and copy the character arguments
c                          
      if (numhdr.lt.n1) return
      ierr = n0
      if (allocated(header)) deallocate(header, stat = ierr)
      if (ierr.ne.n0) return
      allocate (header(numhdr), stat = ierr)
      if (ierr.ne.n0) return
      option = w_dbleup(option_in)
      do i = n1, numhdr
         header(i) = w_dbleup(header_in(i)) 
      enddo
c
c Decide if Times Roman font is to be used
c 
      roman = roman_1     
c    
c Scale the font sizes
c
      fixed = .false.
      high = .true.
      call use_windows95_font@()
      call w_syspar (i, c)
      correction = dble(i)/percent
      size_courier = correction*size_courier_1
      if (roman) then
         size_roman = correction*size_roman_1
      else
         size_roman = size_courier
      endif       
c
c the general case ... leave the caption in
c
      yes_or_no = yes 
      i = winio@('%sy[no_sysmenu]&')
      i = winio@('%ca[Simfit: decision]&')
c
c restore the next line for a %ww type window instead of a dialogue window
c*****i = winio@('%ww[topmost]&')
c
      if (numhdr.le.20) call add_stop_option (isend)
c
c now set the font and background and text colours depending on icolor
c
      if (fixed) then
         i = winio@('%fn[Courier New]&')
      else        
         if (roman) then
            i = winio@('%fn[Arial]&')
            i = winio@('%ts&', size_roman) 
         else   
            i = winio@('%`sf&')
         endif   
      endif
c
c choose colours
c
      if (icolor.eq.0) then
         k = winio@('%bg[black]&')
      elseif (icolor.eq.1) then
         k = winio@('%bg[blue]&')
      elseif (icolor.eq.2) then
         k = winio@('%bg[green]&')   
      elseif (icolor.eq.3) then
         k = winio@('%bg&', rgb@(0,167,167))      
      elseif (icolor.eq.4) then
         k = winio@('%bg[red]&')
      elseif (icolor.eq.9) then
         k = winio@('%bg[white]&')
      else
         k = winio@('%bg&', rgb@(240,240,240))
      endif
c
c output the header
c
      if (numhdr.gt.n0) then
         do i = n1, numhdr
            if (i.eq.1) then
               k = winio@('%fn[Arial]&')
               k = winio@('%ts&', size_roman) 
            else  
               k = winio@('%sf&')
            endif   
            if (numbld(i).eq.n0) then
               if (icolor.eq.0) then
                  k = winio@('%tc[grey]&')
               elseif (icolor.eq.1) then
                  k = winio@('%tc[grey]&')
               elseif (icolor.eq.2) then
                  k = winio@('%tc[black]&')   
               elseif (icolor.eq.4) then
                  k = winio@('%tc[grey]&')
               else
                  k = winio@('%tc[black]&')
               endif
            elseif (numbld(i).eq.2) then
               k = winio@('%it&')  
               if (icolor.eq.0) then
                  k = winio@('%tc[grey]&')
               elseif (icolor.eq.1) then
                  k = winio@('%tc[grey]&')
               elseif (icolor.eq.2) then
                  k = winio@('%tc[black]&')   
               elseif (icolor.eq.4) then
                  k = winio@('%tc[grey]&')
               else
                  k = winio@('%tc[black]&')
               endif
            elseif (numbld(i).eq.4) then    
               k = winio@('%bf&')
               if (icolor.eq.0) then
                  k = winio@('%tc[white]&')
               elseif (icolor.eq.1) then
                  k = winio@('%tc[white]&')
               elseif (icolor.eq.2) then
                  k = winio@('%tc[black]&')   
               elseif (icolor.eq.4) then
                  k = winio@('%tc[white]&')
               else
                  k = winio@('%tc[black]&')
               endif
            else   
               if (icolor.eq.0) then
                  k = winio@('%tc[yellow]&')
               elseif (icolor.eq.1) then
                  k = winio@('%tc[yellow]&')
               elseif (icolor.eq.2) then
                  k = winio@('%tc[blue]&')   
               elseif (icolor.eq.4) then
                  k = winio@('%tc[yellow]&')
               elseif (icolor.eq.9) then
                  k = winio@('%tc[red]&')
               else
                  k = winio@('%tc[blue]&')
               endif
            endif
            if (fixed) then
               if (i.eq.numhdr) then
                  line =
     +            '%ts'//header(i)(n1:x_len200(header(i)))//'&'
               else
                  line = 
     +            '%ts'//header(i)(n1:x_len200(header(i)))//'%nl&'
               endif
               k = winio@(line, size_courier)
            else    
               k = winio@('%ts&', size_roman)
               l = x_len200(header(i))
               line = blank
               j = index(header(i), '`')
               if (j.ge.n1) then
                  line = header(i)(n1:j - n1)//'&'
                  k = winio@(line(n1:j))
                  tab = correction*factor*dble(j)
                  k = winio@('%`1tl&', tab)
                  line = blank
                  if (i.eq.numhdr) then
                     line = '%ta'//header(i)(j + n1:l)//'&'
                     k = winio@(line(n1:l - j + n4))
                  else
                     line = '%ta'//header(i)(j + n1:l)//'%nl&'
                     k = winio@(line(n1:l - j + n7))
                  endif
               else
                  if (i.eq.numhdr) then
                     line = header(i)(n1:l)//'&'
                     k = winio@(line(n1:l + n1))
                  else
                     line = header(i)(n1:l)//'%nl&'
                     k = winio@(line(n1:l + n4))
                  endif
               endif
            endif
         enddo
         k = winio@('%ff&')
         k = winio@('%nl   &')
      endif
c
c now the final question highlighted if high = .true.
c
      if (fixed) then
         i = winio@('%fn[Courier New]&')
         i = winio@('%ts&', size_courier)
      else   
         if (roman) then
            i = winio@('%fn[Arial]&')
            i = winio@('%ts&', size_roman) 
         else   
            i = winio@('%`sf&')         
            i = winio@('%ts&', size_courier)
         endif   
      endif
      if (high) then
         if (icolor.eq.0) then
            k = winio@('%tc[yellow]&')
         elseif (icolor.eq.1) then
            k = winio@('%tc[yellow]&')
         elseif (icolor.eq.4) then
            k = winio@('%tc[yellow]&')
        elseif (icolor.eq.9) then
            k = winio@('%tc[red]&')
         else
            k = winio@('%tc[blue]&')
         endif
      endif

      k = winio@('%nl&')
      
      line = '%si?'//option(n1:x_len200(option))//'&'
      k = winio@(line)
      k = winio@('%`sf&')
      k = winio@('%tc[black]%ts&', size_courier)
      if (yes) then
         line = space//'%6`^bt[&Yes]'//space//'%6^bt[&No]&'
      else
         line = space//'%6^bt[&Yes]'//space//'%6`^bt[&No]&'
      endif
      if (numhdr.le.nmax) k = winio@('%ff%nl&')
      k = winio@(line, i_simfit_answer_yes, i_simfit_answer_no)
      
      k = winio@('%ff&')
c      k = winio@('%nl  &')
      k = winio@('%nl  ')
      
      yes = yes_or_no
c
c deallocate
c           
      deallocate(header, stat = ierr)
      end
c
c
      recursive integer function i_simfit_answer_no()
      implicit none
      logical  yes_or_no
      common / simfit_answer / yes_or_no
      yes_or_no = .false.
      i_simfit_answer_no = 0
      end
c
c
      recursive integer function i_simfit_answer_yes()
      implicit none
      logical  yes_or_no
      common / simfit_answer / yes_or_no
      yes_or_no = .true.
      i_simfit_answer_yes = 0
      end
c
c
c

        